/* debug.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "make.h"
#include "instboot.h"
#include "library.h"
#include "boot.h"
#include "debug.h"

#if DEBUG & DEBUG_STACK
#define DBG_MSG_STACK "STACK, "
#else
#define DBG_MSG_STACK ""
#endif

#if DEBUG & DEBUG_ADDR
#define DBG_MSG_ADDR "ADDR, "
#else
#define DBG_MSG_ADDR ""
#endif

#if DEBUG & DEBUG_DISK
#define DBG_MSG_DISK "DISK, "
#else
#define DBG_MSG_DISK ""
#endif

#if DEBUG & DEBUG_VIDEO
#define DBG_MSG_VIDEO "VIDEO, "
#else
#define DBG_MSG_VIDEO ""
#endif

#if DEBUG & DEBUG_SERIAL
#define DBG_MSG_SERIAL "SERIAL, "
#else
#define DBG_MSG_SERIAL ""
#endif

#if DEBUG & DEBUG_FS
#define DBG_MSG_FS "FS, "
#else
#define DBG_MSG_FS ""
#endif

#if DEBUG & DEBUG_MOUSE
#define DBG_MSG_MOUSE "MOUSE, "
#else
#define DBG_MSG_MOUSE ""
#endif

#if DEBUG & DEBUG_LOADER
#define DBG_MSG_LOADER "LOADER, "
#else
#define DBG_MSG_LOADER ""
#endif

#define DBG_MSG DBG_MSG_STACK DBG_MSG_ADDR \
		DBG_MSG_DISK DBG_MSG_VIDEO DBG_MSG_SERIAL \
		DBG_MSG_FS DBG_MSG_MOUSE DBG_MSG_LOADER

#if ((DEBUG & DEBUG_OUTPUT) == COM1) || ((DEBUG & DEBUG_OUTPUT) == COM2) \
    || ((DEBUG & DEBUG_OUTPUT) == COM3) || ((DEBUG & DEBUG_OUTPUT) == COM4)

#if (DEBUG & DEBUG_OUTPUT) > 4
#error That will not work
#endif

static const serialconf_t dbg_sconf = {
    1, 1, 0, 0, 0, B9600, 0,	/* COMx,n,8,1 */
    };

unsigned short dbg_sport = (DEBUG & DEBUG_OUTPUT) - 1;	/* 0..3 */

__attribute__ ((noinline)) static void dbg_puts (const char *str)
  {
  unsigned short dummy1, dummy2;

  asm volatile (
"dbg_puts_loop:				\n"
"	lodsb	%%ds:(%%si),%%al	\n"
"	cmp	$0,%%al			\n"
"	je	dbg_puts_end		\n"
"	mov	$0x01,%%ah		\n"
"	int	$0x14			\n"
"	jmp	dbg_puts_loop		\n"
"dbg_puts_end:				\n"
	: "=S" (dummy1), "=a" (dummy2)
	: "S" (str), "d" (dbg_sport)
	);
  }

void dbg_init (void)
  {
  struct {
      unsigned char	timeout		: 1;
      unsigned char	TXregempty	: 1;
      unsigned char	buffempty	: 1;
      unsigned char	ErrBreak	: 1;
      unsigned char	ErrFramming	: 1;
      unsigned char	ErrParity	: 1;
      unsigned char	ErrOverflow	: 1;
      unsigned char	dataready	: 1;
      unsigned char	gap		: 8;
      } __attribute__ ((packed)) status;

  asm volatile (
"	int	$0x14					\n"
"	mov	$0x0120,%%ax	# print a space		\n"
"	int	$0x14					\n"
	: "=a" (status)
	: "d" (dbg_sport), "a" (dbg_sconf)
	);

#if   ((DEBUG & DEBUG_OUTPUT) == COM1)
  BOOT1_putstr ("debug " DBG_MSG "on COM1.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == COM2)
  BOOT1_putstr ("debug " DBG_MSG "on COM2.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == COM3)
  BOOT1_putstr ("debug " DBG_MSG "on COM3.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == COM4)
  BOOT1_putstr ("debug " DBG_MSG "on COM4.\r\n");
#endif

  if (status.timeout)
      BOOT1_putstr ("DEBUG: serial timeout !\r\n");

  dbg_puts ("       \r\nDebug active\r\n");
  }

void dbg_end (void)
  {
  dbg_puts ("\r\nDebug end.\r\n");
  BOOT1_putstr ("\r\n\r\nDebug end.\r\n");
  }

#elif ((DEBUG & DEBUG_OUTPUT) == LPT1) || ((DEBUG & DEBUG_OUTPUT) == LPT2) \
      || ((DEBUG & DEBUG_OUTPUT) == LPT3) || ((DEBUG & DEBUG_OUTPUT) == LPT4)

#if (DEBUG & 0xFC) != 0x80
#error That will not work
#endif

__attribute__ ((noinline)) static void dbg_puts (const char *str)
  {
  unsigned short dummy1, dummy2, dummy3;

  asm volatile (
"dbg_puts_loop:				\n"
"	lodsb   %%ds:(%%si),%%al	\n"
"	cmp     $0,%%al			\n"
"	je      dbg_puts_end		\n"
"	mov	%3,%%dx			\n"
"	xor     %%ah,%%ah		\n"
"	int     $0x17			\n"
"	jmp     dbg_puts_loop		\n"
"dbg_puts_end:				\n"
	: "=S" (dummy1), "=a" (dummy2), "=d" (dummy3)
	: "g" ((DEBUG & 0xFF) - 0x80), "S" (str)
	);
  }

void dbg_init (void)
  {
  struct {
      unsigned char	notbusy		: 1;
      unsigned char	acknowledge	: 1;
      unsigned char	outofpaper	: 1;
      unsigned char	selected	: 1;
      unsigned char	IOerror		: 1;
      unsigned char	unused		: 2;
      unsigned char	timeout		: 1;
      unsigned char	gap		: 8;
      } __attribute__ ((packed)) status;

  asm volatile (
"	int	$0x17					\n"
"	mov	$0x0020,%%ax	# print a space		\n"
"	int	$0x17					\n"
	: "=a" (status)
	: "d" ((DEBUG & 0xFF) - 0x80), "a" (0x0100)
	);

#if   ((DEBUG & DEBUG_OUTPUT) == LPT1)
  BOOT1_putstr ("debug " DBG_MSG "on LPT1.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == LPT2)
  BOOT1_putstr ("debug " DBG_MSG "on LPT2.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == LPT3)
  BOOT1_putstr ("debug " DBG_MSG "on LPT3.\r\n");
#elif ((DEBUG & DEBUG_OUTPUT) == LPT4)
  BOOT1_putstr ("debug " DBG_MSG "on LPT4.\r\n");
#endif

  if (status.timeout)
      BOOT1_putstr ("DEBUG: printer timeout !\r\n");
  if (status.IOerror)
      BOOT1_putstr ("DEBUG: printer I/O error !\r\n");

  dbg_puts ("       \r\nDebug active\r\n");
  }

void dbg_end (void)
  {
  dbg_puts ("\r\nDebug end.\r\n");
  BOOT1_putstr ("\r\nDebug end.\r\n");
  }

#elif ((DEBUG & DEBUG_OUTPUT) == SCREEN)
#include "bios.h"
__attribute__ ((noinline)) static void dbg_puts (const char *str)
  {
  unsigned short dummy1, dummy2;

  asm volatile (
"dbg_puts_loop:								\n"
"	lodsb	%%ds:(%%si),%%al					\n"
"	cmp	$0,%%al							\n"
"	je	dbg_puts_end						\n"
"	pushw	%%ds							\n"
"	pusha								\n"
"	mov	$0x07,%%bx	# only for int 0x10, display page 0, color 7.\n"
"	mov	$0x0E,%%ah						\n"
"	int	$0x10		# int 0x10 may destroy %%bp if scroll...\n"
"	popa	# take care of popa bug on i386				\n"
"	popw	%%ds							\n"
"	jmp	dbg_puts_loop						\n"
"dbg_puts_end:								\n"
	: "=S" (dummy1), "=a" (dummy2)
	: "S" (str)
	);
  while (_BIOS_get_shift_flags().control_key_pressed)
      continue;
  if (_BIOS_get_shift_flags().alternate_key_pressed)
      _BIOS_wait (100000);
  }

/* ... Sirius Cybernetics Corporation products ...
   It is very easy to be blinded to the essential uselessness of them
   by the sense of achievement you get from getting them to work at all */

void dbg_init (void)
  {
  BOOT1_putstr ("debug " DBG_MSG "on screen.\r\n");
  dbg_puts ("       \r\nDebug active\r\n");
  }

void dbg_end (void)
  {
  dbg_puts ("\r\nDebug end.\r\n");
  BOOT1_putstr ("\r\nDebug end.\r\n");
  }

#elif ((DEBUG & DEBUG_OUTPUT) == DOSFILE)
#include "dos.h"

static unsigned short dbg_handle;

static void dbg_puts (const char *str)
  {
  unsigned short nbwrite_errorcode;
  if (dbg_handle)
      DOS_WriteFile (dbg_handle, str, strlen(str), &nbwrite_errorcode);
  }

void dbg_init (void)
  {
  static const char *const str = "       \r\nDebug v2.8.7 active (compiled by " __VERSION__ ")\r\n";	/*VERSION*/
  char path[64] = {};
  unsigned len = strlen(str);
  unsigned short nbwrite_errorcode;
  struct DOS_attribute_str on_disk_attribute = {};

  BOOT1_putstr ("debug " DBG_MSG "on DOS file \"");
  if (DOS_GetCwd (0, path, &nbwrite_errorcode) != 0) {
      BOOT1_putstr ("\\");
      BOOT1_putstr (path);
      }
  BOOT1_putstr ("dbg\": ");

  if (DOS_CreateFile ("dbg", on_disk_attribute, &dbg_handle) != 0) {
      BOOT1_putstr ("failed to create file!\r\n");
      dbg_handle = 0;
      return;
      }
  if (DOS_WriteFile (dbg_handle, str, len, &nbwrite_errorcode) != 0) {
      BOOT1_putstr ("failed to write to file!\r\n");
      DOS_CloseFile (dbg_handle, &nbwrite_errorcode);
      dbg_handle = 0;
      return;
      }
  if (nbwrite_errorcode != len)
      BOOT1_putstr ("strange nb of bytes written!\r\n");
    else
      BOOT1_putstr ("OK.\r\n");
  }

void dbg_end (void)
  {
  unsigned short errorcode;

  if (dbg_handle) {
      dbg_puts ("\r\nDebug end.\r\n");
      if (DOS_CloseFile (dbg_handle, &errorcode) != 0)
	  BOOT1_putstr ("failed to close debug file!\r\n");
	else
	  BOOT1_putstr ("\r\nDebug end.\r\n");
      }
  dbg_handle = 0;
  }

#else /* (DEBUG & DEBUG_OUTPUT) */
#if ((DEBUG & ~(DEBUG_ADDR|DEBUG_STACK)) != 0) && DEBUG != 0
#error Invalid value of DEBUG (destination of messages)
#endif
#endif

#if DEBUG & DEBUG_OUTPUT

void dbg_printf (const char *format, ...)
  {
  static char buffer[256];

#if SETUP & XDATA_SEGMENT
  if (*format == '\xFF') {
      static char newformat[256];
      lmemcpy (newformat, STATE.xdataseg_adr + *(farptr *)(format + 1), sizeof(newformat));
      newformat[sizeof(newformat)-1] = '\0';
      format = newformat;
      }
#endif

  dbg_puts (snprtf (buffer, sizeof(buffer), format,
		(void **)((unsigned)&format + sizeof(const char *)) ));
  }

#endif

