/* cleandisk.c */

/*
 * This standalone "cleandisk.kgz" utility is a demo on how to create
 * simple real-mode utilities directly runnable from the Gujin bootloader.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * This utility display content of the "unused/unreserved sectors" of a disk,
 * that is sectors not mapped into a partition, sectors at the beginning
 * of the disk (usually LBA 1..63) or at the beginning of an extended
 * partition, sectors at the end of the disk (when HPA not set, see
 * "ignore kernel IDE options" and GZIP comment "option=0x007B").
 * It does not display sector containning only zero bytes, so search
 * the first non empty sector - and automatically finishes when reaching
 * the last sector of the disk.
 * The sectors at beginning of FAT partitions (extra sectors of FAT32,
 * copy of boot record,...) or the first Kbyte of EXT2/3 are _not_
 * considered as "unused sectors" and so are not displayed.
 * This utility can also clear these sectors to zero, but then take
 * care because some bootloader (other than Gujin) are using those
 * reserved sectors.
 * The function called to clear those sectors is the Gujin "write_disk"
 * function, so Gujin "write enable" setup box has to be ticked in the
 * setup menu.
 * This utility assumes a sector size smaller than 4 Kbytes, so
 * CDROMs/DVDs should behave correctly with their 2 Kbytes sectors.
 * This utility is a lot quicker in text modes (versus graphic modes).
 * Because this utility uses Gujin variables (hopefully read-only),
 * it has to be distributed only on a way authorised by the
 * GPL License, as described in the "gzcopy" in Makefile.
 *
 * Pattern to add to your Makefile to regenerate this cleandisk.kgz executable:
 * (See also the cleandisk.ld linker file for a complete cleandisk package)
##
## Standalone KGZ utilities:
##
cleandisk.kgz:	cleandisk.c cleandisk.ld gzcopy
	gcc -Os -fomit-frame-pointer -mrtd -march=i386 -fno-gcse -fgcse-lm -fno-builtin -fno-optimize-sibling-calls -S cleandisk.c -o cleandisk.S
	-@ echo "Greeping indirect calls through 32 bits register, there shall be none (lcallw replacement impossible):"
	-(grep 'call	\*%e' cleandisk.S && echo "ERROR: the application will not work, did you remove -fno-gcse?")
	sed 's/call\t\(.*_far_.*\)/lcallw\t\1/' < cleandisk.S > cleandisk.s
	as cleandisk.s -o cleandisk.o
	ld -T cleandisk.ld -nostdlib -Map=cleandisk.map cleandisk.o -o cleandisk.elf
	objcopy -O binary -R .note -R .comment -S  cleandisk.elf cleandisk.bin
	gzip -9 -f cleandisk.bin
	./gzcopy cleandisk.bin.gz -f cleandisk.kgz \
		-c="This software is distributed under the GPL license." \
		-al="Its source code is at: http://gujin.org" \
		-al="min_gujin_version=0x105 " \
		-as="realfct_size=`grep _gujin_size cleandisk.map | awk '{print $$1}'`"
	rm -f cleandisk.S cleandisk.s cleandisk.o cleandisk.elf cleandisk.bin cleandisk.bin.gz
	rm -f cleandisk.map
	# rm -f cleandisk.kgz
 */

#ifndef __ELF__
#error Gujin can only load ELF files, and this compiler is not producing ELF files
#endif

/*
 * Do not touch - used to generate real mode ia32 code and manage
 * the inter-segment call transparently, after maximum 4 Kbytes
 * of real mode rodata/data has been copied in %ds segment.
 *
 * Having constant/initialised variable in here is a bit tricky,
 * it uses the fact that since Gujin-1.0, the segment ".fourKsegment"
 * containning the only variable "fourKbuffer" (of size 4 Kbytes)
 * is located at %ds:0 i.e. before the ".rodata" of Gujin.
 * Note that here the code segment is local but the data segment
 * (and the stack segment) is the one of Gujin for obvious reasons.
 * The first person modifying data at address over %ds:0x1000
 * get all the blame for any bug happening in Gujin, in between
 * the last and the next big bang, and it is a pretty long time.
 * <smile>You don't like my memory protection system?</smile>
 */
asm("	.psize 0				\n"
    "	.code16gcc				\n"
    "	.section .init.text,\"ax\",@progbits	\n"
    "	.global entry				\n"
    "entry:					\n"
//  "int $3 # if debugging with borland td.exe	\n"
    "	push	%di				\n"
    "	push	%si				\n"
    "	mov	$_sdata16,%si			\n"
    "	mov	%si,%di				\n"
    "	mov	$_edata16,%cx			\n"
    "	and	$0xFFF,%cx			\n"
    "	sub	%si,%cx				\n"
    "	shr	$2,%cx				\n"
    "	rep movsl %cs:(%si),%es:(%di)		\n"
    "	pop	%si				\n"
    "	pop	%di				\n"
    "	calll	start				\n"
//  "int $3 # if debugging with borland td.exe	\n"
    "	lretw					\n"
    "	.previous				\n"
);

/**
 ** Structure exported from the Gujin project, Gujin will
 ** try not to modify any of the documented features:
 **/

// simple stuff from Gujin/library.h:
#define FASTCALL        __attribute__ (( fastcall ))

#define data_adr(adr) ({ \
	unsigned dataseg;				\
	asm ("mov %%ds,%0" : "=r" (dataseg));		\
	/*return*/ (dataseg << 16) + (unsigned)adr;	\
	})

typedef unsigned farptr;

extern inline unsigned peekb (farptr adr)
  {
  unsigned returned;

  asm (
"	pushl	%1		\n"
"	popw	%w0		\n"
"	popw	%%fs		\n"
"	movzbl	%%fs:(%w0),%0	\n"
	: "=bSDB" (returned) : "g" (adr)
	);
  return returned;
  }

// other simple stuff:
struct linux_param;

struct LOADER_str;

struct loader_t {
    enum { LILO = 0, Loadlin, bootsect,
	SYSLINUX, ETHERBOOT, ELILO, UNKNOWN,
	GRUB, UBOOT, XEN, GUJIN
	} type:4;
    unsigned char version : 4;
    } __attribute__ ((packed));


typedef struct { unsigned short x, y; } __attribute__ ((packed)) coord;

struct mousefieldattr_str {
    unsigned short fullscreenfield	: 1;	/* for "Press a key to continue" */
    unsigned short upperfield	: 1;
    unsigned short mainfield	: 1;
    unsigned short active	: 1;
    unsigned short pressed	: 1;
    unsigned short checkbox	: 1;
    unsigned short greyed	: 1;
    };

struct attribute_str {
    unsigned char underline	:1;
    unsigned char reverse	:1;
    unsigned char blink		:1;
    unsigned char brighter	:1;
    unsigned char darker	:1;
    unsigned char transparent	:1; /* no background drawn in VESA */
    unsigned char reserved	:2;
    } __attribute__ ((packed));

struct registers;
struct gujin_param_attrib {
    unsigned reserved : 32;
    } __attribute__ ((packed));

// The structure given to GPL compliant kernel:
struct gpl_compliant_str {
    unsigned signature;
    unsigned version;
    unsigned feature;
    unsigned size;

    struct { /* not initialised for real mode */
	unsigned esi, edi, ebp, ebx;
	unsigned short gs, fs, es, ds;
	unsigned short flags, ip, cs, sp, ss;
	} __attribute__ ((packed)) jmpbuf;

    unsigned filename_array, gdt, regs, fourKbuffer;
    unsigned loader, state, ui, mouse, di, util, bootway;

    /* Those are real mode seg:off addresses: */
    farptr font8x8_farptr, font8x14_farptr, font8x16_farptr,
	 font10x20_farptr, font12x24_farptr, reserved_farptr[3];

    struct real_mode_fct { /* Those are for real-mode *.kgz use, to call only by lcallw */
	void     (*putstr) (const char *str);
	int      (*printf) (const char *format, ...);
	unsigned (*getkey) (unsigned timeout); /* timeout in ticks, 18/s, 0 = none */
	unsigned (*get_number) (unsigned deflt, unsigned max);
	void     (*reset_field) (struct mousefieldattr_str attr);
	void     (*print_field) (const char *str, struct mousefieldattr_str attr,
		unsigned short key, unsigned short row, unsigned short col);
	void     (*clearscreen) (void);
	void     (*draw_bg_box) (unsigned baserow, unsigned height, unsigned color);
	/* Cursor: home at (0,0) */
	unsigned (*setcursor) (unsigned char row, unsigned char col);
	unsigned (*getcursor) (unsigned char *row, unsigned char *col);
	/* Colors are set by functions, and get directly from structure: */
	unsigned (*setfgcolor) (unsigned color);
	unsigned (*setbgcolor) (unsigned color);
	void     (*setattribute) (struct attribute_str attr);

	/* WARNING: NO range check is done (on xy or xend) to be quicker: */
	void     (*setpixel) (coord xy, unsigned color) FASTCALL;
	unsigned (*getpixel) (coord xy) FASTCALL;
	void     (*plotHline) (coord xy, unsigned short xend, unsigned color);

	/* This function ptr is null if (and only if) we are on a serial interface: */
	unsigned __attribute__ ((const)) (*VESA_color) (unsigned char _red, unsigned char _green, unsigned char _blue);
	unsigned (*final_loadrun) (unsigned index, struct registers *regs, struct gujin_param_attrib gujin_attr); /* not tested */
	unsigned (*get_line) (char *buffer, unsigned maxlen, char display, unsigned char lowlimit, unsigned char highlimit); /* Gujin-1.6+ */
	unsigned reserved[6];
	} fct;

    /* Those are simply local address: */
    const struct {
	unsigned        identification; /* do not have linear or clear bit */
	unsigned        nbcolor;
	unsigned short  width, height;
	unsigned	base_address;
	unsigned char   row, col;
	unsigned char   nbrow, nbcol;
	unsigned char   charwidth, charheight;
	struct {
	    unsigned char isgraphic		: 1;
	    /*
	     * Some VGA cards, in VGA mode, can accept 256 colors background,
	     * color given by the "page" parameter, VESA card have always
	     * this flag set when graphic (at least >= 16 colors):
	     */
	    unsigned char graphicbgcolor	: 1;
	    /* Else only the PC font is available, always set for serial: */
	    unsigned char ansi_font		: 1;
	    /* null used for text mode mouse, del used as a ticker for tick-box */
	    unsigned char null_del_available	 : 1;
	    /* Always set when scroll, never unset but by application: */
	    unsigned char has_scrolled	 : 1;
	    /* Always set when one char has been written on screen, never unset but by application: */
	    unsigned char has_outputed	 : 1;
	    } attr;
	/* rest of the structure video_parameter_str undocumented */
	} *current_param;
    struct mouse_color_str {
	unsigned deflt		: 4;
	unsigned leftbutton	: 4;
	unsigned rightbutton	: 4;
	unsigned middlebutton	: 4;
	unsigned topbutton	: 4;
	unsigned twobutton	: 4;
	unsigned invalid	: 4;
	unsigned infield	: 4;
	} *mouse_colors;
    void *reserved_ptr[6];
    };

// from Gujin/fs.h:
struct desc_str {
    unsigned inode;
    unsigned filesize;
    unsigned last_modification_date; /* in seconds since Jan 1st, 2001 (i.e. after Feb 29th, 2000) */
    unsigned char variant_number;
    unsigned char searchpath;  /* =0 for copy_gujin_param.scanpath, =1 for install.386, =2 for install.amd, =3 for isolinux, =4 for casper, =5 for live */
    unsigned char variant_timeout;
    unsigned char unused;
    unsigned iso_filesize;	/* if inSlashBoot or inIsoSlashBoot */
    unsigned iso_inode;	/* if inSlashBoot or inIsoSlashBoot */
    unsigned reserved;
    unsigned char disk;
    unsigned char partition;
    unsigned char name_offset;	/* to after the pattern */
    unsigned char inRoot : 1;
    unsigned char inISO : 1;
    enum boottype_e { is_MBR, is_PBR, is_linux, is_elf,
		is_initrd, is_bdi_file, is_el_torito, is_initramfs, is_multiboot } boottype : 4;
    unsigned char ReadOnly	 : 1;	/* mostly for BDI files */
#define NAME_LENGTH	64
    char filename[NAME_LENGTH];
    } __attribute__ ((packed));

// from Gujin/user.h:
enum stdcolor_enum {
    black,    blue,         green,      cyan,
    red,      magenta,      brown,      lightgray,
    darkgray, lightblue,    lightgreen, lightcyan,
    lightred, lightmagenta, yellow,     white
    };

const struct user_interface_str {
    unsigned reserved;	/* pre gujin-v1.5:    unsigned (*getkey) (unsigned timeout); */
    unsigned fgcolor, bgcolor;
    unsigned stdcolor[16];
    struct {
	struct attribute_str current, valid;
	} attributes;
    /* rest of the structure video_parameter_str undocumented */
    } *_UI;

// getkey() returns:
#define FCT_KEYCODE(i)	(((i <= 10)? (0x3AU + i) : (0x85U + i - 11)) << 8)

#define ESC_KEYCODE()		(0x0100 | '\033')
#define SPACE_KEYCODE()		(0x3900 | ' ')
#define BACKSPACE_KEYCODE()	(0x0E00 | '\010')
#define RETURN_KEYCODE()	(0x1C00 | '\r')
#define KP_RETURN_KEYCODE()	(0xE000 | '\r') /* only extkbd */
#define KP_MINUS_KEYCODE()	(0x4A00 | '-')
#define MINUS_KEYCODE()		0x0C2D
#define KP_PLUS_KEYCODE()	(0x4E00 | '+')
#define PLUS_KEYCODE()		0x0D2B
#define DIVIDE_KEYCODE()	(0x3500 | '/')
#define KP_DIVIDE_KEYCODE()	(0xE000 | '/') /* only extkbd */
#define KP_MULTIPLY_KEYCODE()	(0x3700 | '*')
#define MULTIPLY_KEYCODE()	0x092A
#define DOT_KEYCODE()		(0x3400 | '.')
#define KP_DOT_KEYCODE()	(0x5300 | '.')
#define DIGIT_KEYCODE(i) \
    (((i? i+1 : 0x0B)<<8) | ('0' + i))
#define NUMPAD_DIGIT_KEYCODE(i) \
    ((((i == 0)? 0x52 : (0x4E - ((i - 1)/3 * 7) + i)) << 8) | ('0' + i))

#define PAGEUP_KEYCODE()	(0x4900)
#define PAGEDOWN_KEYCODE()	(0x5100)
#define HOME_KEYCODE()		(0x4700)
#define END_KEYCODE()		(0x4F00)
#define UP_KEYCODE()		(0x4800)
#define DOWN_KEYCODE()		(0x5000)
#define RIGHT_KEYCODE()		(0x4D00)
#define LEFT_KEYCODE()		(0x4B00)
#define TAB_KEYCODE()		(0x0F09)

#define CTRL_L_KEYCODE()	(0x260C)
#define CTRL_R_KEYCODE()	(0x1312)
#define CTRL_P_KEYCODE()	(0x1910)
#define CTRL_X_KEYCODE()	(0x2D18)

#define CTRLENTER		(0x1C0A)	/* PC keyboard only */
#define TIMEOUT_KEYCODE()	0xFFFE		/* Fake key for timeouts */

// A very long structure defining disks from Gujin/disk.h - you can skip it at first reading.
//   most of it shall be considered "documented", but check the sizeof_* if digging too deep.
enum IDE_security_enum {
    IDE_security_unlock_user = 0,
    IDE_security_unlock_master = 1
    };
struct diskparam_str;
const struct disk_interface_str {
    unsigned		nbdisk;

    unsigned char	max_disk, max_partition, max_freelist, max_IDE_found;
    unsigned short	sizeof_diskparam_str,	sizeof_partition_str;
    unsigned short	sizeof_freelist_str,	sizeof_IDE_found_str;
    unsigned char	nb_bios_fd, nb_bios_hd, cannot_reset_floppy, nb_bios_blacklist;
    unsigned char	bios_blacklist[4];
    unsigned		reserved1[2];

    /* if (partition < 0), standard disk access based at lba == 0,
	else add partition base and check partition limit: */
    unsigned char	(*readsector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
    unsigned char	(*writesector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
    int			(*ideSecurity) (struct diskparam_str *dp, char password[32],
			enum IDE_security_enum action);
    unsigned long long	unused;

    struct diskparam_str {
	enum {	bios_chs, ebios_lba,
		hardide_chs, hardide_lba, hardide_lba48, hardide_atapi,
		dos_part
		} access : 8;
	unsigned char	disknb;	/* BIOS number or DOS letter
				   or 0=master, non zero=slave */
	unsigned char	biostype;	/* 1:floppy w/o change detect, 2: with, 3: HD */
	unsigned char	drivetype;	/* 1: 360K, 2: 1.2M, 3: 720K, 4: 1.44M, 6: 2.88M, 0x10: ATAPI */
	unsigned char	diskname[32];

	struct ide_attribute_str {
	    unsigned smart			: 1;
	    unsigned host_protected_area	: 1;
	    unsigned security			: 1;
	    unsigned lba48			: 1;
	    unsigned removable			: 1;
	    unsigned SAORAB			: 1;
	    unsigned config_overlay		: 1;
	    unsigned reserved			: 25;
	    } ide_attribute;
	struct error_log_str {
	    unsigned read_media			: 1;
	    unsigned write_media		: 1;
	    unsigned access_over_disk_tryed	: 1;
	    unsigned access_over_partition_tryed: 1;
	    unsigned no_ebios_fct		: 1;
	    unsigned ebios_size_zero		: 1;
	    unsigned chs_bios_part_mismatch	: 1;
	    unsigned chs_ajusted_bootsect	: 1;
	    unsigned diskheader_ignored		: 1;
	    unsigned disk_locked		: 1;
	    unsigned disk_was_pw_locked		: 1;
	    unsigned SMART_disabled		: 1;
	    unsigned SMART_failure		: 1;
	    unsigned analyse_partition_failed	: 1;
	    unsigned NB_PARTITION_exceded	: 1;
	    unsigned partition_overlap		: 1;
	    unsigned partition_over_limit	: 1;
	    unsigned beer_checksum_error	: 1;
	    unsigned beer_in_extended_partition	: 1;
	    unsigned reserved			: 13;
	    } error_log;
	unsigned short	bytepersector;
	unsigned short	nbhead;		  /* DOS: MediaDescriptorByte */
	unsigned	nbcylinder;       /* DOS: ClustersOnDisk */
	unsigned	BEER_size;
	unsigned long long	BEER_sector;
	unsigned long long	BEER_HostProtectedArea_start;
	unsigned long long	BEER_ReservedAreaBootCodeAddress;
	unsigned long long	BEER_disksize;
	  signed long long	fulldisk_offset;	/* Nb sectors before MBR */
	unsigned long long	fulldisk_size;		/* only if MBR is offseted */
	unsigned long long	nbtotalsector;
	unsigned long long	config_max_lba;
	unsigned short	nbsectorpertrack; /* DOS: SectorPerCluster */

	unsigned short	infobit;	/* Int13/48 extensions */
	unsigned short	ebios_version;	/* if present, used if access = ebios */
	struct ebios_bitmap_u {
	    unsigned short extended_fct	: 1;
	    unsigned short removable	: 1;
	    unsigned short enhanced	: 1;
	    unsigned short reserved	: 13;
	    } __attribute__ ((packed)) ebios_bitmap;
	unsigned short	reserved3;	/* ebios_bitmap is 16 bits */

	unsigned char  ebios_bustype[4];	/* exactly "ISA" or "PCI" */
	unsigned char  ebios_Interface[8];	/* exactly "ATA", "ATAPI", "SCSI", "USB"...*/
	union interface_path_u {
	    struct {
		unsigned short addr;
		unsigned char pad[6];
		} __attribute__ ((packed)) isa;
	    struct {
		unsigned char busnr;
		unsigned char devicenr;
		unsigned char functionnr;
		unsigned char pad[5];
		} __attribute__ ((packed)) pci;
	    } ebios_bus;
	union device_path_u {
	    struct {
		unsigned char slave; /* 0 if master, 1 if slave */
		unsigned char pad[7];
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) ata;
	    struct {
		unsigned char slave; /* 0 if master, 1 if slave */
		unsigned char logical_unit_nr;
		unsigned char pad[6];
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) atapi;
	    struct {
		unsigned char logical_unit_nr;
		unsigned char pad[7];
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) scsi;
	    struct {
		unsigned char tobedefined;
		unsigned char pad[7];
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) usb;
	    struct {
		unsigned long long FireWireGeneralUniqueID;
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) ieee1394;
	    struct {
		unsigned long long WorldWideNumber;
		unsigned long long  reserved; // Not present in some docs...
		} __attribute__ ((packed)) FibreChannel;
	    } ebios_device;

	unsigned short	reserved4;

	unsigned short	ide_master_password_revision; /* if command_supported1.security */
	unsigned short	ideIOadr;	/* if known, else 0 */
	unsigned short	ideIOctrladr;	/* if known, else 0 */
	unsigned char	lba_slave_mask;	/* see Phenix Bios (head | mask) */
	unsigned char	irq;		/* if known, else 0 */
	unsigned char	multiplemode;	/* int 0x13/0x21,0x22,0x24 or hardide,
						set but not used. */
	unsigned char	BEER_device_index;
	struct config_multiword_dma_s {
	    unsigned short  mode0    : 1;
	    unsigned short  mode1    : 1; /* and below */
	    unsigned short  mode2    : 1; /* and below */
	    unsigned short  reserved : 13;
	    } __attribute__ ((packed)) initial_multiDMA, maximum_multiDMA;
	struct config_ultra_dma_s {
	    unsigned short  mode0    : 1;
	    unsigned short  mode1    : 1; /* and below */
	    unsigned short  mode2    : 1; /* and below */
	    unsigned short  mode3    : 1; /* and below */
	    unsigned short  mode4    : 1; /* and below */
	    unsigned short  mode5    : 1; /* and below */
	    unsigned short  reserved : 10;
	    } __attribute__ ((packed)) initial_ultraDMA, maximum_ultraDMA;
	struct config_feature_s {
	    unsigned short	smart_feature	: 1;
	    unsigned short	smart_selftest	: 1;
	    unsigned short	smart_errorlog	: 1;
	    unsigned short	security	: 1;
	    unsigned short	standby_powerup	: 1;
	    unsigned short	RW_DMA_queued	: 1;
	    unsigned short	acoustic	: 1;
	    unsigned short	host_prot_area	: 1;
	    unsigned short	lba48		: 1;
	    unsigned short	reserved	: 7;
	    } __attribute__ ((packed)) initial_feature, maximum_feature;
	unsigned	CalibrationIdeloop; /* nanoseconds (10^-9) per loop */

	/* Following three only for C/H/S partition, from standard BIOS: */
	unsigned short	bios_nbcylinder;	/* max 1024 */
	unsigned char	bios_maxhead;		/* max 255 */
	unsigned char	bios_nbsectorpertrack;	/* max 63 */

	unsigned char	bootable : 7;	/* nonzero: bootable, > 1 when special bootsector recognised */
	unsigned char	sigAA55  : 1;	/* has signature 0xAA55, can contain partition */
	unsigned char	OSdisknumber;	/* as written in the boot block */

	unsigned short	nbpartition;
	unsigned short	nbOSnumber;
	unsigned short	unused;
	/* Following value is current extended primary partition,
	   if there is more than one extended primary partition. */
	unsigned	first_ext_partition_start; /* NOT A LONG LONG */
	struct partition_str {
	    unsigned long long	start, length;
	    /* OSnumber should be equal to the # in /dev/hda#: */
	    unsigned char	type, OSnumber;
	    struct partition_misc_str {
		unsigned char order		: 6;
		unsigned char fsanalyse_toobig	: 1;
		unsigned char fsanalyse_error	: 1;
		unsigned char maybe_root	: 1;
		unsigned char fat_bootable	: 1;
		unsigned char beer_partition	: 1;
		unsigned char swap_partition	: 1;
		unsigned char reserved		: 2;
		enum part_state {
		    part_invalid = 0, part_active, part_inactive, part_extended
		    } active			: 2;
		} __attribute__ ((packed)) misc;
#define MAX_DISKNAMESIZE	(64-20-4) /* bigger than "floppy" */
	    unsigned char	name[MAX_DISKNAMESIZE];
	    struct ISO9660_Default_entry {
		unsigned char  boot_indicator; /* 0x88: bootable, 0x00: non bootable */
		enum { boot_media_noemul = 0, boot_media_120 = 1, boot_media_144 = 2,
			boot_media_288 = 3, boot_media_hd = 4 } boot_media_type : 8;
		unsigned short load_segment; /* 0x7c0 if null */
		unsigned char  system_type;
		unsigned char  unused;
		unsigned short sector_count;
		unsigned       sector_lba;
		unsigned char  reserved[0x1F - 0x0C +1];
		} __attribute__ ((packed)) *ElToritoCatalog; /* first one is ISO9660_Validation_entry */
	    } __attribute__ ((packed)) *partition;

	unsigned	nbfreelist;
	struct freelist_str {
	    unsigned long long start, length;
	    } *freelist;
	char set_max_password[32];
	} *param;

    unsigned	nb_IDE_found;
    struct IDE_found_str {
	unsigned short	ideIOadr;
	unsigned short	ideIOctrladr;
	unsigned char	irq;
	unsigned char	bios_order;
	unsigned short	reserved;
	} *IDE_found;
    } *_DI;
// End of the very long structure, resume read-mode...

/**
 ** The adaptation layer Gujin/application:
 ** Functions of Gujin are "far" functions and have to be called through pointers,
 **	take care of optimisations: forbid inlining init_global()!
 **    Note also the line in Makefile, all function with a name containing
 **      the chars "_far_" will be called by "lcallw" instead of "call":
 **    sed 's/call\t\(.*_far_.*\)/lcallw\t\1/' < cleandisk.S > cleandisk.s
 **/
unsigned char	(*_far_readsector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
unsigned char	(*_far_writesector) (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer);
int		(*_far_ideSecurity) (struct diskparam_str *dp, char password[32],
			enum IDE_security_enum action);
 /* Previous three function pointer can be null, for instance in mingujin.exe:
    so we provide dummy functions (could also be done for print_field() when no mouse).
    Testing if pointer is null before calling function would be optimised
	by "mov fctptr,%edx ; cmp $0,%edx ; call %edx" which cannot work here */
unsigned char	dummy_readsector (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer)
	{ unsigned char ret; asm (" mov $0xFF,%%eax ; lretw $24 " : "=a" (ret)); return ret; }
unsigned char	dummy_writesector (struct diskparam_str *dp, int partition,
			unsigned long long lba, unsigned number, farptr buffer)
	{ unsigned char ret; asm (" mov $0xFF,%%eax ; lretw $24 " : "=a" (ret)); return ret; }
int		dummy_ideSecurity (struct diskparam_str *dp, char password[32],
			enum IDE_security_enum action)
	{ int ret; asm (" mov $0xFF,%%eax ; lretw $12 " : "=a" (ret)); return ret; }
unsigned char DISK_is_initialised = 0;

unsigned char MOUSE_is_initialised = 0;

struct real_mode_fct _far_fctarray;

__attribute__ ((noinline))
static void init_global (const struct gpl_compliant_str *togpl, unsigned cur_cs_segment)
  {
  unsigned *src, *dst, cpt;

  /* _far_fctarray = togpl->fct; // would call memcpy and not inline it */
  for (cpt = sizeof(_far_fctarray) / sizeof(unsigned),
	src = (unsigned *)&togpl->fct,
	dst = (unsigned *)&_far_fctarray; cpt != 0; cpt--)
      *dst++ = *src++;
  _UI = (const struct user_interface_str *)togpl->ui;
  _DI = (const struct disk_interface_str *)togpl->di;

  if (togpl->fct.reset_field && togpl->fct.print_field)
      MOUSE_is_initialised = 1;
  if (_DI->readsector && _DI->writesector)
      DISK_is_initialised = 1;

  _far_readsector = _DI->readsector;
  if (((unsigned)_far_readsector & 0xFFFF) == 0)
      _far_readsector = (typeof(&dummy_readsector))((cur_cs_segment << 16) | (unsigned)dummy_readsector);
  _far_writesector = _DI->writesector;
  if (((unsigned)_far_writesector & 0xFFFF) == 0)
      _far_writesector = (typeof(&dummy_writesector))((cur_cs_segment << 16) | (unsigned)dummy_writesector);
  _far_ideSecurity = _DI->ideSecurity;
  if (((unsigned)_far_ideSecurity & 0xFFFF) == 0)
      _far_ideSecurity = (typeof(&dummy_ideSecurity))((cur_cs_segment << 16) | (unsigned)dummy_ideSecurity);
  }

#define UI	(*_UI)
#define DI	(*_DI)
#define DI_readsector	(*_far_readsector)
#define DI_writesector	(*_far_writesector)
#define DI_ideSecurity	(*_far_ideSecurity)

#define getkey		(*_far_fctarray.getkey)
#define get_number	(*_far_fctarray.get_number)
#define putstr		(*_far_fctarray.putstr)
#define printf		(*_far_fctarray.printf)
#define reset_field	(*_far_fctarray.reset_field)	/* CHECK that pointer is not null (MOUSE_is_initialised) */
#define print_field	(*_far_fctarray.print_field)	/* CHECK that pointer is not null (MOUSE_is_initialised) */
#define clearscreen	(*_far_fctarray.clearscreen)
#define draw_bg_box	(*_far_fctarray.draw_bg_box)
#define setcursor	(*_far_fctarray.setcursor)
#define getcursor	(*_far_fctarray.getcursor)
#define setfgcolor	(*_far_fctarray.setfgcolor)
#define setbgcolor	(*_far_fctarray.setbgcolor)
#define setattribute	(*_far_fctarray.setattribute)
#define setpixel	(*_far_fctarray.setpixel)
#define getpixel	(*_far_fctarray.getpixel)
#define plotHline	(*_far_fctarray.plotHline)
#define get_line	(*_far_fctarray.get_line)

#define print	putstr
#define TICKS_PER_SECOND	18	/* for getkey timeout */

inline void puts(const char *msg) { print (msg); print ("\r\n"); }

/**
 ** The intermediate function to call the application of this file:
 ** You can define "VERBOSE" to see which data Gujin provides.
 **/
#define VERBOSE

static void application (char *buffer4K);

__attribute__ ((cdecl, regparm (0)))
unsigned start (unsigned local_return_address,
	unsigned totalmem,
	struct loader_t bootloader_type,
	struct linux_param *LnxParam,
	const struct LOADER_str *loader,
	const struct gpl_compliant_str *togpl,
	const struct desc_str *system_desc,
	int proposed_row,
	const char **stringptr,
	unsigned char *proposed_cmdline)
  {
//  if (!togpl) {
//      *stringptr = "This application is either not GPL or its source download location is unknown.";
//      return 0x80000001; /* Go back to Gujin menu system without loading a Linux kernel, pre-printing " Error: " */
//      }
  if (togpl->version < 0x105) {
      *stringptr = "This application needs at least Gujin version 1.5";
      return 0x80000001; /* Go back to Gujin menu system without loading a Linux kernel, pre-printing " Error: " */
      }
  if (togpl->font8x16_farptr == 0) { /* else USER_SUPPORT == 0 */
      *stringptr = "Loaded by mingujin.exe/mingujin.com, no user interface initialised, no retf in functions...";
      return 0x80000001; /* Go back to Gujin menu system without loading a Linux kernel, pre-printing " Error: " */
      }

  init_global(togpl, local_return_address >> 16);

#ifdef VERBOSE
  {
  unsigned char row, col;
  unsigned color;
  getcursor (&row, &col);
  setcursor (row + 1, 4);
  print ("Welcome to ");
  setfgcolor (UI.stdcolor[brown]); setbgcolor (UI.stdcolor[lightgray]);
  print ("the verbose standalone ");
//  setattribute (((struct attribute_str) {1, 0, 0, 0, 0, 0})); gcc version 4.6.0 bug
  { struct attribute_str tmp = {1, 0, 0, 0, 0, 0}; setattribute (tmp); }
  print ("application");
  setfgcolor (UI.stdcolor[brown]); setbgcolor (UI.stdcolor[lightgray]);
//  setattribute (((struct attribute_str) {0, 0, 0, 0, 0, 0})); gcc version 4.6.0 bug
  { struct attribute_str tmp = {0, 0, 0, 0, 0, 0}; setattribute (tmp); }
  puts (", really!");
  print ("\r\n");
  //color = getpixel ((coord){13, row+2}); gcc version 4.6.0 bug
  { coord tmp = {13, row+2}; color = getpixel (tmp); }
  /* plotHline works also in text mode, but color which shall be ((color<<8)|char_written) */
//  plotHline ((coord){10, row+2}, 20, UI.stdcolor[green]); gcc version 4.6.0 bug
  { coord tmp = {13, row+2}; setpixel (tmp, color); }
//  setpixel ((coord){13, row+2}, color); gcc version 4.6.0 bug
  { coord tmp = {13, row+2}; setpixel (tmp, color); }
  setfgcolor (UI.stdcolor[cyan]); setbgcolor (UI.stdcolor[yellow]);
  puts ("Some colors combination are not easy to read, isn't it!");
  setfgcolor (UI.stdcolor[blue]); setbgcolor (UI.stdcolor[lightgray]);
  }

  printf ("Application has been loaded after Gujin stack, starting a far addr 0x%X\r\n", local_return_address);
  printf ("The total memory available for the load subsystem is %u Kb\r\n", totalmem);
  printf ("The loader is 0x%X (0xA reserved for Gujin) version %u\r\n", bootloader_type.type, bootloader_type.version);
  printf ("LnxParam is some (4 Kbytes) useable data space in stack at address 0x%X\r\n", LnxParam);
  printf ("loader is a (not too interresting) read only structure at 0x%X\r\n", loader);
  if (togpl) {
      printf ("Because this software is GPL, Gujin is wanting to share its read only data:\r\n");
      printf ("  signature %u, version 0x%X, feature 0x%X, size %u\r\n", togpl->signature, togpl->version, togpl->feature, togpl->size);
      printf ("  filename_array %u, gdt 0x%X, regs 0x%X, fourKbuffer 0x%X\r\n", togpl->filename_array, togpl->gdt, togpl->regs, togpl->fourKbuffer);
      printf ("  LOADER 0x%X, STATE 0x%X, UI 0x%X, MOUSE 0x%X,\r\n  DI 0x%X, UTIL 0x%X, BOOTWAY 0x%X\r\n",
		togpl->loader, togpl->state, togpl->ui, togpl->mouse, togpl->di, togpl->util, togpl->bootway);
      }
  printf ("This application is stored in file %s of size %u, inode %u\r\n", system_desc->filename, system_desc->filesize, system_desc->inode);
  printf ("  of disk %u in partition %u (initrd name_offset %u)\r\n", system_desc->disk, system_desc->partition, system_desc->name_offset);
#if defined (DI)
  if (system_desc->disk < DI.nbdisk) {
      static const char *access_type[] = {
	  "bios_chs", "ebios_lba",
	  "hardide_chs", "hardide_lba", "hardide_lba48", "hardide_atapi",
	  "dos_part"
	  };

      printf ("  disk name: '%s' accessed by '%s'", DI.param[system_desc->disk].diskname, access_type[DI.param[system_desc->disk].access]);
      if (system_desc->partition < DI.param[system_desc->disk].nbpartition) {
	  printf (" partition name '%s'\r\n", DI.param[system_desc->disk].partition[system_desc->partition].name);
	  //printf (" partition type 0x%X", DI.param[system_desc->disk].partition[system_desc->partition].type);
	  }
      }
#endif
  if (system_desc->boottype == is_elf && !system_desc->inISO) {
      if (system_desc->inRoot)
	  puts ("  the *.kgz file has been found in the ROOT directory of this filesystem");
	else
	  puts ("  the *.kgz file has been found in a subdirectory of this filesystem");
      }
    else
      printf ("  boottype %u in ISO image\r\n", system_desc->boottype);
  printf ("The proposed row to write message is %u\r\n", proposed_row);
  printf ("The pointer to gujin-extra command line (64 bytes) is 0x%X pointing to:\r\n", *stringptr);
  if (*stringptr == 0)
      print ("zero pointer\r\n");
    else {
      /* This field in gujin_param_t is 64 bytes long: */
      unsigned char gujin_cmd_line[64], *ptr = gujin_cmd_line;
      farptr fptr = (farptr)*stringptr;
      while ((*ptr = peekb (fptr++)) != '\0' && ptr - gujin_cmd_line < 63)
	  ptr++;
      *ptr = '\0';
      printf ("   '%s'\r\n", gujin_cmd_line);
      }
  printf ("The pointer to gujin-proposed command line (256 bytes) is 0x%X pointing to:\r\n   '%s'\r\n", proposed_cmdline, proposed_cmdline);

  if (*_far_writesector == 0)
      puts ("Gujin compiled without 'WRITE_ENABLE'!");
  if (*_far_ideSecurity == 0)
      puts ("Gujin compiled without 'IDE_SUPPORT' or without 'USER_SUPPORT'!");

  if (MOUSE_is_initialised) {
      unsigned char row, col;
      unsigned key;

      /* A problem for generic text or 3D buttons: the border of the 3D button are located
	on the character _following_ the field, so you should not write a space directly
	after the field, it will erase the right border - but for text modes there isn't
	any 3D buttons and you _have_ to write a space just after the field to set the
	background color... Just always write spaces to fill the background color
	from one char _before_ to one char _after_ for such fields before print_field(),
	Gujin let you decide on which background color you want there. */
      setfgcolor (UI.stdcolor[brown]); setbgcolor (UI.stdcolor[lightgray]);
      print ("Mouve the mouse and click");
      getcursor (&row, &col);
      print ("      ");	/* write the char before and after the field with the current background */
      setcursor (row, col + 1);
//      print_field ("HERE", (struct mousefieldattr_str) { 0, 0, 1, 1, 0, 0, 0 }, SPACE_KEYCODE(), row, col + 1); gcc version 4.6.0 bug
      { struct mousefieldattr_str tmp = { 0, 0, 1, 1, 0, 0, 0 }; print_field ("HERE", tmp, SPACE_KEYCODE(), row, col + 1); }
      setcursor (row, col + 1 + sizeof("HERE")); /* Shall not overflow onto next line, and _no_ scroll screen... */
      print ("or press a key to continue"); /* No puts(), do not scroll screen when mouse fields defined! */
      key = getkey(30 * TICKS_PER_SECOND);
      if (key == TIMEOUT_KEYCODE())
	  puts ("\r\n You did not press any key within 30 seconds, continue anyway.");
	else if (key == SPACE_KEYCODE())
	  puts ("\r\n you either pressed SPACE or clicked the field, continuing.");
	else
	  puts ("\r\n you pressed another key than SPACE, continuing.");
//      reset_field ((struct mousefieldattr_str) { 0, 0, 1, 0, 0, 0, 0 }); gcc version 4.6.0 bug
      { struct mousefieldattr_str tmp = { 0, 0, 1, 0, 0, 0, 0 }; reset_field (tmp); }
      setfgcolor (UI.stdcolor[blue]); setbgcolor (UI.stdcolor[lightgray]);
      }

  if (togpl->version >= 0x106) {
      char buffer[10];
      unsigned ret;
      print ("Test input line editing, enter the max line length: ");
      buffer[0] = '\0';
      ret = get_line (buffer, sizeof(buffer), 0, '0', '9'); // hexa also possible, limits: 'f','0'
      if (ret == 0) {
	  unsigned maxlen = 0, index = 0;
	  while (buffer[index] >= '0' && buffer[index] <= '9')
	      maxlen = 10 * maxlen + buffer[index++] - '0';
	  if (maxlen == 0 || maxlen > 256)
	      printf ("\r\n  do not test when maximum line length is %u\r\n", maxlen);
	  else {
	      char buffer2[maxlen];

	      // That is not soo good a demo, multiline input not nicely working when screen scrolls...
	      printf ("\r\nEnter/edit test line (max_len=%u): ", maxlen);
	      buffer2[0] = '\0';
	      ret = get_line (buffer2, sizeof(buffer2), 0, ' ', '~');
	      printf ("\r\nget_line() returned %d, buffer: '%s'\r\n", ret, buffer2);
	      }
	  }
      }

#endif /* VERBOSE */

  puts (""); /* In the middle of the line if not "force last text mode at startup" */

  application ((char *)LnxParam);

  *stringptr = "\r\nThis application has ended";
//  return 0x80000000; /* Go back to Gujin menu system without loading a Linux kernel */
  return 0xC0000000; /* Go back to Gujin menu system without loading a Linux kernel or clearing screen */
  }

/**
 ** The real application:
 **/
#include "messages.h"
const char *const itoa_array =      "0123456789ABCDEF0123456789abcdef";
static void do_clean_disk (char *buffer4K);

static void application (char *buffer4K)
  {
  if (!DISK_is_initialised) {
      puts ("Gujin compiled without read or write disk support, cannot run this demo.");
      return;
      }
  do_clean_disk (buffer4K);
  }

static void do_clean_disk (char *buffer4K)
  {
  struct diskparam_str *dp;
  unsigned nb, freeblock;
  unsigned short key = 0;

  printf (FL_FREELIST_DISK_0_N, DI.nbdisk);
  for (dp = &DI.param[0]; dp < &DI.param[DI.nbdisk]; dp++) {
      printf (INDEX_DISK_NAME, dp - &DI.param[0], dp->diskname);
      for (freeblock = 0; freeblock < dp->nbfreelist; freeblock++)
	  printf ("%llU + %llu; ", dp->freelist[freeblock].start, dp->freelist[freeblock].length);
      printf (FL_TOTAL_DISK_SIZE, dp->nbtotalsector);
      }
  printf (FL_TYPE_DISK_INDEX);
  nb = get_number (DI.nbdisk, DI.nbdisk); /* invalid default for ESC */
  if (nb < DI.nbdisk) {
      dp = &DI.param[nb];
      printf (FL_CHECKING_CONTENT_OF_DISK, dp->diskname);
      for (freeblock = 0; freeblock < dp->nbfreelist && key != '\033'; freeblock++) {
	  unsigned long long i;
	  for (i = 0; i < dp->freelist[freeblock].length; i++) {
	      printf (FL_CHECKING_SECTOR_AT_LBA, dp->freelist[freeblock].start + i);
	      if (DI_readsector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(buffer4K)) != 0) {
		  puts (FL_READ_SECTOR_FAILED);
		  break;
		  }
	      if (getkey(0) != TIMEOUT_KEYCODE()) {
		  puts (FL_INTERRUPTED);
		  key = ' ';	/* stop CTRLENTER */
		  }
		else {
		  unsigned short j;

		  for (j = 0; j < dp->bytepersector / sizeof(unsigned); j++)
		      if (((unsigned *)buffer4K)[j] != 0)
			  break;
		  if (j >= dp->bytepersector / sizeof(unsigned))
		      continue;
		  if (key == CTRLENTER)
		      puts (FL_AUTOCLEAR);
		    else
		      puts (FL_NOT_EMPTY);
		  }
	      for (;;) {
		  if (key != CTRLENTER) {
		      puts (FL_TYPE_SPACE_ENTER_TAB);
		      key = getkey(30 * TICKS_PER_SECOND);
		      if (key == CTRLENTER) {
			  unsigned short key2;
			  print (FL_CONFIRM_CLEAR_ALL);
			  key2 = getkey(30 * TICKS_PER_SECOND) & 0xFF;
			  if (key2 != 'y' && key2 != 'Y') {
			      key = ' ';
			      continue;
			      }
			  }
			else
			  key &= 0xFF;
		      }
		  if (key == ' ') {
		      unsigned short j = 0;
		      if (DI_readsector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(buffer4K)) != 0) {
			  printf (FL_REREAD_DISK_FAILED, dp->diskname, dp->freelist[freeblock].start + i);
			  break;
			  }
		      while (j < dp->bytepersector) {
			  extern const char *const itoa_array;
			  const unsigned strlen_pattern = sizeof("0x000: 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................\r\n") - 1;
			  /* 4 lines at a time for scrolling speed: */
			  char buf[4 * strlen_pattern + 1], *endbuf = &buf[4 * strlen_pattern], *ptr = buf;

			  while (ptr < endbuf) {
			      unsigned k;
			      *ptr++ = '0';
			      *ptr++ = 'x';
			      *ptr++ = itoa_array[j / 256];
			      *ptr++ = itoa_array[(j / 16) % 16];
			      *ptr++ = '0';
			      *ptr++ = ':';
			      for (k = 0; k < 16; k++) {
				  unsigned char c = buffer4K[j+k];
				  *ptr++ = ' ';
				  *ptr++ = itoa_array[c / 16];
				  *ptr++ = itoa_array[c % 16];
				  }
			      *ptr++ = ' ';
			      *ptr++ = ' ';
			      for (k = 0; k < 16; k++) {
				  unsigned char c = buffer4K[j++];
				  if (c >= ' ' && c <= '~')
				      *ptr++ = c;
				     else
				      *ptr++ = '.';
				  }
			      *ptr++ = '\r';
			      *ptr++ = '\n';
			      }
			  *ptr = '\0';
			  print (buf);
			  }
		      }
		    else if (key == '\r' || key == '\n' || key == CTRLENTER) {
		      unsigned short j;
		      for (j = 0; j < dp->bytepersector / sizeof(unsigned); j++)
			  ((unsigned *)buffer4K)[j] = 0;
		      if (DI_writesector (dp, -1, dp->freelist[freeblock].start + i, 1, data_adr(buffer4K)) != 0) {
			  printf (FL_WRITING_DISK_FAILED, dp->diskname, dp->freelist[freeblock].start + i);
			  break;
			  }
			else if (key == CTRLENTER)
			  break;
		      }
		    else
		      break;
		  } /* for (;;) */
	      if (key != '\t' && key != CTRLENTER) {
		  printf (FL_SKIP_TO_NEXT_BLOCK, freeblock);
		  break;
		  }
	      }
	  }
      printf (FL_FINISHED_TREATMENT_DISK, dp->diskname);
      }
  }
