/* boot.h */
#ifndef BOOT_H
#define BOOT_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

extern inline unsigned char
partition_hide (unsigned char part_type)
  {
  return part_type | 0x10;
  }

extern inline unsigned char
partition_unhide (unsigned char part_type)
  {
  return part_type & ~0x10;
  }

extern inline unsigned
partition_has_bootsector (unsigned char part_type)
  {
  if (   part_type == DOS_extended
      || part_type == Win95_extendedLBA
      || part_type == EMPTY
      || part_type == Linux_Swap)
      return 0;
    else
      return 1;
  }

extern inline unsigned
partition_is_FAT12 (unsigned char part_type)
  {
#define P(name, val, str)	if (part_type == val) return 1;
  DOS_FAT12_PARTITIONLIST()
#undef P
  return 0;
  }

extern inline unsigned
partition_is_FAT16 (unsigned char part_type)
  {
#define P(name, val, str)	if (part_type == val) return 1;
  DOS_FAT16_PARTITIONLIST()
#undef P
  return 0;
  }

extern inline unsigned
partition_is_FAT32 (unsigned char part_type)
  {
#define P(name, val, str)	if (part_type == val) return 1;
  DOS_FAT32_PARTITIONLIST()
#undef P
  return 0;
  }

extern inline unsigned short
begin_cylinder (const struct bootsect_partition_str *bp)
  {
  union {
      struct {
	  unsigned char lsb;
	  unsigned char msb;
	  } bits;
      unsigned short all;
      } tmp = { bits: {
      lsb: bp->begin_cylinder_lsb,
      msb: bp->begin_cylinder_msb
      }};

  return tmp.all;
  }

extern inline unsigned short
end_cylinder (const struct bootsect_partition_str *bp)
  {
  union {
      struct {
	  unsigned char lsb;
	  unsigned char msb;
	  } bits;
      unsigned short all;
      } tmp = { bits: {
      lsb: bp->end_cylinder_lsb,
      msb: bp->end_cylinder_msb
      }};

  return tmp.all;
  }

/*
 * A sample boot sector:
 */
typedef struct {
    bootbefore_t	before;
    unsigned char	code_part[512 - sizeof(bootbefore_t)
					- sizeof(boot1param_t)
					- sizeof (serialconf_t)
					- sizeof(bootafter_t) ];
    boot1param_t	param;
    serialconf_t        serial_configuration;
    bootafter_t		after;
    } __attribute__ ((packed)) bootsector_t;

typedef struct {
    bootbefore_FAT32_t	before;
    unsigned char	code_part[512 - sizeof(bootbefore_FAT32_t)
					- sizeof(boot1param_t)
					- sizeof (serialconf_t)
					- sizeof(bootafter_t) ];
    boot1param_t	param;
    serialconf_t        serial_configuration;
    bootafter_t		after;
    } __attribute__ ((packed)) bootsector_FAT32_t;

/**
 ** The interface to boot1 when in boot2:
 **/

/*
 * The "main()" function of boot level two is called
 * with this structure as parameter:
 * (These registers are those boot level one has found, but %%ss:%%esp !)
 */
struct registers {
	unsigned edi, esi, ebp, initsp;
	unsigned ebx, edx, ecx, eax;
	unsigned short gs, fs, aligngap, es, ds;
	unsigned short ip, cs, flags;
	} __attribute__ ((packed));

/* while we are here, these few obvious things: */
static inline int
displayconfig_changed (struct gujin_param_attrib first,
		       struct gujin_param_attrib second)
  {
  union {
      struct gujin_param_attrib bits;
      unsigned all;
      } one, two, mask = { bits: {
	lock_text_graphic: 1,
	lock_bpp: 1,
	VGA_interface: 1,
	use_gujin_embedded_font: 1,
	VESA_interface: 1,
	enable_VESA_hardwin : 1,
	VESA2_interface: 1,
	}};
  one.bits = first;
  two.bits = second;

  return (one.all & mask.all) != (two.all & mask.all);
  }

static inline int
diskconfig_changed (struct gujin_param_attrib first,
		    struct gujin_param_attrib second)
  {
  union {
      struct gujin_param_attrib bits;
      unsigned all;
      } one, two, mask = { bits: {
#define apply(name, deflt)	name : 1,
	SEARCH_APPLY_ATTRIBUTE_ORDER(apply)
	PROBE_APPLY_ATTRIBUTE_ORDER(apply)
	probe_file_in_iso_image : 1,
	IDE_in_BIOS_order : 1
	}};
  one.bits = first;
  two.bits = second;

  return (one.all & mask.all) != (two.all & mask.all);
  }

static inline int
searchconfig_changed (struct gujin_param_attrib first,
		      struct gujin_param_attrib second)
  {
  union {
      struct gujin_param_attrib bits;
      unsigned all;
      } one, two, mask = { bits: {
	search_disk_mbr:	1,
	search_part_mbr:	1,
	keep_all_part_mbr:	1,
	search_topdir_files:	1,
	search_subdir_files:	1,
	}};
  one.bits = first;
  two.bits = second;

  return (one.all & mask.all) != (two.all & mask.all);
  }

/*
 * The structure of long filenames, used only by fs.c
 * but linked to the "directory_t" structure:
 */
typedef struct {
    struct {
	unsigned char	entry	: 5;
	unsigned char	unused	: 1;
	unsigned char	last	: 1;
	unsigned char	deleted	: 1;
	} __attribute__ ((packed)) sequence;
    unsigned short		name1[5];
    struct file_attr_str	attributes;
    unsigned char		reserved; /* = 0 */
    unsigned char		checksum;
    unsigned short		name2[6];
    unsigned short		cluster; /* = 0 */
    unsigned short		name3[2];
    } __attribute__ ((packed)) long_filename_t;


extern inline unsigned
FAT_attribute_equal (struct file_attr_str attr1, struct file_attr_str attr2)
  {
  union {
      struct file_attr_str fields;
      unsigned char        all;
      } all1 = {fields: attr1}, all2 = {fields: attr2};
  return all1.all == all2.all;
  }

/*
 * Interface functions:
 */
void BOOT1_putstr (const char *str);
awk_farcall (BOOT1_putstr);

extern const char ansi2pc[];
void BOOT1_ansi_putstr (const char *str);
awk_farcall (BOOT1_ansi_putstr);

#if SETUP & MULTILINGUAL
void update_msg_exiting(void);
awk_farcall (update_msg_exiting);
#endif

#if SETUP & UPDATE_BOOTPARAM
unsigned char BOOT1_uninstall_mbr (void);
awk_farcall (BOOT1_uninstall_mbr);
unsigned char BOOT1_update_bootparam (void);
awk_farcall (BOOT1_update_bootparam);
#endif

extern const char *const recognise_MBR_string[];
unsigned recognise_MBR (const bootsector_t *bootsector);
awk_farcall (recognise_MBR);

/*
 * Few inlines, working even if called from diskcodeseg/fscodeseg/loadcodeseg/usercodeseg,
 * because code_adr() references the initial %cs segment:
 */
#if 0 /* optimise them to reduce code size */

#if SETUP & BIOS_ONLY
#define BOOT1_DOS_running()     0
#else
extern inline const unsigned char BOOT1_DOS_running (void)
  {
  extern unsigned char dospatch4adr[];

  return codeseg_peekl(&dospatch4adr) == 0x21CD4CB4;
  }
#endif

extern inline const signed short BOOT1_COM_port (void)
  {   /* < 0 if screen interface, always sign extend it ! */
  extern const unsigned short serialconf_port_plus_1;

  return codeseg_peekw(&serialconf_port_plus_1) - 1;
  }

#else /* 0 */

#if SETUP & BIOS_ONLY
#define BOOT1_DOS_running()	0
#else
#define BOOT1_DOS_running()	STATE.dos_running
#endif

#define FIRST_BOOT1_COM_port()	STATE.serial_port /* (signed)-1 for screen */

#if (USER_SUPPORT == 0) || (USER_SUPPORT & SERIAL_SUPPORT)
#define BOOT1_COM_port()	FIRST_BOOT1_COM_port()
#else
#define BOOT1_COM_port()	-1
#endif

#endif /* 0 */

extern inline void BOOT1_setcolor (unsigned char color)
  {  /* Only in graphic modes (BIOS) */
  codeseg_pokeb (STATE.color_overwrite_adr, color);
  }

extern inline unsigned getTFflagCounter (void)
  {
  extern unsigned TFflagCounter;

  return codeseg_peekl (&TFflagCounter);
  }

extern inline void setTFflagCounter (unsigned val)
  {
  extern unsigned TFflagCounter;

  codeseg_pokel (&TFflagCounter, val);
  }

extern gujin_param_t copy_gujin_param;

#if USER_SUPPORT & VGA_SUPPORT
extern inline unsigned VIDEO_mode_is_valid (unsigned char mode)
  {
  return !!(copy_gujin_param.VGA_valid_mode[mode/8] & (1 << (mode % 8)));
  }

extern inline void VIDEO_mode_invalidate (unsigned char mode)
  {
  copy_gujin_param.VGA_valid_mode[mode/8] &= ~(1 << (mode % 8));
  }

extern inline void VIDEO_mode_revalidate (unsigned char mode)
  {
  copy_gujin_param.VGA_valid_mode[mode/8] |= (1 << (mode % 8));
  }

extern inline void
VGA_get_saved_mode (unsigned index, struct vga_mode_str *mode)
  {
  *mode = copy_gujin_param.vga_mode[index];
  }

extern inline void
VGA_set_saved_mode (unsigned index, struct vga_mode_str mode)
  {
  copy_gujin_param.vga_mode[index] = mode;
  }

extern inline unsigned VIDEO_mode_max (void)
  {
#if (USER_SUPPORT & VGA_SUPPORT)
  return 8 * sizeof (copy_gujin_param.VGA_valid_mode) - 1; /* 8 bits per byte */
#else
  return 0x7F;
#endif
  }

extern inline void VIDEO_mode_initname (farptr cardname)
  {
  unsigned i;

  for (i = 0; i < nbof (copy_gujin_param.cardname); i++) {
      copy_gujin_param.cardname[i] = peekb (cardname++);
      if (copy_gujin_param.cardname[i] == '\0')
	  break;
      }
  }

extern inline int VIDEO_mode_checkname (farptr cardname)
  {
  unsigned i;

  for (i = 0; i < nbof (copy_gujin_param.cardname); i++) {
      unsigned char tmp = peekb (cardname++);
      if (copy_gujin_param.cardname[i] != tmp)
	  return 1;
      if (tmp == '\0')
	  break;
      }
  return 0;
  }

extern inline void VIDEO_mode_reset (void)
  {
  unsigned char cpt;

  for (cpt = 0; cpt < sizeof (copy_gujin_param.VGA_valid_mode); cpt++)
      copy_gujin_param.VGA_valid_mode[cpt] = 0xFF;
  for (cpt = 0; cpt <= nbof (copy_gujin_param.vga_mode); cpt++)
      copy_gujin_param.vga_mode[cpt] = (struct vga_mode_str) {};
  copy_gujin_param.default_graphic_video_mode = 0xFFFF;
  copy_gujin_param.default_text_video_mode = 0xFFFF;
  copy_gujin_param.default_video_mode = 0xFFFF;
  }

#else /* VGA_SUPPORT */
extern inline void VIDEO_mode_reset (void)
  {
  copy_gujin_param.default_graphic_video_mode = 0xFFFF;
  copy_gujin_param.default_text_video_mode = 0xFFFF;
  copy_gujin_param.default_video_mode = 0xFFFF;
  }
#endif /* VGA_SUPPORT */

#endif /* BOOT_H */
