/* boot.c */

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * I repeat, Gujin is only available under the GPL license version 2.
 *
 * Just few extracts of file GPL.txt included in this package, for you
 * and your lawyer:
 *
 * - You are not required to accept this License, since you have not
 * signed it.  However, nothing else grants you permission to modify or
 * distribute the Program or its derivative works.
 *
 * - If you cannot distribute so as to satisfy simultaneously your obligations
 * under this License and any other pertinent obligations, then as a
 * consequence you may not distribute the Program at all.
 *
 * - You may not copy, modify, sublicense, or distribute the Program
 * except as expressly provided under this License.  Any attempt
 * otherwise to copy, modify, sublicense or distribute the Program is
 * void, and will automatically terminate your rights under this License.
 *
 * So, if you still have rights under the GPL License, you are welcome
 * to continue reading, patching and distributing, else you are in violation
 * of copyright laws under the terms of the Berne Convention.
 *
 * If you have special license requirements, like if you want to base some
 * of your non-GPL work on Gujin, or want to use more than the standard
 * 512 bytes MBR interface to load your non-GPL work, you have first to
 * contact the author (me).
 *
 * Etienne Lorrain.
 */
#include "make.h"
#include "instboot.h"
#include "library.h"	/* for farptr and UPDATE_BOOTPARAM */
#include "boot.h"
#include "debug.h"	/* for applications calling boot2 interface */
#include "messages.h"

#if (__GNUC__ >= 5) && (__GNUC_MINOR__ >= 0) /* gonna be fixed by then */
#define _packed		packed, aligned(1)	// as documented in variable attributes
#else
#define _packed		aligned(1)
#endif

/* This is due to the handling of separate sections in GCC-3.4 */
#if __GNUC__ == 3 && __GNUC_MINOR__ < 4
#define ONE_TEXT_START_SECTION
#define SECTTYPE	", \"a\""
#else
#define SECTTYPE	", \"ax\""	/* warning mix code/data problem if in one single section:
			 ignoring changed section attributes for .text_start */
#endif

#ifdef ONE_TEXT_START_SECTION
#define SECTNAME(order)	".text_start"
#else
#define SECTNAME(order)	".text_start" #order
#endif

/*
 * The (false) compilation switch "DOSPATCH" is there to show
 * what to patch to have a DOS bootloader (gujin.com/gujin.exe), it
 * should not be defined (I do not know how to remove optimisation
 * "byte constant coded on 1 byte instead of 2" in "addw $deltaseg,%sp"
 */

/**
 ** This is the beginning of the boot sector
 **/
/* Default for a 1.44M floppy, anyways init'ed by instboot.c
   The loader chain and checksums are not initialised in file boot.bin,
   so boot.bin is not equivalent to gujin.com (checksums of everything
   in less than 64 Kbytes crunchs) */
/* Warning: for CDROM boots in no-emulation mode with mkisofs -boot-info-table
   parameter, the bytes between offset 8 and 0x48 are re-written.
   mkisofs just fills in interresting parameters in the first four 32bits
   words and erases all the rest to zero - so the next structure and the
   beginning of the code are erased (anybody wants to add an option to
   mkisofs just to fill in NbHiddensector?). So we will still work
   because this area is not too much used in case of CDROM boot (it uses
   the BIOS interface, not the IDE one) but do not expect the right messages.
   on screen.
   Note that I am not aware of a PC needing "mkisofs -boot-info-table option"
   because you can detect the CDROM sector to use with the CDROM BIOS/0x4B01
   and if this BIOS call is not implemented then it seems that the
   CDROM BIOS call to read next sectors just reads the following sectors
   of the file (it has just loaded the first sector so knows where it is)
   whatever the LBA given. Some BIOSes have CDROM BIOS/0x4B01 only working
   once and then (probably) unhooking the interrupt handler so the call
   disappear.
   Note that the strange order of the assembly code is due to the fact
   that a jmp with a displacement in between -128..127 bytes is coded on
   two bytes and with a bigger displacement (or a section change) is coded
   on three bytes. We are counting bytes here.
 */
/* Note that %dl is preserved all over boot1 for BIOS or EBIOS boot (only),
   but can be forced by the bootchain content and the use of
   "mov %ax,%dx = 0x89C2" instead of "mov %ah,%dh = 0x88E6".
   The CDROM BIOS interface may also set %dl register */
/* "const" to have the good section attribute for .text_startA */
static volatile const bootbefore_t bootbefore
	 __attribute__ ((section (SECTNAME(A)), _packed, attr_used)) = {
    .part1 = {
#if 1
	.jmpinstruction =	{ 0xEB, 0x6F }, /* jmp relative start -- start at 0x71 */
	.EBIOSaccess =		0x90,	/* 0x90 = nop if BIOS only; 0x0E = Win95_FAT16LBA if EBIOS accessible */
#else /* E96F00 : long form of jmp 0x71, non relative (Only then MSDOS recognise the device as B: and does not answer:
		Probing disk B: INT0x24 called with AH=0x1A, DI=0x8, BP:SI=0x70006B: sector not found) */
	.jmpinstruction =	{ 0xE9, 0x6F },
	.EBIOSaccess =		0x00,
#endif
	.String =		"Gujin1\0\0", /* HAS TO be zero terminated */
	.Bytepersector =	512, /* it can provide zero terminator of previous
				string if multiple of 256 in little endian. */
	.Sectorpercluster =	1,
	.Reservedsector =	1,	/* nb sectors before FAT */
	.NbFAT =		2,
	.NbRootdirentry =	224,
	.NbTotalsector =	2 * 18 * 80,
	.Mediadescriptor =	0xF0,
	.NbSectorperFAT =	9,
	.NbSectorpertrack =	18,
	.NbHead =		2,
	.NbHiddensector =	0,	/* nb of sectors before this partition */
	.NbTotalsector2 =	0	/* if NbTotalsector == 0 */
	},
    .part2 = {
	.PhysicaldriveNb =	0,
	.FATreserved =		0,
	.Signaturebyte0x29 =	0x29,
	.VolumeserialNb = 0x39394C45,
	.Volumelabel =	"gujin2\0\0\0\0\0", /* zero terminated */
	.FileSysName =	"FAT12   "	/* 8 char blank padded */
	}
    };

asm (
"	.section        " SECTNAME(B) SECTTYPE ",@progbits		\n"
#ifdef ONE_TEXT_START_SECTION
".org 0x3E	# end of MSDOS reserved area				\n"
"checksum_start:							\n"
#else
" checksum_start = 0x3E							\n"
#endif
"									\n"

//#define INT13SINGLE	/* read one sector at a time only for INT13/0x02 */
//$ diff /tmp/tmp1 /tmp/tmp2
//6,7c6,7
//< 0000050: 4a 6e 5a e2 f3 6e ec ec a8 81 e0 fb e3 38 a8 08  JnZ..n.......8..
//< 0000060: 74 f5 8b 4c 02 52 83 ea 07 f3 6d 5a ff 0c 75 e6  t..L.R....mZ..u.
//---
//> 0000050: 4a 2e fe 0e 0a 00 74 0b fe c7 fe c7 fe c1 b8 01  J.....t.........
//> 0000060: 02 eb 52 8b 5c 04 c3 89 5c 04 2e a2 0a 00 b0 01  ..R.\...\.......
//12c12
//< 00000b0: 44 06 89 5c 04 cd 13 73 05 31 c0 cd 13 f9 c3 ac  D..\...s.1......
//---
//> 00000b0: 44 06 e8 b2 ff cd 13 73 98 31 c0 cd 13 f9 c3 ac  D......s.1......

//#define ONLY_PRINT_GEOMETRY
//#define ONLY_PRINT_GEOMETRY_ALL_DISKS
//#define ONLY_PRINT_GEOMETRY_CDROM
/*
 * This is just a dirty patch to display (and freeze just after) the simulated
 * disk managed by the motherboard BIOS to boot from a USB (pen) hard disk.
 * We want to know which BIOS number (probably set in %dx), the size of
 * sectors and the number of cylinder/head/sectorpertrack, and if the
 * BIOS is extended, i.e. we can read using LBA.
 * The motherboard BIOS may use the header of the MBR, so we want to be
 * as close as the real Gujin boot conditions, "bootbefore" shall contain
 * the same values as we will get in the real Gujin boot.
 * Everything shall fit in 512 bytes because we may not be able to access
 * more because we need first to load the disk and geometry...
 * Only the stack is reliable now. The emulated disk may disappear
 * at the first BIOS_resetdisk or the like.
 * Note that if nothing is displayed, this disk has not been considered bootable
 * by the motherboard BIOS (maybe your motherboard BIOS can only boot
 * from a CDROM driver attached to USB?)
 * You can just do (note that the disk number 'PhysicaldriveNb' will be wrong):
 * dd if=/dev/sda of=~/save_mbr_sda bs=410 count=1
 * make clean dep boot.144 CFLAGS=-DONLY_PRINT_GEOMETRY
 * dd if=boot.144 of=/dev/sda bs=410 count=1
 * and to restore after test:
 * dd if=~/save_mbr_sda of=/dev/sda bs=410 count=1
 * Booting from this modified device will display some registers (look at INT0x13/8
 * BIOS description) and try to read with increasing sector and increasing head up
 * to an error happens. The last displayed word shall correspond to the maximum
 * value of sector (low byte) and head (high byte) which still work.
 */
#if !defined (ONLY_PRINT_GEOMETRY) && !defined (ONLY_PRINT_GEOMETRY_ALL_DISKS) && !defined (ONLY_PRINT_GEOMETRY_CDROM)
"L_waitcontinue:							\n"
"	inb	%dx,%al							\n"
"	test	$0xD0,%al	# clear carry				\n"
"	jle	L_waitdiskready	# Don't bother, I understood that for you!\n"
"	jnp	L_waitdiskready						\n"
"	mov	$2,%cx							\n"
"L_lba48_fifo:								\n"
"	push	%dx		# %dx points to status/cmd register	\n"
"	dec	%dx							\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .cylinder_high			\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .cylinder_low				\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .sector				\n"
"	dec	%dx							\n"

"SingleSectorPatch1:							\n"
#ifndef INT13SINGLE
"	outsb	%ds:(%si),%dx	# .nb_sect				\n"
"	pop	%dx							\n"
"	loop	L_lba48_fifo	# %cx null after			\n"
"	outsb	%ds:(%si),%dx	# .ide_command				\n"
"L_loopnextsector:							\n"
"	inb	%dx,%al		# short wait				\n"
"L_waitcommandcomplete:		# %cx null before			\n"
"	inb	%dx,%al							\n"
"	test	$0x81,%al	# clear carry				\n"
// end for FAT32 header
"	loopnz	L_waitcommandcomplete					\n"
"	jcxz	L_ide_error	# timeout/error/no DRQ			\n"
"	test	$0x08,%al						\n"
"	jz	L_waitcommandcomplete					\n"
"	mov	2(%si),%cx						\n"
"	push	%dx							\n"
"	sub	$7,%dx		# point to data register and clear carry\n"
"	rep insw %dx,%es:(%di)	# %cx null after			\n"
"	pop	%dx							\n"
// The command block is modified by following instruction, do not re-use
// it or use a copy... Note that .nb_sect_read is a copy of .nb_sect
"	decw	(%si)	# .nb_sect_read, %ah = 0 (but if 256 sectors)	\n"
"	jnz	L_loopnextsector					\n"
#else
"L_int13_ok:								\n"
"	decb	%cs:10							\n"
"	jz	int13done						\n"
"	inc	%bh							\n"
"	inc	%bh							\n"
"	inc	%cl							\n"
"	mov	$0x0201,%ax						\n"
"	jmp	proceed_int13						\n"
"int13done:								\n"
"	mov	4(%si),%bx						\n"
"	retw								\n"
"INT13SINGLE_preint13:							\n"
"	mov	%bx,4(%si)	# .offset	(0x89,0x5C,0x04)	\n"
"	mov	%al,%cs:10	# 'Gujin1\\0?'				\n"
"	mov	$1,%al							\n"
#endif
"	retw								\n"

#else	/* !ONLY_PRINT_GEOMETRY && !ONLY_PRINT_GEOMETRY_ALL_DISKS && !ONLY_PRINT_GEOMETRY_CDROM */

"print_quartet_in_al:							\n"
"	andw	$0x000F,%ax	# ALcarry				\n"
"	sahf			# clear AF				\n"
"	aaa								\n"
"	aad	$0x11							\n"
"	add	$0x0E30,%ax						\n"
"	int	$0x10							\n"
"	retw								\n"
"									\n"
"print_string_in_si_then_number_in_cx:					\n"
"	callw	s_puts							\n"
"print_number_in_cx:							\n"
"	pushaw								\n"
"	mov	$print_quartet_in_al,%bp				\n"
"	mov	$0x07,%bx	# display page 0, color 7 if graphic mode\n"
"	mov	%cx,%ax							\n"
"	shr	$12,%ax							\n"
"	callw	*%bp							\n"
"	mov	%cx,%ax							\n"
"	shr	$8,%ax							\n"
"	callw	*%bp							\n"
"	mov	%cx,%ax							\n"
"	shr	$4,%ax							\n"
"	callw	*%bp							\n"
"	mov	%cx,%ax							\n"
"	callw	*%bp							\n"
"	popaw								\n"
"	retw								\n"
"	nop								\n"

#endif	/* !ONLY_PRINT_GEOMETRY && !ONLY_PRINT_GEOMETRY_ALL_DISKS  && !ONLY_PRINT_GEOMETRY_CDROM */

"									\n"
"									\n"
"start:									\n"
"	# Do not know what is CS and IP, usual hardware address 0007Cxx,\n"
"	# i.e. 07C0:00xx or 0000:7Cxx for cold boot; but if we are	\n"
"	# building a DOS bootloader we cannot use absolute values...	\n"
"	# The stack segment will be defined once for the whole loader	\n"
"	# here to be able to save everything, the stack offset will be	\n"
"	# set to zero to get a maximum working space.			\n"
"	# N.B.  push: predecrement and write, pop: read and postincr	\n"
"	# Note: start has to be aligned like boot1param.boot1_checksum	\n"
"									\n"
"	cli								\n"
#if defined (ONLY_PRINT_GEOMETRY) || defined (ONLY_PRINT_GEOMETRY_ALL_DISKS)
"	mov	%ss,%cx							\n"
"	mov	%sp,%bx							\n"
#endif
"	mov	%cs,%sp							\n"
"									\n"
#ifndef DOSPATCH
/*
 * Arggg... CDROM El-Torito booting, loader like SYSLINUX want 4 sectors
 * of 2048 bytes (CDROM sectors and not 512 bytes disk sectors) at 0x7C00,
 * and to simplify some BIOS boot CDROM jumping at address 0x0000:0x7C00,
 * that is %cs = 0. Then we do not have these 8 Kbytes space to load
 * CDROM booting in no emulation mode if we use 0x840.
 * Lets use 0xBC0 here to enable Gujin to load 8 2048 bytes sectors.
 * Adjust also the value verified by instboot at dospatch2adr.
 * Grub El-torito loader needs 30 sectors (not all PC BIOS can do that),
 * so leave again more space: 32 sectors so 0x17C0 instead of 0xBC0.
 */
"									\n"
"dospatch2adr = . + 2							\n"
#if SETUP & CODE_SEGMENT
"	addw	$0x17C0+deltaseg,%sp	# far away, even if %cs is null	\n"
#else
"	addw	$0x17C0,%sp		# far away, even if %cs is null	\n"
#endif
"									\n"
#else
"									\n"
"	# do not really move the software (debugger):			\n"
"	# HERE (0x10+deltaseg) has to be a TWO BYTES constant,		\n"
"	# even if (deltaseg == 0)!					\n"
"#	addw	$0x10+deltaseg,%sp	# ONLY for gujin.com files	\n"
"	addw	$deltaseg,%sp		# ONLY for gujin.exe/.PIC files	\n"
"									\n"
#endif /* DOSPATCH */
"									\n"
"	mov	%sp,%ss							\n"
"	xor	%sp,%sp							\n"
"	sti								\n"
"									\n"

#if defined (ONLY_PRINT_GEOMETRY) || defined (ONLY_PRINT_GEOMETRY_ALL_DISKS) || defined (ONLY_PRINT_GEOMETRY_CDROM)
#ifdef ONLY_PRINT_GEOMETRY_CDROM
"ljmpw	$0x7C0,$1f							\n"
"hello_msg: .asciz	\"\\r\\nInitial DX= 0x\"					\n"
"int13_msg: .asciz	\"\\r\\nINT13/4B01: carry \"					\n"
"	1:								\n"
"	pushw	%cs							\n"
"	popw	%ds							\n"
"	pushw	%cs							\n"
"	popw	%es							\n"
"	pushw	%dx							\n"

"	mov	$hello_msg,%si						\n"
"	mov	%dx,%cx							\n"
"	mov	$print_string_in_si_then_number_in_cx,%bp			\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	$0x4B01,%ax						\n"
"	mov	$0x200,%si						\n"
"	int	$0x13							\n"
"	setc	%cl							\n"
"	mov	$0,%ch							\n"
"	mov	$int13_msg,%si						\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	$0x200,%bx						\n"
"	2:								\n"
"	mov	(%bx),%cl						\n"
"	mov	$hello_msg + 13,%si					\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	inc	%bx							\n"
"	cmp	$0x213,%bx						\n"
"	je	.							\n"
"	jmp	2b							\n"
"	.long 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0			\n"
"	.long 0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0				\n"

#else /* ONLY_PRINT_GEOMETRY_CDROM */

"ljmpw	$0x7C0,$1f						\n"
"hello_msg: .ascii	\"\\r\\nInitial\"	# not zero ended	\n"
"reg_msg: .asciz	\" DX= 0x\"					\n"
"stack_msg: .asciz	\" SS= 0x\"					\n"
"int13_msg: .asciz	\"\\r\\nINT13/08:\"				\n"
"head_sect_msg: .asciz	\", C/H/S: 00\"					\n"
"	1:								\n"
"	pushw	%cs							\n"
"	popw	%ds							\n"
"	pushw	%cs							\n"
"	popw	%es							\n"
"	pushw	%dx							\n"

"	mov	$hello_msg,%si						\n"
"	xchgw	%dx,%cx							\n"
"	mov	$print_string_in_si_then_number_in_cx,%bp		\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	xchgw	%dx,%cx							\n"
"	mov	$stack_msg,%si						\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	movb	$'P',stack_msg+2					\n"
"	mov	%bx,%cx							\n"
"	mov	$stack_msg,%si						\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	$int13_msg,%si						\n"
"	callw	s_puts							\n"
"	mov	$0x08,%ah						\n"
"	int	$0x13							\n"
"	setc	%bh		# CARRY seen in %bh			\n"
"8:									\n"
"	mov	$reg_msg,%si						\n"
"	xchgw	%dx,%cx							\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	xchgw	%dx,%cx							\n"
"	mov	$reg_msg,%si						\n"
"	decb	1(%si)							\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	%bx,%cx							\n"
"	mov	$reg_msg,%si						\n"
"	decb	1(%si)							\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	%ax,%cx							\n"
"	mov	$reg_msg,%si						\n"
"	decb	1(%si)							\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	%di,%cx							\n"
"	mov	$reg_msg,%si						\n"
"	movw	$('D' + ('I' << 8)),1(%si)				\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"

#ifdef ONLY_PRINT_GEOMETRY_ALL_DISKS
"	mov	$crlf,%si						\n"
"	callw	s_puts							\n"
"	mov	$0x10,%ah						\n"
"	int	$0x16							\n"
"	mov	$reg_msg,%si						\n"
"	movw	$('D' + ('X' << 8)),1(%si)				\n"
"9:									\n"
"	incb	cpt							\n"
"	jz	.							\n"
"	popw	%dx							\n"
"	inc	%dx							\n"
"	mov	$0,%dh							\n"
"	pushw	%dx							\n"
"	mov	%dx,%cx							\n"
"	mov	$reg_msg,%si						\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	mov	$0x08,%ah						\n"
"	stc								\n"
"	int	$0x13							\n"
"	jc	9b							\n"
"	jmp	8b							\n"
"cpt:	.byte 0								\n"
"crlf:	.asciz \" press key\\r\\n\"		\n"
"	nop				\n"

#else /* ONLY_PRINT_GEOMETRY_ALL_DISKS */
"	nop								\n"
//"	xor	%ax,%ax	;	int	$0x13				\n"

/* confirm the number of sector per track and heads by reading
   increasing sector C/H/S up to the BIOS error: */
"	popw	%dx							\n"
"	mov	$int13_msg,%si						\n"
"	movb	$'2',int13_msg+9					\n"
"	callw	s_puts							\n"
"	mov	$0,%dh							\n"
//"	mov	$0x0000,%cx						\n"
"	xor	%cx,%cx							\n"
//"	mov	$0x0003,%cx						\n"
"next_sector:								\n"
"	inc	%cl							\n"
"	callw	read_sector						\n"
"	jnc	next_sector						\n"
"	dec	%cl							\n"
"	mov	$error_msg,%si						\n"
"	callw	s_puts							\n"
"	callw	s_puts	# just after is a crlf				\n"
"next_head:								\n"
"	inc	%dh							\n"
"	callw	read_sector						\n"
"	jnc	next_head						\n"
"	dec	%dh							\n"
"	mov	$error_msg,%si						\n"
"	callw	s_puts							\n"
"	callw	s_puts	# just after is a crlf				\n"
"	callw	read_sector						\n"
"	1:								\n"
"	rep	\n	nop						\n"
"	jmp	1b							\n"
#endif /* ONLY_PRINT_GEOMETRY_ALL_DISKS */
"									\n"
"read_sector:								\n"
"	mov	$head_sect_msg,%si					\n"
"	xchg	%dh,%ch							\n"
"	callw	*%bp	# print_string_in_si_then_number_in_cx		\n"
"	xchg	%dh,%ch							\n"
"	pushw	%cs							\n"
"	popw	%es							\n"
"	mov	$0x0201,%ax						\n"
"	mov	$0x0200,%bx	# %es == %cs, 0x07E00 : no DMA problem	\n"
"	int	$0x13							\n"
"	nop								\n"
"	retw								\n"
#endif /* ONLY_PRINT_GEOMETRY_CDROM */

"s_puts:	# parameter: zero ended string at %ds:%si		\n"
"	lodsb	%ds:(%si),%al						\n"
"	cmp	$0,%al		# instruction coded on 2 bytes	(aad $0)\n"
"	je	L_s_puts_end						\n"
"	pushaw			# %dx, %bp and %bx would be a minimum	\n"
"	mov	$0x07,%bx	# display page 0, color 7 if graphic mode\n"
"	mov	$0x0E,%ah						\n"
"	int	$0x10		# int 0x10 may destroy %bp if scroll...	\n"
"	popaw	# take care of popa bug on i386				\n"
"	jmp	s_puts							\n"
"L_s_puts_end:								\n"
"	retw								\n"

"USBHDD_patch:			\n"
"SingleSectorPatch1:		\n"
"SingleSectorPatch2:		\n"
"partition_relative_read_disk:	\n"
"read_disk:	lodsw ; movw %ax,%dx	# need retw just before	\n"
"serialconf_port_plus_1:	\n"
"simple_puts:			\n"
"color_overwrite:		\n"
"bootloader2_return:		\n"
"detect_usbhdd_patch:		\n"

#else	/* ONLY_PRINT_GEOMETRY || ONLY_PRINT_GEOMETRY_ALL_DISKS || ONLY_PRINT_GEOMETRY_CDROM */

"	# save everything in 16 bit, top of stack with 'iretw' format	\n"
"									\n"
"	pushfw								\n"
"	pushw	%cs	# 'pushw %cs; callw puship' = 'lcallw cs:puship'\n"
"	callw	L_puship	# push	%ip				\n"
"L_just_after_puship:							\n"
"									\n"
"#********************************************************************	\n"
"#* What you were looking for - REMEMBER IT IS PROTECTED BY GPL -    *	\n"
"#* see top of this file. Copyrighted by Etienne LORRAIN, 1999-2013. *	\n"
"#*                                                                  *	\n"
"#*   YOU ARE NOT AUTORISED TO READ THAT CODE BEFORE HAVING READ     *	\n"
"#*    AND UNDERSTOOD THE GPL, i.e. the General Public License.      *	\n"
"#* See http://www.gnu.org/copyleft/gpl.html to read the GPL or      *	\n"
"#*              its translation in your language.                   *	\n"
"#********************************************************************	\n"
"# Reading IDE status:							\n"
"# b0: general error, b1: 0, b2: data successfully corrected,		\n"
"# b3: data request, b4: disk ready, b5: write fault, b6: IDE ready,	\n"
"# b7: IDE busy (no bit valid)						\n"
"# More info on Linux file hd.c						\n"
"L_ide_hard:			# access IDE directly			\n"
//#define USBHDD_PATCH
"USBHDD_patch:								\n"
#ifdef USBHDD_PATCH
//$ diff /tmp/boot.xxd /tmp/boot_init.xxd
//< 0000080: 5c00 9008 f675 0e2e 8a36 1a00 08ed 7503  \....u...6....u.
//< 0000090: 80e9 40fe cdfe cec3 a11c 0001 440c a11e  ..@.........D...
//< 00000a0: 0011 440e ad89 c2ad 8b0c 80fc f074 d3e8  ..D..........t..
//< 00000b0: d0ff 895c 04cd 1373 0531 c0cd 13f9 c3ac  ...\...s.1......
//---
//> 0000080: 5c00 89df eead 89c2 31c9 eca8 80e0 fbe3  \.......1.......
//> 0000090: 054a 6e42 e2a8 f9c3 a11c 0001 440c a11e  .JnB........D...
//> 00000a0: 0011 440e ad89 c2ad 8b0c 80fc f074 d38c  ..D..........t..
//> 00000b0: 4406 895c 04cd 1373 0531 c0cd 13f9 c3ac  D..\...s.1......
//28c28
//< 0000190: 4552 524f 5221 000d 0a00 f36e c603 c008  ERROR!.....n....
//---
//> 0000190: 4552 524f 5221 000d 0a00 a9c2 c603 c008  ERROR!..........
// Last diff line is checksum at 0x19A (get it with instboot)
// First difference at 0x82: "\x89\xdf\xee\xad\x89\xc2\x31\xc9\xec\xa8\x80\xe0\xfb\xe3\x05\x4a\x6e\x42\xe2\xa8\xf9"
//                       to: "\x90\x08\xf6\x75\x0e\x2e\x8a\x36\x1a\x00\x08\xed\x75\x03\x80\xe9\x40\xfe\xcd\xfe\xce"
// Second difference at 0xAF: "\x8c\x44\x06" to : "\xe8\xd0\xff"
//"	mov	%es,6(%si)	# .segment	(0x8C,0x44,0x06)	\n"
"	nop								\n"
"	or	%dh,%dh							\n"
"	jnz	no_change_cylinder					\n"
"	mov	%cs:0x001A,%dh		# heads				\n"
"	or	%ch,%ch							\n"
"	jnz	no_dec_cly_msb						\n"
"	sub	$64,%cl							\n"
"no_dec_cly_msb:							\n"
"	dec	%ch							\n"
"no_change_cylinder:							\n"
"	dec	%dh							\n"
"	retw								\n"
"L_waitdiskready = . - 4;						\n"
"L_ide_error = . - 2;							\n"
#else
"	mov	%bx,%di		# %bx should not be modified in this fct\n"
"	outb	%al,%dx		# %al = 0x0A, %dx = 0x3F6 (for instance)\n"
"	lodsw	%ds:(%si),%ax	# .base_ide_plus7 (that is already in	\n"
"				# cx, but it also increment %si and	\n"
"				# initialise %ah=1 !)			\n"
"	mov	%ax,%dx		# for instance 0x1F7 (cmd/status register)\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms		\n"
"L_waitnotbusy:								\n"
"	inb	%dx,%al							\n"
"	test	$0x80,%al	# busy					\n"
"	loopnz	L_waitnotbusy						\n"
"	jcxz	L_ide_error	# timeout: %ah=MSB(ideaddr), often 1	\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .lba_head				\n"
"	inc	%dx							\n"
"L_waitdiskready:							\n"
"	loop	L_waitcontinue	# timeout BUSY or !DRDY or !RDY: %ah = 1\n"
"L_ide_error:								\n"
"	stc								\n"
"	retw	# retw before partition_relative_read_disk label	\n"
#endif
"									\n"
"									\n"
"	# for EBIOS only, add NbHiddensector to be independant		\n"
"	# of the position of the partition on the disk & CDROM used.	\n"
"	# Could be used for IDE/LBA48 by changing 12(%si) and 14(%si)	\n"
"	#  and for IDE/LBA by also changing mov $2,%cx to mov $1,%cx	\n"
// NOTE: this changes bootchain[] so take care for self install...
"partition_relative_read_disk:						\n"
"	movw	bootbefore_part1_NbHiddensector,%ax			\n"
"	addw	%ax,12(%si)						\n"
"	movw	2+bootbefore_part1_NbHiddensector,%ax			\n"
"	adcw	%ax,14(%si)						\n"
"# Parameters: %es:%bx points to buffer, %ds:%si points to the union	\n"
"read_disk:			# BIOS INT13 %ah=0x02			\n"
"	lodsw	%ds:(%si),%ax	# .reg_dx / .ide_dcr_address		\n"
"	mov	%ax,%dx							\n"
// mov %ax,%dx = 0x89C2 ; mov %ah,%dh = 0x88E6 only for BIOS or EBIOS
"	lodsw	%ds:(%si),%ax	# .reg_ax				\n"
"	mov	(%si),%cx	# .reg_cx / .nb_word_to_read (unused 13/42)\n"
"	cmp	$0xF0,%ah						\n"
"	je	L_ide_hard						\n"
"	#cmp	$0x42,%ah	We do not really care to modify memory	\n"
"	#jne	proceed_int13	at 6(%si) and 4(%si) for int1302...	\n"
"	#								\n"
"	#  hope this INT13 %ah=42 keeps %es:%bx				\n"
"	#  here  %si==&(bootloader2.int1342.cst_16)			\n"
"detect_usbhdd_patch:							\n"
#ifndef USBHDD_PATCH
"	mov	%es,6(%si)	# .segment	(0x8C,0x44,0x06)	\n"
#else
"	callw	USBHDD_patch						\n"
#endif
"SingleSectorPatch2:							\n"
#ifndef INT13SINGLE
"	mov	%bx,4(%si)	# .offset	(0x89,0x5C,0x04)	\n"
#else
"	callw	INT13SINGLE_preint13					\n"
#endif
"proceed_int13:								\n"
"	int	$0x13							\n"
"	jnc	L_int13_ok						\n"
"init_bios_disk:							\n"
"	xor	%ax,%ax							\n"
"	int	$0x13		# reset disks (known by BIOS)		\n"
"	stc			# shall remember the error...		\n"
#ifndef INT13SINGLE
"L_int13_ok:								\n"
#endif
"	retw								\n"
"									\n"
"									\n"
"# register %di is clearly modified by the BIOS of 3DLabs Permedia 2,	\n"
"# VIPER II AGP Rev-2.1.0 product P2-85 in VESA modes... (hwinfo.exe)	\n"
"# Result is the same as Trident/SVGA video mode/%ds : crash when scroll\n"
"#									\n"
"# The video BIOSes are just crap these days, execute a 'pusha', but	\n"
"# after having realigned the stack which is currently only on two	\n"
"# bytes boundary.							\n"
"simple_puts:		# parameter: zero ended string at %ds:%si	\n"
"	lodsb	%ds:(%si),%al						\n"
"	cmp	$0,%al		# instruction coded on 2 bytes	(aad $0)\n"
"	je	L_simple_puts_end					\n"
"	pushw	%ds							\n"
"	pushaw			# %dx, %bp and %bx would be a minimum	\n"
"serialconf_port_plus_1 = . + 1	# then no need of %cs prefix		\n"
"	movw	$0,%dx		# 0 for screen, 1 for COM1 ...		\n"
"	dec	%dx							\n"
"	js	L_screen_output	# jmp if sign bit set			\n"
"	mov	$0x01,%ah	# Max 256 COM ports in %dx here		\n"
"	int	$0x14		# output char				\n"
"	jmp	L_was_serial_output					\n"
"L_screen_output:							\n"
"	mov	$0x07,%bx	# display page 0, color 7 if graphic mode\n"
"color_overwrite = . - 2;						\n"
"	mov	$0x0E,%ah						\n"
"	int	$0x10		# int 0x10 may destroy %bp if scroll...	\n"
"L_was_serial_output:							\n"
"	popaw	# take care of popa bug on i386				\n"
"	popw	%ds							\n"
"	jmp	simple_puts						\n"
"L_simple_puts_end:							\n"
"	retw								\n"
"									\n"
"									\n"
"L_puship:								\n"
"	pushw	%ds							\n"
"	pushw	%es							\n"
"	pushaw		# push ax, cx, dx, bx, initsp, bp, si, di (16 bytes)	\n"
"	cld		# needed for lods[bw] (post-increment)		\n"
"									\n"
"	# normalise %cs and setup %ds,%es				\n"
"	mov	%sp,%bp	# %sp cannot be indexed (%bp used with %ss)	\n"
"	# if not modified, 'iretw' will restart all:			\n"
"	subw	$L_just_after_puship,20(%bp)				\n"
"	mov	20(%bp),%ax	# 20(bp) is the pushed %ip (callw nextip)\n"
"	shr	$4,%ax		# segment/offset for 80x86 real mode	\n"
"	add	22(%bp),%ax	# the pushed %cs			\n"
"	mov	%ax,%ds							\n"
"	mov	%ax,%es		# just for 'read_disk'			\n"
"	push	%ax							\n"
"	pushw	$L_normalise_jump					\n"
"	lretw								\n"
"	inc	%bp							\n"
"	dec	%sp							\n"
"L_normalise_jump:							\n"
"	# What is the output system (serial/screen) ?			\n"
"	pushw	%dx							\n"
"	movw	serialconf_port_plus_1,%dx	# cs = ds from before	\n"
"	dec	%dx		# initialise serial port COM%dx+1	\n"
"	js	L_noserialbios	# jmp if sign bit set			\n"
"	mov	password_serial,%ax					\n"
"	int	$0x14		# init serial line: %ah==0 %al=params	\n"
"L_noserialbios:							\n"
"	popw	%dx							\n"
"	movw	$simple_puts,%bp	# save few bytes, %bp constant	\n"
"	movw	$crlf_msg,%si		# parameter in %ds:%si		\n"
"	callw	*%bp			# callw	simple_puts		\n"
"	movw	$boot1_msg,%si						\n"
"	callw	*%bp			# callw	simple_puts		\n"
"									\n"
"	xor	%cx,%cx							\n"
"	movw	$checksum_start,%si	# i.e. 0x3E			\n"
"L_boot1_checksum_loop:							\n"
"	lodsw	%ds:(%si),%ax						\n"
"	add	%ax,%cx							\n"
"	cmpw	$bootend,%si						\n"
"	jl	L_boot1_checksum_loop					\n"
"	jcxz	L_boot1_checksum_ok					\n"
"	movw	$checksum_msg,%si					\n"
"	callw	*%bp			# callw	simple_puts		\n"
"	callw	*%bp		# print the string declared just after	\n"
"L_boot1_checksum_ok:							\n"
"					# %es already setup		\n"
"	movw	$loadnextsector,%bx	# checksum/load just after boot1\n"
"	pushw	$bootloader2						\n"
"	movw	$comma_msg,%si						\n"
"	callw	*%bp			# callw	simple_puts		\n"
"	mov	$boot2_msg,%si						\n"
"	jmp	L_loop_loader_entry					\n"
"									\n"
"L_error_checksum:							\n"
"	movw	$checksum_msg,%si					\n"
"	callw	*%bp			# callw	simple_puts		\n"
"L_error_disk:								\n"
"	movw	$error_msg,%si						\n"
"bootloader2_return:							\n"
"	callw	*%bp			# callw	simple_puts		\n"
"	pop	%si			# realign stack			\n"
"									\n"
#endif	/* ONLY_PRINT_GEOMETRY || ONLY_PRINT_GEOMETRY_ALL_DISKS || ONLY_PRINT_GEOMETRY_CDROM*/
#ifndef DOSPATCH
"									\n"
"L_retry_all:								\n"
#if 0 // !(SETUP & BOOT_NEVER_RETRY)
"	# 4 bytes, that is 0xCF1F0761					\n"
"dospatch4adr = .							\n"
"	popaw	# take care of popa bug on i386				\n"
"	popw	%es							\n"
"	popw	%ds							\n"
"	iretw								\n"
#else
"	# 4 bytes, that is 0x18CD19CD					\n"
"dospatch4adr = .							\n"
"	int	$0x19							\n"
"	int	$0x18							\n"
#endif
"									\n"
#else
"									\n"
"	# 4 bytes, that is 0x21CD4CB4					\n"
"	mov	$0x4C,%ah						\n"
"	int	$0x21							\n"
#endif
"									\n"
"L_loop_loader:								\n"
#if !defined (ONLY_PRINT_GEOMETRY) && !defined (ONLY_PRINT_GEOMETRY_ALL_DISKS) && !defined (ONLY_PRINT_GEOMETRY_CDROM)
"	pushw	(%si)			# .next				\n"
"	lodsw	%ds:(%si),%ax		# same as 'add $2,%si'		\n"
"	lodsw	%ds:(%si),%ax						\n"
"	push	%ax			# .checksum			\n"
"	lodsw	%ds:(%si),%ax						\n"
"	push	%ax			# .nbword			\n"
#endif
"									\n"
#ifndef DOSPATCH
"									\n"
"dospatch3adr = . + 1							\n"
"	callw	read_disk						\n"
"									\n"
#else
"	# carry cleared here so we cal call a ret			\n"
"	callw	partition_relative_read_disk-1	# which is a ret	\n"
"	# callw	partition_relative_read_disk	# if instboot option	\n"
"	# callw read_disk_1st	# for CDROM noemul			\n"
"									\n"
#endif
#if !defined (ONLY_PRINT_GEOMETRY) && !defined (ONLY_PRINT_GEOMETRY_ALL_DISKS) && !defined (ONLY_PRINT_GEOMETRY_CDROM)
"									\n"
"	pop	%cx			# .nbword			\n"
"	pop	%ax			# .checksum			\n"
"	jc	L_error_disk		# I: %si, %es:%bx /O: %es:%bx	\n"
"L_boot2_checksum_loop:							\n"
"	inc	%bx							\n"
"	inc	%bx							\n"
"	addw	%es:-2(%bx),%ax						\n"
"	loop	L_boot2_checksum_loop	# this do not modify flags	\n"
"	jne	L_error_checksum					\n"
"	testw	%bx,%bx							\n"
"	jnz	L_noincseg						\n"
"	mov	%es,%cx							\n"
"	addb	$0x10,%ch						\n"
"	mov	%cx,%es							\n"
"L_noincseg:								\n"
"	movw	$dot_msg,%si						\n"
"L_loop_loader_entry:							\n"
"	callw	*%bp			# callw	simple_puts		\n"
"	pop	%si							\n"
"	testw	%si,%si			# clears carry			\n"
"	jnz	L_loop_loader						\n"
"									\n"
"	movw	$go_msg,%si						\n"
"	callw	*%bp			# callw	simple_puts		\n"
"#									\n"
"#	Hey Bit?							\n"
"#	Yes!								\n"
"#	Do you think we can merge with this memory?			\n"
"#	Yes!								\n"
"#									\n"
"	jmp	bootloader2_entrypoint					\n"
"									\n"
#endif /* !ONLY_PRINT_GEOMETRY && !ONLY_PRINT_GEOMETRY_ALL_DISKS && !ONLY_PRINT_GEOMETRY_CDROM */
	);

static const char comma_msg[] __attribute__ ((section (SECTNAME(C)), _packed, attr_used))
	= ", ";

static const char checksum_msg[] __attribute__ ((section (SECTNAME(D)), _packed, attr_used))
	= ": chksum";	/* position dependancy with next msg */

static const char error_msg[] __attribute__ ((section (SECTNAME(E)), _packed, attr_used))
	= " ERROR!";	/* position dependancy with previous and next msg */

static const char crlf_msg[] __attribute__ ((section (SECTNAME(F)), _packed, attr_used))
	= "\r\n";	/* position dependancy with next msg */

/*
 * If you are using a debugger to get the checksum, take care:
 * it calculates the checksum with the soft breakpoint instruction added!
 * The checksum is set so as to have sum of words between (included)
 * 0x03E and (included) 0x1B6 equals zero, and is updated by 'instboot.c'
 * to take care of 'deltaseg' which is a 32 bit number known at link time.
 */
#ifdef ONE_TEXT_START_SECTION
asm (" .org 0x19A ");
#endif

//static // main.c displays "%cs:boot1param.bootloader2.bootloader2_cmd.int1342.reg_dx"
volatile const boot1param_t boot1param __attribute__ ((section (SECTNAME(G)), _packed, attr_used))
	= {
#if 0 //!(SETUP & BOOT_NEVER_RETRY)
		.boot1_checksum = 0x7511,
#else
		.boot1_checksum = 0x18E7,
#endif
	};

#ifdef ONE_TEXT_START_SECTION
asm (".org 0x1B6	# Max end is 0x1B6 for some kind of password ?	\n");
#endif

/* See vmlinuz.c to change this: */
volatile const union {
    unsigned short	some_kind_of_password;
    serialconf_t	serial_configuration;
    } password_serial __attribute__ ((section (SECTNAME(H)), _packed, attr_used))
    = {0};

#ifdef ONE_TEXT_START_SECTION
asm (".org 0x1B8	# Max end is 0x1B8 for Windows NT marker + partition tables \n");
#endif


static volatile const bootafter_t bootafter __attribute__ ((section (SECTNAME(I)), _packed, attr_used)) = {
	.Signature0xAA55 = 0xAA55
	};

/**
 ** This is the beginning of the second level bootloader:
 **/

#ifdef ONE_TEXT_START_SECTION
asm (".org 0x200	# no more, that is the end of boot1 / start of boot2.	\n");
#endif

#ifdef ONE_TEXT_START_SECTION
asm ("loadnextsector:		# address hardcoded in instboot.c	\n");
#else
asm ("loadnextsector = 0x200	# address hardcoded in instboot.c	\n");
#endif

extern const instboot_info_t instboot_info;
extern unsigned char color_overwrite;

static const instboot_info_t *const
	__attribute__ ((section (SECTNAME(J)), _packed, attr_used))
	instboot_info_ptr = &instboot_info;


asm (
"	.section        " SECTNAME(K) SECTTYPE ",@progbits		\n"
"bootloader2_entrypoint:						\n"
// " mov $0x0100,%ax ; int $0xE6 # DOSEMU dump registers \n"
"	# some output to know boot2 has been reached			\n"
"	movw	$runloader2_msg,%si					\n"
"	callw	*%bp		# callw	simple_puts			\n"
"	jmp	bootloader2_continue					\n"
"									\n"
"runloader2_msg:							\n"
"	.ascii	\"\\r\\nGujin v2.8.7 \"		\n" /*VERSION*/
"runloader2_endmsg:							\n"
"	sub	%al,0x29(%bp,%di)					\n"
"	and	%al,0x74(%di)						\n"
"	imul	$0x656E,0x6E(%di),%sp					\n"
"	and	%cl,0x4F(%si)						\n"
"	push	%dx							\n"
"	push	%dx							\n"
"	inc	%cx							\n"
"	dec	%cx							\n"
"	dec	%si							\n"
"	and	%dh,(%bp,%si)						\n"
"	xor	%dh,(%bx,%di)						\n"
"	.byte	0x33, 0x2e, 0x0d, 0x0a, 0, 0				\n"
"	.align 2							\n"
	);

/*
 * The variable up to 'gujin_param' can be modified by the boot level 2,
 * to save default state when outside of any operating system,
 * because it is in THE sector described in boot1 (easy access).
 */
gujin_param_t volatile const gujin_param	/* aligned for checksum calculus */
	__attribute__ ((section (SECTNAME(L)), aligned (2), _packed)) = {
    .magic   = 16980327,
    .version = 0x0208,	/*VERSION*/ /* to abort ./instboot on a wrong version */
    .compilation_attrib = {
	.vga_support = !!(USER_SUPPORT & VGA_SUPPORT),
	.big_malloc  = !!(SETUP & BIG_MALLOC),
	.bios_only   = !!(SETUP & BIOS_ONLY),
#if defined (INITRDNAME) || defined (INITRDNAME2)
	.initrd_loader = 1,
#else
	.initrd_loader = 0,
#endif
#if defined (LINUZNAME1) || defined (LINUZNAME2) || defined (LINUZNAME3)
	.vmlinuz_loader = 1,
#else
	.vmlinuz_loader = 0,
#endif
#ifdef KERNELEXT
	.elf_loader = 1,
#else
	.elf_loader = 0,
#endif
#ifdef GRUB_MULTIBOOT
	.multiboot_loader = 1,
#else
	.multiboot_loader = 0,
#endif
#if SETUP & MULTILINGUAL
	.multilingual = 1,
#endif
#if SETUP & UNICODE_FONT
	.unicode = 1,
#endif
	},
    .dot_msg = ".",	/* 3 bytes */
    .MltStrLanguage = 0,	/* MLTSTR_ENGLISH */
    .go_msg  = " go!",	/* 8 bytes, a space before */
    .scanpath = DEFAULTSEARCHPATH,	/* directory name to scan for kernels, case insensitive, default "boot" */
    /* The following can be moved without changing checksum of BOOT1: */
    .default_video_mode = 0xFFFF,
    .default_text_video_mode = 0xFFFF,
    .default_graphic_video_mode = 0xFFFF,
    .stop_emulation = 1, /* stop emulation of CDROM FD/HD early */
    .attrib = {
	/* Can be changed and saved at execution time: */
#define Apply(field, deflt)	.field = deflt,
	APPLY_ATTRIBUTE_ORDER(Apply)
#undef Apply
	.write_disabled_written_once = 0
	},
    .kbdmap = kbd_unknown,
    .extra_cmdline = "",

#if USER_SUPPORT & VGA_SUPPORT
    .VGA_valid_mode = {
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF
	},
    .vga_mode = {{0}},
#endif

    .partial_checksum = 0
    };
 /* This is the end of config variables */

#if DISK_SUPPORT & CDBIOS_SUPPORT
/*
 * This small assembly function detects which BIOS number the bootable
 * CDROM is emulating - and so can be removed if boot.elf is not meant
 * to boot from CDROM. Because CDROMs have a 2Kbytes sector size,
 * this code is already loaded in boot1 - and the first bootchain load does
 * not need to do anything (only checksum verification of bytes 512..2048).
 * Unfortunately this code is not checksumed _before_ being used...
 */
/* Exact mapping of mkisofs -boot-info-table in decimal, little endian:
 *	offset  8: LBA of primary volume descriptor
 *	offset 12: LBA of boot file	(the one we use)
 *	offset 16: boot file length in bytes	(we already know that)
 *	offset 20: 32 bits checksum	(we already check the 16 bits checksum)
 *	offset 24..64 : reset to zero	(erases "Gujin2" string)
 *  32 bits checksum is for all the words after offset 64 = 0x40.
 *  CD LBA sectors are 2048 bytes length
 */
/* "	movb	%ss:0xFFF0,%dl	# initial %dl in boot1	\n" */

asm (
"	.section        " SECTNAME(M) SECTTYPE ",@progbits			\n"
"read_disk_1st:									\n"
"	# %di points to boot1param.bootloader2.bootloader2_cmd.int1342.reg_dx	\n"
"	mov	%si,%di								\n"
"	lea	0x4000(%bx),%si	# free area of memory, after current 2K sector	\n"
"	xor	%dh,%dh								\n"
"	movb	%dl,3(%si)	# %dl is preserved from boot			\n"
"	movw	12,%ax		# if INT13/0x4B01 not implemented		\n"
"	movw	%ax,4(%si)	# set default sector loaded as			\n"
"	movw	14,%ax		# a mkisofs -boot-info-table would		\n"
"	movw	%ax,6(%si)	# have set them					\n"
"										\n"
"	pushaw									\n"
"	mov	$bootchain+520,%si	# &bootchain[20]			\n"
"	lodsw	%ds:(%si),%ax	# restore part overwritten by -boot-info-table	\n"
"	cmpw	$0xFEED,%ax	# if instboot has set the magic value		\n"
"	jne	1f								\n"
"	lodsw	%ds:(%si),%ax							\n"
"	mov	%ax,%di								\n"
"	lodsw	%ds:(%si),%ax							\n"
"	mov	%ax,%cx		# cx will be poped from stack			\n"
"	rep	movsw %ds:(%si),%es:(%di)					\n"
"	1:									\n"
"	popaw									\n"
"retry:										\n"
"	movb	$0x13,(%si)							\n"
"	mov	$0x4B01,%ax							\n"
"	int	$0x13								\n"
"	jnc	success								\n"
"	dec	%dl								\n"
"	inc	%dh								\n"
"	jnz	retry		# max 256 tries					\n"
"success:									\n"
"	pushw	%ds								\n"
"	pushw	$0								\n"
"	popw	%ds								\n"
"	movb	$0,0x474		# status of last hard disk operation	\n"
"	popw	%ds								\n"
"										\n"
"	lodsw	%ds:(%si),%ax							\n"
"	lodsw	%ds:(%si),%ax							\n"
"	mov	%al,%dl		# this new %dl will be used from now on		\n"
"	movb	%al,(%di)	# save for reference & boot1 parameters		\n"
"	movw	$bootbefore_part1_NbHiddensector,%di				\n"
"	movsw	%ds:(%si),%es:(%di)						\n"
"	movsw	%ds:(%si),%es:(%di)						\n"
"	subw	$read_disk_1st,dospatch3adr					\n"
"	addw	$partition_relative_read_disk,dospatch3adr			\n"
"	mov	$repring_msg,%si						\n"
"	callw	*%bp			# callw	simple_puts			\n"
"	clc									\n"
"	retw									\n"
"repring_msg:									\n"
"	.asciz	\"Gujin1, gujin2\"						\n"
	);
extern const unsigned char read_disk_1st[];
#endif

/* Each of the following can be defined or not defined to display early debug on screen: */
//#define DBG_ASMBEGIN(letter)		"movb $" #letter ",%al ; callw dbg_print_letter \n"
//#define DBG_BEFORE_ASMREPSTOSL()	"callw dbg_print_before_repstosl \n"
//#define DBG_BEFORE_ASMREPMOVSL()	"callw dbg_print_before_repmovsl \n"
//#define DBG_AFTER_ASMREP()		"callw dbg_print_after_rep \n"

/*
 * The first element has to be in the first sector loaded of
 * the bootloader, Fat/Root blocks are not loaded.
 *
 * Take care, some BIOS can not read across track boundary... and
 * some may be limited (to what?) in the number of sectors they can
 * read at once.
 *
 * There is also the problem of 64K crossing DMA: if we are loading
 * boot2 sectors just after boot1 (at 0x0000:0x7C00), we can load
 * (0x10000 - 0x7C00)/512 = 66 sectors before the DMA cross.
 *
 * Note that the DOS bootloader will not load boot2, the operating
 * system has already loaded everything. Note also that the
 * relocation to segment 0x0840 is done after the loading phase, and
 * can move only 64 K in real mode.
 *
 * For BIOS, the sector 0 does not exists, sector 1 is the boot sector;
 * LBA starts at 0. We already have read 1 sectors, see 'boot1param',
 * so the (beginning of the) following array is now in memory:
 */
/* initialised by external software, size defined at compilation:
   There isn't any trick on the size - just define it as small as
   possible considering compiler version and floppy format (720 K)
   you want to support, and recompile:
   26 bytes per array element, 26 * (14+8) = 572.
   There may be 3 element array at index 20 for the copy of the
   area erased by -boot-info-table for CDROMs, magic 0xFEED
   There may be 2 element array at end for the default IDE
   password, magic 0xBEAF followed by 16 bits reserved length
   and 32 bytes of password itself - stops at first '\0' */
static volatile const __attribute__ ((section (SECTNAME(N)), _packed)) bootloader2_t
	bootchain[24
#if (USER_SUPPORT != 0) || !(DISK_SUPPORT & NO_PROBE_PARTITIONS) || defined (DBG_ASMBEGIN)
			+ 20
#endif
#if SETUP & MULTILINGUAL
			+ 10
#endif
				] = {};

// For instance to see the code here, do "make boot.144" and then:
// objdump -D -m i8086 -b binary --start-address 0x272a boot.144 | less
// or "make gujin.exe" and then:
// objdump -D -m i8086 -b binary --start-address 0x74a gujin.exe | less
// objdump -D -m i8086 -b binary --start-address 0x72c /mnt/cdrom/gujin.bcd | less
asm (
"	.section        " SECTNAME(O) SECTTYPE ",@progbits		\n"
"	.global eltoritodisk						\n"
"eltoritodisk:		# disk used to load if not regs->dl		\n"
"	.byte 0x5E							\n"
"bootloader2_continue:							\n"
"	movb	%dl,eltoritodisk	# save for reference		\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('a')
#endif
"	# Detect < 386 processors:					\n"
"	pushfw								\n"
"	popw	%ax							\n"
"	testw	$0x8000,%ax						\n"
"	jz	processor_not_8086					\n"
"	movw	$processor_8086_msg,%si					\n"
"	callw	*%bp		# callw	simple_puts			\n"
"	jmp	.							\n"
"processor_not_8086:							\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('b')
#endif
"	testw	$0x7000,%ax						\n"
"	jnz	processor_sure_80386					\n"
"	orw	$0x2000,%ax						\n"
"	pushw	%ax							\n"
"	popfw								\n"
"	pushfw								\n"
"	popw	%ax							\n"
"	testw	$0x7000,%ax						\n"
"	jnz	processor_80386						\n"
"	movw	$processor_80286_msg,%si				\n"
"	callw	*%bp		# callw	simple_puts			\n"
"	jmp	.							\n"
"processor_80386:							\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('c')
#endif
"	andw	$~0x2000,%ax						\n"
"	pushw	%ax							\n"
"	popfw								\n"
"processor_sure_80386:							\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('d')
#endif
// Search 0xBEFC1F03 in disk.c for info
"	cmpl	$0xBEFC1F03,%cs:0x0A					\n"
"	jne	1f							\n"
"	movl	$0x00020000,%cs:0x0A					\n"
"	1:								\n"

"	# Clear NT and IOPL flags					\n"
"	pushf								\n"
"	popw	%ax							\n"
"	andw	$0x0FFF,%ax						\n"
"	pushw	%ax							\n"
"	popf								\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('e')
#endif
"	# restore 16 bit registers and store 32 bit registers as	\n"
"	# structure parameter to 'main()'				\n"
"	popaw	# take care of popa bug on i386				\n"
"	pushw	$0x454C		# Align stack to go faster ...		\n"
"	pushw	%fs							\n"
"	pushw	%gs							\n"
"	pushal								\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('f')
#endif
"	# Now it is time to move out of the way to be able		\n"
"	# to load another bootloader at physical address 0x7C00		\n"
"	# By the way, align %ds and %ss to be able to do in C:		\n"
"	#	void fct (int *a) {(*a)++;}				\n"
"	#	void caller (void) { int x; fct(&x); }			\n"
"									\n"
"	# clear the stack to be able to measure its used part,		\n"
"	# and clear BSS:						\n"
"	mov	%ss,%ax							\n"
"	mov	%ax,%es							\n"
"	xorl	%eax,%eax						\n"
"	std			# top to bottom				\n"
"	mov	%sp,%di							\n"
"	and	$0xFFFC,%di						\n"
"	mov	$_sbss,%cx						\n"
"	neg	%cx							\n"
"	sub	$4,%di							\n"
"	add	%di,%cx							\n"
"	shr	$2,%cx							\n"
#ifdef DBG_BEFORE_ASMREPSTOSL
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('\r')
	DBG_ASMBEGIN('\n')
#endif
	DBG_BEFORE_ASMREPSTOSL()
#endif
"	rep stosl %eax,%es:(%di)					\n"
#ifdef DBG_BEFORE_ASMREPSTOSL
#ifdef DBG_AFTER_ASMREP
	DBG_AFTER_ASMREP()
#endif
#endif

"									\n"
"	# Copy .rodata and .data (and .text if no CODE_SEGMENT):	\n"

#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('g')
#endif

"	mov	$deltaseg,%si						\n"
"	pushw	$1f							\n"
"copysegment:								\n"
"	mov	%ss,%ax							\n"
"	add	%si,%ax							\n"
"	sub	$deltaseg,%ax						\n"
"	mov	%ax,%es							\n"
"	pushw	%ds							\n"
"	mov	%ds,%ax							\n"
"	add	%si,%ax							\n"
"	mov	%ax,%ds							\n"
"	mov	%di,%cx							\n"
"	mov	%di,%si							\n"
"	shr	$2,%cx							\n"
#ifdef DBG_BEFORE_ASMREPMOVSL
	DBG_BEFORE_ASMREPMOVSL()
#endif
"	rep movsl %ds:(%si),%es:(%di)					\n"
"	nop								\n"
#ifdef DBG_BEFORE_ASMREPMOVSL
#ifdef DBG_AFTER_ASMREP
	DBG_AFTER_ASMREP()
#endif
#endif
"	popw	%ds							\n"
"	retw								\n"
"	1:								\n"

#if SETUP & CODE_SEGMENT

#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('h')
#endif

#if SETUP & XSTRING_SEGMENT
"	# Copy .xstring2						\n"
"	mov	$xstring2seg,%si					\n"
"	mov	$_exstring2,%di						\n"
"	callw	copysegment						\n"
"	# Copy .xstring1						\n"
"	mov	$xstring1seg,%si					\n"
"	mov	$_exstring1,%di						\n"
"	callw	copysegment						\n"
#endif

#if SETUP & XDATA_SEGMENT
"	# Copy .xdata							\n"
"	mov	$xdataseg,%si						\n"
"	mov	$_exdata,%di						\n"
"	callw	copysegment						\n"
#endif

#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('i')
#endif

#if SETUP & XCODE_SEGMENT
"	# Copy .usercode						\n"
"	mov	$usercodeseg,%si					\n"
"	mov	$_eusertext,%di						\n"
"	callw	copysegment						\n"
"									\n"
"	# Copy .loadcode						\n"
"	mov	$loadcodeseg,%si					\n"
"	mov	$_eloadtext,%di						\n"
"	callw	copysegment						\n"
"									\n"
"	# Copy .fscode							\n"
"	mov	$fscodeseg,%si						\n"
"	mov	$_efstext,%di						\n"
"	callw	copysegment						\n"
"									\n"
"	# Copy .diskcode						\n"
"	mov	$diskcodeseg,%si					\n"
"	mov	$_edisktext,%di						\n"
"	callw	copysegment						\n"
#endif /* XCODE_SEGMENT */

#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('j')
#endif

"									\n"
"	# Copy .text: (if not copied with .data/.rodata)		\n"
"	mov	$0,%si							\n"
"	mov	$_etext,%di						\n"
"	callw	copysegment						\n"

#endif /* CODE_SEGMENT */

#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('k')
#endif
"									\n"
"	# Realign code:							\n"
"	pushw	%es							\n"
"	pushw	$align_mover						\n"
"	lretw								\n"

/* The "lretw" before shall be below 0x8400 in case we started at
 * cs:ip = 0x0000:0x7c00 and so the stack is on segment 0x840+deltaseg
 * and so we have relocated the code to start at 0x8400:0 . If code
 * before lretw is too big we overwrite the lretw before executing it...
 */

/*
 * The Debug stuff if needed:
 * Warning: do not assume %ds nor %es points anywhere sensible in here,
 * so do not access _any_ variable like a string...
 */
#ifdef DBG_ASMBEGIN
"dbg_print_letter_in_al:		\n"
"	mov	$0x0E,%ah		\n"
"	int	$0x10			\n"
"	retw				\n"
"					\n"
"dbg_print_letter:			\n"
"	pushw %ds			\n"
"	pushw %es			\n"
"	pushw %fs			\n"
"	pushw %gs			\n"
"	pushfl				\n"
"	pushal				\n"
"	mov	$0x07,%bx		\n"
"	callw	dbg_print_letter_in_al	\n"
"	popal				\n"
"	popfl				\n"
"	popw %gs			\n"
"	popw %fs			\n"
"	popw %es			\n"
"	popw %ds			\n"
"	retw				\n"
#endif	/* DBG_ASMBEGIN */

#if defined (DBG_BEFORE_ASMREPSTOSL) || defined (DBG_BEFORE_ASMREPMOVSL) || defined  (DBG_AFTER_ASMREP)
#if 1
/* AAA assembly instruction:
IF ((AL AND 0FH) > 9) OR (AF = 1)
THEN
  AL <- AL + 6;
  AH <- AH + 1;
  AF <- 1;
  CF <- 1;
ELSE
  AF <- 0;
  CF <- 0;
FI;
AL <- AL AND 0FH;
  That is false (Intel info) - there are over books
 with "ALcarry <- AL > 0xF9" added to %ah in THEN part.
 *
 * AAD assembly instruction:
regAL = AL;
regAH = AH;
AL <- (regAH * imm8 + regAL) and 0xFF
AH <- 0
 *
 * SAHF (Store AH into Flags) transfers bits 7, 6, 4, 2, and 0
 from AH into SF, ZF, AF, PF, and CF, respectively.
*/
"dbg_print_quartet_in_al:		\n"
"	andw	$0x000F,%ax # ALcarry	\n"
//"	addb	$0x0,%al # clear AF	\n"
"	sahf		# clear AF	\n"
"	aaa				\n"
"	aad	$0x11			\n"
"	add	$0x0E30,%ax		\n"
"	int	$0x10			\n"
"	retw				\n"
#else
"dbg_print_quartet_in_al:		\n"
"	and	$0x0F,%al		\n"
"	cmp	$0x09,%al		\n"
"	jle	1f			\n"
"	add	$'A' - '9' - 1,%al	\n"
"	1:				\n"
"	add	$0x30,%al		\n"
"	jmp	dbg_print_letter_in_al	\n"
#endif

  /* Breathe-o-Smart:
  The major difference between a thing that might go wrong and a
  thing that cannot possibly go wrong is that when a thing that
  cannot possibly go wrong goes wrong it usually turn out to be
  impossible to get at or repair. */
"dbg_print_number_in_cx:		\n"
"	mov	%cx,%ax			\n"
"	shr	$12,%ax			\n"
"	callw	dbg_print_quartet_in_al	\n"
"	mov	%cx,%ax			\n"
"	shr	$8,%ax			\n"
"	callw	dbg_print_quartet_in_al	\n"
"	mov	%cx,%ax			\n"
"	shr	$4,%ax			\n"
"	callw	dbg_print_quartet_in_al	\n"
"	mov	%cx,%ax			\n"
"	callw	dbg_print_quartet_in_al	\n"
"	retw				\n"
#endif	/* DBG_BEFORE_ASMREPSTOSL || DBG_BEFORE_ASMREPMOVSL || DBG_AFTER_ASMREP */

/* Why put DBG_AFTER_ASMREP first? The answer is left as
	an exercise for the reader ... */
#ifdef DBG_AFTER_ASMREP
"dbg_print_after_rep:					\n"
"	pushw %ds					\n"
"	pushw %es					\n"
"	pushw %fs					\n"
"	pushw %gs					\n"
"	pushfl						\n"
"	pushal						\n"
"	mov	$0x07,%bx				\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al \n"
"	mov	$'c',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'x',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'d',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'i',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%di,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$'\r',%al ; callw dbg_print_letter_in_al \n"
"	mov	$'\n',%al ; callw dbg_print_letter_in_al \n"
"	popal						\n"
"	popfl						\n"
"	popw %gs					\n"
"	popw %fs					\n"
"	popw %es					\n"
"	popw %ds					\n"
"	retw						\n"
#endif

#ifdef DBG_BEFORE_ASMREPMOVSL
"dbg_print_before_repmovsl:				\n"
"	pushw %ds					\n"
"	pushw %es					\n"
"	pushw %fs					\n"
"	pushw %gs					\n"
"	pushfl						\n"
"	pushal						\n"
"	mov	$0x07,%bx				\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'m',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'o',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'v',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'l',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'c',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'x',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'d',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%ds,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'i',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%si,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'e',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%es,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'d',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'i',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%di,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$';',%al ; callw dbg_print_letter_in_al \n"
"	popal						\n"
"	popfl						\n"
"	popw %gs					\n"
"	popw %fs					\n"
"	popw %es					\n"
"	popw %ds					\n"
"	retw						\n"
#endif

#ifdef DBG_BEFORE_ASMREPSTOSL
"dbg_print_before_repstosl:				\n"
"	pushw %ds					\n"
"	pushw %es					\n"
"	pushw %fs					\n"
"	pushw %gs					\n"
"	pushfl						\n"
"	pushal						\n"
"	mov	$0x07,%bx				\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'t',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'o',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'l',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'c',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'x',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'e',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'s',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%es,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$' ',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'d',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'i',%al ; callw dbg_print_letter_in_al	\n"
"	mov	$'=',%al ; callw dbg_print_letter_in_al	\n"
"	mov	%di,%cx					\n"
"	callw	dbg_print_number_in_cx			\n"
"	mov	$';',%al ; callw dbg_print_letter_in_al \n"
"	popal						\n"
"	popfl						\n"
"	popw %gs					\n"
"	popw %fs					\n"
"	popw %es					\n"
"	popw %ds					\n"
"	retw						\n"
#endif
	);

/* This contains the previous MBR, to uninstall from Gujin:
 * The first is "where to read", the second is "where to read
 * current partition table", the third is "where to write"
 */
volatile const __attribute__ ((section (SECTNAME(P)), _packed))
bootloader2_t uninstall_mbr[3] = {
    { .header = { .nbword = 0 }}, /* will be gray in menu by default */
    };


/**
 ** The 80x86 instruction set does not like too much having big
 ** structures in stack, the hexadecimal encoding of offsets to
 ** register %esp is quite big. It is declared here to be the first
 ** variable in BSS, so could be used to set %ds for overloaded
 ** linux_set_params() and so have a constant+data section
 ** statically linked at zero, while keeping all other data
 ** in BSS accessible (DI, UI, ...). This later stuff still has
 ** to be writen and tested.
 **/
#if SETUP & CODE_SEGMENT
__attribute__ ((section (".fourKsegment"))) /* will be at zero (%ds:0) */
#else
#warning No CODE_SEGMENT, cannot guaranty 4 Kb data for linux_set_param()
#endif
unsigned char fourKbuffer[DEFAULT_BUFFER_SIZE];

/*
 * General state variable, also initied by assembly, in %ds segment:
 * Because segment values are initialised before main() is called,
 * we need to declare this structure volatile const.
 */
volatile const __attribute__ ((section(".rodata #"))) struct state_str STATE = {
    /* Because of the inter-segment protection of the linker,
	we can only, from diskcodeseg/fscodeseg/loadcodeseg, access gujin_param indirectly:
	(used also in vmlinuz_getparams_run()) */
    .gujin_param_ptr = &gujin_param,
    .color_overwrite_adr = &color_overwrite,
    /* same for : uninstall_mbr */
    .uninstall_mbr_ptr = uninstall_mbr,
#if DEBUG & DEBUG_ADDR
    .data_limit = { .low_limit = (unsigned)_srodata , .high_limit = 0x10000 -4 },
#endif
#if DEBUG & DEBUG_STACK
    .stack_limit = { .low_limit = (unsigned)_edata + stack_safety, .high_limit = 0x10000 -4 },
#endif
    };

/*
 * With GCC, we cannot access structure with asm(""), only with asm("": : :)
 * which has to be in a function, so do NOT call the following:
 * Why do I need to say section .text for this one?
 */
__attribute__((weak)) void dummy_do_not_call (void)
  {
  asm volatile (" bootbefore_part2_PhysicaldriveNb = %c0 " : : "p" (&bootbefore.part2.PhysicaldriveNb));
  asm volatile (" bootbefore_part1_NbHiddensector = %c0 " : : "p" (&bootbefore.part1.NbHiddensector));
  asm volatile (" bootend = %c0 " : : "p" ((unsigned)&boot1param + sizeof (boot1param)));
  asm volatile (" bootloader2 = %c0 "    : : "p" (&boot1param.bootloader2));
  asm volatile (" boot1_msg = %c0 "      : : "p" (bootbefore.part1.String));
  asm volatile (" boot2_msg = %c0 "      : : "p" (bootbefore.part2.Volumelabel));
  asm volatile (" dot_msg = %c0 "        : : "p" (gujin_param.dot_msg) );
  asm volatile (" go_msg = %c0 "         : : "p" (gujin_param.go_msg) );
  asm volatile (" dataseg_adr = %c0 "	: : "p" (&STATE.dataseg_adr));
  asm volatile (" codeseg_adr = %c0 "	: : "p" (&STATE.codeseg_adr));
  asm volatile (" copy_serial_port = %c0 ": : "p" (&STATE.serial_port));
#if SETUP & XCODE_SEGMENT
  asm volatile (" diskcodeseg_adr = %c0 "	: : "p" (&STATE.diskcodeseg_adr));
  asm volatile (" fscodeseg_adr = %c0 "	: : "p" (&STATE.fscodeseg_adr));
  asm volatile (" loadcodeseg_adr = %c0 "	: : "p" (&STATE.loadcodeseg_adr));
  asm volatile (" usercodeseg_adr = %c0 "	: : "p" (&STATE.usercodeseg_adr));
#endif
#if SETUP & XSTRING_SEGMENT
  asm volatile (" xstring1seg_adr = %c0 "	: : "p" (&STATE.xstring1seg_adr));
  asm volatile (" xstring2seg_adr = %c0 "	: : "p" (&STATE.xstring2seg_adr));
#endif
#if SETUP & XDATA_SEGMENT
  asm volatile (" xdataseg_adr = %c0 "	: : "p" (&STATE.xdataseg_adr));
#endif
#if !(SETUP & BIOS_ONLY)
  asm volatile (" dos_running = %c0 "	: : "p" (&STATE.dos_running));
#endif
  asm volatile (" .global boot1param_reg_dx \nboot1param_reg_dx = %c0 "	: : "p" (&boot1param.bootloader2.bootloader2_cmd.int1342.reg_dx));
  asm volatile (" .global bootchain_reg_dx \nbootchain_reg_dx = %c0 "	: : "p" (&bootchain[0].bootloader2_cmd.int1302.reg_dx));
  }

/* The Interrupts / exceptions stuff if needed: */
asm (
".text									\n"
#ifdef TREAT_COUNTER_TRACE_FLAG
"									\n"
"	.GLOBAL	TFflagHandler, TFflagCounter				\n"
"	.align 4							\n"
"TFflagCounter:								\n"
"	.long   0							\n"
"TFflagHandler:	# This ignores any other debugging interrupt...		\n"
"	incl	%cs:TFflagCounter					\n"
"	iretw								\n"
"									\n"
#endif /* TREAT_COUNTER_TRACE_FLAG */

#ifdef TREAT_EXCEPTION
"									\n"
"	# handler for all exceptions:					\n"
"	# function 'exceptions' accept parameters			\n"
"	.GLOBAL exceptions						\n"
"	.GLOBAL divide_error						\n"
"divide_error:								\n"
"	pushl	$0							\n"
"common_exception:							\n"
"	pushl	%es							\n"
"	pushl	%ds							\n"
"	pushal								\n"
"	movw	%cs,%ax							\n"
#if SETUP & CODE_SEGMENT
"	addw	$deltaseg,%ax						\n"
#endif
"	movw	%ax,%ds							\n"
"	movw	%ax,%es							\n"
"	calll	exceptions	# it shall not use %ebp as a pointer	\n"
"				# because %ss not set			\n"
"	popal			# take care of popa bug on i386		\n"
"	popl	%ds							\n"
"	popl	%es							\n"
"	add	$4,%sp							\n"
"	iretw								\n"
"									\n"

#if DEBUG & (DEBUG_ADDR | DEBUG_STACK)
"									\n"
"	.GLOBAL	bound_error						\n"
"bound_error:								\n"
"	pushl	$5							\n"
"	jmp	common_exception					\n"
"									\n"
#endif	/* DEBUG_ADDR | DEBUG_STACK */

"									\n"
"	.GLOBAL instruction_invalid					\n"
"instruction_invalid:							\n"
"	pushl	$6							\n"
"	jmp	common_exception					\n"
"	.GLOBAL stack_segment_fault					\n"
"stack_segment_fault:							\n"
"	pushl	$12							\n"
"	jmp	common_exception					\n"
"	.GLOBAL general_protection					\n"
"general_protection:	# NO error_code IN REAL MODE !			\n"
"	pushl	$13							\n"
"	jmp	common_exception					\n"
"									\n"

#endif /* TREAT_EXCEPTION */

"									\n"
"	# Back to the initialisation treatments:			\n"
"									\n"
"align_mover:								\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('A')
#endif
"	cld								\n"
"	movw	%ds,%dx							\n"
"	movw	%es,%cx							\n"
"	movw	%dx,%es							\n"
"	movzwl	%dx,%edx						\n"
"	movzwl	%cx,%ecx						\n"
"	sub	%edx,%ecx						\n"
"	xorl	%eax,%eax						\n"
"	movl	$512,%edi						\n"
"	sub	$32,%ecx						\n"
// Maybe GAS could generate a warning if two local numeric label are
// located in the same 128 bytes - could save some debug time...
"	jle	5f							\n"
"	shl	$2,%ecx							\n"
"	mov	%ecx,%edx						\n"
"	7:								\n"
"	cmp	$0x2000,%ecx						\n"
"	jb	6f							\n"
"	mov	$0x2000,%ecx						\n"
"	6:								\n"
"	sub	%ecx,%edx						\n"
#ifdef DBG_BEFORE_ASMREPSTOSL
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('\r')
	DBG_ASMBEGIN('\n')
#endif
	DBG_BEFORE_ASMREPSTOSL()
#endif
"	rep stosl %eax,%es:(%edi)					\n"
#ifdef DBG_BEFORE_ASMREPSTOSL
#ifdef DBG_AFTER_ASMREP
	DBG_AFTER_ASMREP()
#endif
#endif
"	mov	%es,%cx							\n"
"	add	$0x0800,%cx						\n"
"	mov	%cx,%es							\n"
"	sub	$0x8000,%edi						\n"
"	mov	%edx,%ecx						\n"
"	jecxz	5f							\n"
"	jmp	7b							\n"
"	5:								\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('S')
#endif
"	movw	%ss,%ax							\n"
"	movw	%ax,%ds							\n"
"	movw	%ax,%es							\n"
"									\n"
"	# For optimisation reasons, following variables are declared	\n"
"	# extern const in C, so we shall initialise them in assembler	\n"
"	# to prevent warnings:						\n"
"	# (initialise BEFORE calling install_exceptions,		\n"
"	#   and after %ds is set-up)					\n"
"									\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('C')
#endif
"	movw	%cs:serialconf_port_plus_1,%ax				\n"
"	dec	%ax			# COM port = -1	for screen	\n"
"	movw	%ax,copy_serial_port					\n"
"									\n"

#if !(SETUP & BIOS_ONLY)
"									\n"
"	movl	%cs:dospatch4adr,%eax					\n"
"	cmpl	$0x21CD4CB4,%eax					\n"
"	sete	dos_running						\n"
"									\n"
#endif

"	mov	%ds,dataseg_adr+2	# lower part is already zero	\n"
"									\n"
"	.global reloc_text_start, reloc_text_nb				\n"
"	mov	%cs,%ax							\n"
"	mov	%ax,codeseg_adr+2	# lower part is already zero	\n"
"	movw	$reloc_text_start,%di					\n"
"	movw	$reloc_text_nb,%cx					\n"
"	pushw	$3f							\n"
"reloc:									\n"
"	jcxz	2f							\n"
"	1:								\n"
"#	addw	%ax,2(%di)						\n"
"	movw	%ax,2(%di)	# can relocate multiple times		\n"
"	add	$4,%di							\n"
"	loop	1b							\n"
"	2:								\n"
"	retw								\n"
"	3:								\n"

#if SETUP & XCODE_SEGMENT
"	.global reloc_diskcode_start, reloc_diskcode_nb			\n"
"	mov	%cs,%ax							\n"
"	addw	$diskcodeseg,%ax					\n"
"	mov	%ax,diskcodeseg_adr+2	# lower part is already zero	\n"
"	movw	$reloc_diskcode_start,%di				\n"
"	movw	$reloc_diskcode_nb,%cx					\n"
"	callw	reloc							\n"
"	.global reloc_fscode_start, reloc_fscode_nb			\n"
"	mov	%cs,%ax							\n"
"	addw	$fscodeseg,%ax						\n"
"	mov	%ax,fscodeseg_adr+2	# lower part is already zero	\n"
"	movw	$reloc_fscode_start,%di					\n"
"	movw	$reloc_fscode_nb,%cx					\n"
"	callw	reloc							\n"
"	.global reloc_loadcode_start, reloc_loadcode_nb			\n"
"	mov	%cs,%ax							\n"
"	addw	$loadcodeseg,%ax					\n"
"	mov	%ax,loadcodeseg_adr+2	# lower part is already zero	\n"
"	movw	$reloc_loadcode_start,%di				\n"
"	movw	$reloc_loadcode_nb,%cx					\n"
"	callw	reloc							\n"
"	.global reloc_usercode_start, reloc_usercode_nb			\n"
"	mov	%cs,%ax							\n"
"	addw	$usercodeseg,%ax					\n"
"	mov	%ax,usercodeseg_adr+2	# lower part is already zero	\n"
"	movw	$reloc_usercode_start,%di				\n"
"	movw	$reloc_usercode_nb,%cx					\n"
"	callw	reloc							\n"
#endif

#if SETUP & XDATA_SEGMENT
"	.global reloc_xdata_start, reloc_xdata_nb			\n"
"	mov	%cs,%ax							\n"
"	addw	$xdataseg,%ax						\n"
"	mov	%ax,xdataseg_adr+2	# lower part is already zero	\n"
"	# if some functions are in this segment:			\n"
"	movw	$reloc_xdata_start,%di					\n"
"	movw	$reloc_xdata_nb,%cx					\n"
"	callw	reloc							\n"
#endif

#if SETUP & XSTRING_SEGMENT
"	mov	%cs,%ax							\n"
"	addw	$xstring1seg,%ax					\n"
"	mov	%ax,xstring1seg_adr+2	# lower part is already zero	\n"
"	mov	%cs,%ax							\n"
"	addw	$xstring2seg,%ax					\n"
"	mov	%ax,xstring2seg_adr+2	# lower part is already zero	\n"
#endif

#if ASSEMBLY_TYPE == ASSEMBLY_DSES
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('O')
#endif
"									\n"
"	# easy one: (call long because of its retl)			\n"
"	calll	initial_open_door					\n"
"	testl	%eax,%eax	# value returned			\n"
"	jnz	1f	# with DOS, when cannot open segs doors		\n"
"									\n"
#endif

#ifdef TREAT_EXCEPTION
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('V')
#endif
"									\n"
"	# another easy one: (call long because of its retl)		\n"
"	calll	setup_vectors						\n"
"									\n"
#endif

"									\n"
"	# now it is time to do serious things				\n"
"	xorl	%ebp,%ebp						\n"
"	xorl	%edi,%edi						\n"
"	xorl	%esi,%esi						\n"
"	xorl	%edx,%edx						\n"
"	xorl	%ecx,%ecx						\n"
"	xorl	%ebx,%ebx						\n"
"	mov	%esp,%eax						\n"
"	pushl	$0x33221100						\n"
"	pushl	$0x372E382E						\n"
"	pushl	$0x32205620						\n"
"	pushl	$0x4E494152						\n"
"	pushl	$0x524F4C20						\n"
"	pushl	$0x656E6E65						\n"
"	pushl	$0x69744520						\n"
"	pushl	$0x32313032						\n"
"	pushl	$0x20294328						\n"
"	pushl	%eax		# main's parameter			\n"
#ifdef DBG_ASMBEGIN
	DBG_ASMBEGIN('M')
	DBG_ASMBEGIN('\r')
	DBG_ASMBEGIN('\n')
#endif
"	calll	main							\n"
"	# clear the stack and BSS:					\n"
"	pushl	%eax							\n"
"	mov	%ss,%ax							\n"
"	mov	%ax,%es							\n"
"	xorl	%eax,%eax						\n"
"	std			# top to bottom				\n"
"	mov	%sp,%di							\n"
"	and	$0xFFFC,%di						\n"
"	mov	$_sbss,%cx						\n"
"	neg	%cx							\n"
"	sub	$4,%di							\n"
"	add	%di,%cx							\n"
"	shr	$2,%cx							\n"
"	rep stosl %eax,%es:(%di)					\n"
"	popl	%eax							\n"
"	#								\n"
"	addl	$40,%esp						\n"
"	testl	%eax,%eax						\n"
"	1:								\n"
"	pushw	%cs							\n"
"	popw	%ds							\n"
"	popal	# take care of popa bug on i386				\n"
"	cld								\n"
"	popw	%gs							\n"
"	popw	%fs							\n"
"	popw	%es	# crashes %es, to pop alignment word		\n"
"	pushaw								\n"
"	pushw	%si			# it will be restored		\n"
"	movw	$msg_exiting_ok,%si					\n"
"	jz	no_error						\n"
"	movw	$msg_exiting_err,%si					\n"
"no_error:								\n"
"	movw	$simple_puts,%bp	# it has been erased,		\n"
"					# so set it again		\n"
/* This can be equivalent of "jmp bootloader2_return" but this area
   at bootloader2_return may be patched... note that most DOS executable
   we will not come here because we exit by ^C or ^D which calls
   BIOS_killme() of library.c and then _DOS_stop_exe() of bios.h,
   or exit by executing a kernel file - but mingujin.exe comes here when
   called without parameters at all...
   We arrive here either because something else is loaded at 0x7c0:0x0000
   and we want to jump to this new MBR, or we are exiting a DOS program.
   Typing Escape in the menu directly calls _BIOS_failedboot(), i.e. INT0x18 */
"	cmpl	$0x18CD19CD,dospatch4adr				\n"
"	jne	bootloader2_return					\n"
//#if SETUP & BOOT_NEVER_RETRY
"	callw	*%bp			# callw	simple_puts		\n"
"	pop	%si			# realign stack			\n"
"	popaw	# take care of popa bug on i386				\n"
"	popw	%es							\n"
"	popw	%ds							\n"
"	iretw								\n"
//#endif
	);

/* No need to translate those two, anyway they will not happen... */
static const char processor_8086_msg[]  __attribute__ ((section (".text.start"), _packed, attr_used))
	= "Gujin can not run on a 8086!\r\n";
static const char processor_80286_msg[]  __attribute__ ((section (".text.start"), _packed, attr_used))
	= "Gujin can not run on a 80286!\r\n";
/* Those two will be overwritten as soon as Gujin can to adjust their language: */
static const char msg_exiting_ok[80]  __attribute__ ((section (".text.start"), _packed, attr_used))
	= "\r\nExiting Gujin without error...\r\n";
static const char msg_exiting_err[80]  __attribute__ ((section (".text.start"), _packed, attr_used))
	= "\r\nExiting Gujin with error...\r\n";

extern unsigned char checksum_start[], start[], bootend[];
extern unsigned char dospatch2adr[], dospatch3adr[], dospatch4adr[];
extern const unsigned short serialconf_port_plus_1;
extern const unsigned char partition_relative_read_disk[], read_disk[];
extern const unsigned char detect_usbhdd_patch[], USBHDD_patch[];
extern const unsigned char SingleSectorPatch1[], SingleSectorPatch2[];

/*
 * This is not static because its address is passed to
 * the kernel at startup (%cs relative):
 */
const instboot_info_t instboot_info __attribute__ ((section (".text.start"), _packed, attr_used)) = {
    .checksum_start_adr = (unsigned) checksum_start,
    .boot1param_adr = (unsigned) &boot1param,
    .bootend_adr = (unsigned) bootend,
    .uninstall_mbr_adr = (unsigned)uninstall_mbr,
    .bootchain_adr = (unsigned) bootchain,
    .bootchain_nbblock = (unsigned) nbof (bootchain),
    .gujin_param_adr = (unsigned) &gujin_param,
    .code_start_adr = (unsigned) start,
    .patch2adr = (unsigned) dospatch2adr,
    .patch3adr = (unsigned) dospatch3adr,
    .patch4adr = (unsigned) dospatch4adr,
    .sbss = (unsigned) _sbss,
    .edata = (unsigned) _edata,
    .password_serial_adr = (unsigned)&password_serial,
    .serialconf_port_plus_1_adr = (unsigned)&serialconf_port_plus_1,
#if SETUP & CODE_SEGMENT
    .deltaseg = (unsigned) deltaseg,
#endif
#if SETUP & XCODE_SEGMENT /* else not initialised, so zero */
    .diskcodeseg = (unsigned) diskcodeseg,
    .fscodeseg = (unsigned) fscodeseg,
    .loadcodeseg = (unsigned) loadcodeseg,
    .usercodeseg = (unsigned) usercodeseg,
#endif
#if SETUP & XSTRING_SEGMENT
    .xstring1seg = (unsigned) xstring1seg,
    .xstring2seg = (unsigned) xstring2seg,
#endif
#if SETUP & XDATA_SEGMENT /* else not initialised, so zero */
    .xdataseg = (unsigned) xdataseg,
#endif
    .partition_relative_read_disk_adr = (unsigned)partition_relative_read_disk,
    .read_disk_adr = (unsigned)read_disk,
#if DISK_SUPPORT & CDBIOS_SUPPORT
    .read_disk_1st_adr = (unsigned)read_disk_1st,
#endif
    .usbhdd_patch1 = (unsigned)USBHDD_patch,
    .usbhdd_patch2 = (unsigned)detect_usbhdd_patch,
    .singlesectorload_patch1 = (unsigned)SingleSectorPatch1,
    .singlesectorload_patch2 = (unsigned)SingleSectorPatch2,
    };

void *volatile const default_ide_password_codeptr __attribute__ ((section(".rodata #"))) = (void *)&bootchain[nbof(bootchain)-3];
gujin_param_t copy_gujin_param;

/*
 * The BOOT1 "library" functions:
 */
#if SETUP & MULTILINGUAL
void update_msg_exiting(void)
  {
  const char *msg = MSG_EXITING_OK, *end = msg;
  while (*end && end - msg < (int)sizeof (msg_exiting_ok) - 1)
      end++;
  codeseg_pokeb((char *)&msg_exiting_ok[end - msg], '\0');
  while (end-- > msg)
      codeseg_pokeb((char *)&msg_exiting_ok[end - msg], *end);

  end = msg = MSG_EXITING_ERR;
  while (*end && end - msg < (int)sizeof (msg_exiting_err) - 1)
      end++;
  codeseg_pokeb((char *)&msg_exiting_err[end - msg], '\0');
  while (end-- > msg)
      codeseg_pokeb((char *)&msg_exiting_err[end - msg], *end);
  }
#endif

void __attribute__((noinline)) BOOT1_putstr (const char *str)
  {bound_stack();{
  unsigned short dummy1;

  asm volatile (
"	pushw	%%si		# only 16 bits, aligned stack for int10	\n"
"	callw	simple_puts						\n"
"	popw	%%si							\n"
	: "=a" (dummy1)
	: "S" (str)
	);
  }}

const char ansi2pc[] =
	/* 0x00 */ "??????????\n??\r??"
	/* 0x10 */ "?????????????"
	/* 0x20 */ " !\"#$%&'()*+,-./"
	/* 0x30 */ "0123456789:;<=>?"
	/* 0x40 */ "@ABCDEFGHIJKLMNO"
	/* 0x50 */ "PQRSTUVWXYZ[\\]^_"
	/* 0x60 */ "`abcdefghijklmno"
	/* 0x70 */ "pqrstuvwxyz{|}~\x1B"	/* DEL: anything but \x7F because \xFB */
	/* 0x80 */ ""
	/* 0x90 */ "ܢ??"
	/* 0xA0 */ "Ѫ?"
	/* 0xB0 */ "???|????????????"	/* B3: vertical line */
	/* 0xC0 */ "????_???????????"	/* C4: horizontal line */
	/* 0xD0 */ "????????????????" /* text mode mouse: '\xDB' is used directly by UI_setpixel(), no need to treat here */
	/* 0xE0 */ "?????????????"
	/* 0xF0 */ "???????\x7F???"; /* checked box '\xFB' */

void BOOT1_ansi_putstr (const char *str)
  {bound_stack();{
  const char *ptr = str;

  while (*ptr != '\0')
      ptr++;
  if (ptr != str) {
      char ansi_str[ptr - str + 1], *ansi_ptr = ansi_str;

      do {
	  const char *p = ansi2pc;

	  while (*p && *p != *str)
	      p++;
	  *ansi_ptr++ = (*p == '\0') ? '?' : (p - ansi2pc);
	  } while (*++str != '\0');
      *ansi_ptr = '\0';
      BOOT1_putstr (ansi_str);
      }
  }}

#if SETUP & UPDATE_BOOTPARAM
//#define DBG_RW_BOOTPARAM
//#ifdef DBG_RW_BOOTPARAM
#include "bios.h"	// for _BIOS_wait()
//#endif
#include "util.h"	// UTIL.boot_device_read_only

extern inline unsigned char
BOOT1_diskread (union diskcmd *str, farptr buffer)
  {
  unsigned short status, dummy;
  unsigned char  returned;

  if ((str->hardide.reg_ax >> 8) == 0xF0)
      str->hardide.nb_sect_read = str->hardide.nb_sect;
#ifdef DBG_RW_BOOTPARAM
  if ((str->hardide.reg_ax >> 8) == 0xF0)
      printf (" [read at 0x%X IDE: ide_command 0x%X at 0x%X,0x%X, %u sectors at LBA28 0x%X:0x%X:0x%X:0x%X ",
		buffer,
		str->hardide.ide_command,
		str->hardide.base_ide_plus7 - 7,
		str->hardide.ide_dcr_address,
		str->hardide.nb_sect,
		str->hardide.lba_head,
		str->hardide.cylinder_high,
		str->hardide.cylinder_low,
		str->hardide.sector
		);
    else if ((str->hardide.reg_ax >> 8) == 0x02)
      printf (" [read at 0x%X BIOS: disk 0x%X %u sectors at C/H/S %u/%u/%u to 0x%X ",
		buffer,
		str->int1302.reg_dx & 0xFF,
		str->int1302.reg_ax & 0xFF,
		4 * (str->int1302.reg_cx & 0x00C0) + (str->int1302.reg_cx >> 8),
		str->int1302.reg_dx >> 8,
		str->int1302.reg_cx & 0x003F,
		str->int1302.will_be_set_to_esbx
		);
    else if ((str->hardide.reg_ax >> 8) == 0x42)
      printf (" [read at 0x%X EBIOS: disk 0x%X %u sectors at 0x%llX to 0x%X:0x%X",
		buffer,
		str->int1342.reg_dx & 0xFF,
		str->int1342.nb_block,
		str->int1342.lba,
		str->int1342.segment,
		str->int1342.offset
		);
    else
      printf (" [read called with bad args! ");
#endif
  /* also %dx as input in case there is the "mov %ah,%dh" patch */
  asm volatile (
"	movw	%%es,%%ax						\n"
"	pushl	%4							\n"
"	popw	%%bx							\n"
"	popw	%%es							\n"
"	pushw	%%ax							\n"
"	callw	read_disk						\n"
"	popw	%%es							\n"
"	setc	%%dl	# set dest to 1 if carry, else clear dest	\n"
	: "=&a" (status), "=d" (returned), "=S" (dummy)
	: "S" (str), "g" (buffer), "d" (str->int1342.reg_dx), ASM_STRUCT_INPUT (*str)
	: "ebx", "edi", "ecx", "memory"
	);
#ifdef DBG_RW_BOOTPARAM
  if (returned != 0)
      print (" failed] ");
    else
      print (" OK] ");
  _BIOS_wait (100000); /* 100 ms */
#endif
  if (returned != 0)
      ZDBG ((" [Error carry in 'diskread', ax=0x%X] ", status));
  return returned;
  }

asm (
"	.GLOBAL	write_disk						\n"
"write_disk:			# BIOS INT13 %ah=0x03			\n"
"	lodsw	%ds:(%si),%ax	# .reg_dx / .ide_dcr_address		\n"
"	mov	%ax,%dx							\n"
"	lodsw	%ds:(%si),%ax	# .reg_ax				\n"
"	mov	(%si),%cx	# .reg_cx / .nb_word_to_write (unused 13/43)\n"
"	cmp	$0xF0,%ah						\n"
"	je	Lw_ide_hard						\n"
"	mov	%es,6(%si)						\n"
"	cmpw	$0xD0E8,%cs:detect_usbhdd_patch				\n"
"	jne	not_usbhdd	# never patched when EBIOS		\n"
"	or	%dh,%dh							\n"
"	jnz	no_change_cylinder_write				\n"
"	mov	%cs:0x001A,%dh		# heads				\n"
"	or	%ch,%ch							\n"
"	jnz	no_dec_cly_msb_write					\n"
"	sub	$64,%cl							\n"
"no_dec_cly_msb_write:							\n"
"	dec	%ch							\n"
"no_change_cylinder_write:						\n"
"	dec	%dh							\n"
"not_usbhdd:								\n"
"	mov	%bx,4(%si)						\n"
"	int	$0x13							\n"
"	# do not reset here						\n"
"	retw								\n"
"									\n"
"# Copyrighted  Etienne LORRAIN, 1999-2013.				\n"
"# If you are reading that code before reading the GPL license,		\n"
"# you are actually fucking me.						\n"
"Lw_ide_hard:			# access IDE directly			\n"
"	mov	%bx,%di		# %bx should not be modified in this fonction\n"
"	outb	%al,%dx		# %al = 0x0A, %dx = 0x3F6 (for instance)\n"
"	lodsw	%ds:(%si),%ax	# .base_ide_plus7 (that is already in cx, but\n"
"				# it also increment %si and initialise %ah=1 !)\n"
"	mov	%ax,%dx		# for instance 0x1F7 (command/status register)\n"
"	xor	%cx,%cx		# 65536*1us (approx) = 65 ms		\n"
"Lw_waitnotbusy:							\n"
"	inb	%dx,%al							\n"
"	test	$0x80,%al	# busy					\n"
"	loopnz	Lw_waitnotbusy						\n"
"	jcxz	Lw_ide_error	# timeout: %ah=MSB(ideaddr), often 1	\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .lba_head				\n"
"	inc	%dx							\n"
"Lw_waitdiskready:							\n"
"	loop	Lw_waitcontinue	# timeout BUSY or !DRDY or !RDY: %ah = 1\n"
"Lw_ide_error:								\n"
"	stc								\n"
"	retw								\n"
"Lw_waitcontinue:							\n"
"	inb	%dx,%al							\n"
"	test	$0xD0,%al	# clear carry				\n"
"	jle	Lw_waitdiskready	# I have understood that for you!\n"
"	jnp	Lw_waitdiskready					\n"
"	mov	$2,%cx							\n"
"Lw_lba48_fifo:								\n"
"	push	%dx		# %dx points to status/cmd register	\n"
"	dec	%dx							\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .cylinder_high			\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .cylinder_low				\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .sector				\n"
"	dec	%dx							\n"
"	outsb	%ds:(%si),%dx	# .nb_sect				\n"
"	pop	%dx							\n"
"	loop	Lw_lba48_fifo	# %cx null after			\n"
"	outsb	%ds:(%si),%dx	# .ide_command				\n"
"Lw_loopnextsector:							\n"
"	inb	%dx,%al		# short wait				\n"
"Lw_waitcommandcomplete:	# %cx null before			\n"
"	inb	%dx,%al							\n"
"	test	$0x81,%al	# clear carry				\n"
"	loopnz	Lw_waitcommandcomplete					\n"
"	jcxz	Lw_ide_error	# timeout/error/no DRQ			\n"
"	test	$0x08,%al						\n"
"	jz	Lw_waitcommandcomplete					\n"
"	mov	2(%si),%cx						\n"
"	push	%dx							\n"
"	sub	$7,%dx		# point to data register and clear carry\n"
"	xchgw	%si,%di			#				\n"
"	rep outsw %es:(%si),%dx		# the only R/W difference	\n"
"	xchgw	%si,%di			#				\n"
"	pop	%dx							\n"
"	decw	(%si)							\n"
"	jnz	Lw_loopnextsector	# %cx null after (rep)		\n"
"	retw								\n"
	);

extern inline unsigned char
BOOT1_diskwrite (union diskcmd *str, farptr buffer)
  {
  unsigned short status, dummy;
  unsigned char  returned;

  if ((str->hardide.reg_ax >> 8) == 0xF0)
      str->hardide.nb_sect_read = str->hardide.nb_sect;
#ifdef DBG_RW_BOOTPARAM
  if ((str->hardide.reg_ax >> 8) == 0xF0)
      printf (" [write from 0x%X IDE: ide_command 0x%X at 0x%X,0x%X, %u sectors at LBA28 0x%X:0x%X:0x%X:0x%X ",
		buffer,
		str->hardide.ide_command,
		str->hardide.base_ide_plus7 - 7,
		str->hardide.ide_dcr_address,
		str->hardide.nb_sect,
		str->hardide.lba_head,
		str->hardide.cylinder_high,
		str->hardide.cylinder_low,
		str->hardide.sector
		);
    else if ((str->hardide.reg_ax >> 8) == 0x03)
      printf (" [write from 0x%X BIOS: disk 0x%X %u sectors at C/H/S %u/%u/%u to 0x%X ",
		buffer,
		str->int1302.reg_dx & 0xFF,
		str->int1302.reg_ax & 0xFF,
		4 * (str->int1302.reg_cx & 0x00C0) + (str->int1302.reg_cx >> 8),
		str->int1302.reg_dx >> 8,
		str->int1302.reg_cx & 0x003F,
		str->int1302.will_be_set_to_esbx
		);
    else if ((str->hardide.reg_ax >> 8) == 0x43)
      printf (" [write from 0x%X EBIOS: disk 0x%X %u sectors at 0x%llX to 0x%X:0x%X",
		buffer,
		str->int1342.reg_dx & 0xFF,
		str->int1342.nb_block,
		str->int1342.lba,
		str->int1342.segment,
		str->int1342.offset
		);
    else
      printf (" [read called with bad args! ");
#endif
  asm volatile (
"	movw	%%es,%%ax						\n"
"	pushl	%4							\n"
"	popw	%%bx							\n"
"	popw	%%es							\n"
"	pushw	%%ax							\n"
"	callw	write_disk						\n"
"	popw	%%es							\n"
"	setc	%%dl	# set dest to 1 if carry, else clear dest	\n"
	: "=&a" (status), "=d" (returned), "=S" (dummy)
	: "S" (str), "g" (buffer), ASM_STRUCT_INPUT (*str)
	: "ebx", "edi", "ecx"
	);
#ifdef DBG_RW_BOOTPARAM
  if (returned != 0)
      print ("failed] ");
    else
      print ("OK] ");
  _BIOS_wait (100000); /* 100 ms */
#endif
  if (returned != 0)
      ZDBG ((" [Error carry in '%s', ax=0x%X] ", __FUNCTION__, status));
  return returned;
  }

struct fourKstruct { char data[4096]; };

unsigned char BOOT1_uninstall_mbr (void)
  {bound_stack();{
  bootloader2_t copy_in_dataseg;
  unsigned char disk_overwrite = 0;

  /* load the uninstall, with potential wrong partitions: */
  cmemcpy (&copy_in_dataseg, &uninstall_mbr[0], sizeof (bootloader2_t));
  if (copy_in_dataseg.header.nbword == 0 || copy_in_dataseg.header.nbword % 256 != 0 || copy_in_dataseg.header.nbword > 64*1024/2) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_NO_BACKUP_OF);
      return 1;
      }
  if (copy_in_dataseg.bootloader2_cmd.int1342.reg_ax == 0x4200 || copy_in_dataseg.bootloader2_cmd.int1302.reg_ax == 0x0201) {
      unsigned short movaxdx = cs_peekw (((char *)read_disk) + 1);
      if (movaxdx == 0xC289) /* mov %ax,%dx */
	  ZDBG ((" [%s: standard mov %%ax,%%dx] ", __FUNCTION__));
	else if (movaxdx == 0xE688) { /* mov %ah,%dh */
	  disk_overwrite = 1;
	  ZDBG ((" [%s: mov %%ah,%%dh, used initial %%dl = 0x%X as disk] ", __FUNCTION__, UTIL.bios_boot_dl));
	  }
	else {
	  ZDBG ((" [%s: mov %%a?,%%d? unrecognized - abort] ", __FUNCTION__));
	  printf (MSG_UNINSTALL_MBR_ERROR, MSG_NO_BACKUP_OF);
	  return 1;
	  }
      }
  if (disk_overwrite) {
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx &= 0xFF00;
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx |= UTIL.bios_boot_dl;
      }
  {
  unsigned short address_called = ((unsigned)dospatch3adr & 0x0000FFFF) + cs_peekw (dospatch3adr) + 2;
  if (address_called == ((unsigned)read_disk & 0x0000FFFF)) {
      ZDBG ((" [%s: absolute LBA] ", __FUNCTION__));
      }
    else if (address_called == ((unsigned)partition_relative_read_disk & 0x0000FFFF)) {

      unsigned partition_offset = cs_peekl (&bootbefore.part1.Reservedsector);
      ZDBG ((" [%s: relative LBA, add %u] ", __FUNCTION__, partition_offset));
      if (copy_in_dataseg.bootloader2_cmd.int1342.reg_ax == 0x4200)
	  copy_in_dataseg.bootloader2_cmd.int1342.lba += partition_offset;
	else {
#if 0
	  union { unsigned char array[8]; unsigned long long lba; } tmp = { .array = {
		copy_in_dataseg.bootloader2_cmd.hardide.sector,
		copy_in_dataseg.bootloader2_cmd.hardide.cylinder_low,
		copy_in_dataseg.bootloader2_cmd.hardide.cylinder_high,
		copy_in_dataseg.bootloader2_cmd.hardide.lba48_sector,
		copy_in_dataseg.bootloader2_cmd.hardide.lba48_cylinder_low,
		copy_in_dataseg.bootloader2_cmd.hardide.lba48_cylinder_high,
		0,0
		}};
	  tmp.lba += partition_offset;
	  copy_in_dataseg.bootloader2_cmd.hardide.sector		= tmp.array[0];
	  copy_in_dataseg.bootloader2_cmd.hardide.cylinder_low		= tmp.array[1];
	  copy_in_dataseg.bootloader2_cmd.hardide.cylinder_high		= tmp.array[2];
	  copy_in_dataseg.bootloader2_cmd.hardide.lba48_sector		= tmp.array[3];
	  copy_in_dataseg.bootloader2_cmd.hardide.lba48_cylinder_low	= tmp.array[4];
	  copy_in_dataseg.bootloader2_cmd.hardide.lba48_cylinder_high	= tmp.array[5];
#else
	  *(unsigned long long *)&copy_in_dataseg.bootloader2_cmd.hardide += partition_offset;
#endif
	  }
      }
    else {
      ZDBG ((" [%s: neither absolute nor relative LBA, address_called 0x%X not 0x%X nor 0x%X] ",
		__FUNCTION__, address_called,
		(unsigned short)read_disk, (unsigned short)partition_relative_read_disk));
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_NO_BACKUP_OF);
      return 1;
      }
  }

  unsigned char space[3 * 4096], *sector, *reread_sector;
  farptr sector_fptr, reread_sector_fptr;

  /* This set segment bit 4..11 to zero, each page is 64Kb, only one may cross DMA: */
  sector = space;
  sector_fptr = linear2farptr (farptr2linear (stack_adr(sector)));
  reread_sector = &space[4096];
  reread_sector_fptr = linear2farptr (farptr2linear (stack_adr(reread_sector)));
  if ((sector_fptr & 0xFFFFU) + 4096 > 0x10000) { /* DMA barrier */
      sector = &space[2*4096];
      sector_fptr = linear2farptr (farptr2linear (stack_adr(sector)));
      }
    else if ((reread_sector_fptr & 0xFFFFU) + 4096 > 0x10000) { /* DMA barrier */
      reread_sector = &space[2 * 4096];
      reread_sector_fptr = linear2farptr (farptr2linear (stack_adr(reread_sector)));
      }

  if (BOOT1_diskread (&copy_in_dataseg.bootloader2_cmd, sector_fptr) != 0) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_READING);
      return 1;
      }
  asm ("" : "=m" (*(struct fourKstruct *)sector)); /* GCC-4.6, do not try to know/optimise what *sector contains */
  if (peekw (sector_fptr + 0x1FE) != 0xAA55) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_CHECKING);
      return 1;
      }
  /* Index 1 load the real MBR at zero to get partitions: */
  cmemcpy (&copy_in_dataseg, &uninstall_mbr[1], sizeof (bootloader2_t));
  if (disk_overwrite) {
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx &= 0xFF00;
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx |= UTIL.bios_boot_dl;
      }
  if (BOOT1_diskread (&copy_in_dataseg.bootloader2_cmd, reread_sector_fptr) != 0) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_READING);
      return 1;
      }
  asm ("" : "=m" (*(struct fourKstruct *)reread_sector)); /* GCC-4.6, do not try to know/optimise what *reread_sector contains */
  if (peekw (reread_sector_fptr + 0x1FE) != 0xAA55) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_CHECKING);
      return 1;
      }
  /* copy partition table: we have 16 bytes per partition
     plus 6 bytes at beginning for WindowsNTmarker and Unknown,
     plus two byte after for signature 0xAA55:
     4 * 16 + 6 + 2 = 72 bytes. */
  fmemcpy (sector_fptr + 0x200 - 72, reread_sector_fptr + 0x200 - 72, 72);
  /* Index 2 is writing the data to the real MBR */
  cmemcpy (&copy_in_dataseg, &uninstall_mbr[2], sizeof (bootloader2_t));
  if (disk_overwrite) {
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx &= 0xFF00;
      copy_in_dataseg.bootloader2_cmd.int1342.reg_dx |= UTIL.bios_boot_dl;
      }
  asm ("" : : "m" (*(struct fourKstruct *)sector)); /* GCC-4.6, do not try to know/optimise what *sector contains */
  if ( // !copy_gujin_param.attrib.disk_write_enable ||
      BOOT1_diskwrite (&copy_in_dataseg.bootloader2_cmd, sector_fptr) != 0) {
      printf (MSG_UNINSTALL_MBR_ERROR, MSG_WRITING);
      return 1;
      }
  printf (MSG_UNINSTALL_MBR_SUCCESS);
  asm volatile (" movw $1367,%c0 \n" : : "g" (&STATE.has_just_been_uninstalled));
  return 0;
  }}

/* Have you got your daily B.B.I. yet ? */
/* note: B.B.I. : Brain Botox Injection */
unsigned char BOOT1_update_bootparam (void)
  {bound_stack();{
  union diskcmd cmdblock;	/* in current data/stack segment */
  unsigned char check_partition_relative = 1;
  smemcpy(code_adr(&gujin_param), &copy_gujin_param, sizeof (copy_gujin_param));

  if (sizeof(bootsector_t) != 512)
      __ERROR();
  if (sizeof(gujin_param_t) % 2 != 0)
      __ERROR();

  if (BOOT1_DOS_running())
      return 1;

  if (!copy_gujin_param.attrib.disk_write_enable && copy_gujin_param.attrib.write_disabled_written_once) {
      ZDBG ((" [%s: write disabled and already written once] ", __FUNCTION__));
      /* so no need to try to read - for instance a removed floppy */
      return 2;
      }

  cmemcpy (&cmdblock, &boot1param.bootloader2.bootloader2_cmd, sizeof (cmdblock));
  unsigned sectorsize = 2 * codeseg_peekw(&boot1param.bootloader2.header.nbword);

  if (sectorsize % 512) {
      ZDBG ((" [%s: disable writing boot device, sectorsize %u] ", __FUNCTION__, sectorsize));
      UTIL.boot_device_read_only = 13;
      return 14;
      }

  if (cmdblock.int1302.reg_ax == 0x0100) { /* "status of last HD operation" */
      sectorsize += 512;
      cmdblock.int1342.reg_ax = 0x4200;
      cmdblock.int1342.reg_dx = cmdblock.int1342.cst_0; /* saved diskindex in set_bootchain() */
      cmdblock.int1342.nb_block = 1;
      cmdblock.int1342.cst_0 = 0;
      check_partition_relative = 0;
      }

  if (cmdblock.int1302.reg_ax != 0x0201 && cmdblock.int1342.reg_ax != 0x4200 && cmdblock.hardide.reg_ax != 0xF00A) {
      /* Also handle sector size != 512, first cmdblock do nothing
	 but check checksum of end of the sector */
      ZDBG ((" [%s: disable writing boot device, not 0x0201/0x4200/0xF00A] ", __FUNCTION__));
      UTIL.boot_device_read_only = 3;
      return 3;
      }

  if (   cmdblock.hardide.reg_ax == 0xF00A
      && cmdblock.hardide.ide_command != 0x20
      && cmdblock.hardide.ide_command != 0x24) {
      ZDBG ((" [%s: disable writing boot device, 0xF00A but not 0x20/0x24] ", __FUNCTION__));
      UTIL.boot_device_read_only = 4;
      return 4;
      }

  {
  unsigned short movaxdx = cs_peekw (((char *)read_disk) + 1);
  if (movaxdx == 0xC289) /* mov %ax,%dx */
      ZDBG ((" [%s: standard mov %%ax,%%dx] ", __FUNCTION__));
    else if (movaxdx == 0xE688) { /* mov %ah,%dh */
      ZDBG ((" [%s: mov %%ah,%%dh, used initial %%dl = 0x%X as disk] ", __FUNCTION__, UTIL.bios_boot_dl));
      cmdblock.int1342.reg_dx &= 0xFF00;
      cmdblock.int1342.reg_dx |= UTIL.bios_boot_dl;
      }
    else {
      ZDBG ((" [%s: disable writing boot device, mov %%a?,%%d? unrecognized] ", __FUNCTION__));
      UTIL.boot_device_read_only = 5;
      return 5;
      }
  }

  if (check_partition_relative) {
      unsigned short address_called = ((unsigned)dospatch3adr & 0x0000FFFF) + cs_peekw (dospatch3adr) + 2;
      if (address_called == ((unsigned)read_disk & 0x0000FFFF)) {
	  ZDBG ((" [%s: absolute LBA] ", __FUNCTION__));
	  }
	else if (address_called == ((unsigned)partition_relative_read_disk & 0x0000FFFF)) {
	  unsigned partition_offset = cs_peekl (&bootbefore.part1.Reservedsector);
	  ZDBG ((" [%s: relative LBA, add %u] ", __FUNCTION__, partition_offset));
	  if (cmdblock.int1342.reg_ax == 0x4200)
	      cmdblock.int1342.lba += partition_offset;
	    else {
#if 0
	      union { unsigned char array[8]; unsigned long long lba; } tmp = { .array = {
		cmdblock.hardide.sector,
		cmdblock.hardide.cylinder_low,
		cmdblock.hardide.cylinder_high,
		cmdblock.hardide.lba48_sector,
		cmdblock.hardide.lba48_cylinder_low,
		cmdblock.hardide.lba48_cylinder_high,
		0,0
		}};
	      tmp.lba += partition_offset;
	      cmdblock.hardide.sector			= tmp.array[0];
	      cmdblock.hardide.cylinder_low		= tmp.array[1];
	      cmdblock.hardide.cylinder_high		= tmp.array[2];
	      cmdblock.hardide.lba48_sector		= tmp.array[3];
	      cmdblock.hardide.lba48_cylinder_low	= tmp.array[4];
	      cmdblock.hardide.lba48_cylinder_high	= tmp.array[5];
#else
	      *(unsigned long long *)&cmdblock.hardide += partition_offset;
#endif
	      }
	  }
	else {
	  ZDBG ((" [%s: disable writing boot device, neither absolute nor relative LBA, address_called 0x%X not 0x%X nor 0x%X] ",
		__FUNCTION__, address_called, (unsigned short)read_disk, (unsigned short)partition_relative_read_disk));
	  UTIL.boot_device_read_only = 6;
	  return 6;
	  }
      }

  {
  unsigned char space[3 * 4096], *sector, *reread_sector;
  farptr sector_fptr, reread_sector_fptr;
  unsigned short cpt;

  /* This set segment bit 4..11 to zero, each page is 64Kb, only one may cross DMA: */
  sector = space;
  sector_fptr = linear2farptr (farptr2linear (stack_adr(sector)));
  reread_sector = &space[4096];
  reread_sector_fptr = linear2farptr (farptr2linear (stack_adr(reread_sector)));
  if ((sector_fptr & 0xFFFFU) + 4096 > 0x10000) { /* DMA barrier */
      sector = &space[2*4096];
      sector_fptr = linear2farptr (farptr2linear (stack_adr(sector)));
      }
    else if ((reread_sector_fptr & 0xFFFFU) + 4096 > 0x10000) { /* DMA barrier */
      reread_sector = &space[2 * 4096];
      reread_sector_fptr = linear2farptr (farptr2linear (stack_adr(reread_sector)));
      }

  cpt = 4;
  for (;;) {
      union diskcmd cpycmdblock = cmdblock;
      if (BOOT1_diskread (&cpycmdblock, sector_fptr) == 0)
	  break;
      asm ("" : "=m" (*(struct fourKstruct *)sector)); /* GCC-4.6, do not try to know/optimise what *sector contains */
      if (--cpt == 0) {
	  ZDBG ((" [%s: disable writing boot device, error reading] ", __FUNCTION__));
	  UTIL.boot_device_read_only = 7;
	  return 7;
	  }
      }

  {
  gujin_param_t *param;

  /* if (sectorsize <= 512) // to not rely on comment: "Search 0xBEFC1F03 in disk.c for info", if "instboot --hpcompaqbug" */
  if (cs_peekw (&bootbefore.part1.Bytepersector) <= 512)
      param = (gujin_param_t *) ((unsigned)sector + (unsigned)&gujin_param - 512); /* second sector */
    else
      param = (gujin_param_t *) ((unsigned)sector + (unsigned)&gujin_param); /* first sector */

  if (param->magic != 16980327 || param->version != 0x0208 /*VERSION*/) { /* changed floppy disk ? */
      ZDBG ((" [%s: disable writing boot device, bad magic] ", __FUNCTION__));
      UTIL.boot_device_read_only = 8;
      return 8;
      }

  if (copy_gujin_param.attrib.disk_write_enable) {
      if (!param->attrib.disk_write_enable)
	  ZDBG ((" [%s: re-enabling write / retest UTIL.boot_device_read_only] ", __FUNCTION__));
	else if (UTIL.boot_device_read_only) {
	  /* Re-check boot_device_read_only only when re-enabling write */
	  ZDBG ((" [%s: UTIL.boot_device_read_only set to %u, retest now] ", __FUNCTION__, UTIL.boot_device_read_only));
	  }
      }
    else {
      UTIL.boot_device_read_only = 0;
      if (!param->attrib.disk_write_enable) {
	  ZDBG ((" [%s: disk write disabled in setup and on disk already] ", __FUNCTION__));
	  return 10;
	  }
      ZDBG ((" [%s: force once disabling write on disk] ", __FUNCTION__));
      }

  copy_gujin_param.attrib.write_disabled_written_once = !copy_gujin_param.attrib.disk_write_enable;

#if 0
  unsigned short delta;
  for (cpt = 0, delta = 0; cpt < sizeof(gujin_param_t)/2; cpt++) {
      unsigned short tmp = codeseg_peekw ((char *)&gujin_param + 2*cpt);
      delta += ((unsigned short *)param)[cpt] - tmp;
      ((unsigned short *)param)[cpt] = tmp;
      }
  param->partial_checksum += delta;
#else
  for (cpt = 0; cpt < sizeof(gujin_param_t)/2 - 1; cpt++) {
      unsigned short tmp = codeseg_peekw ((char *)&gujin_param + 2*cpt);
      param->partial_checksum += ((unsigned short *)param)[cpt] - tmp;
      ((unsigned short *)param)[cpt] = tmp;
      }
#endif
  }

  {
  union diskcmd write_cmdblock = cmdblock;
#if 0
  if (write_cmdblock.int1302.reg_ax == 0x0201)
      write_cmdblock.int1302.reg_ax = 0x0301;
    else if (write_cmdblock.int1302.reg_ax == 0x4200)
      write_cmdblock.int1342.reg_ax = 0x4300;
    else { /* if (write_cmdblock.hardide.reg_ax == 0xF00A) */
      if (write_cmdblock.hardide.ide_command == 0x20)
	  write_cmdblock.hardide.ide_command = 0x30;
	else /* if (write_cmdblock.hardide.ide_command == 0x24) */
	  write_cmdblock.hardide.ide_command = 0x34;
      }
#else
  if (write_cmdblock.hardide.reg_ax == 0xF00A)
      write_cmdblock.hardide.ide_command += 0x10;
    else
      write_cmdblock.int1302.reg_ax += 0x0100;
#endif

  asm ("" : : "m" (*(struct fourKstruct *)sector)); /* GCC-4.6, do not try to know/optimise what *sector contains */
  cpt = 4;
  for (;;) {
      union diskcmd cpycmdblock = write_cmdblock;
      if (BOOT1_diskwrite (&cpycmdblock, sector_fptr) == 0)
	  break;
      if (--cpt == 0) {
	  ZDBG ((" [%s: disable writing boot device, writing error] ", __FUNCTION__));
	  UTIL.boot_device_read_only = 10;
	  return 11;
	  }
      }
  }

  cpt = 4;
  for (;;) {
      union diskcmd cpycmdblock = cmdblock;
      if (BOOT1_diskread (&cpycmdblock, reread_sector_fptr) == 0)
	  break;
      asm ("" : "=m" (*(struct fourKstruct *)reread_sector)); /* GCC-4.6, do not try to know/optimise what *reread_sector contains */
      if (--cpt == 0) {
	  ZDBG ((" [%s: disable writing boot device, error re-reading] ", __FUNCTION__));
	  UTIL.boot_device_read_only = 11;
	  return 12;
	  }
      }
  for (cpt = 0; cpt < sectorsize; cpt++) {
      if (sector[cpt] != reread_sector[cpt]) {
	  ZDBG ((" [%s: disable writing boot device, re-reading check failed at byte %u] ", __FUNCTION__, cpt));
	  UTIL.boot_device_read_only = 12;
	  return 13;
	  }
      }
  UTIL.boot_device_read_only = 0;
  }

  return 0;
  }}
#endif /* UPDATE_BOOTPARAM */

/*
Standart DOS MBR:
0x10F : "Invalid partition table"
0x127 : "Error loading operating system"
0x146 : "Missing operating system"
0x1DC..0x1FD : filled with zeroes
0x1FE : 0x55 0xAA

Standart DOS bootsector (not MBR):
0x003 : "MSDOS5.0"
0x02B : 'NO NAME    ' (not zero terminated)
0x036 : 'FAT12   '
0x1A0 : "Disque non-syst\x8Ame/erreur disque\r\nRemplacez et pressez touche\r\n"
0x1E0 : 'IO      SYS'
0x1EB : 'MSDOS   SYS'
0x1FE : 0x55 0xAA

Standart LILO MBR and bootsector:
0x006 : 'LILO'
0x1FE : 0x55 0xAA

Standart LILO in LBA MBR and bootsector:
0x003 : 'lbaLILO'
0x1FE : 0x55 0xAA

Another LILO MBR and bootsector:
0x005 : 'zLILO'
0x1FE : 0x55 0xAA

Extended partition by fdisk:
0x000..0x1BD : 0
0x1FE : 0x55 0xAA

Swap:
0x000 : 0xFE
0x001..0x200 : 0xFF

Standart DOSEMU MBR:
0x000 : "DOSEMU"
0x1FE : 0x00 0x00

Standart DOSEMU bootsector:
0x003 : "DOSEMU  "
0x02B : 'DOSEMU     '
0x036 : 'FAT12   '
0x043..0x1FD : filled with 0
0x1FE : 0x55 0xAA

Standart FREEDOS MBR:
0x000 : "DOSEMU"
0x1FE : 0x00 0x00

Standart FREEDOS bootsector:
0x003 : zeroes
0x02B : 'FREEDOS    '
0x036 : 'FAT12   '
0x1E4 : 'Boot error' (no terminator)
0x1EE : "IPL     SYS"
0x1FE : 0x55 0xAA

XOSLOAD:
0x181 : 'XOSLLOAD'

else if (bootsect->before.part1.NbSectorpertrack == 0x1DEA) // offset 0x18
ptr = "OSL2000";
 */

const char *const recognise_MBR_string[] = {
	"noAA55", "unknown", "nullname", "Gujin", "Grub",
	"zLILO", "lbaLILO", "LILO", "oldLILO", "GAG", "XoslLoad", "Windows",
	"EBIOS MBR", "BIOS MBR", "SYSLINUX", "ISOLINUX"
	};

unsigned recognise_MBR (const bootsector_t *bootsector)
  {bound_stack();{
  unsigned *ptr = (unsigned *)((char *)bootsector + 0x170);

  while (ptr < (unsigned *)((char *)bootsector + 0x1A0)) {
      if (*ptr == *(unsigned *)"GRUB")
	  break;
      ptr = (unsigned *)((char *)ptr + 1);
      }

  if (bootsector->after.Signature0xAA55 != 0xAA55 || bootsector->before.part1.jmpinstruction[0] == 0)
      return 0;
    else if (*ptr == *(unsigned *)"GRUB")
      return 4;	/* also String is kept to previous value (maybe null) */
    else if (STRBEGINEQL ((char *)bootsector + 0x1D, "\x88\x16\x00\x08\xB4\x08\xCD\x13")
	  || STRBEGINEQL ((char *)bootsector + 0x1E7, "Boot error\r\n"))
      return 14;
    else if (*CAST (unsigned long long *, bootsector->before.part1.String) == 0)
      return 2;
    else if (STRBEGINEQL (bootsector->before.part1.String, "Gujin"))
      return 3;
    else if (STRBEGINEQL (&bootsector->before.part1.String[2], "zLILO"))
      return 5;
    else if (STRBEGINEQL (bootsector->before.part1.String, "lbaLILO"))
      return 6;
    else if (STRBEGINEQL (&bootsector->before.part1.String[3], "LILO"))
      return 7;
    else if (STRBEGINEQL ((char *)bootsector + 2, "LILO"))
      return 8;
    else if (STRBEGINEQL ((char *)bootsector + 0x55, "GAG: "))
      return 9;
    else if (STRBEGINEQL ((char *)bootsector + 0x181, "XOSLLOAD"))
      return 10;
    else if (STRBEGINEQL (bootsector->before.part1.String, "NTFS    "))
      return 11;
    else if (STRBEGINEQL ((char *)bootsector + 0x113, "\xB4\x42\x8B\xF4\xCD\x13\x61\x61"))
      return 12;
    else if (STRBEGINEQL ((char *)bootsector + 0x63, "\xB8\x01\x02\x57\xCD\x13\x5F"))
      return 13;
    else if (STRBEGINEQL ((char *)bootsector + 0x94, "isolinux.bin"))
      return 15;
    else
      return 1;
  }}

/* We cannot write those in assembly without forcing the calling sequence - and
	we may want to modify it (-mrtd, -mregparm=) */

void codeseg_memcpy (char *dest, const char *src, unsigned nb) asm ("memcpy");
void codeseg_memcpy (char *dest, const char *src, unsigned nb)
  {
  while (nb--)
      *dest++ = *src++;
  }

void codeseg_memset (char *dest, unsigned val, unsigned nb) asm ("memset");
void codeseg_memset (char *dest, unsigned val, unsigned nb)
  {
  while (nb--)
      *dest++ = val;
  }

#if SETUP & XCODE_SEGMENT
__attribute__ ((section (".diskcode_start"), noreturn))
void diskcodeseg_never_call_address_zero (void)
  {
  unsigned short i;
  for (i = 0; i < 5; i++)
      BOOT1_putstr ("PANIC: address zero in diskcode segment called!\007\r\n");
#if 0
  {
  unsigned *stkptr = (unsigned *)getesp();

  printf ("stack: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\r\n",
		stkptr[0], stkptr[1], stkptr[2], stkptr[3],
		stkptr[4], stkptr[5], stkptr[6], stkptr[7]);
  }
#endif

  for (;;) ;
  }

__attribute__ ((section (".fscode_start"), noreturn))
void fscodeseg_never_call_address_zero (void)
  {
  unsigned short i;
  for (i = 0; i < 5; i++)
      BOOT1_putstr ("PANIC: address zero in fscode segment called!\007\r\n");
#if 0
  {
  unsigned *stkptr = (unsigned *)getesp();
  while (*stkptr == 0)
	stkptr++;

  printf ("stack: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\r\n",
		stkptr[0], stkptr[1], stkptr[2], stkptr[3],
		stkptr[4], stkptr[5], stkptr[6], stkptr[7]);
  }
#endif

  for (;;) ;
  }

__attribute__ ((section (".loadcode_start"), noreturn))
void loadcodeseg_never_call_address_zero (void)
  {
  unsigned short i;
  for (i = 0; i < 5; i++)
      BOOT1_putstr ("PANIC: address zero in loadcode segment called!\007\r\n");
#if 0
  {
  unsigned *stkptr = (unsigned *)getesp();

  printf ("stack: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\r\n",
		stkptr[0], stkptr[1], stkptr[2], stkptr[3],
		stkptr[4], stkptr[5], stkptr[6], stkptr[7]);
  }
#endif

  for (;;) ;
  }

__attribute__ ((section (".usercode_start"), noreturn))
void usercodeseg_never_call_address_zero (void)
  {
  unsigned short i;
  for (i = 0; i < 5; i++)
      BOOT1_putstr ("PANIC: address zero in usercode segment called!\007\r\n");
#if 0
  {
  unsigned *stkptr = (unsigned *)getesp();

  printf ("stack: 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X, 0x%X\r\n",
		stkptr[0], stkptr[1], stkptr[2], stkptr[3],
		stkptr[4], stkptr[5], stkptr[6], stkptr[7]);
  }
#endif

  for (;;) ;
  }
#endif /* XCODE_SEGMENT */

