/* bios.h */
#ifndef BIOS_H
#define BIOS_H

/* Gujin is a bootloader, it loads a Linux kernel from cold boot or DOS.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * General BIOS:
 */

struct _BIOS_equipment_flags_str {
    unsigned	floppy_present			: 1; /* can be set without */
    unsigned	coprocessor80x87installed	: 1;
#if 0
    unsigned	nb_64K_bank			: 2;
#else
    unsigned	pointing_device_installed	: 1;
    unsigned	unused1				: 1;
#endif
    enum {EGA = 0, C40x25, C80x25, M80x25} initial_video : 2;
    unsigned	nb_floppy_less_1		: 2;
    unsigned	DMA_support			: 1;
    unsigned	nb_serial_port			: 3;
    unsigned	game_port_installed		: 1;
    unsigned	serial_printer_attached		: 1; /* or internal modem */
    unsigned	nb_parallel_port		: 2;
    /* eax: */
    unsigned	unused2				: 8;
    unsigned	real_mode_weitek		: 1;
    unsigned	weitek_present			: 1;
    unsigned	internal_DMA_parallel_port	: 1;
    unsigned	IRQ7_DMA_parallel_port		: 1; /* else IRQ 5 */
    unsigned	channel_DMA_parallel_port	: 2;
    } __attribute__ ((packed));

extern inline struct _BIOS_equipment_flags_str
_BIOS_equipment_flags (void)
  {
#if 0 /* GCC-2.95.2 generate a memcpy (&eax, &eax, 4) on this one ! */
  unsigned returned;

#if 1
  asm ASMREAD (" int $0x11 # _BIOS_equipment_flags"
		: "=a" (returned) : "a" (0));
#else
  returned = peekw (0x00400010);
#endif
  return *(struct _BIOS_equipment_flags_str *)&returned;
#else
  struct _BIOS_equipment_flags_str returned;
  asm ASMREAD (" int $0x11 # _BIOS_equipment_flags"
		: "=a" (returned) : "a" (0));
  return returned;
#endif
  }

extern inline unsigned short
_BIOS_get_keyboard_id (void)
  {
  unsigned short returned;

  asm ASMREAD (" int $0x16 # _BIOS_get_keyboard_id"
	: "=b" (returned)
	: "a" (0x0A00), "b" (0)
	);
  return returned;
  }

/*
 * The method used by keyb.com:
 */
extern inline unsigned
_BIOS_is_122key_keyboard (void)
  {
  unsigned short result;

  asm ASMREAD (
"	mov	$0xA2,%%ah				\n"
"	int	$0x16	# _BIOS_is_122key_keyboard	\n"
	: "=a" (result)
	);
  return (result >> 8) <= 0x80;
  }

extern inline unsigned
_BIOS_is_extended_keyboard (void)
  {
  unsigned short result;

  asm ASMREAD (
"	mov	$0x92,%%ah				\n"
"	int	$0x16	# _BIOS_is_extended_keyboard	\n"
	: "=a" (result)
	);
  return (result >> 8) <= 0x80;
  }

/*
 * The official method, to be used only if bit 6 of the
 * second feature byte returned by INT 15/AH=C0h is set:
 */
extern inline struct _BIOS_functionality_keyboard_str {
  unsigned char support_0x0300		: 1;
  unsigned char support_0x0304		: 1;
  unsigned char support_0x0305		: 1;
  unsigned char support_0x0306		: 1;
  unsigned char support_0x0A		: 1;
  unsigned char support_0x10_0x12	: 1;
  unsigned char support_0x20_0x22	: 1;
  } __attribute__ ((packed))
_BIOS_functionality_keyboard (void)
  {
  struct _BIOS_functionality_keyboard_str returned;

  asm ASMREAD (
"	mov	$0x09,%%ah			\n"
"	int	$0x16 # _BIOS_functionality_keyboard\n"
	: "=a" (returned)
	);
  return returned;
  }

extern inline unsigned short
_BIOS_getkey (void)
  {
  unsigned short key16bit;

  asm volatile (
//"	inb	$0x92,%%al		\n"
//"	andb	$0xFC,%%al		\n"
//"	pushl	%%eax			\n"
//"	orb	$0x02,%%al		\n"
//"	outb	%%al,$0x92		\n"
"	xorw	%%ax,%%ax		\n"
"	int $0x16	# _BIOS_getkey	\n"
//"	xchgl	%%eax,(%%esp)		\n"
//"	outb	%%al,$0x92		\n"
//"	popl	%%eax			\n"
	: "=a" (key16bit)
	);

  return key16bit;
  }

/* If we do not declare the asm() volatile for _BIOS_checkkey() and _BIOS_extkbd_checkkey(),
   then GCC copy the return value into two local variables and test them, see BIOS_getkey() assembly.
   The problem is that the two _BIOS_checkkey() and _BIOS_extkbd_checkkey() are executed and it seems
   that _BIOS_extkbd_checkkey() may return false if it just follow a true return from _BIOS_checkkey(),
   as if some BIOS clear the "key pressed" flag after _BIOS_checkkey() execution...
   */
extern inline unsigned char
_BIOS_checkkey (void)
  {
  unsigned char key_waiting;

  asm volatile (
// "	 inb	 $0x92,%%al		\n"
// "	 andb	 $0xFC,%%al		\n"
// "	 pushl	 %%eax			\n"
// "	 orb	 $0x02,%%al		\n"
// "	 outb	 %%al,$0x92		\n"
"	mov	$0x01,%%ah		\n"
"	int	$0x16	# _BIOS_checkkey		\n"
// "	 xchgl	 %%eax,(%%esp)		\n"
// "	 outb	 %%al,$0x92		\n"
// "	 popl	 %%eax			\n"
	// we do not use the returned %%ax, so overwrite:
"	setnz	%%al	# ZF clear if key available	\n"
	: "=a" (key_waiting)
	);
  return key_waiting;
  }

extern inline struct shift_flags_str {
    unsigned char right_shift_key_pressed	: 1;
    unsigned char left_shift_key_pressed	: 1;
    unsigned char control_key_pressed		: 1;
    unsigned char alternate_key_pressed		: 1;
    unsigned char scroll_lock_active		: 1;
    unsigned char num_lock_active		: 1;
    unsigned char caps_lock_active		: 1;
    unsigned char insert_active			: 1;
    }
_BIOS_get_shift_flags (void)
  {
  struct shift_flags_str flags;

  asm volatile (
"	mov	$0x02,%%ah			\n"
"	int	$0x16	# _BIOS_get_shift_flags	\n"
	: "=a" (flags)
	);
  return flags;
  }

extern inline unsigned short
_BIOS_extkbd_getkey (void)
  {
  unsigned short key16bit;

  asm volatile (
//"	inb	$0x92,%%al		\n"
//"	andb	$0xFC,%%al		\n"
//"	pushl	%%eax			\n"
//"	orb	$0x02,%%al		\n"
//"	outb	%%al,$0x92		\n"
"	mov	$0x10,%%ah			\n"
"	int	$0x16	# _BIOS_extkbd_getkey	\n"
//"	xchgl	%%eax,(%%esp)		\n"
//"	outb	%%al,$0x92		\n"
//"	popl	%%eax			\n"
	: "=a" (key16bit)
	);
  return key16bit;
  }

extern inline unsigned char
_BIOS_extkbd_checkkey (void)
  {
  unsigned char key_waiting;

  asm volatile (
// "	  inb	  $0x92,%%al		\n"
// "	  andb	  $0xFC,%%al		\n"
// "	  pushl   %%eax			\n"
// "	  orb	  $0x02,%%al		\n"
// "	  outb	  %%al,$0x92		\n"
"	mov	$0x11,%%ah			\n"
"	int	$0x16	# _BIOS_extkbd_checkkey	\n"
// "	  xchgl   %%eax,(%%esp)		\n"
// "	  outb	  %%al,$0x92		\n"
// "	  popl	  %%eax			\n"
	// we do not use the returned %%ax, so overwrite:
"	setnz	%%al	# ZF clear if key available	\n"
	: "=a" (key_waiting)
	);

  return key_waiting;
  }

extern inline unsigned
_BIOS_extkbd_ungetkey (unsigned short key16bit)
  {
  unsigned short command = 0x0500;

  asm volatile (" int $0x16 # _BIOS_extkbd_ungetkey"
	: "+a" (command)
	: "c" (key16bit)
	);
  return (command & 0xFF) == 0; /* true if successfull */
  }

extern inline void
_BIOS_set_repeat_rate_delay (unsigned char repeat, unsigned char delay)
  {
  unsigned short cmd = 0x0305;
  asm volatile (" int	$0x16	# _BIOS_set_repeat_rate_delay "
	: "+a" (cmd)
	: "b" (((unsigned short)delay << 8) | repeat)
	);
  }

extern inline unsigned
_BIOS_ungetkey (unsigned short key16bit)
  {
  unsigned short head, queue, nexthead, keyb_start, keyb_end;

  if (_BIOS_checkkey ())
      return 1;	/* do it only once */

  if (   _BIOS_is_extended_keyboard ()
      && _BIOS_extkbd_ungetkey(key16bit))
      return 0;

  head = peekw (0x0040001C);
  nexthead = head + 2;
  keyb_end= peekw (0x00400082);
  if (nexthead >= keyb_end) {
      keyb_start = peekw (0x00400080);
      nexthead = keyb_start;
      }
  queue = peekw (0x0040001A);

  if (nexthead == queue)
      return 1; /* queue full */

  /* binutils-2.10.0 and before was generating bad hexadecimal
     for "leal 4194428(%eax),%edi", but now I have better pokew()
     which do not produce such an instruction. */
  pokew (0x00400000 + head, key16bit);
  pokew (0x0040001C, nexthead);

  return 0;
  }

extern inline void
_BIOS_reset_control_break (void)
  {
  unsigned char val = peekb (0x00400071) & ~0x80;
  pokeb (0x00400071, val);
  }

extern inline unsigned
_BIOS_control_break_pressed (void)
  {
  return !!(peekb (0x00400071) & 0x80);
  }

extern inline unsigned short _BIOS_power_OFF(void)
  {
  unsigned status;
  asm volatile (
"	int	$0x15	# _BIOS_power_OFF	\n"
	: "=a" (status) : "a" (0x4200) /* or 0x4201 ? */
	);
  return status;
  }

extern inline unsigned
_BIOS_wait (unsigned nb_microsec)
  {
  unsigned char error;
  unsigned short status;

  asm volatile (
"	int	$0x15	# _BIOS_wait			\n"
"	setc	%0	# CF clear if successful	\n"
	: "=qm" (error), "=a" (status)
	: "a" (0x8600), "d" (nb_microsec & 0xFFFF),
		"c" (nb_microsec >> 16)
	);
  return error;
  }

extern inline unsigned
_BIOS_nbtick (void) /* since midnight */
  {
  return peekl (0x0040006C);
  }
#define TICKS_PER_SECOND	18

struct biostime_t {
    unsigned char hour, minute, second, daylight_saving;
    } __attribute__((packed));

extern inline unsigned _BIOS_gettime (struct biostime_t *biostime)
  {
  unsigned char error;
  unsigned short tmpcx, tmpdx;
  asm (
"	int	$0x1A	\n"
"	setc	%0	# CF clear if successful	\n"
	: "=qm" (error), "=c" (tmpcx), "=d" (tmpdx)
	: "a" (0x0200));
  if (error)
	return 1;
  biostime->hour = tmpcx >> 8;
  biostime->minute = tmpcx & 0xFF;
  biostime->second = tmpdx >> 8;
  biostime->daylight_saving = tmpdx & 0xFF;
#if 0
  /* They are in BCD, convert to number: */
  biostime->hour = 10 * (biostime->hour >> 4) + (biostime->hour & 0x0F);
  if (biostime->hour > 23)
	return 2;
  biostime->minute = 10 * (biostime->minute >> 4) + (biostime->minute & 0x0F);
  if (biostime->minute > 63)
	return 3;
  biostime->second = 10 * (biostime->second >> 4) + (biostime->second & 0x0F);
  if (biostime->second > 63)
	return 3;
#endif
  return 0;
  }

struct JoystickButton_str {
    unsigned char unused	: 4;
    unsigned char button_A	: 1;
    unsigned char button_B	: 1;
    unsigned char button_C	: 1;
    unsigned char button_D	: 1;
    }  __attribute__ ((packed));

extern inline unsigned char
JoystickButton (struct JoystickButton_str *button)
  {
  unsigned char error;

  /* In this function, the returned %ah seems to be 0x48,
     is that a kind of standart identification ? */
  /* if no Joystick, all button are set, read 0xF0 */
  asm (
"	int	$0x15	# JoystickButton	\n"
"	setc	%0				\n"
	: "=qm" (error), "=a" (*button)
	: "a" (0x8400), "d" (0)
	);
  return error;
  }

extern inline unsigned char
JoystickButtonEqual (const struct JoystickButton_str bt1, const struct JoystickButton_str bt2)
  {
  const union {
      struct JoystickButton_str bits;
      unsigned char all;
      } tmp1 = { .bits = bt1}, tmp2 = { .bits = bt2};
  return tmp1.all == tmp2.all;
  }

extern inline unsigned char
JoystickPosition (unsigned short *Ax, unsigned short *Ay,
		  unsigned short *Bx, unsigned short *By)
  {
  unsigned char error;
  /* usually in the range 0..1A0 with 250 kOhm */

  /* I have a cheap one here, non proportionnal:
     Left: Ax = 4, Middle: Ax = 67, Right: Ax = 131.
     Up:   Ay = 4, Middle: Ay = 69, Down:  Ay = 134
     By is nearly always 68-69, Bx depends on two
     other "fast shoot buttons" */
  /* Presence detect: at least one A{x,y} or B{x,y} different of 0 */
  asm (
"	int	$0x15	# JoystickPosition	\n"
"	setc	%0				\n"
//	: "=qm" (error), /* use cx or dx if *Bx or *By not used, do not compile with gcc-2.95.3 */
	: "=m" (error),
	  "=a" (*Ax), "=b" (*Ay),
	  "=c" (*Bx), "=d" (*By)
	: "a" (0x8400), "d" (1)
	);
  return error;
  }

/*
 * Serial BIOS interface:
 */
typedef struct {
    union {
	unsigned char charread;
	struct {
	    unsigned char delta_cts : 1;
	    unsigned char delta_dsr : 1;
	    unsigned char teri      : 1;
	    unsigned char delta_cd  : 1;
	    unsigned char cts       : 1;
	    unsigned char dsr       : 1;
	    unsigned char ri        : 1;
	    unsigned char cd        : 1;
	    } __attribute__ ((packed)) modem;
	} al;
    unsigned char receive_data_ready : 1;
    unsigned char overrun_error      : 1;
    unsigned char parity_error       : 1;
    unsigned char framming_error     : 1;
    unsigned char break_detected     : 1;
    unsigned char tx_holding_empty   : 1;
    unsigned char tx_shift_empty     : 1;
    unsigned char timeout            : 1;
    } __attribute__ ((packed)) _BIOS_serialstatus;

struct serial_setup_str {
    enum { serial_7bit = 2, serial_8bit = 3 } word_length : 2;
    enum { serial_1stopbit, serial_2stopbit } stop_bit : 1;
    enum { serial_none = 0, serial_odd = 1, serial_even = 3} parity : 2;
    enum { B110b, B150b, B300b, B600b, B1200b, B2400b, B4800b, B9600b } speed : 3;
    } __attribute__ ((packed));

extern inline _BIOS_serialstatus
_BIOS_initserial (unsigned short port, struct serial_setup_str byte)
  {
  _BIOS_serialstatus returned;

  asm volatile (" int $0x14 # _BIOS_initserial"
	: "=a" (returned)
	: "a" (0x0000 | *(unsigned char *)&byte), "d" (port)
	);
  return returned;
  }

extern inline _BIOS_serialstatus
_BIOS_putserial (unsigned short port, unsigned char byte)
  {
  _BIOS_serialstatus returned;

  asm volatile (" int $0x14 # _BIOS_putserial"
	: "=a" (returned)
	: "a" (0x0100 | byte), "d" (port)
	);
  return returned;
  }

extern inline _BIOS_serialstatus
_BIOS_getserial (unsigned short port)
  {
  _BIOS_serialstatus returned;

  /* WARNING: This function will reset RTS if it timeouts ! */

  asm volatile (" int $0x14 # _BIOS_getserial"
	: "=a" (returned)
	: "a" (0x0200), "d" (port)
	);
  return returned;
  }

extern inline _BIOS_serialstatus
_BIOS_getserialstatus (unsigned short port)
  {
  _BIOS_serialstatus returned;

  asm ASMREAD (" int $0x14 # _BIOS_getserialstatus"
	: "=a" (returned)
	: "a" (0x0300), "d" (port)
	);
  return returned;
  }

enum _BIOS_ext_serial_parity {
    no_parity, odd_parity, even_parity, stick_parity_odd, stick_parity_even
    };

enum _BIOS_ext_serial_stopbit { one_stop, two_stop  };

enum _BIOS_ext_serial_wordlen { fivebit, sixbit, sevenbit, eightbit };

enum _BIOS_ext_serial_speed {
    B110e, B150e, B300e, B600e, B1200e, B2400e, B4800e, B9600e, B19200e
    };

extern inline _BIOS_serialstatus
_BIOS_extended_initserial (unsigned short port,
			   unsigned char break_set,
			   enum _BIOS_ext_serial_parity  parity,
			   enum _BIOS_ext_serial_stopbit stopbit,
			   enum _BIOS_ext_serial_wordlen wordlen,
			   enum _BIOS_ext_serial_speed   speed
			   )
  {
  _BIOS_serialstatus returned;

  asm volatile (" int $0x14 # _BIOS_initserial"
	: "=a" (returned)
	: "a" (0x0400 | !!break_set), "d" (port),
	  "b" (((unsigned short)parity << 8) | stopbit),
	  "c" (((unsigned short)wordlen << 8) | speed)
	);
  return returned;
  }

struct serial_modem_str {
    unsigned char dtr       : 1;
    unsigned char rts       : 1;
    unsigned char out1      : 1;
    unsigned char out2      : 1;
    unsigned char loopback  : 1;
    unsigned char reserved0 : 3;
    } __attribute__ ((packed));

extern inline _BIOS_serialstatus
_BIOS_extended_get_modem_control (unsigned short port,
				  struct serial_modem_str *modem)
  {
  _BIOS_serialstatus returned;
  /* This could be declared "inline const struct" in GCC version ?
     static const struct serial_modem_str full = {1,1,1,1,1,7}; */

  asm ASMREAD (" int $0x14 # _BIOS_extended_get_modem_control "
	: "=a" (returned), "=b" (*modem)
	: "a" (0x0500), "d" (port),
	  "b" ((struct serial_modem_str) {1,1,1,1,1,7})
	);
  return returned;
  }

extern inline _BIOS_serialstatus
_BIOS_extended_set_modem_control (unsigned short port,
				  struct serial_modem_str modem)
  {
  _BIOS_serialstatus returned;

  asm volatile (" int $0x14 # _BIOS_extended_set_modem_control "
	: "=a" (returned)
	: "a" (0x0501), "d" (port), "b" (modem)
	);
  return returned;
  }

/*
 * Bios for memory:
 */

extern inline unsigned short
_BIOS_getBaseMemory (void)
  {
  unsigned short sizeinKb;

  asm ASMREAD (" int $0x12 # _BIOS_getBaseMemory" : "=a" (sizeinKb));
  return sizeinKb;
  }

extern inline unsigned short
_BIOS_getEBDA_segment (void)
  {
  unsigned short segment;

  asm ASMREAD (
"	pushl	%%es				\n"
"	xorl	%%eax,%%eax			\n"
"	movw	%%ax,%%es			\n"
"	movb	$0xC1,%%ah			\n"
"	stc					\n"
"	int	$0x15	# _BIOS_getEBDA_segment \n"
"	jc	1f				\n"
"	movw	%%es,%%ax			\n"
"	jmp	2f				\n"
"	1:					\n"
"	xorw	%%ax,%%ax			\n"
"	2:					\n"
"	popl	%%es				\n"
	: "=a" (segment)
	: : "cc");
  return segment;
  }

extern inline unsigned char
_BIOS_getExtendedMemory (unsigned short *sizeinKb)
  {
  unsigned char error;

  asm ASMREAD (
"	int	$0x15	# _BIOS_getExtendedMemory	\n"
"	setc	%0					\n"
	: "=qm" (error), "=a" (*sizeinKb)
	: "a" (0x8800)
	);
  return error; /* 0 if no error */
  }

extern inline unsigned
_BIOS_getPhoenixExtendedMemory (void)
  {
  unsigned char error;
  unsigned short msb_sizeinKb, lsb_sizeinKb;

  asm ASMREAD (
"	int	$0x15	# _BIOS_getPhoenixExtendedMemory	\n"
"	setc	%0						\n"
	: "=qm" (error), "=a" (lsb_sizeinKb), "=d" (msb_sizeinKb)
	: "a" (0x8A00), "d" (0)
	);
  if (error)
      return 0;
  return (((unsigned)msb_sizeinKb) << 16) | lsb_sizeinKb;
  }

extern inline unsigned
_BIOS_getAMIExtendedMemory (void)
  {
  unsigned char error;
  unsigned short check;
  unsigned sizeinKb;

  asm ASMREAD (
"	int	$0x15	# _BIOS_getAMIExtendedMemory	\n"
"	setc	%0					\n"
"	movzbw	%%cl,%%cx				\n"
"	shll	$16,%%ecx				\n"
"	movw	%%bx,%%cx				\n"
	: "=qm" (error), "=a" (check), "=c" (sizeinKb)
	: "a" (0xDA88)
	: "ebx"
	);
  if (error || check != 0)
      return 0;
  return sizeinKb;
  }

extern inline unsigned char
_BIOS_getExtendedE801Memory (unsigned short *low16Mb_inKb,
			      unsigned short *high16Mb_in64Kb)
  {
  unsigned char error;
  /* The *_config returned values are just copies.
     When a memory hole is selected in the BIOS,
     high16Mb_in64Kb usually returns zero */
  /* max of low16Mb_inKb is 15Mb, i.e. 0x3C00 (< 0xE801). */

#if (__GNUC__ > 3) || (__GNUC__ == 3 && __GNUC_MINOR__ >= 1)
  /* I think I should thank someone for GCC-3.0.2 */
  unsigned short _low16Mb_inKb, _high16Mb_in64Kb = 0;
  unsigned short _low16Mb_inKb_config = 0, _high16Mb_in64Kb_config = 0;

  asm (
"	int     $0x15	# _BIOS_getExtendedE801Memory	\n"
"	setc    %0					\n"
	: "=m" (error), "=a" (_low16Mb_inKb), "+b" (_high16Mb_in64Kb),
	  "+c" (_low16Mb_inKb_config), "+d" (_high16Mb_in64Kb_config)
	: "a" (0xE801)
	);
#else
  unsigned short _low16Mb_inKb, _high16Mb_in64Kb;
  unsigned short _low16Mb_inKb_config, _high16Mb_in64Kb_config;

  asm (
"	xorl	%%ebx,%%ebx				\n"
"	xorl	%%ecx,%%ecx				\n"
"	xorl	%%edx,%%edx				\n"
"	int     $0x15	# _BIOS_getExtendedE801Memory	\n"
"	setc    %0					\n"
	: "=m" (error), "=a" (_low16Mb_inKb), "=b" (_high16Mb_in64Kb),
	  "=c" (_low16Mb_inKb_config), "=d" (_high16Mb_in64Kb_config)
	: "a" (0xE801)
	);
#endif
  if (_low16Mb_inKb_config != 0 && _low16Mb_inKb_config <= 0x3C00 && !error) {
      /* Linux used to get values from dx:cx, use them
	 if low16Mb_inKb_config valid and not zero: */
      *low16Mb_inKb = _low16Mb_inKb_config;
      *high16Mb_in64Kb = _high16Mb_in64Kb_config;
      }
    else {
      *low16Mb_inKb = _low16Mb_inKb;
      *high16Mb_in64Kb = _high16Mb_in64Kb;
      }
  return error || (*low16Mb_inKb > 0x3C00); /* 0 if no error */
  }

struct e820map_info {
    unsigned long long base, length;
    enum { MemAvailable = 1, MemSpecial = 2,
		Mem_ACPI_reclaim = 3, Mem_ACPI_NVS = 4 } type : 32;
    } __attribute__ ((packed));

extern inline unsigned
_BIOS_QueryMemoryMap (unsigned *cont_val, unsigned bufsize, struct e820map_info *buffer)
{
       unsigned smapsig = 0x534D4150;       // 'SMAP' in big endian, can be modified by some BIOS
       unsigned smap_res = 0xE820U, actual_len;

       asm volatile (
"	int     $0x15   # _BIOS_QueryMemoryMap          \n"
"	jnc     1f                                      \n"
"	xorl    %%eax,%%eax                             \n"
"	1:                                              \n"
	: "+a" (smap_res), "+b" (*cont_val), "=c" (actual_len), "+d" (smapsig), ASM_STRUCT_OUTPUT (*buffer)
	: "c" (bufsize), "D" (buffer)      /* in fact %%es:%%di */
	: "%esi"	/* clobbered */
	);
       if (smap_res != smapsig)
	       return 0;
       return actual_len;
}

extern inline unsigned short
_BIOS_copy_extended_memory (unsigned dst, unsigned src, unsigned nbword)
  {
  unsigned short status;
  unsigned limit = 2 * nbword - 1;
  struct {
      /* see library.h for "dataseg_t" */
      dataseg_t dummy, temporary; /* used by BIOS */
      dataseg_t src, dst;
      dataseg_t cs, ss;
      } gdt = {
      src: {
	limit: limit, limit_msb: limit >> 16,
	base: src, base_msb: src >> 24,
	// Why Phoenix ROM BIOS PLUS Version 1.10 A18 on a
	// Dell System OptiPlex GL133 needs write access here?
	writable: 1,
	accessed: 1, expansion_direction: 0, cste_10: 2, dpl: 0, present: 1,
	granularity: 0, big: 0, cste_0: 0, available: 0
	},
      dst: {
	limit: limit, limit_msb: limit >> 16,
	base: dst, base_msb: dst >> 24,
	writable: 1,
	accessed: 1, expansion_direction: 0, cste_10: 2, dpl: 0, present: 1,
	granularity: 0, big: 0, cste_0: 0, available: 0
	}
      };

  asm ASMREAD (
"	int     $0x15	# _BIOS_copy_extended_memory	\n"
"	setc    %%al					\n"
	: "=a" (status)
	: "a" (0x8700), "c" (nbword), "S" (&gdt) /* in fact %%es:%%si */
	  , ASM_STRUCT_INPUT (gdt)
	);
/* returns:
 0000h	source copied into destination
 0101h	parity error
 0201h	interrupt error
 0301h	address line 20 gating failed
 8001h	invalid command (PC,PCjr)
 8601h	unsupported function (XT,PS30)
 xx01h  ?unknown error?

  BUG? DOSEMU set carry and returns %ah = 0 ?
 */
  return status;
  }

/*
 * Basic DPMI interface:
 */
extern inline unsigned short
_DPMI_installation_check (void)
  {
  struct {
      unsigned short support_32bit_program : 1;
      } flags;
  enum { UP_80286 = 2, UP_80386 = 3, UP_80486 = 4 } processor_t;
  struct {
      unsigned char twodigit_minor_version;
      unsigned char major_version;
      } version;
  unsigned short number_of_paragraphs_of_DOS_extender_private_data;
  farptr DPMI_mode_switch_entry_point;
  unsigned short not_installed;

  asm ASMREAD (
"	pushl	%%es		\n"
"	int	$0x2F		\n"
"	pushw	%%es		\n"
"	pushw	%%di		\n"
"	popl	%%edi		\n"
"	popl	%%es		\n"
	: "=a" (not_installed), /* zero if installed */
	  "=b" (flags), "=c" (processor_t), "=d" (version),
	  "=S" (number_of_paragraphs_of_DOS_extender_private_data),
	  "=D" (DPMI_mode_switch_entry_point)
	: "a" (0x1687)
	);
  return not_installed;
  }

extern inline unsigned short
_DPMI_int0x31_available (void)
  {
  unsigned short returned = 0x1686;

  asm ASMREAD (" int	$0x2F	# _DPMI_int0x31_available "
	: "+a" (returned)
	);
  return !returned;
  }

extern inline unsigned char
_DPMI_get_real_mode_interrupt_vector (unsigned char nb, farptr *vector)
  {
  unsigned char error;

  asm ASMREAD (
"	stc								\n"
"	int     $0x31	# _DPMI_get_real_mode_interrupt_vector (DPMI 0.9+)\n"
"	pushw	%%cx	# segment					\n"
"	pushw	%%dx	# offset					\n"
"	popl	%%edx							\n"
"	setc    %%cl							\n"
	: "=d" (*vector), "=c" (error)
	: "a" (0x0200), "b" (nb), "c" (0), "d" (0)
	: "cc"
	);
  return error;
  }

extern inline unsigned char
_DPMI_set_real_mode_interrupt_vector (unsigned char nb, farptr vector)
  {
  unsigned char error;

  /* Note:   all memory that may be touched by a hardware interrupt handler
	 must be locked down with INT 31/AX=0600h */

  asm volatile (
"	stc								\n"
"	int     $0x31   # _DPMI_set_real_mode_interrupt_vector (DPMI 0.9+)\n"
"	setc    %%cl							\n"
	: "=c" (error)
	: "a" (0x0201), "b" (nb), "c" (vector >> 16), "d" (vector)
	: "cc"
	);
  return error;
  }

/*
 * A20 playing is not supported by every BIOS...
 */

/*
 * First method:
 */
extern inline farptr
GetExternalA20Handler (unsigned char *can_report)
  {
  unsigned char status;
  farptr returned;

  asm ASMREAD (
"	pushl	%%es		\n"
"	xorw	%%bx,%%bx	\n"
"	movw	%%bx,%%es	\n"
"	xorb	%%cl,%%cl	\n"
"	int	$0x2F		\n"
"	pushw	%%es		\n"
"	pushw	%%bx		\n"
"	popl	%%ebx		\n"
"	popl	%%es		\n"
	: "=a" (status), "=c" (*can_report), "=b" (returned)
	: "a" (0x4330)
	: "cc"
	);
  if (status != 0x80)
      return 0;
  return returned;
  }

extern inline unsigned char
SetA20 (farptr A20Handler, unsigned enable)
  {
  unsigned short status;

  asm volatile (
"	lcallw	*%a2	\n"	// it can be 16(%esp)
	: "=a" (status)
	: "a" (!!enable), "g" (&A20Handler)
	);
  return status; /* = 1 if success */
  }

extern inline unsigned char
GetA20 (farptr A20Handler) /* only if *can_report */
  {
  unsigned short status;

  asm volatile (
"	lcallw	*%a2	\n"	// it can be 16(%esp)
	: "=a" (status)
	: "a" (2), "g" (&A20Handler)
	);
  return status; /* = 1 if A20 enabled */
  }

/*
 * Second method:
 */
struct PS2_A20_str {
    unsigned short kbd_ctrl    : 1;
    unsigned short IO0x92_bit1 : 1;
    unsigned short unknown     : 14;
    };

extern inline unsigned short
PS2_A20_supported (struct PS2_A20_str *status)
  {
  unsigned short returned;

  asm ASMREAD (
"	int	$0x15 # PS2_A20_supported	\n"
"	setc	%%al  # error code in %%ah	\n"
	: "=a" (returned), "=b" (*status)
	: "a" (0x2403), "b" (0)
	: "cc"
	);
  return returned; /* 0 if OK */
  }

extern inline unsigned short
PS2_get_A20 (unsigned char *enabled)
  {
  unsigned short returned, timeout;
  unsigned char error;

  asm ASMREAD (
"	int	$0x15 # PS2_get_A20	\n"
"	setc	%%dl			\n"
	: "=d" (error), "=c" (timeout), "=a" (returned)
	: "a" (0x2402)
	: "cc"
	);
  if (error) /* carry set */
      return 0xFF00 | (returned >> 8); /* 0xFF86 -> unsupported */
  if ((returned & 0xFF00) != 0) /* undocumented error */
      return 0xFE00 | (returned >> 8);
  if (timeout == 0xFFFF) /* keyboard controller ready timeout */
      return 0xFFFF;
  *enabled = returned;
  return 0;
  }

extern inline unsigned short
PS2_set_A20 (unsigned char enabled)
  {
  unsigned short returned;

  asm ASMREAD (
"	int	$0x15 # PS2_set_A20	\n"
"	setc	%%al			\n"
	: "=a" (returned)
	: "a" (0x2400 | !!enabled)
	: "cc"
	);
  return returned; /* 0 if OK, 0x8601 if unsupported or 0x0101 if error */
  }

/*
 * Some configuration stuff:
 */
struct PC_configuration_str {
    unsigned short	nb_bytes_following;
    unsigned char	model;
    unsigned char	submodel;
    unsigned char	BIOS_revision;
    struct {
	unsigned char	dual_bus			: 1;
	unsigned char	MicroChannel_bus		: 1;
	unsigned char	extendedBIOSarea_allocated	: 1;
	unsigned char	wait_for_external_event_supported	: 1;
	unsigned char	INT15ah4F_called_upon_INT09	: 1;
	unsigned char	RealTimeClock_installed		: 1;
	unsigned char	second_8259_installed		: 1;
	unsigned char	DMA_channel3_used_by_HDbios	: 1;
	} feature1;
    struct {
	unsigned char	reserved			: 1;
	unsigned char	data_streaming_supported	: 1;
	unsigned char	non_8042_keyboard_controller	: 1;
	unsigned char	INT15ahC8_supported		: 1;
	unsigned char	INT15ahC7_supported		: 1;
	unsigned char	INT15ahC6_supported		: 1;
	unsigned char	INT16ah09_supported		: 1;
	unsigned char	DMA32bit_supported		: 1;
	} feature2;
    struct {
	unsigned char	SCSI_in_IML_supported		: 1;
	unsigned char	InitialMachineLoad_system	: 1;
	unsigned char	information_pannel_installed	: 1;
	unsigned char	SCSI_on_system_board		: 1;
	unsigned char	POST_ROMtoRAM_support		: 1;
	unsigned char	reserved1			: 1;
	unsigned char	reserved2			: 1;
	unsigned char	unused				: 1;
	} feature3;
    struct {
	unsigned char	POSTEXT_supported_by_POST	: 1;
	unsigned char	memory_split16M_supported	: 1;
	unsigned char	said_private			: 1;
	enum { not_supported = 1,
	       supportedInROM, supportedInRAM } ABIOS	: 3;
	unsigned char	system_has_EEPROM		: 1;
	unsigned char	IBM_private			: 1;
	} feature4;
    struct {
	unsigned char	FLASH_EPROM	: 1;
	unsigned char	enhanced_mouse	: 1;
	unsigned char	reserved	: 3;
	unsigned char	IBM_private	: 3;
	} feature5;
    unsigned char BIOS_dependant[6];
    } __attribute__ ((packed));

extern inline farptr _BIOS_getConfiguration (unsigned short *XtAt_carry)
  {
  farptr conf;

  asm ASMREAD (
"	pushl	%%es						\n"
"	xorl	%%ebx,%%ebx					\n"
"	movw	%%bx,%%es					\n"
"	stc							\n"
"	int	$0x15		# _BIOS_getConfiguration	\n"
"	setc	%%al						\n"
"	pushw	%%es						\n"
"	pushw	%%bx						\n"
"	popl	%%ebx						\n"
"	popl	%%es						\n"
	: "=a" (*XtAt_carry), "=b" (conf)
	: "a" (0xC000)
	: "cc"
	);
  return conf;
  }

/*
 * Auto Power Mode interface:
 */
extern inline unsigned char
_APM_installation_check (unsigned short *BCD_version,
			unsigned short *flags)
  {
  unsigned char error;
  unsigned short sig;

  asm ASMREAD (
"	int     $0x15	# _APM_installation_check	\n"
"	setc    %0					\n"
	: "=qm" (error), "=a" (*BCD_version),
	  "=b" (sig), "=c" (*flags)
	: "a" (0x5300), "b" (0)
	);
  if (error) {
      *BCD_version &= 0xFF00; /* keep there error code */
      *flags = 0;
      }
  return error || sig != 0x504D; /* "PM" */
  }

extern inline unsigned short
_APM_disconnect (void)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# _APM_disconnect	\n"
"	setc    %%al				\n"
	: "=a" (error)
	: "a" (0x5304), "b" (0)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  return error;
  }

extern inline unsigned char
_APM_connect_realmode (void)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# _APM_connect_realmode	\n"
"	setc    %%al				\n"
	: "=a" (error)
	: "a" (0x5301), "b" (0)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  error >>= 8;
  if (error == 0)
      return 0xFF;
    else
      return error;
  }

extern inline unsigned char
_APM_connect_32bit (unsigned short *RM_segbase_32,
		   unsigned	  *entry_point_offset_32,
		   unsigned short *RM_segbase_code16,
		   unsigned short *RM_segbase_data16,
		   /* APM v1.1: */
		   unsigned short *APM_BIOS_codeseg_len,
		   unsigned short *APM_BIOS_dataseg_len)
  {
  unsigned char error;

  asm ASMREAD (
"	xorl	%%edi,%%edi			\n"
"	xorl	%%esi,%%esi			\n"
"	int     $0x15	# _APM_connect_32bit	\n"
"	setc    %0				\n"
	: "=qm" (error),
	  "=a" (*RM_segbase_32),
	  "=b" (*entry_point_offset_32),
	  "=c" (*RM_segbase_code16),
	  "=d" (*RM_segbase_data16),
	  "=S" (*APM_BIOS_codeseg_len),
	  "=D" (*APM_BIOS_dataseg_len)
	: "a" (0x5303), "b" (0)
	);
  return error;
  }

/* Only for APM v1.1+ : */
extern inline unsigned char
_APM_declare_driver_version (unsigned bcd_version)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# _APM_declare_driver_version	\n"
"	# ignore value returned in %%ax			\n"
"	setc    %%al					\n"
	: "=a" (error)
	: "a" (0x530E), "b" (0), "c" (bcd_version)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  return error >> 8;
  }

enum APM_CPU_enum { APM_CPU_idle = 0, APM_CPU_busy = 1};

extern inline unsigned char
_APM_CPU (enum APM_CPU_enum state)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# APM_CPU_{idle,busy}	\n"
"	setc    %%al				\n"
	: "=a" (error)
	: "a" (0x5305 + state), "b" (0)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  error >>= 8;
  if (error == 0)
      return 0xFF;
    else
      return error;
  }

enum APM_POWER_STATE_enum {
    APM_ready	= 0,	/* not for APM_alldevice */
    APM_standby	= 1,
    APM_suspend	= 2,
    APM_off	= 3	/* not for APM_alldevice */
    };

enum APM_POWER_DEVICE_enum {
    APM_system		= 0, /* : not sure of this one */
    APM_alldevice	= 1,
    APM_alldisplay	= 0x1FF,
    APM_allsecstorage	= 0x2FF,
    APM_allparport	= 0x3FF,
    APM_allserport	= 0x4FF,
	// more for APM v1.1:
    APM_allnetwork	= 0x5FF,
    APM_allpcmcia	= 0x6FF,
	// more for APM v1.2:
    APM_allbattery	= 0x5FF,

    APM_allOEM		= 0xEFF
    };

extern inline unsigned char
_APM_state (enum APM_POWER_DEVICE_enum device, enum APM_POWER_STATE_enum state)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# APM_STATE	\n"
"	setc    %%al			\n"
	: "=a" (error)
	: "a" (0x5307), "b" (device), "c" (state)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  error >>= 8;
  if (error == 0)
      return 0xFF;
    else
      return error;
  }

enum APM_POWER_MANAGE_enum {
    APM_disable = 0, APM_enable = 1
    };

extern inline unsigned char
_APM_power_manage (enum APM_POWER_DEVICE_enum device,
		   enum APM_POWER_MANAGE_enum state)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# APM_power_manage	\n"
"	setc    %%al				\n"
	: "=a" (error)
	: "a" (0x530D), "b" (device), "c" (state)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  error >>= 8;
  if (error == 0)
      return 0xFF;
    else
      return error;
  }

enum APM_POWER_ENGAGED_enum {
    APM_disengage = 0, APM_engage = 1
    };

extern inline unsigned char
_APM_power_engage (enum APM_POWER_DEVICE_enum device,
		   enum APM_POWER_ENGAGED_enum state)
  {
  unsigned short error;

  asm volatile (
"	int     $0x15	# APM_power_engage	\n"
"	setc    %%al				\n"
	: "=a" (error)
	: "a" (0x530F), "b" (device), "c" (state)
	: "cc"
	);
  if ((error & 0xFF) == 0)
      return 0;
  error >>= 8;
  if (error == 0)
      return 0xFF;
    else
      return error;
  }

/*
 * Which comment to put here? This beer was nice.
 */

extern inline void
_BIOS_failedboot (void)
  {
  /* May try to boot from the network card BIOS or something else... */
  asm volatile (" int $0x18 # _BIOS_failedboot" : : "a" (0));
  }

extern inline void
_DOS_stop_exe (void)
  {
  asm volatile (" int $0x21 # _DOS_stop_exe" : : "a" (0x4C00));
  }

extern inline void
_DOS_stop_com (void)
  {
  asm volatile (" int $0x20 # _DOS_stop_com");
  }

extern inline void
_BIOS_sigsegv (void)
  {
  asm volatile (" movw %%cs:(0xFFFF),%%ax " : : : "ax");
  }

extern inline void
_BIOS_halt (void)
  {
  asm volatile (" hlt ");
  }

extern inline int
DPMS_check (void)
  {
  unsigned ax = 0x43E0, bx = 0;

  asm ASMREAD (
"	pushl	%%es					\n"
"	pushl	%%edi					\n"
"	int	$0x2F					\n"
"	setc	%%al	# ax = 0 if installed		\n"
"	popl	%%edi					\n"
"	popl	%%es					\n"
	: "+a" (ax), "+b" (bx)
	: "c" (0x4450), "d" (0x4D53)
	: "cc"
	);
  return ax == 0;
  }

/*
 * DOSEMU stuff:
 */

extern inline unsigned
_DOSEMU_check (unsigned short *version, unsigned short *patchlevel)
  {
  unsigned short signature;

  if (peekl (0xE6 * 4) >> 16 != 0xF000)
      return 2; /* bad vector address */

  asm ASMREAD (" int $0xE6 # _DOSEMU_check"
	: "=a" (signature), "=b" (*version), "=c" (*patchlevel)
	: "a" (0x0000) );
  if (signature == 0xAA55)
      return 0; /* present */
    else
      return 1;
  }

extern inline void
_DOSEMU_stop (void)
  {
  asm volatile (" int $0xE6 # _DOSEMU_stop" : : "a" (0xFF00) );
  }

extern inline void
_DOSEMU_dumpreg (void)
  {
  asm volatile (" int $0xE6 # _DOSEMU_dumpreg" : : "a" (0x0100) );
  }

extern inline void
_DOSEMU_enable_ioports (unsigned short start, unsigned short nb)
  {
  asm volatile (" stc ; int $0xE6 # _DOSEMU_enable_ioports"
	: /* nothing */
	: "a" (0x0200), "b" (start), "c" (nb)
	: "cc"
	);
  }

extern inline void
_DOSEMU_disable_ioports (unsigned short start, unsigned short nb)
  {
  asm volatile (" clc ; int $0xE6 # _DOSEMU_disable_ioports"
	: /* nothing */
	: "a" (0x0200), "b" (start), "c" (nb)
	: "cc"
	);
  }

/*
 * Pre VGA BIOS:
 */
union EGA_bx_union {
    struct {
	enum { EGA_64K, EGA_128K, EGA_192K, EGA_256K } memory : 8;
	enum { EGA_COLOR_3Dx, EGA_MONO_3Bx } IOaddr : 8;
	} enums;
    unsigned short IOaddr_memory;
    } __attribute__ ((packed));

struct EGA_cx_str {
    unsigned short	switch1off : 1;
    unsigned short	switch2off : 1;
    unsigned short	switch3off : 1;
    unsigned short	switch4off : 1;
    unsigned short	switch_unused : 4;
    unsigned short	FEAT1line_state2 : 1;
    unsigned short	FEAT0line_state2 : 1;
    unsigned short	FEAT1line_state1 : 1;
    unsigned short	FEAT0line_state1 : 1;
    unsigned short	FEAT_unused : 4;
    } __attribute__ ((packed));

extern inline void
_EGA_getdisplay (union EGA_bx_union *EGA_bx, struct EGA_cx_str *EGA_cx)
  {
  unsigned short destroyed_tseng_ET4000_V8_00;
  asm ASMREAD (" int	$0x10	# _EGA_getdisplay "
	: "=b" (*EGA_bx), "=c" (*EGA_cx),
	  "=a" (destroyed_tseng_ET4000_V8_00)
	: "a" (0x1200), "b" (0xFF10) /* %bh = 0xFF to see if supported */
	);
  }

extern inline unsigned
_VIDEO_cold_initialise (void)
  {
  if (peekw (0xC0000000) == 0xAA55) {
      asm volatile (
"	pushl	%ds					\n"
"	pushl	%es					\n"
"	pushl	%gs					\n"
"	pushal						\n"
"#	mov	%ss,%cs:1f				\n"
"#	mov	%sp,%cs:2f				\n"
"#	jmp	3f					\n"
"#	1:	.word 0					\n"
"#	2:	.word 0					\n"
"#	3:						\n"
"	pushw	$0x40					\n"
"	popw	%ds					\n"
"	pushw	$0					\n"
"	popw	%es					\n"
"	xorw	%bp,%bp					\n"
"	lcallw	$0xC000,$0x0003				\n"
"#	mov	%cs:1b,%ss				\n"
"#	mov	%cs:2b,%sp				\n"
"	mov	%bp,%gs	# zero if success		\n"
"	popal	# take care of popa bug on i386		\n"
"	movw	%gs,%ax	# ignored now but...		\n"
"	popl	%gs					\n"
"	popl	%es					\n"
"	popl	%ds					\n"
	);
      return 0;
      }
    else
      return 1;
  }

/*
 * PS mouse/pointing device BIOS:
 */
enum PS_error {
    PS_successfull = 0,
    PS_invalid_fct,
    PS_invalid_input,
    PS_interface_error,
    PS_need_resend,
    PS_no_device_handler
    };

struct PS_resetID {
    unsigned char reset_value; /* 0xAA if mouse */
    unsigned char device_id;
    } __attribute__ ((packed));

struct PS_status {
    unsigned char right_button_pressed	: 1;
    unsigned char reserved1		: 1;
    unsigned char left_button_pressed	: 1;
    unsigned char reserved2		: 1;
    unsigned char twoTOone_scaling	: 1; /* else 1:1 scaling */
    unsigned char enabled		: 1;
    unsigned char remote_mode		: 1; /* else stream mode */
    unsigned char reserved3		: 1;
    } __attribute__ ((packed));

extern inline unsigned char _PS_enable (unsigned char enable /* if 1 */)
  {
  unsigned char status;

  asm volatile (
//#define PATCH_QEMU	/* corrected in Qemu CVS approx october 20th, 2007 */
#ifdef PATCH_QEMU
//#define KBD_CCMD_MOUSE_DISABLE	0xA7	/* Disable mouse interface */
//#define KBD_CCMD_MOUSE_ENABLE		0xA8	/* Enable mouse interface */
//#define KBD_CCMD_KBD_DISABLE		0xAD	/* Keyboard interface disable */
//#define KBD_CCMD_KBD_ENABLE		0xAE	/* Keyboard interface enable */
"	mov	$0xAD,%%bl		\n"
"	xchgb	%%bl,%%al		\n"
"	outb	%%al,$0x64		\n"
"	xchgb	%%bl,%%al		\n"
#endif
"	int	$0x15		# Pointing device enable/disable	\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	/* "Knowing is not enough, we must apply." (Bruce Lee) */
#ifdef PATCH_QEMU
"	mov	$0xAE,%%ah		\n"
"	xchgb	%%ah,%%al		\n"
"	outb	%%al,$0x64		\n"
"	xchgb	%%ah,%%al		\n"
#endif
	: "=a" (status)
	: "a" ((unsigned short)0xC200), "b" ((unsigned short)enable << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_reset (struct PS_resetID *result)
  {
  unsigned char status;

  asm volatile (
"	int	$0x15		# Pointing device reset			\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status), "=b" (*result)
	: "a" ((unsigned short)0xC201)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_set_sampling (unsigned char rate)
  {
  unsigned char status;

  /* rate = 0: 10/sec,  1: 20/sec,  2: 40/sec,  3: 60/sec,
	    4: 80/sec   5: 100/sec, 6: 200/sec */

  asm volatile (
"	int	$0x15		# Pointing device set sampling rate	\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status)
	: "a" ((unsigned short)0xC202), "b" ((unsigned short)rate << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_set_resolution (unsigned char resol)
  {
  unsigned char status;

  /* resol = 0: 1 per milimeter,  1: 2/mm,  2: 4/mm,  3: 8/mm */

  asm volatile (
"	int	$0x15		# Pointing device set resolution	\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status)
	: "a" ((unsigned short)0xC203), "b" ((unsigned short)resol << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_get_type (unsigned char *device)
  {
  unsigned char status;
  unsigned short device_id;

  asm ASMREAD (
"	int	$0x15		# Pointing device get type		\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status), "=b" (device_id)
	: "a" ((unsigned short)0xC204)
	: "cc"
	);
  *device = device_id >> 8;
  return status;
  }

extern inline unsigned char _PS_initialise (unsigned char data_package_size)
  {
  unsigned char status;
  /* 1 <= data_package_size <= 8 */

  asm volatile (
"	int	$0x15		# Pointing device initialise		\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status)
	: "a" ((unsigned short)0xC205),
	  "b" ((unsigned short)data_package_size << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_device_status (struct PS_status *psstat,
					       unsigned char *resolution,
					       unsigned char *sample_rate)
  {
  unsigned char status;

  asm ASMREAD (
"	int	$0x15		# Pointing device status		\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status), "=b" (*psstat), "=c" (*resolution), "=d" (*sample_rate)
	: "a" ((unsigned short)0xC206), "b" ((unsigned short)0 << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_set2to1scaling (unsigned char twoTOone /* else 1:1 */)
  {
  unsigned char status;

  asm volatile (
"	int	$0x15		# Pointing device set squaling 2:1 or 1:1\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
	: "=a" (status)
	: "a" ((unsigned short)0xC206),
	  "b" ((unsigned short)(twoTOone + 1) << 8)
	: "cc"
	);
  return status;
  }

extern inline unsigned char _PS_set_handler (farptr handler)
  {
  unsigned char status;

  /* handler can be zero to cancel */

  asm volatile (
"	pushl	%%es							\n"
"	roll	$16,%%ebx	# change carry				\n"
"	movw	%%bx,%%es						\n"
"	roll	$16,%%ebx						\n"
"	int	$0x15		# Pointing device handler		\n"
"	setc	%%al							\n"
"	shlb	$7,%%al							\n"
"	orb	%%ah,%%al	# %%al bit 7 set if major error		\n"
"	popl	%%es							\n"
	: "=a" (status)
	: "a" ((unsigned short)0xC207), "b" (handler)
	: "cc"
	);
  return status;
  }

extern inline void _PS_write (unsigned char byte)
  {
  asm volatile (" int	$0x15	# Pointing device write "
	: /* nothing */
	: "a" ((unsigned short)0xC208), "b" (byte)
	);
  }

extern inline void _PS_read (unsigned char *first_byte,
			     unsigned char *second_byte,
			     unsigned char *third_byte)
  {
  asm ASMREAD (" int	$0x15	# Pointing device read "
	: "=b" (*first_byte), "=c" (*second_byte), "=d" (*third_byte)
	: "a" ((unsigned short)0xC209)
	);
  }


/*
 * Disk BIOS:
 */

/*
 * Unfortunately, we need this data from the BIOS; it is not implemented
 * in DOSEMU, the probing may then fail:
 */
extern inline unsigned char
_BIOSDISK_get_nb_harddisk (void)
  {
  return peekb ((farptr)0x00400075);
  }

/*
 * This structure returned by INT13, %ah = 0x25 or directly asked
 * to the hardware by the IDE driver.
 */
typedef struct {
    struct {
	unsigned short	reserved : 1;
	unsigned short	hard_sectored : 1;
	unsigned short	soft_sectored : 1;
	unsigned short	not_MFM_encoded : 1;
	unsigned short	head_switch_time_more15us : 1;
	unsigned short	spindle_motor_control_option : 1;
	unsigned short	fixed_drive : 1;
	unsigned short	removable_cartridge_drive : 1;

	unsigned short	disk_transfert_rate_less5Mbs : 1;
	unsigned short	disk_transfert_rate_more5Mbs : 1;
	unsigned short	disk_transfert_rate_more10Mbs : 1;
	unsigned short	rotation_speed_tolerace : 1;
	unsigned short	data_strobe_offset_option : 1;
	unsigned short	track_offset_option : 1;
	unsigned short	format_speed_tolerance_gap : 1;
	unsigned short	nonmagnetic_reserved : 1;
	} __attribute__ ((packed)) config;
    unsigned short	number_of_cylinder;
    unsigned short	reserved;
    unsigned short	number_of_head;
    unsigned short	number_of_unformated_byte_per_track;
    unsigned short	number_of_unformated_byte_per_sector;
    unsigned short	number_of_sector_per_track;
    unsigned char	vendor_unique[6];
    unsigned char	ASCII_serial_number[20]; /* 0 : not specified */
    unsigned short	buffer_type;
    unsigned short	buffer_size_in_512bytes; /* 0 : not specified */
    unsigned short	number_of_ECC_bytes; /* 0 : not specified */
    unsigned char	ASCII_firmware_revision[8];
    unsigned char	ASCII_model_vendor[40];
    unsigned char	RWmultiple_max_sector; /* little endian */
    unsigned char	vendor_unique1;
    unsigned short	can_perform_doubleword_io; /* 1 if yes */
    struct {
	unsigned short	vendor_unique : 8;
	unsigned short	dma_supported : 1;
	unsigned short	lba_supported : 1;
	unsigned short	reserved : 6;
	} __attribute__ ((packed)) capabilities;
    unsigned short	reserved2;
    unsigned char	PIOvendor_unique;
    unsigned char	PIOdata_transfert_cycle_timing_mode;
    unsigned char	DMAvendor_unique;
    unsigned char	DMAdata_transfert_cycle_timing_mode;
    struct {
	unsigned short	translation_fields_valid : 1;
	unsigned short	description_reserved : 15;
	} __attribute__ ((packed)) description;
    unsigned short	number_of_current_cylinder;
    unsigned short	number_of_current_head;
    unsigned short	number_of_current_sector_per_track;
    unsigned		current_capacity_in_sector;
    unsigned short	reserved3;
    unsigned		lba_capacity;	/* BIG disk, > 20 Go [2*60] */
    unsigned char	undefined_ATA_2_6_part1[80];
    unsigned long long	lba48_capacity;	/* 48bits IDE [2*100] */
    unsigned char	undefined_ATA_2_6_part2[44];
    unsigned char	vendor_unique2[64];
    unsigned short	reserved4[96];
    } __attribute__ ((packed)) ATAstruct_t;

/*
 * The Disk Base Table return by "get drive parameter" INT13/08
 * (usually only for floppy, but "bytepersector" used if present & valid)
 */
struct DBT_str {
    unsigned short	specify;
    unsigned char	motorshutofftime;	/* in ticks */
    unsigned char	bytepersector;	/* 0=128, 1=256, 2=512, 3=1024 */
    unsigned char	sectorpertrack;	/* i.e. last sector number */
    unsigned char	intersectorgap;
    unsigned char	datalength;	/* if sector length not specified */
    unsigned char	formatgaplength;
    unsigned char	fillbyteafterformat;
    unsigned char	headsettletime;		/* in ms */
    unsigned char	motorstartuptime;	/* in 1/8 of a second */
    } __attribute__ ((packed)) ;


extern inline unsigned char
_BIOSDISK_reset_floppy (unsigned char *status)
  {
  unsigned char carry;

  asm volatile (
"	int	$0x13	# _BIOSDISK_reset_floppy	\n"
"	mov	%%ah,%1					\n"
"	setc	%0					\n"
	: "=qm" (carry), "=qm" (*status)
	: "a" ((unsigned short)0x0000), "d" (0)
	);
  return carry;
  }

extern inline unsigned char
_BIOSDISK_reset_alldisks (unsigned char *status)
  {
  unsigned char carry;

  asm volatile (
"	int	$0x13	# _BIOSDISK_reset_alldisks	\n"
"	mov	%%ah,%1					\n"
"	setc	%0					\n"
	: "=qm" (carry), "=qm" (*status)
	: "a" ((unsigned short)0x0000), "d" (0x80)
	);
  return carry;
  }

extern inline unsigned char
_BIOSDISK_getstatus (unsigned char drive, unsigned char *status)
  {
  unsigned char error;

  asm ASMREAD (
"	int	$0x13	# _BIOSDISK_getstatus		\n"
"	mov	%%ah,%1					\n"
"	setc	%0					\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x0100), "d" (drive)
	);
  return error;
  }

extern inline unsigned char
_BIOSDISK_RWsector (unsigned char  drive,
		    unsigned short cylinder,
		    unsigned char  head,
		    unsigned char  sector,
		    unsigned char  nbsector,
		    farptr         buffer,
		    const unsigned read,
		    unsigned char *nbprocessed)
  {
  unsigned short result_nbread;
  unsigned char carry;

  asm volatile (
"	pushl	%%es					\n"
"	pushl	%%ebx					\n"
"	popw	%%bx					\n"
"	popw	%%es					\n"
"	stc		# buggy BIOSes			\n"
"	int	$0x13   # _BIOSDISK_RWsector		\n"
"	sti		# buggy BIOSes			\n"
"	popl	%%es					\n"
"	setc	%1					\n"
	: "=a" (result_nbread), "=qm" (carry)
	: "a" ((read? 0x0200 : 0x0300) | nbsector), "b" (buffer),
	  "c" ((cylinder << 8) | ((cylinder >> 2) & 0xC0) | (sector & 0x3F)),
	  "d" ((((unsigned short)head) << 8) | drive)
	: "memory"
	);
  if (nbprocessed)
      *nbprocessed = result_nbread;
  if (carry)
      return result_nbread >> 8;
  return 0;
  }

extern inline unsigned
_BIOSDISK_getparam (unsigned char disk,
		    unsigned short *tmpcx, unsigned short *tmpdx,
		    farptr *DBTadr, unsigned char *CmosDriveType,
		    unsigned char *status)
  {
  unsigned short cmd_status_carry = 0x0800;
  *CmosDriveType = 0;
  *tmpdx = disk;
  *DBTadr = 0;

  asm ASMREAD (
"	pushl	%%es						\n"
"	mov	%%di,%%es	# set it to zero		\n"
"	int	$0x13		# _BIOSDISK_getparam		\n"
"	pushw	%%es						\n"
"	pushw	%%di						\n"
"	popl	%%edi						\n"
"	popl	%%es						\n"
"	setc	%%al						\n"
	: "+a" (cmd_status_carry), "+d" (*tmpdx), "+b" (*CmosDriveType), "+D" (*DBTadr),
	  "=c" (*tmpcx)
	);
  *status = cmd_status_carry >> 8;
  return cmd_status_carry & 0xFF;
  }

extern inline unsigned char
_BIOSDISK_reinitHD (unsigned char drive, unsigned char *status)
  {
  unsigned char error;
  /* This uses the table pointed by INT 41/46 */

  asm volatile (
"	int	$0x13	# _BIOSDISK_reinitHD	\n"
"	mov	%%ah,%1				\n"
"	setc	%0				\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x0900), "d" (drive)
	);
  return error;
  }

extern inline unsigned char
_BIOSDISK_resetHD (unsigned char drive, unsigned char *status)
  {
  unsigned char error;

  asm volatile (
"	int	$0x13	# _BIOSDISK_resetHD	\n"
"	mov	%%ah,%1				\n"
"	setc	%0				\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x0D00), "d" (drive)
	);
  return error;
  }

extern inline unsigned char
_BIOSDISK_recalibrateHD (unsigned char drive, unsigned char *status)
  {
  unsigned char error;

  asm volatile (
"	int	$0x13	# _BIOSDISK_reinitHD	\n"
"	mov	%%ah,%1				\n"
"	setc	%0				\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x1100), "d" (drive)
	);
  return error;
  }

extern inline unsigned char
_BIOSDISK_gettype (unsigned char disk, unsigned *nbsect, unsigned char *status)
  {
  struct { unsigned char carry, status; } ax;

  asm ASMREAD (
"	xor %%dh,%%dh			\n"
"	xor %%cx,%%cx			\n"
"	int $0x13	# _BIOSDISK_gettype	\n"
"	setc %%al				\n"
"	shll $16,%%ecx				\n"
"	movw %%dx,%%cx			\n"
	: "=a" (ax), "=c" (*nbsect), "+d" (disk) /* disk = 0x80..0x8F else buggy Compaq */
	: "a"((unsigned short)0x1500)
	);
  *status = ax.status;
  return ax.carry;
  }

extern inline unsigned char
_BIOSDISK_set_media_type_for_format (unsigned char disk,
		unsigned nbSectPerTrack, unsigned nbCylinder)
  {
  unsigned short command_status = 0x1800;
  unsigned cx, maxCylinder = nbCylinder - 1;
  farptr param_table;

  cx = maxCylinder << 8;
  cx |= nbSectPerTrack & 0x3F;
  cx |= (maxCylinder >> 2) & 0xC0;

  asm volatile (
"	pushl	%%es		\n"
"	int	$0x13		# _BIOSDISK_set_media_type_for_format	\n"
"	pushw	%%es		\n"
"	pushw	%%di		\n"
"	popl	%%edi		\n"
"	popl	%%es		\n"
	: "+a" (command_status), "=D" (param_table)
	: "d" (disk), "c" (cx), "D" (0)
	);
  command_status >>= 8;
/* int	$0x13/0x1800 does not set the INT 1E vector to point at the returned
	parameter table; it is the caller's responsibility to do so: */
  if (command_status == 0)
      pokel (0x1E * 4, param_table);

/* returns: 00h requested combination supported
	    01h function not available
	    0Ch not supported or drive type unknown
	    80h there is no disk in the drive
 */
  return command_status;
  }


/* never seen this one working... Compaq only anyway */
extern inline unsigned char
_BIOSDISK_sensemedia (unsigned char disk, unsigned char *media,
		      unsigned char *err)
  {
  unsigned char error;
  unsigned short err_media;

  asm ASMREAD (
"	int	$13	# _BIOSDISK_sensemedia		\n"
"	setc	%0					\n"
	: "=qm" (error), "=a" (err_media)
	: "a" ((unsigned short)0x2000), "d" ((unsigned short) disk) /* %dh = 0 */
	);

  if (error || err_media == 0x2000)
      return 1;
  *media = (unsigned char) err_media;
  *err = err_media >> 8;
  return 0;
  }

/*
 * We may have read/write multiple available and so provide here
 * the function to enable it: (set "nb" to the value read in
 * structure ATAstruct_t, 0 to disable). the "nb" field
 * is also returned in EBIOS initialisation (Phoenix extension)
 * and used if EBIOS has no "extended_fct" attribute.
 */
extern inline unsigned char
_BIOSDISK_setmultiple (unsigned char drive, unsigned char nb,
			unsigned char *status)
 {
  unsigned char error;

  asm volatile (
"	int	$0x13	# _BIOSDISK_setmultiple	\n"
"	mov	%%ah,%1				\n"
"	setc	%0				\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x2400 + nb), "d" (drive)
	);

  return error;
  }

extern inline unsigned char
_BIOSDISK_getidentity (unsigned char drive, ATAstruct_t *buff,
		unsigned char *status)
  {
  unsigned char error;

  asm ASMREAD (
"	int	$0x13		# _BIOSDISK_getidentity		\n"
"	mov	%%ah,%1						\n"
"	setc	%0						\n"
	: "=qm" (error), "=qm" (*status)
	: "a" ((unsigned short)0x2500), "d" (drive), "b" (buff) /* in fact %es:%bx */
	);

  return error;
  }

/*
 * Extended BIOS for disk:
 */

typedef struct {
    unsigned short extended_fct	: 1;
    unsigned short removable	: 1;
    unsigned short enhanced	: 1;
    unsigned short reserved	: 13;
    } __attribute__ ((packed)) ebios_bitmap_t;

typedef struct {
    unsigned short	buffersize;
    unsigned short	infobit;
    unsigned		nbcylinder;
    unsigned		nbhead;
    unsigned		nbsectorpertrack;
    unsigned long long	nbtotalsector;
    unsigned short	bytepersector;
    /* V2.0+ : */
    farptr	configptr; /* for instance ebiosPhoenix realmode address */
    /* V3.0+ : */
    unsigned short signature0xBEDD;
    unsigned char  lendevicepath; /* 0x24 for v3.0 */
    unsigned char  reserved[3];
    unsigned char  bustype[4]; /* exactly "ISA" or "PCI" */
    unsigned char  Interface[8]; /* exactly "ATA", "ATAPI", "SCSI", "USB"...*/
    union interface_path_u {
	struct {
	    unsigned short addr;
	    unsigned char pad[6];
	    } __attribute__ ((packed)) isa;
	struct {
	    unsigned char busnr;
	    unsigned char devicenr;
	    unsigned char functionnr;
	    unsigned char pad[5];
	    } __attribute__ ((packed)) pci;
	} bus;
    union device_path_u {
	struct {
	    unsigned char slave; /* 0 if master, 1 if slave */
	    unsigned char pad[7];
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) ata;
	struct {
	    unsigned char slave; /* 0 if master, 1 if slave */
	    unsigned char logical_unit_nr;
	    unsigned char pad[6];
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) atapi;
	struct {
	    unsigned char logical_unit_nr;
	    unsigned char pad[7];
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) scsi;
	struct {
	    unsigned char tobedefined;
	    unsigned char pad[7];
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) usb;
	struct {
	    unsigned long long FireWireGeneralUniqueID;
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) ieee1394;
	struct {
	    unsigned long long WorldWideNumber;
	    unsigned long long	reserved; // Not present in some docs...
	  } __attribute__ ((packed)) FibreChannel;
      } device;
    unsigned char zero;
    unsigned char bytechecksum; /* byte sum [0x1E..0x49] = 0 */
    } __attribute__ ((packed)) ebiosinfo_t;

typedef   struct {
    unsigned short	dataIOadr;
    unsigned short	controlIOadr;
    unsigned char	lba_slave_mask; /* to OR at each use */
    unsigned char	shiftneeded; /* in fact, reserved */
    unsigned char	irq;
    unsigned char	multisectorsize;
    unsigned char	DMAtype : 4;
    unsigned char	DMAchannel : 4;
    unsigned char	PIOtype;
    struct {
	unsigned short	fastPIOenabled : 1;
	unsigned short	DMAenabled : 1;
	unsigned short	blockPIOenabled : 1;
	unsigned short	CHSenabled : 1;
	unsigned short	LBAenabled : 1;
	unsigned short	removable : 1;
	unsigned short	ATAPIdevice : 1;
	unsigned short	transfert32bits : 1;
	unsigned short	ATAPIusesDRQ : 1;
	unsigned short	CHStranslationtype : 2;
	unsigned short	reserved : 5;
	} __attribute__ ((packed)) hardoption;
    unsigned short	reserved;
    unsigned char	revision; /* 0x11 now */
    unsigned char	checksum; /* 2's complement of bytes [0..14] */
    } __attribute__ ((packed)) ebiosPhoenix_t;

extern inline unsigned char
_EBIOSDISK_probe (unsigned char disk, unsigned short *version, ebios_bitmap_t *bitmap)
  {
  unsigned char carry;
  unsigned short signat0xAA55;

  asm ASMREAD (
"	int	$0x13		# _EBIOSDISK_probe	\n"
"	setc	%%dl					\n"
	: "=a" (*version), "=b" (signat0xAA55), "=c" (*bitmap), "=d" (carry)
	: "a"((unsigned short)0x4100), "b" ((unsigned short)0x55AA), "d" (disk)
	);

  return carry || (signat0xAA55 != 0xAA55) || (*version == 0x100 /*INT13 invalid function*/);
  }

extern inline unsigned char
_EBIOSDISK_RWsector (unsigned char disk,
		     unsigned long long lba,
		     unsigned char number,
		     farptr buffer,
		     const unsigned read,
		     unsigned short *nbprocessed)
  {
  unsigned char carry;
  unsigned short cmd = read? 0x4200 : 0x4300;
  /* if EBIOS < v2.1, verify write by %al bit 0 */
  /* if EBIOS >= v2.1, verify write by 'cmd += 2 ;' */
  unsigned short errcode;
  struct diskpacket_t {
      unsigned char      packetsize_16;	/* or 24 */
      unsigned char      reserved_0;
      unsigned short     nbblock; /* max 007Fh for Phoenix EDD */
      farptr             buffer;
      unsigned long long lba;
      } __attribute__ ((packed)) diskpacket = {
      .packetsize_16	= 16,
      .reserved_0	= 0,
      .nbblock		= number,
      .buffer		= buffer,
      .lba		= lba
      };

  asm volatile (
"	int	$0x13		# _EBIOSDISK_RWsector	\n"
"	setc	%1					\n"
	: "=a" (errcode), "=qm" (carry)
	: "a"(cmd), "d" (disk), "S" (&diskpacket), /* %ds:%si */
	  ASM_STRUCT_INPUT (diskpacket)
	);

  if (nbprocessed)
      *nbprocessed = diskpacket.nbblock;

  if (carry)
      return errcode >> 8;
  return 0;
  }

enum _EBIOSDISK_lock_unlock {
    _EBIOSDISK_lock = 0, _EBIOSDISK_unlock = 1, _EBIOSDISK_getlock = 2
    };

extern inline unsigned char
_EBIOSDISK_locker (unsigned char disk, enum _EBIOSDISK_lock_unlock lock_operation,
		 unsigned short *returned_status) /* status == 0 if unlock !!! */
  {
  unsigned char carry;

  asm ASMREAD (
"	int	$0x13		# _EBIOSDISK_locker	\n"
"	setc	%0					\n"
	: "=qm" (carry), "=a" (*returned_status)
	: "a"((unsigned short)0x4500 | lock_operation), "d" (disk)
	);

  return carry;	/* if carry = 0, no error, and %ah = 0, %al = status */
  }

extern inline unsigned char
_EBIOSDISK_getparam (unsigned char disk, ebiosinfo_t *ebiosinfo, unsigned char *status)
  {
  unsigned short carry_status;

  asm ASMREAD (
"	push %%es	\n"	// Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13		# _EBIOSDISK_getparam	\n"
"	setc	%%al		# status in %%ah	\n"
"	pop %%es	\n"
	: "=a" (carry_status), "=m" (*ebiosinfo)
	: "a"((unsigned short)0x4800), "d" (disk), "S" (ebiosinfo)
	);

  if (status)
      *status = carry_status >> 8;
  return carry_status & 0xFF;
  }

enum ebios_hw_conf_t {
	enable_prefetch,
	disable_prefetch,
	set_max_PIO,
	set_PIO_mode_0,
	set_default_PIO,
	enable_INT_13h_DMA_max,
	disable_INT_13h_DMA
	};
extern inline unsigned char
_EBIOSDISK_set_hw_conf (unsigned char disk, enum ebios_hw_conf_t conf,
		     unsigned char *status)
  {
  unsigned short errorcode;
  unsigned char carry;

  asm ASMREAD (
"	int	$0x13		# _EBIOSDISK_set_hw_conf	\n"
"	setc	%%dl					\n"
	: "=a" (errorcode), "=d" (carry)
	: "a"((unsigned short)0x4E00 + conf), "d" (disk)
	);

  if (carry)
      *status = errorcode >> 8;
    else
      *status = errorcode; /* 0: command was safe, 1: other devices are affected */
  return carry;
  }

/* Common Access Method SCSI interface rev 2.3 */
extern inline unsigned char
_SCSI_CAM_BIOS_installcheck (void)
  {
  farptr scsi_cam_fptr;
  const union {
      unsigned long long ull; unsigned char c[8];
      } str = { .c = {'S', 'C', 'S', 'I', '_', 'C', 'A', 'M' }};
  unsigned short ax, cx, dx;

  asm ASMREAD (
"	pushl	%%es			\n"
"	int	$0x4F		# _SCSI_CAM_BIOS_installcheck	\n"
"	pushw	%%es			\n"
"	pushw	%%di			\n"
"	popl	%%edi			\n"
"	popl	%%es			\n"
	: "=a" (ax), "=c" (cx), "=d" (dx), "=D" (scsi_cam_fptr)
	: "a"((unsigned short)0x8200), "c" ((unsigned short)0x8765), "d" ((unsigned short)0xCBA9)
	);

  if ((ax >> 8) != 0)
      return 1;
  if (cx != 0x9ABC)
      return 2;
  if (dx != 0x5678)
      return 3;
  if (peekll (scsi_cam_fptr) != str.ull)
      return 4;
  return 0;
  }


/* From Adaptec proposal dated 8/25/2000 EDD 3.0 extension Document No. E00143R0
   found on Internet: */
extern inline unsigned char
_SCSIBIOSDISK_sanitycheck (unsigned char disk, unsigned char *status,
			unsigned short *nbchannel, unsigned short *scsibios_codeseg)
  {
  unsigned char carry;
  unsigned short signature;

  asm ASMREAD (
"	int	$0x13		# _SCSIBIOSDISK_sanitycheck	\n"
"	mov	%%ah,%1						\n"
"	setc	%0						\n"
	: "=qm" (carry), "=qm" (*status), "=c" (signature),
		"=S" (*nbchannel), "=b" (*scsibios_codeseg)
	: "a"((unsigned short)0x5000), "d" (disk), "c" (0x55AA)
	);

  return carry || (signature != 0xAA55);
  }

/*
 * PCI BIOS (V2.0c+):
 */
union PCI_hardware_characteristics_u {
    unsigned char all;
    struct {
	unsigned char configuration_space_access_mechanism_1 : 1;
	unsigned char configuration_space_access_mechanism_2 : 1;
	unsigned char reserved1 : 2;
	unsigned char special_cycle_generation_mechanism_1_supported : 1;
	unsigned char special_cycle_generation_mechanism_2_supported : 1;
	unsigned char reserved2 : 2;
	} __attribute__ ((packed)) bits;
    } __attribute__ ((packed));

enum PCI_error_code {
    PCI_successful = 0x00,
    PCI_unsupported_function = 0x81,
    PCI_bad_vendorID = 0x83,
    PCI_device_not_found = 0x86,
    PCI_bad_register_number = 0x87,
    PCI_set_failed = 0x88,
    PCI_buffer_too_small = 0x89,
    };

struct PCIv3_support {
    unsigned char last_pci_bus;
    unsigned char function_6_D_below_256 : 1;	// GENERATE_SPECIAL_CYCLE..WRITE_CONFIG_DWORD
    unsigned char function_6_D_over_256_POST : 1;
    unsigned char function_E_implemented : 1;	// GET_IRQ_ROUTING_EXPANSIONS
    unsigned char function_F_implemented : 1;	// SET_PCI_HW_INT
    unsigned char function_2_implemented : 1;	// FIND_PCI_DEVICE
    unsigned char function_3_implemented : 1;	// FIND_PCI_CLASS_CODE
    unsigned char BIOS_option_ROM_execution : 1;
    unsigned char BIOS_option_ROM_DMTF_CLP : 1;
    } __attribute((packed));

struct PCI_location {
    unsigned char function	    : 3;
    unsigned char device	    : 5;
    unsigned char bus_number;
    } __attribute__ ((packed));

struct PCI_classcode {
    unsigned char programming_interface;
    unsigned char pci_subclass;
    unsigned char pci_class;
    unsigned char unused;
    } __attribute__ ((packed));

extern inline unsigned char
_PCIBIOS_install_check (union PCI_hardware_characteristics_u *PCI_hardware_characteristics,
		unsigned short *version_bcd, struct PCIv3_support *PCI_support)
  {
  unsigned ICP_signature, phyadr_PMentry;
  struct {
      union PCI_hardware_characteristics_u PCI_hardware_characteristics;
      unsigned char not_installed;
      } __attribute((packed)) ax;

  /* If version 2 only, %ch unmodified so set default as function present: 0x31
	VirtualBox says version 0x210 but clears %%ch, anyway cannot detect PCI so OK */
  asm (
"	int	$0x1A	# _PCIBIOS_install_check	\n"
"	jnc	1f					\n"
"	mov	$0x54,%%ah	# set not_installed	\n"
"	1:						\n"
	: "=a" (ax),
		"=d" (ICP_signature), "=D" (phyadr_PMentry),
		"=b" (*version_bcd), "=c" (*PCI_support)
	: "a"((unsigned short) 0xB101), "D" (0), "c" (0x3100)
	);
  *PCI_hardware_characteristics = ax.PCI_hardware_characteristics;

  return (ax.not_installed != 0) || (ICP_signature != 0x20494350);
  }

extern inline unsigned char
_PCIBIOS_find_device (unsigned short deviceID, /* 0xFFFF wildcard in future implementation */
			unsigned short vendorID,
			unsigned short device_index, /* zero based, when same deviceID & vendorID */
			struct PCI_location *location)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;

  asm (
"	int	$0x1A		# _PCIBIOS_find_device	\n"
"	setc	%%al		\n"
	  /* EAX, EBX, ECX, and EDX may be modified: */
	: "=a" (status), "=b" (*location), "+c" (deviceID), "+d" (vendorID)
	: "a" ((unsigned short) 0xB102), "S" (device_index)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_find_classcode (struct PCI_classcode classcode,
			unsigned short device_index, /* zero based, when same classcode */
			struct PCI_location *location)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_find_classcode	\n"
"	setc	%%al		\n"
	: "=a" (status), "=b" (*location), "+c" (classcode), "=d" (dx)
	: "a" ((unsigned short) 0xB103), "S" (device_index)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_read_config_byte (struct PCI_location location, unsigned short register_addr,
				unsigned char *byte_read)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_read_config_byte	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "=c" (*byte_read), "=d" (dx)
	: "a" ((unsigned short) 0xB108), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_read_config_word (struct PCI_location location, unsigned short register_addr,
				unsigned short *word_read)
  {
  /* register_addr has to be multiple of 2 */
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_read_config_word	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "=c" (*word_read), "=d" (dx)
	: "a" ((unsigned short) 0xB109), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_read_config_dword (struct PCI_location location, unsigned short register_addr,
				unsigned *dword_read)
  {
  /* register_addr has to be multiple of 4 */
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_read_config_dword	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "=c" (*dword_read), "=d" (dx)
	: "a" ((unsigned short) 0xB10A), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_write_config_byte (struct PCI_location location, unsigned short register_addr,
				unsigned char byte_to_write)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_write_config_byte	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "+c" (byte_to_write), "=d" (dx)
	: "a" ((unsigned short) 0xB10B), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_write_config_word (struct PCI_location location, unsigned short register_addr,
				unsigned short word_to_write)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_write_config_word	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "+c" (word_to_write), "=d" (dx)
	: "a" ((unsigned short) 0xB10C), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

extern inline unsigned char
_PCIBIOS_write_config_dword (struct PCI_location location, unsigned short register_addr,
				unsigned dword_to_write)
  {
  struct {
      unsigned char carry;
      enum PCI_error_code errorcode : 8;
      } __attribute__ ((packed)) status;
  unsigned short dx;	/* EAX, EBX, ECX, and EDX may be modified: */

  asm (
"	int	$0x1A		# _PCIBIOS_write_config_dword	\n"
"	setc	%%al		\n"
	: "=a" (status), "+b" (location), "+c" (dword_to_write), "=d" (dx)
	: "a" ((unsigned short) 0xB10D), "D" (register_addr)
	: "cc"
	);

  if (status.carry != 0 && status.errorcode == 0)
      return 0xFF;
  return status.errorcode;
  }

/*
 * Extended CDROM BIOS: (bootable CDROM)
 */
typedef struct {
    unsigned char	packetsize;	/* = 0x13 */
    struct {
	enum { noemul=0, floppy120=1, floppy144=2, floppy288=3, harddisk=4 }
			mediatype : 4;
	unsigned char	reserved  :2;
	unsigned char	ATAPIdrv  :1;
	unsigned char	SCSIdrv	  :1;
	} __attribute__ ((packed)) bootmedia;
    unsigned char	bootdrive;	/* floppy=0, hd=0x80..0xFF */
    unsigned char	CDROMcontrol;	/* controller number */
    unsigned		LBAimage;	/* disk image to emulate */
    unsigned short	devspec;	/* if IDE bit0: slave/master */
					/* if SCSI: BUSnb<<8|LUNPUN */
    unsigned short	buffer3Kseg;	/* segment for caching */
    unsigned short	bootloadseg;	/* 0x07C0 if 0 */
    unsigned short	nb512bytesect;	/* of loader, if INT13/0x4C */
    unsigned char	lsbnbcyl;
    unsigned char	nbsector_msbnbcyl;
    unsigned char	nbhead;
    } __attribute__ ((packed)) CDROMbootspec;

typedef struct {
    unsigned char  packetsize; /* = 8 */
    unsigned char  nbsector;
    void	   *buffer;	/* multiple of 800 bytes */
    unsigned short firstsector;
    } __attribute__ ((packed)) CDROMcatalog;

/*
 * Most of those call are the perfect example why I need
 * to encode a structure as an output parameter, and not
 * its address - but pass its address, like function
 * returning big structures. I need to write:
  asm (
"	int	$0x13	# _BIOSCDROM_getstatus		\n"
"	setc	%0					\n"
	: "=qm" (error), "=a" (*returned_code), "=S" (*spec)
	: "a"(0x4B01), "d" (drivenum)
	);
 * So that register "%[e]si" contains the address of the structure
 * at asm entry, and the structure content is modified by the asm.
 * I cannot use "memory" because the structure may be in the stack.
 * Using "=g"(*spec) generate code to set up the pointer
 */

extern inline unsigned char
_BIOSCDROM_initiate (CDROMbootspec *spec,
		      unsigned short *returned_code)
  {
  unsigned char error;

  asm volatile (
"	push %%es	\n"	// ?? Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13	# _BIOSCDROM_initiate		\n"
"	setc	%0					\n"
"	pop %%es	\n"
	: "=qm" (error), "=a" (*returned_code), ASM_STRUCT_OUTPUT (*spec)
	: "a"(0x4A00), "S" (spec) /* %%ds:%%si */, ASM_STRUCT_INPUT (*spec)
	);

  return error;	/* 1 if error */
  }

extern inline unsigned char
_BIOSCDROM_initiateANDboot (CDROMbootspec *spec,
			unsigned short *returned_code)
  {
  unsigned char error;

  asm volatile (
"	push %%es	\n"	// ?? Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13	# _BIOSCDROM_initiateANDboot	\n"
"	setc	%0					\n"
"	pop %%es	\n"
	: "=qm" (error), "=a" (*returned_code)
	: "a"(0x4C00), "S" (spec) /* %%ds:%%si */, ASM_STRUCT_INPUT (*spec)
	);

  return error;	/* 1 if error */
  }

extern inline unsigned char
_BIOSCDROM_getcatalog (CDROMcatalog *catal, unsigned char drivenum,
			unsigned short *returned_code)
  {
  unsigned char error;

  if (sizeof (CDROMcatalog) != 8)
      __ERROR();
  if (sizeof (CDROMbootspec) != 0x13)
      __ERROR();

  asm (
"	push %%es	\n"	// ?? Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13	# _BIOSCDROM_getcatalog		\n"
"	setc	%0					\n"
"	pop %%es	\n"
	: "=qm" (error), "=a" (*returned_code), ASM_STRUCT_OUTPUT (*catal)
	: "a"(0x4D00), "d" (drivenum), "S" (catal) /* %%ds:%%si */, ASM_STRUCT_INPUT (*catal)
	: "memory");

  return error;	/* 1 if error */
  }

extern inline unsigned char
_BIOSCDROM_getstatus (CDROMbootspec *spec, unsigned char drivenum,
			unsigned short *returned_code)
  {
  unsigned char error;

  asm ASMREAD (
"	push %%es	\n"	// Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13	# _BIOSCDROM_getstatus		\n"
"	setc	%0					\n"
"	pop %%es	\n"
	: "=qm" (error), "=a" (*returned_code), "+m" (*spec)
	: "a"(0x4B01), "d" (drivenum), "S" (spec) /* %%ds:%%si */
	);

  return error;	/* 1 if error */
  }

extern inline unsigned char
_BIOSCDROM_terminate (CDROMbootspec *spec, unsigned char drivenum,
			unsigned short *returned_code)
  {
  unsigned char error;
  /* drivenum can be 0x7F to terminate all emulations
   *  *spec is filled with drive information!
   */

  asm volatile (
"	push %%es	\n"	// ?? Phoenix Award BIOS v6.00PG (C) 1984-2003 for HP (C) 2003 Rev. 3.06 (USB disk boot)
"	int	$0x13	# _BIOSCDROM_terminate		\n"
"	setc	%0					\n"
"	pop %%es	\n"
	: "=qm" (error), "=a" (*returned_code)
	: "a"(0x4B00), "d" (drivenum), "S" (spec) /* %%ds:%%si */, ASM_STRUCT_INPUT (*spec)
	);

  return error;	/* 1 if error */
  }

/*
 * Network BIOS:
 */
extern inline unsigned char
_BIOS_NETWORK_installed (void)
  {
  unsigned short ax = 0;
  unsigned short segment = peekw(4 * 0x2A + 2);

  if (segment < 0x100 || segment > 0xFFF0)
	return 0;
  asm ASMREAD ("   int	$0x2A	# _BIOS_NETWORK_installed \n" : "+a" (ax));

  return (ax >> 8) != 0;	/* not null if network present */
  }

extern inline unsigned short
_BIOS_NETWORK_execute_NCB (void *ncb)
  {
  unsigned short ax = 0x0100;

  asm ASMREAD (
"	int  $0x2A   # _BIOS_NETWORK_execute_NCB \n"
	: "+a" (ax) : "b" (ncb) /* %es:%bx */);

  return ax;        /* ah == 0 if no error */
  }


extern inline unsigned short
_BIOS_NETWORK_execute_NCB_retry (void *ncb, unsigned retry)
  {
  unsigned short ax = 0x0400 + !retry;

  asm ASMREAD (
"	int  $0x2A   # _BIOS_NETWORK_execute_NCB_retry \n"
	: "+a" (ax) : "b" (ncb) /* %es:%bx */);

  return ax;        /* ax == 0 if no error */
  }

struct NDIS_s {
    unsigned short command;
    unsigned short status;
    unsigned param1;
    unsigned param2;
    unsigned short extra;
    };
extern inline unsigned char
_BIOS_NDIS_execute (unsigned short PROTMAN_handle, struct NDIS_s *NDIS_request, unsigned short *error_nbbytes)
  {
  unsigned char carry;

  asm ASMREAD (
"	int  $0x21   # _BIOS_NDIS_execute	\n"
"	setc	%0				\n"
	: "=qm" (carry), "=a" (*error_nbbytes)
	: "a" (0x4402), "b" (PROTMAN_handle), "c" (0x000E), "d" (NDIS_request) /* %ds:%dx */);

  return carry;
  }

#endif /* BIOS_H */
