/* addhole.c is a utility to prepend a hole on a file of a E2/3FS filesystem.
 * Copyright (C) 1999-2013 Etienne Lorrain, fingerprint (2D3AF3EA):
 *   2471 DF64 9DEE 41D4 C8DB 9667 E448 FF8C 2D3A F3EA
 * E-Mail: etienne@gujin.org
 * This work is registered with the UK Copyright Service: Registration No:299755
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define AUTO_RETRY 16	/* less than ('Z' - 'A'), tmp file created "tmp[A-Z]" */

int file_mapping (int file_desc, unsigned file_len)
{
	unsigned bmap_block_size, nb_block, block_index, current_block = 0;
	int start_hole = 0;

	if (ioctl (file_desc, FIGETBSZ, &bmap_block_size)) {
		perror ("FIGETBSZ");
		return -1;
		}

	nb_block = (file_len + bmap_block_size - 1) / bmap_block_size;

	for (block_index = 0; block_index < nb_block; block_index++) {
		unsigned long block = block_index;

		if (ioctl (file_desc, FIBMAP, &block)) {
			perror ("FIBMAP");
			return -2;
			}
		if (block == 0 && start_hole >= 0) {
			start_hole++;
			continue;
			}
		if (block == 0) {
//			printf ("FAILED: another hole present\n");
			return 1;
			}
		if (start_hole >= 0) {
//			printf ("Target file begins with a hole of %u blocks\n", start_hole);
			if (start_hole != 12)
				return 2;
			start_hole = -1;
			current_block = block;
			continue;
			}
		if (block != current_block + 1) {
//			printf ("FAILED: at least two segments, %u follow %u\n", block, current_block);
			return 3;
			}
		current_block = block;
		}
//	printf ("Success: file correctly mapped\n");
	return 0;
}

int main (int argc, char **argv)
{
	int fout_desc, fin_desc, fs_block_size, ret = 0;
	char *buffer, *fout_name, *fin_name;
	off_t fin_len;
	struct stat statbuf;

	if (argc != 3) {
		printf ("Utility to copy a file, prepending a hole of 12 filesystem blocks at its begin\n");
		printf ("USAGE: %s <input_file> <output_file>\n", argv[0]);
		return -1;
		}
	fin_name = argv[1];
	fout_name = argv[2];

	if ((fout_desc = open (fout_name, O_CREAT|O_WRONLY|O_EXCL|O_SYNC, S_IRWXU)) == -1) {
		perror (fout_name);
		return -2;
		}
	if (fstat (fout_desc, &statbuf)) {
		perror ("fstat");
		if (close (fout_desc) != 0)
			perror (fout_name);
		return -3;
		}
#if 0
	{
	int device_desc;
	char pattern[] = "tmp_dev";
	if (mknod (pattern, S_IFBLK | 0600, statbuf.st_dev) != 0) {
		perror("mknod for BLKBSZGET [need to be root]");
		return -3;
		}
	if ((device_desc = open (pattern, O_RDONLY)) == -1) {
		perror(pattern);
		if (unlink (pattern))
			perror ("unlink");
		return -3;
		}
	if (ioctl (device_desc, BLKBSZGET, &fs_block_size)) {
		perror ("BLKBSZGET");
		if (close (fout_desc) != 0)
			perror (fout_name);
		return -3;
		}
	if (close (device_desc))
		perror ("close");
	if (unlink (pattern))
		perror ("unlink");
	}
#else
	fs_block_size = statbuf.st_blksize;
#endif
	if ((fin_desc = open(fin_name, O_RDONLY)) == -1) {
		perror (fin_name);
		if (close (fout_desc) != 0)
			perror (fout_name);
		return -4;
		}
	fin_len = lseek(fin_desc, 0, SEEK_END);
	if (fin_len == (off_t)-1 || lseek(fin_desc, 0, SEEK_SET) == (off_t)-1) {
		perror (fin_name);
		if (close (fout_desc) != 0)
			perror (fout_name);
		if (close (fin_desc) != 0)
			perror (fin_name);
		return -5;
		}
//	printf ("FS block size: %u, input file size: %u\n", fs_block_size, (unsigned)fin_len);
	if (fin_len == 0 || (buffer = malloc(fin_len)) == 0) {
		perror ("input file empty or too big");
		if (close (fout_desc) != 0)
			perror (fout_name);
		if (close (fin_desc) != 0)
			perror (fin_name);
		return -6;
		}
	if (read (fin_desc, buffer, fin_len) != fin_len) {
		perror (fin_name);
		if (close (fout_desc) != 0)
			perror (fout_name);
		if (close (fin_desc) != 0)
			perror (fin_name);
		return -7;
		}

#ifdef AUTO_RETRY
		{
		char *fout_desc_failed[AUTO_RETRY];
		unsigned fout_desc_failed_nb = 0, cpt;
		for (;;) {
#endif

#define HOLE_AT_BEGIN_SIZE	(fs_block_size * 12) /* E2FS dependant: nbof(i_direct_block) */
	if (lseek(fout_desc, HOLE_AT_BEGIN_SIZE, SEEK_SET) != HOLE_AT_BEGIN_SIZE) {
		perror (fout_name);
		if (close (fout_desc) != 0)
			perror (fout_name);
		if (close (fin_desc) != 0)
			perror (fin_name);
		return -8;
		}

	if (write (fout_desc, buffer, fin_len) != fin_len) {
		perror (fout_name);
		if (close (fout_desc) != 0)
			perror (fout_name);
		if (close (fin_desc) != 0)
			perror (fin_name);
		return -9;
		}

#ifdef AUTO_RETRY
		if ((ret = file_mapping (fout_desc, fin_len + HOLE_AT_BEGIN_SIZE)) > 0) {
			if (fout_desc_failed_nb < AUTO_RETRY) {
				unsigned begin_filename;

				printf ("Warning: created file with multiple segments, ");
				if (close (fout_desc) != 0) {
					perror (fout_name);
					ret = -10;
					goto return_error;
					}
				fout_desc_failed[fout_desc_failed_nb] = malloc (strlen (fout_name) + strlen ("tmpA") + 1);
				strcpy (fout_desc_failed[fout_desc_failed_nb], fout_name);
				begin_filename = strlen (fout_name);
				for (;;) {
					if (begin_filename == 0)
						break;
					if (   fout_desc_failed[fout_desc_failed_nb][begin_filename] == '/'
					    || fout_desc_failed[fout_desc_failed_nb][begin_filename] == '\\')
						break;
					begin_filename--;
					}
				strcpy (&fout_desc_failed[fout_desc_failed_nb][begin_filename+1], "tmpA");
				fout_desc_failed[fout_desc_failed_nb][begin_filename+4] += fout_desc_failed_nb;
				printf ("renaming to '%s' and retrying\n", fout_desc_failed[fout_desc_failed_nb]);

				if (rename (fout_name, fout_desc_failed[fout_desc_failed_nb]) != 0) {
					perror (fout_desc_failed[fout_desc_failed_nb]);
					ret = -10;
					goto return_error;
					}
				fout_desc_failed_nb++;
				if ((fout_desc = open (fout_name, O_CREAT|O_WRONLY|O_EXCL|O_SYNC, S_IRWXU)) == -1) {
					perror (fout_name);
					ret = -2;
					goto return_error;
					}
				}
			else {
				printf ("ERROR: did not acheive to create single segment file, cleaning\n");
				if (unlink (fout_name) != 0)
					perror (fout_name);
				ret = -10;
return_error:
				if (close (fin_desc) != 0)
					perror (fin_name);
				for (cpt = 0; cpt < fout_desc_failed_nb; cpt++) {
					if (unlink (fout_desc_failed[cpt]) != 0)
						perror (fout_desc_failed[cpt]);
					free (fout_desc_failed[cpt]);
					}
				return ret;
				}
			}
		else {
			if (fout_desc_failed_nb)
				printf ("Success: cleaning intermediate files\n");
			for (cpt = 0; cpt < fout_desc_failed_nb; cpt++) {
				if (unlink (fout_desc_failed[cpt]) != 0)
					perror (fout_desc_failed[cpt]);
				free (fout_desc_failed[cpt]);
				}
			break;
			}
		}
		}
#endif

	if (close (fout_desc) != 0)
		perror (fout_name);
	if (close (fin_desc) != 0)
		perror (fin_name);
	return ret;
}

