/*
Copyright 2011, 2012 Elias Aebi

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

using Gtk;
using Cairo;

[CCode (cname = "g_param_spec_boxed")]
public extern unowned ParamSpec param_spec_boxed (string name, string nick, string blurb, Type boxed_type, ParamFlags flags);
[CCode (cname = "g_param_spec_int")]
public extern unowned ParamSpec param_spec_int (string name, string nick, string blurb, int minimum, int maximum, int default_value, ParamFlags flags);

public class Solidity : ThemingEngine {
	private Gdk.RGBA color;
	private Gdk.RGBA background_color;
	private Gdk.RGBA border_color;
	private double radius;
	private Gdk.RGBA gradient_top_color;
	private Gdk.RGBA gradient_bottom_color;
	private Gdk.RGBA reflection_top_color;
	private Gdk.RGBA reflection_bottom_color;
	private Gdk.RGBA highlight_color;
	private Gdk.RGBA shadow_color;
	private int style;
	private Orientation orientation;
	private Gdk.RGBA inner_border_color;
	
	// constructor
	public Solidity () {
		this.register_property ("solidity", null, param_spec_boxed ("gradient-top-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_boxed ("gradient-bottom-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_boxed ("reflection-top-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_boxed ("reflection-bottom-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_boxed ("highlight-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_boxed ("shadow-color", "", "", typeof (Gdk.RGBA), 0));
		this.register_property ("solidity", null, param_spec_int ("style","","",0,4,1,0));
	}
	
	private bool has_state (StateFlags state) {
		return (bool) (this.get_state() & state);
	}
	
	// private drawing methods
	enum Corner {
		NONE = 0,
		TOPLEFT = 1 << 0,
		TOPRIGHT = 1 << 1,
		BOTTOMLEFT = 1 << 2,
		BOTTOMRIGHT = 1 << 3,
		ALL = 15
	}
	
	private void get_options (double width, double height) {
		color = this.get_color (this.get_state());
		this.background_color = this.get_background_color (this.get_state());
		this.border_color = this.get_border_color (this.get_state());
		radius = (double)this.get_property("border-radius", this.get_state()).get_int();
		if (this.has_class(STYLE_CLASS_RADIO) || radius<0.0 || radius>width/2.0 || radius>height/2.0)
			radius = (height<width)?(height/2.0):(width/2.0);
		if (this.has_class(STYLE_CLASS_SCALE) && (this.has_class(STYLE_CLASS_TROUGH)||this.has_class(STYLE_CLASS_PROGRESSBAR)) && radius>3.0)
			radius = 3.0;
		gradient_top_color = (Gdk.RGBA)this.get_property ("-solidity-gradient-top-color", this.get_state());
		gradient_bottom_color = (Gdk.RGBA)this.get_property ("-solidity-gradient-bottom-color", this.get_state());
		reflection_top_color = (Gdk.RGBA)this.get_property ("-solidity-reflection-top-color", this.get_state());
		reflection_bottom_color = (Gdk.RGBA)this.get_property ("-solidity-reflection-bottom-color", this.get_state());
		highlight_color = (Gdk.RGBA)this.get_property ("-solidity-highlight-color", this.get_state());
		shadow_color = (Gdk.RGBA)this.get_property ("-solidity-shadow-color", this.get_state());
		if (this.get_property("-solidity-style", this.get_state()).holds(typeof(int)))
			style = this.get_property("-solidity-style", this.get_state()).get_int();
		if (
			(this.has_class(STYLE_CLASS_BUTTON) && this.has_state(StateFlags.ACTIVE)) ||
			this.has_class(STYLE_CLASS_ENTRY) ||
			this.has_class(STYLE_CLASS_TROUGH) ||
			((this.has_class(STYLE_CLASS_CHECK) || this.has_class(STYLE_CLASS_RADIO)) && this.has_state(StateFlags.SELECTED))
		)
			inner_border_color = shadow_color;
		else
			inner_border_color = highlight_color;
		if (this.has_class(STYLE_CLASS_VERTICAL)) orientation = Orientation.VERTICAL;
		else orientation = Orientation.HORIZONTAL;
	}
	
	private void draw_line (Context cr, double x1, double y1, double x2, double y2) {
		cr.new_path ();
		cr.set_line_width (1.0);
		cr.set_line_cap (LineCap.BUTT);
		
		if ((x2-x1) > (y2-y1)) this.orientation = Orientation.HORIZONTAL;
		if ((y2-y1) > (x2-x1)) this.orientation = Orientation.VERTICAL;
		
		if (style==0) {
			cr.set_source_rgb (color.red, color.green, color.blue);
			if (orientation==Orientation.HORIZONTAL) {
				cr.move_to (x1, y1+0.5);
				cr.line_to (x2, y2+0.5);
			}
			if (orientation==Orientation.VERTICAL) {
				cr.move_to (x1+0.5, y1);
				cr.line_to (x2+0.5, y2);
			}
			cr.stroke ();
		}
		
		if (style==1 || style==2) {
			// draw the highlight
			if (style==1)
				cr.set_source_rgba (highlight_color.red, highlight_color.green, highlight_color.blue, highlight_color.alpha);
			if (style==2) {
				var highlight_pattern = new Pattern.linear (x1, y1, x2, y2);
				highlight_pattern.add_color_stop_rgba (0.0, highlight_color.red, highlight_color.green, highlight_color.blue, 0.0);
				highlight_pattern.add_color_stop_rgba (0.3, highlight_color.red, highlight_color.green, highlight_color.blue, highlight_color.alpha);
				highlight_pattern.add_color_stop_rgba (0.7, highlight_color.red, highlight_color.green, highlight_color.blue, highlight_color.alpha);
				highlight_pattern.add_color_stop_rgba (1.0, highlight_color.red, highlight_color.green, highlight_color.blue, 0.0);
				cr.set_source (highlight_pattern);
			}
			if (orientation==Orientation.HORIZONTAL) {
				cr.move_to (x1, y1+0.5);
				cr.line_to (x2, y2+0.5);
			}
			if (orientation==Orientation.VERTICAL) {
				cr.move_to (x1+0.5, y1);
				cr.line_to (x2+0.5, y2);
			}
			cr.stroke ();
			// draw shadow
			if (style==1)
				cr.set_source_rgba (shadow_color.red, shadow_color.green, shadow_color.blue, shadow_color.alpha);
			if (style==2) {
				var shadow_pattern = new Pattern.linear (x1, y1, x2, y2);
				shadow_pattern.add_color_stop_rgba (0.0, shadow_color.red, shadow_color.green, shadow_color.blue, 0.0);
				shadow_pattern.add_color_stop_rgba (0.3, shadow_color.red, shadow_color.green, shadow_color.blue, shadow_color.alpha);
				shadow_pattern.add_color_stop_rgba (0.7, shadow_color.red, shadow_color.green, shadow_color.blue, shadow_color.alpha);
				shadow_pattern.add_color_stop_rgba (1.0, shadow_color.red, shadow_color.green, shadow_color.blue, 0.0);
				cr.set_source (shadow_pattern);
			}
			if (orientation==Orientation.HORIZONTAL) {
				cr.move_to (x1, y1-0.5);
				cr.line_to (x2, y2-0.5);
			}
			if (orientation==Orientation.VERTICAL) {
				cr.move_to (x1-0.5, y1);
				cr.line_to (x2-0.5, y2);
			}
			cr.stroke ();
		}
	}
	
	private void draw_button (Context cr, Corner rounded_corners, double x, double y, double width, double height) {
		cr.new_path ();
		cr.set_line_width (1.0);
		cr.set_line_cap (LineCap.SQUARE);
		// draw the background
		if ((bool)rounded_corners&Corner.TOPLEFT && radius>0.5)
			cr.arc (x+radius, y+radius, radius-0.5, Math.PI, 1.5*Math.PI);
		else cr.move_to (x+0.5, y+0.5);
		if ((bool)rounded_corners&Corner.TOPRIGHT && radius>0.5)
			cr.arc (x+width-radius, y+radius, radius-0.5, -0.5*Math.PI, 0);
		else cr.line_to (x+width-0.5, y+0.5);
		if ((bool)rounded_corners&Corner.BOTTOMRIGHT && radius>0.5)
			cr.arc (x+width-radius, y+height-radius, radius-0.5, 0, 0.5*Math.PI);
		else cr.line_to (x+width-0.5, y+height-0.5);
		if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>0.5)
			cr.arc (x+radius, y+height-radius, radius-0.5, 0.5*Math.PI, Math.PI);
		else cr.line_to (x+0.5, y+height-0.5);
		cr.close_path ();
		var path = cr.copy_path ();
		cr.set_source_rgb (background_color.red, background_color.green, background_color.blue);
		cr.fill_preserve ();
		// draw the gradient
		if (gradient_top_color.alpha!=0.0 || gradient_bottom_color.alpha!=0.0) {
			Pattern pattern;
			if (orientation==Orientation.VERTICAL)
				pattern = new Pattern.linear (x, y, x+width, y);
			else pattern = new Pattern.linear (x, y, x, y+height);
			pattern.add_color_stop_rgba (0.0, gradient_top_color.red, gradient_top_color.green, gradient_top_color.blue, gradient_top_color.alpha);
			pattern.add_color_stop_rgba (1.0, gradient_bottom_color.red, gradient_bottom_color.green, gradient_bottom_color.blue, gradient_bottom_color.alpha);
			cr.set_source (pattern);
			cr.fill ();
		}
		// draw the reflection
		cr.new_path ();
		if (reflection_top_color.alpha>0.0 || reflection_bottom_color.alpha>0.0) {
			if (orientation==Orientation.VERTICAL) {
				if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>0.5)
					cr.arc (x+radius, y+height-radius, radius-0.5, 0.5*Math.PI, Math.PI);
				else cr.move_to (x+0.5, y+height-0.5);
				if ((bool)rounded_corners&Corner.TOPLEFT && radius>0.5)
					cr.arc (x+radius, y+radius, radius-0.5, Math.PI, 1.5*Math.PI);
				else cr.line_to (x+0.5, y+0.5);
				cr.line_to (Math.round(x+width*0.5), y+0.5);
				cr.line_to (Math.round(x+width*0.5), y+height-0.5);
			}
			else {
				if ((bool)rounded_corners&Corner.TOPLEFT && radius>0.5)
					cr.arc (x+radius, y+radius, radius-0.5, Math.PI, 1.5*Math.PI);
				else cr.move_to (x+0.5, y+0.5);
				if ((bool)rounded_corners&Corner.TOPRIGHT && radius>0.5)
					cr.arc (x+width-radius, y+radius, radius-0.5, -0.5*Math.PI, 0);
				else cr.line_to (x+width-0.5, y+0.5);
				cr.line_to (x+width-0.5, Math.round(y+height*0.5));
				cr.line_to (x+0.5, Math.round(y+height*0.5));
			}
			cr.close_path ();
			
			Pattern pattern;
			if (orientation==Orientation.VERTICAL)
				pattern = new Pattern.linear (x, y, Math.round(x+width*0.5), y);
			else pattern = new Pattern.linear (x, y, x, Math.round(y+height*0.5));
			pattern.add_color_stop_rgba (0.0, reflection_top_color.red, reflection_top_color.green, reflection_top_color.blue, reflection_top_color.alpha);
			pattern.add_color_stop_rgba (1.0, reflection_bottom_color.red, reflection_bottom_color.green, reflection_bottom_color.blue, reflection_bottom_color.alpha);
			cr.set_source (pattern);
			cr.fill ();
		}
		// draw the highlight/shadow
		if (this.style == 1) {
			if ((bool)rounded_corners&Corner.TOPLEFT && radius>1.5)
				cr.arc (x+radius-1.0, y+radius, radius-1.5, Math.PI, 1.5*Math.PI);
			else cr.move_to (x+1.5, y+1.5);
			if ((bool)rounded_corners&Corner.TOPRIGHT && radius>1.5)
				cr.arc (x+width-radius+1.0, y+radius, radius-1.5, -0.5*Math.PI, 0);
			else cr.line_to (x+width-1.5, y+1.5);
		}
		if (this.style == 2) {
			if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>1.5)
				cr.arc (x+radius, y+height-radius+1.0, radius-1.5, 0.5*Math.PI, Math.PI);
			else cr.move_to (x+1.5, y+height-1.5);
			if ((bool)rounded_corners&Corner.TOPLEFT && radius>1.5)
				cr.arc (x+radius, y+radius, radius-1.5, Math.PI, 1.5*Math.PI);
			else cr.line_to (x+1.5, y+1.5);
			if ((bool)rounded_corners&Corner.TOPRIGHT && radius>1.5)
				cr.arc (x+width-radius+1.0, y+radius, radius-1.5, -0.5*Math.PI, 0);
			else cr.line_to (x+width-1.5, y+1.5);
		}
		if (this.style == 3) {
			if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>1.5)
				cr.arc (x+radius, y+height-radius+1.0, radius-1.5, 0.5*Math.PI, Math.PI);
			else cr.move_to (x+1.5, y+height-1.5);
			if ((bool)rounded_corners&Corner.TOPLEFT && radius>1.5)
				cr.arc (x+radius, y+radius, radius-1.5, Math.PI, 1.5*Math.PI);
			else cr.line_to (x+1.5, y+1.5);
			if ((bool)rounded_corners&Corner.TOPRIGHT && radius>1.5)
				cr.arc (x+width-radius, y+radius, radius-1.5, -0.5*Math.PI, 0);
			else cr.line_to (x+width-1.5, y+1.5);
			if ((bool)rounded_corners&Corner.BOTTOMRIGHT && radius>1.5)
				cr.arc (x+width-radius, y+height-radius+1.0, radius-1.5, 0, 0.5*Math.PI);
			else cr.line_to (x+width-1.5, y+height-1.5);
		}
		if (this.style == 4) {
			if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>1.5)
				cr.arc (x+radius, y+height-radius, radius-1.5, 0.5*Math.PI, Math.PI);
			else cr.move_to (x+1.5, y+height-1.5);
			if ((bool)rounded_corners&Corner.TOPLEFT && radius>1.5)
				cr.arc (x+radius, y+radius, radius-1.5, Math.PI, 1.5*Math.PI);
			else cr.line_to (x+1.5, y+1.5);
			if ((bool)rounded_corners&Corner.TOPRIGHT && radius>1.5)
				cr.arc (x+width-radius, y+radius, radius-1.5, -0.5*Math.PI, 0);
			else cr.line_to (x+width-1.5, y+1.5);
			if ((bool)rounded_corners&Corner.BOTTOMRIGHT && radius>1.5)
				cr.arc (x+width-radius, y+height-radius, radius-1.5, 0, 0.5*Math.PI);
			else cr.line_to (x+width-1.5, y+height-1.5);
			cr.close_path ();
		}
		cr.set_source_rgba (inner_border_color.red, inner_border_color.green, inner_border_color.blue, inner_border_color.alpha);
		cr.stroke ();
		// draw the border
		cr.append_path (path);
		cr.set_source_rgb (border_color.red, border_color.green, border_color.blue);
		cr.stroke ();
	}
	
	private void draw_bar (Context cr, double x, double y, double width, double height) {
		cr.new_path ();
		cr.set_line_width (1.0);
		cr.set_line_cap (LineCap.SQUARE);
		// draw the background
		cr.rectangle (x, y, width, height);
		cr.set_source_rgb (background_color.red, background_color.green, background_color.blue);
		cr.fill_preserve ();
		var pattern = new Pattern.linear (x, y, x, y+height);
		pattern.add_color_stop_rgba (0.0, gradient_top_color.red, gradient_top_color.green, gradient_top_color.blue, gradient_top_color.alpha);
		pattern.add_color_stop_rgba (1.0, gradient_bottom_color.red, gradient_bottom_color.green, gradient_bottom_color.blue, gradient_bottom_color.alpha);
		cr.set_source (pattern);
		cr.fill ();
		// draw the highlight
		cr.move_to (x+0.5, y+0.5);
		cr.line_to (x+width-0.5, y+0.5);
		cr.set_source_rgba (highlight_color.red, highlight_color.green, highlight_color.blue, highlight_color.alpha);
		cr.stroke ();
		// draw the shadow
		cr.move_to (x+0.5, y+height-0.5);
		cr.line_to (x+width-0.5, y+height-0.5);
		cr.set_source_rgba (shadow_color.red, shadow_color.green, shadow_color.blue, shadow_color.alpha);
		cr.stroke ();
	}
	
	private void draw_tab (Context cr, double x, double y, double width, double height, PositionType gap_side) {
		if (gap_side==PositionType.BOTTOM||gap_side==PositionType.TOP)
			height += 1.0;
		if (gap_side==PositionType.RIGHT||gap_side==PositionType.LEFT) {
			width += 1.0;
			this.orientation = Orientation.VERTICAL;
		}
		if (gap_side==PositionType.TOP)
			y -= 1.0;
		if (gap_side==PositionType.LEFT)
			x -= 1.0;
		this.draw_button (cr, Corner.NONE, x, y, width, height);
	}
	
	private void draw_spinner (Context cr, double x, double y, double width, double height) {
		double radius = (width>height)?(height/2.0):(width/2.0);
		Gdk.RGBA color = this.get_color (this.get_state());
		double progress;
		this.state_is_running (StateType.ACTIVE, out progress);
		
		for (double i=0.0; i<2.0*Math.PI; i+=Math.PI/3.0) {
			double offset = progress*Math.PI/3.0;
			double alpha = (i-offset)/(2.0*Math.PI)-progress;
			if (alpha<=0.0) alpha += 1.0;
			cr.set_source_rgba (color.red, color.green, color.blue, alpha);
			cr.arc (x+width/2.0, y+height/2.0, radius, i-offset, i+0.25*Math.PI-offset);
			cr.arc_negative (x+width/2.0, y+height/2.0, radius*2.0/3.0, i+0.25*Math.PI-offset, i-offset);
			cr.close_path ();
			cr.fill ();
		}
		
		/* an alternate spinner
		double pos1 = progress*4.0*Math.PI;
		double pos2 = (1.0-progress)*2.0*Math.PI;
		cr.set_source_rgb (color.red, color.green, color.blue);
		cr.arc (x+width/2.0, y+height/2.0, radius, pos1, pos1+0.5*Math.PI);
		cr.arc_negative (x+width/2.0, y+height/2.0, radius*0.75, pos1+0.5*Math.PI, pos1);
		cr.close_path ();
		cr.fill ();
		cr.arc (x+width/2.0, y+height/2.0, radius*0.75, pos2, pos2+0.5*Math.PI);
		cr.arc_negative (x+width/2.0, y+height/2.0, radius*0.5, pos2+0.5*Math.PI, pos2);
		cr.close_path ();
		cr.fill ();
		*/
	}
	
	private void draw_bottom_highlight (Context cr, Corner rounded_corners, double x, double y, double width, double height) {
		cr.new_path ();
		cr.set_line_width (1.0);
		cr.set_line_cap (LineCap.SQUARE);
		
		if ((bool)rounded_corners&Corner.BOTTOMRIGHT && radius>0.5)
			cr.arc (x+width-radius, y+height-radius, radius-0.5, 0, 0.5*Math.PI);
		else cr.move_to (x+width-0.5, y+height-0.5);
		if ((bool)rounded_corners&Corner.BOTTOMLEFT && radius>0.5)
			cr.arc (x+radius, y+height-radius, radius-0.5, 0.5*Math.PI, Math.PI);
		else cr.line_to (x+0.5, y+height-0.5);
		cr.set_source_rgba (highlight_color.red, highlight_color.green, highlight_color.blue, highlight_color.alpha);
		cr.stroke ();
	}
	
	// override the virtual methods
	public override void render_line (Context cr, double x0, double y0, double x1, double y1) {
		this.get_options (0, 0);
		this.draw_line (cr, x0, y0, x1, y1);
	}
	
	public override void render_background (Context cr, double x, double y, double width, double height) {
		this.get_options (width, height);
		// Buttons
		if (this.has_class(STYLE_CLASS_BUTTON)) {
			// ComboBoxText
			if (this.get_path().iter_has_class(this.get_path().length()-2,STYLE_CLASS_COMBOBOX_ENTRY)) {
				this.draw_bottom_highlight (cr, Corner.TOPRIGHT|Corner.BOTTOMRIGHT, x, y, width, height);
				this.draw_button (cr, Corner.TOPRIGHT|Corner.BOTTOMRIGHT, x, y, width, height-1);
			}
			
			// SpinButton
			else if (this.has_class(STYLE_CLASS_SPINBUTTON)) {
				
			}
			
			// TreeView
			else if (this.get_path().has_parent(typeof(TreeView))) {
				if ((bool)this.get_state() & StateFlags.ACTIVE)
					this.draw_button (cr, Corner.NONE, x-1, y-1, width+2, height+2);
				else
					this.draw_bar (cr, x, y, width, height);
			}
			
			// Scrollbar (don't draw anything)
			else if (this.has_class(STYLE_CLASS_SCROLLBAR));
			
			// other buttons
			else {
				this.draw_bottom_highlight (cr, Corner.ALL, x, y, width, height);
				this.draw_button (cr, Corner.ALL, x, y, width, height-1);
			}
		}
		
		// Entry
		// the Buttons of a SpinButtons have the entry class. work around this
		if (this.has_class(STYLE_CLASS_ENTRY) && !(this.has_class(STYLE_CLASS_SPINBUTTON) && this.has_class(STYLE_CLASS_BUTTON))) {
			// ComboBoxText
			if (this.get_path().iter_has_class(this.get_path().length()-2,STYLE_CLASS_COMBOBOX_ENTRY)) {
				this.draw_bottom_highlight (cr, Corner.TOPLEFT|Corner.BOTTOMLEFT, x, y, width+1, height);
				this.draw_button (cr, Corner.TOPLEFT|Corner.BOTTOMLEFT, x, y, width+1, height-1);
			}
			
			// other Entries
			else {
				this.draw_bottom_highlight (cr, Corner.ALL, x, y, width, height);
				this.draw_button (cr, Corner.ALL, x, y, width, height-1);
			}
		}
		
		// troughs
		if (this.has_class(STYLE_CLASS_TROUGH)) {
			// Scrollbar
			if (this.has_class(STYLE_CLASS_SCROLLBAR)) {
				this.draw_button (cr, Corner.ALL, x, y, width, height);
			}
			
			// Scale
			if (this.has_class(STYLE_CLASS_SCALE)) {
				if (this.has_class(STYLE_CLASS_VERTICAL)) {
					double middle = x+(width*0.5);
					this.draw_bottom_highlight (cr, Corner.ALL, middle-3, y, 6, height);
					this.draw_button (cr, Corner.ALL, middle-3, y, 6, height-1);
				}
				if (this.has_class(STYLE_CLASS_HORIZONTAL)) {
					double middle = y+(height*0.5);
					this.draw_bottom_highlight (cr, Corner.ALL, x, middle-3, width, 7);
					this.draw_button (cr, Corner.ALL, x, middle-3, width, 6);
				}
			}
			// ProgressBar and Switch
			if (this.get_path().is_type(typeof(ProgressBar))||this.get_path().is_type(typeof(Switch))) {
				this.draw_bottom_highlight (cr, Corner.ALL, x, y, width, height);
				this.draw_button (cr, Corner.ALL, x, y, width, height-1);
			}
		}
		
		// Menus
		if (this.has_class(STYLE_CLASS_MENUITEM)) {
			if (this.has_class(STYLE_CLASS_MENUBAR)) {
				this.draw_button (cr, Corner.TOPLEFT|Corner.TOPRIGHT, x, y, width, height+1);
			}
			else {
				this.draw_button (cr, Corner.NONE, x, y, width, height);
			}
		}
		
		// ToolBar, MenuBar and InfoBar
		if (this.has_class(STYLE_CLASS_TOOLBAR) || this.has_class(STYLE_CLASS_INFO) || (this.has_class(STYLE_CLASS_MENUBAR) && !this.has_class(STYLE_CLASS_MENUITEM))) {
			if (style==0) {
				cr.set_source_rgb (background_color.red, background_color.green, background_color.blue);
				cr.rectangle (x, y, width, height);
				cr.fill ();
			}
			if (style==1)
				this.draw_bar (cr, x, y, width, height);
		}
		
		// TreeView, Notebook and Window background
		if (this.has_class(STYLE_CLASS_CELL) || this.has_class(STYLE_CLASS_NOTEBOOK) || this.has_class(STYLE_CLASS_BACKGROUND)) {
			cr.set_source_rgb (background_color.red, background_color.green, background_color.blue);
			cr.rectangle (x, y, width, height);
			cr.fill ();
		}
	}
	
	public override void render_frame (Context cr, double x, double y, double width, double height) {
		if (this.has_class(STYLE_CLASS_MENU)||this.has_class(STYLE_CLASS_TOOLTIP)) {
			cr.set_line_width (1.0);
			var color = this.get_border_color (this.get_state());
			cr.set_source_rgb (color.red, color.green, color.blue);
			cr.rectangle (x+0.5, y+0.5, width-1.0, height-1.0);
			cr.stroke ();
		}
		
		// Separator (if -GtkWidget-wide-separators is set to true)
		if (this.has_class(STYLE_CLASS_SEPARATOR)) {
			this.get_options (width, height);
			if (width > height)
				this.draw_line (cr, x, y+1, x+width, y+1);
			if (height > width)
				this.draw_line (cr, x+1, y, x+1, y+height);
		}
	}
	
	// Notebook
	public override void render_frame_gap (Context cr, double x, double y, double width, double height, PositionType gap_side, double xy0_gap, double xy1_gap) {
		// render a frame with a gap
		Gdk.RGBA color = this.get_border_color (this.get_state());
		cr.set_source_rgb (color.red, color.green, color.blue);
		cr.set_line_width (1.0);
		cr.set_line_cap (LineCap.SQUARE);
		if (gap_side==PositionType.TOP) {
			cr.move_to (x+xy1_gap-0.5, y+0.5);
			cr.line_to (x+width-0.5, y+0.5);
			cr.line_to (x+width-0.5, y+height-0.5);
			cr.line_to (x+0.5, y+height-0.5);
			cr.line_to (x+0.5, y+0.5);
			cr.line_to (x+xy0_gap+0.5, y+0.5);
		}
		if (gap_side==PositionType.RIGHT) {
			cr.move_to (x+width-0.5, y+xy1_gap-0.5);
			cr.line_to (x+width-0.5, y+height-0.5);
			cr.line_to (x+0.5, y+height-0.5);
			cr.line_to (x+0.5, y+0.5);
			cr.line_to (x+width-0.5, y+0.5);
			cr.line_to (x+width-0.5, y+xy0_gap+0.5);
		}
		if (gap_side==PositionType.BOTTOM) {
			cr.move_to (x+xy1_gap-0.5, y+height-0.5);
			cr.line_to (x+width-0.5, y+height-0.5);
			cr.line_to (x+width-0.5, y+0.5);
			cr.line_to (x+0.5, y+0.5);
			cr.line_to (x+0.5, y+height-0.5);
			cr.line_to (x+xy0_gap+0.5, y+height-0.5);
		}
		if (gap_side==PositionType.LEFT) {
			cr.move_to (x+0.5, y+xy1_gap-0.5);
			cr.line_to (x+0.5, y+height-0.5);
			cr.line_to (x+width-0.5, y+height-0.5);
			cr.line_to (x+width-0.5, y+0.5);
			cr.line_to (x+0.5, y+0.5);
			cr.line_to (x+0.5, y+xy0_gap+0.5);
		}
		cr.stroke ();
	}
	
	public override void render_extension (Context cr, double x, double y, double width, double height, PositionType gap_side) {
		// clip
		cr.rectangle (x, y, width, height);
		cr.clip ();
		
		// render a tab
		this.get_options (width, height);
		this.draw_tab (cr, x, y, width, height, gap_side);
	}
	
	// CheckButton
	public override void render_check (Context cr, double x, double y, double width, double height) {
		this.get_options (width, height);
		this.draw_bottom_highlight (cr, Corner.ALL, x, y, width, height);
		this.draw_button (cr, Corner.ALL, x, y, width, height-1);
		if ((bool)this.get_state()&StateFlags.ACTIVE) {
			Gdk.RGBA color = this.get_color (this.get_state());
			cr.set_source_rgb (color.red, color.green, color.blue);
			cr.set_line_width (2.0);
			cr.set_line_cap (LineCap.ROUND);
			cr.move_to (x+width*0.25, y+height*0.5);
			cr.line_to (x+width*(5.0/12.0), y+height*0.75);
			cr.line_to (x+width*0.75, y+height*0.25);
			cr.stroke ();
		}
		else if ((bool)this.get_state()&StateFlags.INCONSISTENT) {
			Gdk.RGBA color = this.get_color (this.get_state());
			cr.set_source_rgb (color.red, color.green, color.blue);
			cr.set_line_width (2.0);
			cr.set_line_cap (LineCap.ROUND);
			cr.move_to (x+width*0.25, y+height*0.5);
			cr.line_to (x+width*0.75, y+height*0.5);
			cr.stroke ();
		}
	}
	
	// RadioButton
	public override void render_option (Context cr, double x, double y, double width, double height) {
		this.get_options (width, height);
		double r = (height-1.0)/2.0;
		this.draw_bottom_highlight (cr, Corner.ALL, x, y, 2.0*r, 2.0*r+1.0);
		this.draw_button (cr, Corner.ALL, x, y, r*2.0, r*2.0);
		if ((bool)this.get_state()&StateFlags.ACTIVE) {
			Gdk.RGBA color = this.get_color (this.get_state());
			cr.set_source_rgb (color.red, color.green, color.blue);
			cr.arc (x+r, y+r, r-3.0, 0, Math.PI*2.0);
			cr.fill ();
		}
	}
	
	// arrows
	private void draw_arrow (Context cr, double cx, double cy, double size, double angle) {
		cr.save ();
		cr.translate (Math.round(cx), Math.round(cy));
		cr.rotate (angle);
		size = Math.round(size/4.0)*4.0;
		Gdk.RGBA color = this.get_color (this.get_state());
		cr.set_source_rgb (color.red, color.green, color.blue);
		cr.move_to (0.0, -0.25*size);
		cr.line_to (-0.5*size, 0.25*size);
		cr.line_to (0.5*size, 0.25*size);
		cr.fill ();
		cr.restore ();
	}
	
	public override void render_arrow (Context cr, double angle, double x, double y, double size) {
		this.draw_arrow (cr, x+size*0.5, y+size*0.5, size, angle);
	}
	
	public override void render_expander (Context cr, double x, double y, double width, double height) {
		double size = (width>height)?(height*0.8):(width*0.8);
		if ((bool)this.get_state()&StateFlags.ACTIVE)
			this.draw_arrow (cr, x+width*0.5, y+height*0.5, size, Math.PI);
		else
			this.draw_arrow (cr, x+width*0.5, y+height*0.5, size, Math.PI*0.5);
	}
	
	public override void render_focus (Context cr, double x, double y, double width, double height) {
		
	}
	/*
	public override void render_layout (Context cr, double x, double y, Pango.Layout layout) {
		
	}
	*/
	public override void render_slider (Context cr, double x, double y, double width, double height, Orientation orientation) {
		this.get_options (width, height);
		if (this.get_path().is_type(typeof(Switch)))
			this.draw_button (cr, Corner.ALL, x, y, width, height-1);
		else
			this.draw_button (cr, Corner.ALL, x, y, width, height);
	}
	
	public override void render_handle (Context cr, double x, double y, double width, double height) {
		// for example the window resize handle
		var color = this.get_background_color (this.get_state());
		cr.set_source_rgb (color.red, color.green, color.blue);
		cr.rectangle (x, y, width, height);
		cr.fill ();
	}
	
	public override void render_activity (Context cr, double x, double y, double width, double height) {
		this.get_options (width, height);
		
		// Spinner
		if (this.has_class(STYLE_CLASS_SPINNER)) {
			this.draw_spinner (cr, x, y, width, height);
		}
		
		// ProgressBar of Scale
		else if (this.get_path().is_type(typeof(Scale))) {
			if (this.has_class(STYLE_CLASS_VERTICAL)) {
				double middle = x+(width*0.5);
				this.draw_button (cr, Corner.ALL, middle-3, y, 6, height-1);
			}
			if (this.has_class(STYLE_CLASS_HORIZONTAL)) {
				double middle = y+(height*0.5);
				this.draw_button (cr, Corner.ALL, x, middle-3, width, 6);
			}
		}
		
		// other ProgressBars
		else
			this.draw_button (cr, Corner.ALL, x, y, width, height-1);
	}
}

// the following functions are called by GTK when loading and unloading the engine (AFAIK)

[ModuleInit]
public void theme_init (TypeModule module) {
}

public void theme_exit () {
}

public ThemingEngine create_engine () {
	return new Solidity ();
}
