/*
 * Copyright (c) 2011-2012 Lucas Baudin <xapantu@gmail.com>,
 *                         Jaap Broekhuizen <jaapz.b@gmail.com>,
 *                         Victor Eduardo <victoreduardm@gmal.com>,
 *                         Tom Beckmann <tom@elementaryos.org>
 *
 * This is a free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; see the file COPYING.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */

public class Granite.Demo : Granite.Application {

    /**
     * Small helper class for constructing labels a bit faster.
     */
    private class LLabel : Gtk.Label {
        public LLabel (string label) {
            this.label = label;
            halign = Gtk.Align.START;
        }

        public LLabel.indent (string label) {
            this.label = label;
            margin_left = 12;
        }

        public LLabel.markup (string label) {
            this.label = label;
            use_markup = true;
        }

        public LLabel.right (string label) {
            this.label = label;
            halign = Gtk.Align.END;
        }

        public LLabel.right_with_markup (string label) {
            this.label = label;
            halign = Gtk.Align.END;
        }
    }

    /**
     * SourceList item. It stores the number of the corresponding page in the notebook widget.
     */
    private class SourceListItem : Granite.Widgets.SourceList.Item {
        public int page_num { get; set; default = -1; }
        private static Icon? themed_icon;

        public SourceListItem (string title) {
            base (title);
            editable = true;

            if (themed_icon == null)
                themed_icon = new ThemedIcon.with_default_fallbacks ("help-info-symbolic");

            icon = themed_icon;
        }
    }

    private Gtk.Window window; // both NORMAL and DARK window modes

    private Gtk.Grid main_layout; // outer-most container

    /**
     * Basic app information for Granite.Application. This is used by the About dialog.
     */
    construct {
        application_id = "demo.granite.org";
        program_name = "Granite Demo";
        app_years = "2011 - 2012";

        build_version = "0.2";
        app_icon = "text-editor";
        main_url = "https://launchpad.net/granite";
        bug_url = "https://bugs.launchpad.net/granite";
        help_url = "https://answers.launchpad.net/granite";
        translate_url = "https://translations.launchpad.net/granite";

        about_documenters = {"Valadoc", null };
        about_artists = {"Daniel P. Fore", null };
        about_authors = {"Maxwell Barvian <mbarvian@gmail.com>",
                         "Daniel Foré <bunny@go-docky.com>",
                         "Avi Romanoff <aviromanoff@gmail.com>",
                         "Lucas Baudin <xapantu@gmail.com>",
                         "Victor Eduardo <victoreduardm@gmail.com>",
                         "Tom Beckmann <tombeckmann@online.de>",
                         null
                         };

        about_comments = "A demo of the Granite toolkit";
        about_translators = "Launchpad Translators";
        about_license_type = Gtk.License.GPL_3_0;
    }


    public override void activate () {
        window = new Gtk.Window ();

        window.delete_event.connect (() => {
            Gtk.main_quit ();
            return false;
        });

        var main_toolbar = new Gtk.Toolbar ();
        main_toolbar.get_style_context ().add_class (Gtk.STYLE_CLASS_PRIMARY_TOOLBAR);
        main_toolbar.hexpand = true;
        main_toolbar.vexpand = false;

        // SourceList
        var sidebar = new Granite.Widgets.SourceList ();

        var page_switcher = new Gtk.Notebook ();
        page_switcher.show_tabs = false;
        page_switcher.show_border = false;
        page_switcher.expand = true;

        sidebar.item_selected.connect ((item) => {
            var sidebar_item = item as SourceListItem;
            assert (sidebar_item != null);
            page_switcher.set_current_page (sidebar_item.page_num);
        });

        // Main sidebar categories
        var widgets_category = new Granite.Widgets.SourceList.ExpandableItem ("Widgets");
        var services_category = new Granite.Widgets.SourceList.ExpandableItem ("Services");

        // Add and expand categories
        sidebar.root.add (widgets_category);
        sidebar.root.add (services_category);
        sidebar.root.expand_all ();

        var sidebar_paned = new Granite.Widgets.ThinPaned ();
        sidebar_paned.pack1 (sidebar, true, false);
        sidebar_paned.pack2 (page_switcher, true, false);
        sidebar_paned.expand = true;

        // Statusbar
        var statusbar = new Granite.Widgets.StatusBar ();
        statusbar.set_text ("Granite.Widgets.StatusBar");
        statusbar.hexpand = true;
        statusbar.vexpand = false;

        // Main widget structure
        main_layout = new Gtk.Grid ();
        main_layout.expand = true;
        main_layout.orientation = Gtk.Orientation.VERTICAL;
        main_layout.add (main_toolbar);
        main_layout.add (sidebar_paned);
        main_layout.add (statusbar);

        window.add (main_layout);

        // Welcome widget
        var welcome_screen = create_welcome_screen ();
        var welcome_item = new SourceListItem ("Welcome");
        widgets_category.add (welcome_item);
        welcome_item.page_num = page_switcher.append_page (welcome_screen, null);

        // Select welcome widget
        sidebar.selected = welcome_item;

        // Light window
        var light_window_icon = new Gtk.Image.from_icon_name ("document-new", Gtk.IconSize.LARGE_TOOLBAR);
        var light_window_item = new Gtk.ToolButton (light_window_icon, "Show LightWindow");
        light_window_item.icon_name = "document-new";
        light_window_item.tooltip_text = "Show Light Window";
        light_window_item.halign = light_window_item.valign = Gtk.Align.CENTER;
        light_window_item.clicked.connect (show_light_window);

        main_toolbar.insert (light_window_item, -1);

        // StaticNotebook
        var staticnotebook = new Granite.Widgets.StaticNotebook ();
        var pageone = new Gtk.Label ("Page 1");

        staticnotebook.append_page (new Gtk.Label ("Page 1"), pageone);
        staticnotebook.append_page (new Gtk.Label ("Page 2"), new Gtk.Label ("Page 2"));
        staticnotebook.append_page (new Gtk.Label ("Page 3"), new Gtk.Label ("Page 3"));

        staticnotebook.page_changed.connect (() => {
            pageone.set_text ("Page changed");
        });

        var static_notebook_item = new SourceListItem ("StaticNotebook");
        static_notebook_item.page_num = page_switcher.append_page (staticnotebook, null);
        widgets_category.add (static_notebook_item);

        // ModeButton
        var mode_button = new Granite.Widgets.ModeButton ();
        mode_button.valign = Gtk.Align.CENTER;
        mode_button.halign = Gtk.Align.CENTER;

        var normal_mode_index = mode_button.append (new Gtk.Label ("Normal"));
        var dark_mode_index = mode_button.append (new Gtk.Label ("Dark"));

        mode_button.selected = normal_mode_index;

        mode_button.mode_changed.connect (() => {
            var settings = Gtk.Settings.get_default ();
            settings.gtk_application_prefer_dark_theme = (mode_button.selected == dark_mode_index);
        });

        var mode_button_item = new Gtk.ToolItem ();
        mode_button_item.add (mode_button);
        main_toolbar.insert (mode_button_item, -1);

        mode_button_item.halign = mode_button_item.valign = Gtk.Align.CENTER;

        // PopOvers
        var popover_statusbar_item = new Gtk.Button ();
        popover_statusbar_item.relief = Gtk.ReliefStyle.NONE;
        popover_statusbar_item.tooltip_text = "Show PopOver";
        popover_statusbar_item.add (new Gtk.Image.from_icon_name ("help-info-symbolic", Gtk.IconSize.MENU));
        statusbar.insert_widget (popover_statusbar_item);

        popover_statusbar_item.clicked.connect (() => {
            var pop = new Granite.Widgets.PopOver ();

            var pop_hbox = pop.get_content_area () as Gtk.Container;
            pop_hbox.add (new Granite.Widgets.HintedEntry ("This is an HintedEntry"));
            pop_hbox.add (new Gtk.Label ("Another label"));

            var mode_pop = new Granite.Widgets.ModeButton ();
            mode_pop.append (new Gtk.Label ("Mode 1"));
            mode_pop.append (new Gtk.Label ("Mode 2"));
            mode_pop.append (new Gtk.Label ("Mode 3"));

            pop_hbox.add (mode_pop);
            pop_hbox.add (new Granite.Widgets.DatePicker ());

            pop.set_parent_pop (window);
            pop.move_to_widget (popover_statusbar_item);

            pop.show_all ();
            pop.present ();
            pop.run ();
            pop.destroy ();
        });

        // Date widget
        var calendar_tool_item = new Gtk.ToolItem ();
        calendar_tool_item.margin_left = 12;
        var date_button = new Granite.Widgets.DatePicker.with_format ("%d-%m-%y");
        calendar_tool_item.add (date_button);
        main_toolbar.insert (calendar_tool_item, -1);

        // Time widget
        var time_tool_item = new Gtk.ToolItem ();
        time_tool_item.margin_left = 12;
        time_tool_item.valign = Gtk.Align.CENTER;
        var time_button = new Granite.Widgets.TimePicker ();
        time_tool_item.add (time_button);
        main_toolbar.insert (time_tool_item, -1);

        // Dynamic notebook
        var dynamic_notebook = create_dynamic_notebook ();
        var dynamic_notebook_item = new SourceListItem ("DynamicNotebook");
        widgets_category.add (dynamic_notebook_item);
        dynamic_notebook_item.page_num = page_switcher.append_page (dynamic_notebook, null);

        var right_sep = new Gtk.SeparatorToolItem ();
        right_sep.draw = false;
        right_sep.set_expand (true);
        main_toolbar.insert (right_sep, -1);

        // Contractor
        var contractor_tab = new Gtk.Box (Gtk.Orientation.VERTICAL, 0);
        var text_view = new Gtk.TextView ();

        var hash_tables = Granite.Services.Contractor.get_contract ("/.zip", "application/zip");
        foreach (var hash_table in hash_tables) {
            text_view.buffer.text += hash_table.lookup ("Name")
                                  + ": " + hash_table.lookup ("Description")
                                  + " icon: " + hash_table.lookup ("Exec") + "\n";
        }

        contractor_tab.add (text_view);
        contractor_tab.add (new Granite.Widgets.ContractorView ("file:///home/user/file.txt", "text/plain"));
        var contractor_menu = new Granite.Widgets.ContractorMenu ("/home/user/file.txt", "text");
        var contractor_button_image = new Gtk.Image.from_icon_name ("document-export",
                                                                     Gtk.IconSize.LARGE_TOOLBAR);
        var contractor_tool_item = new Granite.Widgets.ToolButtonWithMenu (contractor_button_image,
                                                                            "Share", contractor_menu);
        main_toolbar.insert (contractor_tool_item, -1);

        contractor_tool_item.halign = contractor_tool_item.valign = Gtk.Align.CENTER;

        var contractor_item = new SourceListItem ("Contractor");
        contractor_item.page_num = page_switcher.append_page (contractor_tab, null);
        services_category.add (contractor_item);

        // Search Entry
        var search_entry = new Granite.Widgets.SearchBar ("Search");
        var search_item = new Gtk.ToolItem ();
        search_item.add (search_entry);
        search_item.margin_left = 12;
        main_toolbar.insert (search_item, -1);

        // App Menu (this gives access to the About dialog)
        var main_menu = create_appmenu (new Gtk.Menu ());
        main_menu.margin_left = 12;
        main_toolbar.insert (main_menu, -1);

        window.set_default_size (800, 550);
        window.show_all ();
    }

    private Granite.Widgets.Welcome create_welcome_screen () {
        var welcome = new Granite.Widgets.Welcome ("Granite's Welcome Screen",
                                                    "This Is Granite's Welcome Widget.");

        Gdk.Pixbuf? pixbuf = null;

        try {
            pixbuf = Gtk.IconTheme.get_default ().load_icon ("document-new", 48,
                                                             Gtk.IconLookupFlags.GENERIC_FALLBACK);
        } catch (Error e) {
            warning ("Could not load icon, %s", e.message);
        }

        Gtk.Image? image = new Gtk.Image.from_icon_name ("document-open", Gtk.IconSize.DIALOG);

        // Adding elements. Use the most convenient method to add an icon
        welcome.append_with_pixbuf (pixbuf, "Create", "Write a new document.");
        welcome.append_with_image (image, "Open", "select a file.");
        welcome.append ("document-save", "Save", "With a much longer description.");

        return welcome;
    }

    private void show_light_window () {
        var light_window = new Granite.Widgets.LightWindow ();
        
        var light_window_notebook = new Granite.Widgets.StaticNotebook ();
        var entry = new Gtk.Entry ();
        var open_drop = new Gtk.ComboBoxText ();
        var open_lbl = new LLabel ("Alwas Open Mpeg Video Files with Audience");
        
        var grid = new Gtk.Grid ();
        grid.attach (new Gtk.Image.from_icon_name ("video-x-generic", Gtk.IconSize.DIALOG), 0, 0, 1, 2);
        grid.attach (entry, 1, 0, 1, 1);
        grid.attach (new LLabel ("1.13 GB, Mpeg Video File"), 1, 1, 1, 1);
        
        grid.attach (light_window_notebook, 0, 2, 2, 1);
        
        var general = new Gtk.Grid ();
        general.attach (new LLabel.markup ("<b>Info:</b>"), 0, 0, 2, 1);
        
        general.attach (new LLabel.right ("Created:"), 0, 1, 1, 1);
        general.attach (new LLabel.right ("Modified:"), 0, 2, 1, 1);
        general.attach (new LLabel.right ("Opened:"), 0, 3, 1, 1);
        general.attach (new LLabel.right ("Mimetype:"), 0, 4, 1, 1);
        general.attach (new LLabel.right ("Location:"), 0, 5, 1, 1);
        
        general.attach (new LLabel ("Today at 9:50 PM"), 1, 1, 1, 1);
        general.attach (new LLabel ("Today at 9:50 PM"), 1, 2, 1, 1);
        general.attach (new LLabel ("Today at 10:00 PM"), 1, 3, 1, 1);
        general.attach (new LLabel ("video/mpeg"), 1, 4, 1, 1);
        general.attach (new LLabel ("/home/daniel/Downloads"), 1, 5, 1, 1);
        
        general.attach (new LLabel.markup ("<b>Open with:</b>"), 0, 6, 2, 1);
        general.attach (open_drop, 0, 7, 2, 1);
        general.attach (open_lbl, 0, 8, 2, 1);
        
        light_window_notebook.append_page (general, new Gtk.Label ("General"));
        light_window_notebook.append_page (new Gtk.Label ("More"), new Gtk.Label ("More"));
        light_window_notebook.append_page (new Gtk.Label ("Sharing"), new Gtk.Label ("Sharing"));
        
        open_lbl.margin_left = 24;
        open_drop.margin_left = 12;
        open_drop.append ("audience", "Audience");
        open_drop.active = 0;
        grid.margin = 12;
        grid.margin_top = 24;
        grid.margin_bottom = 24;
        entry.text = "Cool Hand Luke";
        general.column_spacing = 6;
        general.row_spacing = 6;
        
        light_window.add (grid);
        light_window.show_all ();
    }

    private Granite.Widgets.DynamicNotebook create_dynamic_notebook () {
        var dynamic_notebook = new Granite.Widgets.DynamicNotebook ();

        var tab = new Granite.Widgets.Tab ("user@elementaryos: ~",
                                            new ThemedIcon ("empty"),
                                            new Gtk.Label ("Page 1"));
        tab.working = true;

        dynamic_notebook.insert_tab (tab, -1);
        dynamic_notebook.insert_tab (new Granite.Widgets.Tab ("user2@elementaryos: ~",
                                                              new ThemedIcon ("empty"),
                                                              new Gtk.Label ("Page 2")), -1);

        dynamic_notebook.tab_added.connect ((t) => {
        	t.page = new Gtk.Label ("newuser@elementaryos: ~");
        	t.label = "newuser@elementaryos: ~";
    	});

    	dynamic_notebook.tab_moved.connect ((t, p) => {
    	    print ("Moved tab %s to %i\n", t.label, p);
    	});

    	dynamic_notebook.tab_switched.connect ((old_t, new_t) => {
    	    print ("Switched from %s to %s\n", old_t.label, new_t.label);
	    });

		dynamic_notebook.tab_removed.connect ((t) => {
		    print ("Going to remove %s\n", t.label);
		    return true;
	    });

        return dynamic_notebook;
    }

    public static int main (string[] args) {
        new Granite.Demo ().run (args);
        Gtk.main ();
        return 0;
    }
}
