#include <math.h>
#include <stdio.h>
#include <string.h>
#include "externs.h"
#include "fractal_types.h"
#include "palette.h"

#define BUF_SIZE 256

/* keep the current palette's filename around */
static char _filename[BUF_SIZE];

gboolean palette_load(char* filename)
{
    int i,r,g,b;
    char buf[BUF_SIZE];
    
    FILE* fp = fopen(filename, "r");
    if (fp == NULL)
        return FALSE;

    i = 0;
    while (fgets(buf, BUF_SIZE, fp) != NULL) {
        if (i >= 256)
            break;
        if (sscanf(buf, " %d %d %d", &r, &g,&b) != 3)
            break;
        palette[i] = RGB(r,g,b);
        i++;
    }

    fclose(fp);
    
    if(i == 0)
        return FALSE;

    pal_indexes = i;
    strncpy(_filename, filename, BUF_SIZE);
    _filename[BUF_SIZE-1] = '\0';

    return TRUE;
}

char* palette_get_filename(void)
{
    return &_filename[0];
}

void palette_apply(image_info* img, int x0, int y0, int width, int height)
{
    int x, y;
    for (y = y0; y < y0 + height; y++) {
        for (x = x0; x < x0 + width; x++) {
            img->rgb_data[y * width + x] = get_pixel(img, x, y);
        }
    }
}

void palette_randomize(random_palette* rnd_pal)
{
    int i;
    int r_bnd, g_bnd, b_bnd;
    int r_bandcount, g_bandcount, b_bandcount;
    int ar, ab, ag, rr, rg, rb;
    guint32 r, g, b;
    gdouble r_hs, g_hs, b_hs;
    gdouble r_hsm, g_hsm, b_hsm;
    int* rnd_r;
    int* rnd_g;
    int* rnd_b;
    double r_bcsize, g_bcsize, b_bcsize;
    double r_band, g_band, b_band;
    double r_bdif, g_bdif, b_bdif;
    double r_difb, g_difb, b_difb;

    if (rnd_pal == 0)
        return;
    if (rnd_pal->r_strength == 0)
        rnd_pal->r_strength = 0.01;
    r_hs = 1 - rnd_pal->r_strength;
    r_hsm = 128 * rnd_pal->r_strength;
    if (rnd_pal->g_strength == 0)
        rnd_pal->g_strength = 0.01;
    g_hs = 1 - rnd_pal->g_strength;
    g_hsm = 128 * rnd_pal->g_strength;
    if (rnd_pal->b_strength == 0)
        rnd_pal->b_strength = 0.01;
    b_hs = 1 - rnd_pal->b_strength;
    b_hsm = 128 * rnd_pal->b_strength;

    r_bandcount = pal_indexes * rnd_pal->r_bands;
    if (r_bandcount < 1)
        r_bandcount = 1;
    g_bandcount = pal_indexes * rnd_pal->g_bands;
    if (g_bandcount < 1)
        g_bandcount = 1;
    b_bandcount = pal_indexes * rnd_pal->b_bands;
    if (b_bandcount < 1)
        b_bandcount = 1;

    rnd_r = g_malloc(sizeof(int) * (r_bandcount + 1));
    rnd_g = g_malloc(sizeof(int) * (g_bandcount + 1));
    rnd_b = g_malloc(sizeof(int) * (b_bandcount + 1));
    for (i = 0; i < r_bandcount; i++)
        rnd_r[i] = r_hsm - 255 * 
            ((double)rand() / RAND_MAX) * rnd_pal->r_strength;
    for (i = 0; i < g_bandcount; i++)
        rnd_g[i] = g_hsm - 255 * 
            ((double)rand() / RAND_MAX) * rnd_pal->g_strength;
    for (i = 0; i < b_bandcount; i++)
        rnd_b[i] = b_hsm - 255 *
            ((double)rand() / RAND_MAX) * rnd_pal->b_strength;
    rnd_r[r_bandcount] = rnd_r[0];
    rnd_g[g_bandcount] = rnd_g[0];
    rnd_b[b_bandcount] = rnd_b[0];

    r_bcsize = (double)r_bandcount / pal_indexes;
    g_bcsize = (double)g_bandcount / pal_indexes;
    b_bcsize = (double)b_bandcount / pal_indexes;
    r_band = g_band = b_band = 0;
    r_bnd = g_bnd = b_bnd = 0;
    for (i=0; i < pal_indexes; i++) {
        r_bnd = r_band;
        g_bnd = g_band;
        b_bnd = b_band;
        r_bdif = r_band - r_bnd;
        g_bdif = g_band - g_bnd;
        b_bdif = b_band - b_bnd;
        r_difb = 1 - r_bdif;
        g_difb = 1 - g_bdif;
        b_difb = 1 - b_bdif;
        rr = rnd_r[r_bnd] * r_difb + rnd_r[r_bnd + 1] * r_bdif;
        rg = rnd_g[g_bnd] * g_difb + rnd_g[g_bnd + 1] * g_bdif;
        rb = rnd_b[b_bnd] * b_difb + rnd_b[b_bnd + 1] * b_bdif;
        r = ar = r_hsm + RED(palette[i]) * r_hs + rr;
        g = ag = g_hsm + GREEN(palette[i]) * g_hs + rg;
        b = ab = b_hsm + BLUE(palette[i]) * b_hs + rb;
        r_band += r_bcsize;
        g_band += g_bcsize;
        b_band += b_bcsize;
        if (ar < 0 || ar > 255 ||
            ag < 0 || ag > 255 ||
            ab < 0 || ab > 255)
        {
            printf("\nR:%d\tG:%d\tB:%d", ar,ag,ab);
        }
        if (i >= rnd_pal->offset &&
            (i + rnd_pal->offset) % rnd_pal->stripe < rnd_pal->spread)
        {
            palette[i] = RGB(r,g,b);
        }
    }
}

void palette_apply_func(function_palette* funpal)
{
    int i;
    guint32 r, g, b;
    for (i = 0; i < pal_indexes; i++) {
        r = RED(palette[i]);
        g = GREEN(palette[i]);
        b = BLUE(palette[i]);
        if (i >= funpal->offset &&
            (i + funpal->offset) % funpal->stripe < funpal->spread)
        {
            switch(funpal->func) {
                case PF_EX_RG:
                    palette[i] = RGB(g, r, b);
                    break;
                case PF_EX_GB:
                    palette[i] = RGB(r, b, g);
                    break;
                case PF_EX_BR:
                    palette[i] = RGB(b, g, r);
                    break;
                case PF_ROT_RGB:
                    palette[i] = RGB(g, b, r);
                    break;
                case PF_INV_RGB:
                    palette[i] = RGB(255 - r, 255 - g, 255 - b);
                    break;
                case PF_INV_R:
                    palette[i] = RGB(255 - r, g, b);
                    break;
                case PF_INV_G:
                    palette[i] = RGB(r, 255 - g, b);
                    break;
                case PF_INV_B:
                    palette[i] = RGB(r, g, 255 - b);
                    break;
                default:
                    break;
            }
        }
    }
}

void palette_rotate_backward(void)
{
    int i, max;
    guint32 temp;

    max = pal_indexes-1;
    temp = palette[0];
    for (i=0; i < max; i++)
        palette[i] = palette[i+1];
    palette[max] = temp;
}

void palette_rotate_forward(void)
{
    int i, max;
    guint32 temp;

    max = pal_indexes-1;
    temp = palette[max];
    for (i=max; i > 0; i--)
        palette[i] = palette[i-1];
    palette[0] = temp;
}

guint32 get_pixel(image_info* img, int x, int y)
{
    if (img->palette_ip)
      {
        double val,cval,diff,rdiff;
        int ind1,ind2;
        guint32 c1,c2;
        guint8 r,g,b;

        val = img->raw_data[y*img->real_width + x];

        /** FIXME: optimize this */

        cval = ceil(val);
        diff = cval - val;
        rdiff = 1.0 - diff;

        ind1 = ((guint32)floor(val)) % (pal_indexes-1);
        ind2 = (guint32)cval % (pal_indexes-1);

        c1 = palette[ind1];
        c2 = palette[ind2];

        r = (guint8)(diff * RED(c1) + rdiff * RED(c2));
        g = (guint8)(diff * GREEN(c1) + rdiff * GREEN(c2));
        b = (guint8)(diff * BLUE(c1) + rdiff * BLUE(c2));

        return RGB(r,g,b);
    }
    else
    {
        double c;
        int index;

        c = img->raw_data[y*img->real_width + x];
        index = (guint32)c % (pal_indexes-1);

        return palette[index];
    }
}

