#include <stdio.h>
#include "externs.h"
#include "fractal_types.h"
#include "misc.h"
#include "palette.h"

static void do_aa_pixel(image_info* img, int x, int y);

void set_image_info(image_info* img, int w, int h, int aa_factor)
{
    gboolean same_size;

    if ((img->user_width == w) && (img->user_height == h)
         && (img->rgb_data != NULL))
        same_size = TRUE;
    else
        same_size = FALSE;

    if (!same_size)
        g_free(img->rgb_data);
    g_free(img->raw_data);

    img->user_width = w;
    img->user_height = h;
    img->aa_factor = aa_factor;
    img->ratio = (double)w/h;

    if (aa_factor < 1) {
        fprintf(stderr, "aa_factor must be >= 1\n");
        gtk_main_quit();
    }

    img->real_width = w*aa_factor;
    img->real_height = h*aa_factor;

    if (!same_size)
        img->rgb_data = g_malloc(img->user_width*img->user_height*4);
    img->raw_data = g_malloc(img->real_width * img->real_height 
                                * 4 * sizeof(double));
    clear_image(img, TRUE, !same_size);
}

void set_autolay_info(image_info* img, auto_layer* als)
{
    img->auto_settings.in_par        = als->in_par;
    img->auto_settings.out_par       = als->out_par;
    img->auto_settings.in_scale      = als->in_scale;
    img->auto_settings.out_scale     = als->out_scale;
    img->auto_settings.kunge         = als->kunge;
    img->auto_settings.bail          = als->bail;
    img->auto_settings.layer_count   = als->layer_count;
    img->auto_settings.bail_start    = als->bail_start;
    img->auto_settings.bail_end      = als->bail_end;
    img->auto_settings.pert_re_start = als->pert_re_start;
    img->auto_settings.pert_re_end   = als->pert_re_end;
    img->auto_settings.pert_im_start = als->pert_im_start;
    img->auto_settings.pert_im_end   = als->pert_im_end;
    img->auto_settings.count_min_iter= als->count_min_iter;
    img->auto_settings.count_max_iter= als->count_max_iter;
    img->auto_settings.count_min_rt  = als->count_min_rt;
    img->auto_settings.count_max_rt  = als->count_max_rt;
    img->auto_settings.bail_min_iter = als->bail_min_iter;
    img->auto_settings.bail_max_iter = als->bail_max_iter;
    img->auto_settings.bail_min_rt   = als->bail_min_rt;
    img->auto_settings.bail_max_rt   = als->bail_max_rt;
    img->auto_settings.pert_min_iter = als->pert_min_iter;
    img->auto_settings.pert_max_iter = als->pert_max_iter;
    img->auto_settings.pert_re_min_rt= als->pert_re_min_rt;
    img->auto_settings.pert_re_max_rt= als->pert_re_max_rt;
    img->auto_settings.pert_im_min_rt= als->pert_im_min_rt;
    img->auto_settings.pert_im_max_rt= als->pert_im_max_rt;
#ifdef DEBUG_PAR
    printf("\nAutosettings (misc.c)");
    printf("\n---------------------\n");

    printf("\nin_par %s = %s", draw_par_names[img->auto_settings.in_par],
                                        draw_par_names[als->in_par]);
    printf("\nout_par %s = %s",
                            draw_par_names[img->auto_settings.out_par],
                                       draw_par_names[als->out_par]);
    printf("\nin_scale %f = %f", img->auto_settings.in_scale,
                            als->in_scale);
    printf("\nout_scale %f = %f", img->auto_settings.out_scale,
                            als->out_scale);
    printf("\nkunge %s = %s", kung_type_names[img->auto_settings.kunge],
                            kung_type_names[als->kunge]);
    printf("\nbail %s = %s", bail_type_names[img->auto_settings.bail],
                            bail_type_names[als->bail]);
    printf("\nlayers %d = %d", img->auto_settings.layer_count,
                            als->layer_count);
    printf("\nbail start %f = %f", img->auto_settings.bail_start,
                            als->bail_start);
    printf("\nbail end %f = %f", img->auto_settings.bail_end,
                            als->bail_end);
    printf("\npert_re_start %f = %f", img->auto_settings.pert_re_start,
                            als->pert_re_start);
    printf("\npert re end %f = %f", img->auto_settings.pert_re_end,
                            als->pert_re_end);
    printf("\npert_im_start %f = %f", img->auto_settings.pert_im_start,
                            als->pert_im_start);
    printf("\npert im_end %f = %f", img->auto_settings.pert_im_end,
                            als->pert_im_end);
    printf("\ncount min iter %d = %d", img->auto_settings.count_min_iter,
                            als->count_min_iter);
    printf("\ncount max iter %d = %d", img->auto_settings.count_max_iter,
                            als->count_max_iter);
    printf("\ncount min rt %f = %f", img->auto_settings.count_min_rt,
                            als->count_min_rt);
    printf("\ncount max rt %f = %f", img->auto_settings.count_max_rt,
                            als->count_max_rt);
    printf("\nbail min iter %d = %d",img->auto_settings.bail_min_iter,
                            als->bail_min_iter);
    printf("\nbail min iter %d = %d",img->auto_settings.bail_max_iter,
                            als->bail_max_iter);
    printf("\nbail min rt %f = %f", img->auto_settings.bail_min_rt,
                            als->bail_min_rt);
    printf("\nbail max rt %f = %f", img->auto_settings.bail_max_rt,
                            als->bail_max_rt);
    printf("\npert min iter %d = %d",img->auto_settings.pert_min_iter,
                            als->pert_min_iter);
    printf("\npert min iter %d = %d",img->auto_settings.pert_max_iter,
                            als->pert_max_iter);
    printf("\npert re min rt %f = %f", img->auto_settings.pert_re_min_rt,
                            als->pert_re_min_rt);
    printf("\npert re max rt %f = %f", img->auto_settings.pert_re_max_rt,
                            als->pert_re_max_rt);
    printf("\npert im min rt %f = %f", img->auto_settings.pert_im_min_rt,
                            als->pert_im_min_rt);
    printf("\npert im max rt %f = %f", img->auto_settings.pert_im_max_rt,
                            als->pert_im_max_rt);
    printf("\nSome stupid message irrelevant to misc.c goes here...");
#endif
}

void set_fracset_info(image_info* img, fractal_settings* frs)
{
    img->frac_settings.in_par    = frs->in_par;
    img->frac_settings.out_par   = frs->out_par;
    img->frac_settings.in_scale  = frs->in_scale;
    img->frac_settings.out_scale = frs->out_scale;
    img->frac_settings.kunge     = frs->kunge;
    img->frac_settings.bail      = frs->bail;
    img->frac_settings.bail_value= frs->bail_value;
    img->frac_settings.re_pert   = frs->re_pert;
    img->frac_settings.im_pert   = frs->im_pert;
}

void set_userpar_info(image_info* img, user_params* ups)
{
    int i, j;
    dr_par* p_par = 0;
    dr_op*  p_ope = 0;
    dr_par* p_dpar= 0;
    dr_op*  p_dope= 0;

    for (i = 0; i < MAX_PAR_SETS; i++) {
        p_par = img->user_settings.params[i];
        p_ope = img->user_settings.operas[i];
        p_dpar= ups->params[i];
        p_dope= ups->operas[i];
        #ifdef DEBUG_PAR
        printf("\n\tparam set [%d]\n\t", i);
        #endif
        for (j = 0; j < USER_MAX_PARS; j++) {
            p_par[j] = p_dpar[j];
            #ifdef DEBUG_PAR
            if (j % 3 == 0)
                printf("\n");
            printf("\tparam[%d] = ", j);
            printf("%s", draw_par_names[p_par[j]]);
            #endif
            if (j < USER_MAX_OPS) {
                p_ope[j] = p_dope[j];
                #ifdef DEBUG_PAR
                printf("\t%s", draw_op_names[p_ope[j]]);
                #endif
            }
        }
        img->user_settings.params_count[i]   = ups->params_count[i];
        img->user_settings.param_used_frs[i] = ups->param_used_frs[i];
        img->user_settings.param_used_als[i] = ups->param_used_als[i];
        #ifdef DEBUG_PAR
        printf("\n\t\tparam count     %d", 
            img->user_settings.params_count[i]);

        printf("\n\t\tparam used by fractal\t");
        if (ups->param_used_frs[i] == TRUE)
            printf("Y"); else printf("N");

        printf("\t\tparam used by auto layer\t");
        if (ups->param_used_als[i] == TRUE)
            printf("Y"); else printf("N");

        #endif
    }
    for (i = 0; i < MAX_NUMS; i++)
        img->user_settings.numbers[i] = ups->numbers[i];
}

void clear_image(image_info* img, gboolean raw, gboolean rgb)
{
    int i;

    if (raw) {
        for (i=0; i < img->real_width*img->real_height; i++)
            img->raw_data[i] = UINT_MAX;
    }
    if (rgb) {
        for (i=0; i < img->user_width*img->user_height; i++)
            img->rgb_data[i] = palette[0];
    }
}

void do_aa_pixel(image_info* img, int x, int y)
{
    int yoff,xoff;
    int xi,yi;
    guint32 r,g,b,c;

    r = g = b = 0;

    yoff = y*img->aa_factor;
    xoff = x*img->aa_factor;

    for (yi=yoff; yi < yoff+img->aa_factor; yi++) {
        for (xi=xoff; xi < xoff+img->aa_factor; xi++) {
            c = get_pixel(img, xi,yi);
            r += RED(c);
            g += GREEN(c);
            b += BLUE(c);
        }
    }

    xi = img->aa_factor*img->aa_factor;

    r /= xi;
    g /= xi;
    b /= xi;

    img->rgb_data[y*img->user_width + x] = RGB(r,g,b);
}

void do_anti_aliasing(image_info* img, int x0, int y0, int width,
                      int height)
{
    int x,y;

    for (y=y0; y < y0+height; y++) {
        for (x=x0; x < x0 + width; x++) {
            do_aa_pixel(img, x, y);
        }
    }
}

/*
void rgb_invert(image_info* img)
{
    int x,y;
    guint32 c,r,g,b;

    for (y=0; y < img->user_height; y++) {
        for (x=0; x < img->user_width; x++) {
            c = img->rgb_data[y*img->user_width + x];
            r = 255-RED(c);
            g = 255-GREEN(c);
            b = 255-BLUE(c);
            c = RGB(r,g,b);
            img->rgb_data[y*img->user_width + x] = c;
        }
    }
}
*/
