#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include "externs.h"
#include "fractal_types.h"
#include "palette.h"
#include "misc.h"
#include "attr_dlg.h"
#include "autolay_dlg.h"
#include "fracset_dlg.h"
#include "pal_rot_dlg.h"
#include "pal_rnd_dlg.h"
#include "pal_fun_dlg.h"
#include "user_dlg.h"
#include "my_png.h"
#include "gkIIfile.h"
#include "timer.h"
#include "version.h"
#include "../pixmaps/zoom_in.xpm"
#include "../pixmaps/zoom_out.xpm"
#include "../pixmaps/reposition.xpm"

#define ZOOM_INTERVAL 25

/* percentage of image width that zoom box starts at */
#define ZOOM_BOX_WIDTH 0.35

#define DEFAULT_WIDTH    480
#define DEFAULT_HEIGHT   360
#define DEFAULT_AAFACTOR 1

#define MIN_WINDOW_WIDTH 320

/* recalc/stop button texts */
#define TEXT_RECALC     "Recalc"
#define TEXT_STOP       "Stop"

/* julia preview size */
#define JPRE_SIZE    160
#define JPRE_AAFACTOR 2

#define DUMP_COMMAND "--read-dump-from"
#define DEFAULT_PALETTE_FILE \
    "/usr/local/share/gkII/palettes/default.map"

#define DEFAULT_PALETTE_FILE2 \
    "/usr/local/share/gfract/palettes/default.map"

typedef struct {
    int zooming;
    int julia_browsing;
    int repositioning;
    int second_point;

    /* zoom box info */
    int z_x;
    int z_y;
    int z_width;
    int z_height;

    /* repositioning info */
    int cl_x; /* center lines */
    int cl_y;
    int fp_x; /* first point */
    int fp_y;
} status_info;

/* options */
typedef struct {
    /* display amount of time needed for rendering */
    int timing;
} options;

static gint idle_callback(image_info* img);
static void start_rendering(image_info* img);
static void stop_rendering(image_info* img);
static void start_julia_browsing(void);
static void stop_julia_browsing(void);
static void start_reposition(void);
static void stop_reposition(void);
static void draw_center_lines(int draw_cl);
static void process_args(int argc, char** argv);
static void kill_zoom_timers(void);
static void zoom_resize(int arg);
static void resize_preview(void);
static int zoom_is_valid_size(void);
static void quit(void);
static void redraw_image(image_info* img);
static void create_menus(GtkWidget* vbox);
static GtkWidget* menu_add(GtkWidget* menu, char* name, 
                        void (*func)(void));

static void get_coords(long double* x, long double* y);
static GdkRectangle horiz_intersect(GdkRectangle* a1, GdkRectangle* a2);
static void my_fread(void* ptr, int size, FILE* fp);
static void my_fwrite(void* ptr, int size, FILE* fp);
static GtkWidget* create_pixmap(GtkWidget* widget, char** xpm_data);

static gint expose_event(GtkWidget* widget, GdkEventExpose* event,
                         image_info* img);

static gint button_press_event(GtkWidget* widget, GdkEventButton* event);
static gint j_pre_delete(GtkWidget *widget, GdkEvent *event,
                        gpointer data);

static gint child_reaper(gpointer nothing);

/*static void invert(void);*/
static void switch_fractal_type(void);
static void print_help(void);
static void print_version(void);
     
/* DIALOG FUNCTIONS */

/* image attribute */
static void image_attr_ok_cmd(GtkWidget* w,     image_attr_dialog* dl);
static void image_attr_apply_cmd(GtkWidget* w,  image_attr_dialog* dl);
static void do_attr_dialog(void);
static void do_reset_zoom(void);
static void toggle_zoom_new_win(GtkWidget* widget);

/* auto layer settings */
static void autolay_attr_ok_cmd(GtkWidget* w,   autolayer_dialog* ald);
static void autolay_attr_apply_cmd(GtkWidget* w,autolayer_dialog* ald);
static void do_autolay_dialog(void);

/* fractal settings */
static void fracset_attr_ok_cmd(GtkWidget* w,   fracset_dialog* dl);
static void fracset_attr_apply_cmd(GtkWidget* w,fracset_dialog* dl);
static void do_fracset_dialog(void);

/* user params */
static void userpar_attr_ok_cmd(GtkWidget* w,   user_attr_dialog* dl);
static void userpar_attr_apply_cmd(GtkWidget* w,user_attr_dialog* dl);
static void do_userpar_dialog(void);

/* image params save */
static void do_savegkIIfile_dialog(void);
static void do_opengkIIfile_dialog(void);

/* palette loading */
static void reapply_palette(void);
static void load_palette_cmd(void);
static void toggle_palette_ip(GtkWidget* widget);

/* palette cycling */
static void do_pal_rot_dialog(void);
static void do_pal_rnd_dialog(void);
static void do_pal_fun_dialog(void);

/* palette saving */
static void save_cmd(void);

/* general stuff we need to have */
static status_info stat;
static image_info img;
static image_info j_pre;
static random_palette rnd_palette;
static function_palette fun_palette;
static Timer timing_info;
static options opts;
static GtkWidget* j_pre_window = NULL;

GtkWidget* window = NULL;

static GtkWidget* drawing_area = NULL;
static GtkWidget* zoom_in_button = NULL;
static GtkWidget* zoom_out_button = NULL;
static GtkWidget* reposition_button = NULL;
static GtkWidget* recalc_button_label = NULL;
static GtkWidget* depth_spin = NULL;
static GtkWidget* pbar = NULL;
static GtkWidget* switch_menu_cmd = NULL;
static GtkWidget* method_toggle = NULL;
static GtkWidget* palette_ip = NULL;
static GtkWidget* zoom_new_win = NULL;

static int zoom_timer;
static char* program_name = NULL;

/* DIALOG POINTERS */
static image_attr_dialog*       img_attr_dlg    = NULL;
static fracset_dialog*          fracset_attr_dlg= NULL;
static autolayer_dialog*        autolay_attr_dlg= NULL;
static user_attr_dialog*        user_attr_dlg   = NULL;
static palette_rot_dialog*      pal_rot_dlg     = NULL;
static palette_rnd_dialog*      pal_rnd_dlg     = NULL;
static palette_fun_dialog*      pal_fun_dlg     = NULL;

/* The Program Stuff */
void my_fread(void* ptr, int size, FILE* fp)
{
    if (fread(ptr, size, 1, fp) != 1) {
        perror("Can't read file");
        exit(1);
    }
}

void my_fwrite(void* ptr, int size, FILE* fp)
{
    if (fwrite(ptr, size, 1, fp) != 1) {
        perror("Can't write to file");
        exit(1);
    }
}

void print_version(void)
{
    printf("%s %s\n", program_name, VERSION);
}

void print_help(void)
{
    print_version();
    printf("Command-line arguments:\n");
    printf("-h, --help        Help\n");
    printf("--version         Display version\n");
    printf("-t                Display timing information\n");
}

void process_args(int argc, char** argv)
{
    int i;
    for (i=1; i < argc; i++) {
        if ((strcmp("-h", argv[i]) == 0) ||
            (strcmp("--help", argv[i]) == 0)) {
            print_help();
            exit(0);
        } else if (strcmp("-t", argv[i]) == 0) {
            opts.timing = 1;
        } else if (strcmp("--version", argv[i]) == 0) {
            print_version();
            exit(0);
        } else if (strcmp(DUMP_COMMAND, argv[i]) == 0) {
            if ((i+1) == argc) {
                fprintf(stderr, "Filename missing from command line.\n");
                exit(1);
            } else {
                FILE* fp = fopen(argv[i+1], "r");
                if (fp == NULL) {
                    perror("Can't open file");
                    exit(1);
                }
                my_fread(&img, sizeof(image_info), fp);
                my_fread(palette, 256*4, fp);
                my_fread(&rnd_palette, sizeof(random_palette), fp);
                fclose(fp);
                
                if (remove(argv[i+1]) == -1) {
                    perror("Can't delete temp file");
                    exit(1);
                }
                img.idle_id = -1;
                img.rgb_data = NULL;
                img.raw_data = NULL;
                i++;
            }
        } else {
            fprintf(stderr, "Unknown option: %s\n", argv[i]);
            exit(1);
        }
    }
}

/*
void invert(void)
{
    palette_invert();
    if (img.aa_factor == 1)
        palette_apply(&img, 0, 0, img.user_width, img.user_height);
    else
        rgb_invert(&img);
    redraw_image(&img);
}
*/

void save_cmd(void)
{
    do_png_save(&img);
}

void switch_fractal_type(void)
{
    if (img.fr_type == MANDELBROT) {
        if (!stat.julia_browsing)
            start_julia_browsing();
    } else if (img.fr_type == JULIA) {
        img.xmin = img.old_xmin;
        img.xmax = img.old_xmax;
        img.ymax = img.old_ymax;
        img.fr_type = MANDELBROT;
        start_rendering(&img);
    }
}

static void do_savegkIIfile_dialog(void)
{
    do_gkII_save(&img);
}

static void do_opengkIIfile_dialog(void)
{
    GtkWidget* dialog =
        gtk_file_chooser_dialog_new("Open gkII File",
                        (GtkWindow*)window,
                        GTK_FILE_CHOOSER_ACTION_OPEN,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        NULL);
    if (gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT)
    {
        char *filename;
        filename = gtk_file_chooser_get_filename(
                        GTK_FILE_CHOOSER(dialog));
        if (gkII_open_file(&img, filename) == 0) {
            gtk_spin_button_set_value(GTK_SPIN_BUTTON(depth_spin),
                                    img.depth);
            gtk_spin_button_update(GTK_SPIN_BUTTON(depth_spin));
            if (img.user_width < MIN_WINDOW_WIDTH)
                gtk_widget_set_size_request(drawing_area,
                                    MIN_WINDOW_WIDTH, img.user_height);
            else gtk_widget_set_size_request(drawing_area,
                                    img.user_width, img.user_height);
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(palette_ip),
                                    img.palette_ip);
            start_rendering(&img);
            /* update any open dialogs */
            if (img_attr_dlg != NULL)
                attr_dlg_set(&img, img_attr_dlg);
            if (fracset_attr_dlg != NULL)
                fracset_dlg_set(&img.frac_settings, fracset_attr_dlg);
            if (autolay_attr_dlg != NULL)
                autolayer_dlg_set(&img.auto_settings, autolay_attr_dlg);
            if (user_attr_dlg != NULL)
                user_dlg_set(&img.user_settings, user_attr_dlg);
            if (pal_rnd_dlg != NULL)
                palette_rnd_dlg_set(&rnd_palette, pal_rnd_dlg);
        }
        g_free(filename);
    }
    gtk_widget_destroy (dialog);
}


void reapply_palette(void)
{
    if (img.aa_factor == 1)
        palette_apply(&img, 0, 0, img.user_width, img.user_height);
    else
        do_anti_aliasing(&img, 0, 0, img.user_width, img.user_height);
    redraw_image(&img);
}

void load_palette_cmd(void)
{
    GtkWidget* dialog =
        gtk_file_chooser_dialog_new("Open gkII File",
                        (GtkWindow*)window,
                        GTK_FILE_CHOOSER_ACTION_OPEN,
                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                        GTK_STOCK_APPLY, GTK_RESPONSE_APPLY,
                        NULL);
    gint dlg_resp;
    do {
        dlg_resp = gtk_dialog_run(GTK_DIALOG(dialog));
        if ( dlg_resp == GTK_RESPONSE_ACCEPT
          || dlg_resp == GTK_RESPONSE_APPLY)
        {
            char *filename;
            filename = gtk_file_chooser_get_filename(
                                    GTK_FILE_CHOOSER(dialog));
            if (palette_load(filename) == FALSE)
                fprintf(stderr, "Invalid palette file %s\n", filename);
            else {
                if (img.aa_factor == 1)
                    palette_apply(&img, 0, 0, img.user_width, img.user_height);
                else
                    do_anti_aliasing(&img, 0, 0, img.user_width, img.user_height);
                redraw_image(&img);
            }
            g_free(filename);
        }
    } while (dlg_resp == GTK_RESPONSE_APPLY);
    gtk_widget_destroy (dialog);
}

void init_misc(void)
{
    /* main window init */
    img.real_width = img.user_width = DEFAULT_WIDTH;
    img.real_height = img.user_height = DEFAULT_HEIGHT;
    img.aa_factor = DEFAULT_AAFACTOR;
    
    img.depth = 300;
    img.rgb_data = NULL;
    img.raw_data = NULL;

    img.xmin = -2.21;
    img.xmax = 1.0;
    img.ymax = 1.2;

    img.idle_id = -1;
    img.j_pre = FALSE;
    img.fr_type = MANDELBROT;
    img.palette_ip = FALSE;
    img.zoom_new_win = FALSE;
    img.method = 0;

    /* init preview */
    j_pre.depth = 300;
    j_pre.rgb_data = NULL;
    j_pre.raw_data = NULL;

    j_pre.xmin = -2.0;
    j_pre.xmax = 1.5;
    j_pre.ymax = 1.25;

    j_pre.u.julia.c_re = 0.3;
    j_pre.u.julia.c_im = 0.6;
    
    j_pre.idle_id = -1;
    j_pre.j_pre = TRUE;
    j_pre.fr_type = JULIA;
    j_pre.palette_ip = FALSE;
    j_pre.method = 0;

    /* other settings inits */

    init_auto_layer(&img.auto_settings);
    init_auto_layer(&j_pre.auto_settings);
    init_fractal_settings(&img.frac_settings);
    init_fractal_settings(&j_pre.frac_settings);
    init_user_params(&img.user_settings);
    init_user_params(&j_pre.user_settings);

    /* misc init */
    stat.zooming = FALSE;
    stat.julia_browsing = FALSE;
    stat.repositioning = FALSE;
    stat.second_point = FALSE;
    zoom_timer = -1;

    /* default values for options */
    opts.timing = 0;

    img.rnd_pal = &rnd_palette;

    fun_palette.offset = 0;
    fun_palette.stripe = 1;
    fun_palette.spread = 1;
}

/* returns the horizontal intersection part of a1 and a2. if the
   rectangles don't overlap, it returns a rectangle with a width of 0.
*/
GdkRectangle horiz_intersect(GdkRectangle* a1, GdkRectangle* a2)
{
    GdkRectangle tmp;
    int ax1, ax2, bx1, bx2, cx1, cx2;
    
    tmp.width = 0;
    tmp.y = a1->y;
    tmp.height = a1->height;

    ax1 = a1->x;
    ax2 = a1->x + a1->width - 1;

    bx1 = a2->x;
    bx2 = a2->x + a2->width - 1;

    cx1 = (ax1 > bx1) ? ax1 : bx1; /* max */
    cx2 = (ax2 < bx2) ? ax2 : bx2; /* min */

    /* no overlap */
    if (cx2 < cx1)
        return tmp;

    tmp.x = cx1;
    tmp.width = cx2-cx1+1;
        
    return tmp;
}

void redraw_image(image_info* img)
{
    gdk_draw_rgb_32_image(img->drawing_area->window,
                          img->drawing_area->style->white_gc,
                          0, 0, img->user_width, img->user_height,
                          GDK_RGB_DITHER_NONE,
                          (guchar*)img->rgb_data,
                          img->user_width*4);
}

gint do_palette_rotation(gpointer dir)
{
    gboolean forward = *((gboolean*)dir);
    if (forward)
        palette_rotate_forward();
    else
        palette_rotate_backward();
    if (img.aa_factor == 1)
        palette_apply(&img, 0, 0, img.user_width, img.user_height);
    else
        do_anti_aliasing(&img, 0, 0, img.user_width, img.user_height);
    redraw_image(&img);
    return TRUE;
}

void do_palette_randomize(random_palette* rnd_pal)
{
    palette_randomize(rnd_pal);
    if (img.aa_factor == 1)
        palette_apply(&img, 0, 0, img.user_width, img.user_height);
    else
        do_anti_aliasing(&img, 0, 0, img.user_width, img.user_height);
    redraw_image(&img);
}

void do_palette_function(function_palette* fun_pal)
{
    palette_apply_func(fun_pal);
    if (img.aa_factor == 1)
        palette_apply(&img, 0, 0, img.user_width, img.user_height);
    else
        do_anti_aliasing(&img, 0, 0, img.user_width, img.user_height);
    redraw_image(&img);
}

void image_attr_ok_cmd(GtkWidget* w, image_attr_dialog* dl)
{
    image_attr_apply_cmd(w, dl);
    gtk_widget_destroy(dl->dialog);
}

void image_attr_apply_cmd(GtkWidget* w, image_attr_dialog* dl)
{
    set_image_info(&img,
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dl->width)),
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dl->height)),
           gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dl->aa)));

    if (img.user_width < MIN_WINDOW_WIDTH) {
        /* limit minimum window width to MIN_WINDOW_WIDTH and handle
           this case in the expose event */
        gtk_widget_set_size_request(drawing_area,
                              MIN_WINDOW_WIDTH, img.user_height);
    } else
        gtk_widget_set_size_request(drawing_area,
                              img.user_width, img.user_height);

    start_rendering(&img);

    resize_preview();
    if (stat.julia_browsing) {
        gtk_widget_hide(j_pre_window);
        gtk_widget_show(j_pre_window);
        start_rendering(&j_pre);
    }
}

static void autolay_attr_ok_cmd(GtkWidget* w, autolayer_dialog* ald)
{
    autolay_attr_apply_cmd(w, ald);
    gtk_widget_destroy(ald->dialog);
}

static void autolay_attr_apply_cmd(GtkWidget* w,autolayer_dialog* ald)
{
    auto_layer als;
    gboolean update_ups = FALSE;
    gboolean popup_ups = FALSE;
    int in, out;
    /* check if a user parameter set is already in use */
    if (get_param_set_used(img.auto_settings.in_par)  >= 0 ||
        get_param_set_used(img.auto_settings.out_par) >= 0)
        update_ups = TRUE; /* yes */
    /* do the regular autolayer settings stuff */
    get_autolayer_data(&als, ald);
    set_autolay_info(&img, &als);
    set_autolay_info(&j_pre, &als);
    /* back to trying to figure out user parameters */
    in = get_param_set_used(als.in_par);
    out = get_param_set_used(als.out_par);
    /* update_als_used regardless, not worth hassle of logic */
    update_als_used(&img.auto_settings, &img.user_settings);
    if (user_attr_dlg == NULL) {
        popup_ups = FALSE;
        if (in != -1)
            if (img.user_settings.params[in][0]  == PAR_OFF)
                popup_ups = TRUE;
        if (out != -1)
            if (img.user_settings.params[out][0] == PAR_OFF)
                popup_ups = TRUE;
        if (popup_ups == TRUE)
            do_userpar_dialog();
    }
    else if (update_ups == TRUE || in != -1 || out != -1)
        user_dlg_update(user_attr_dlg);
    /* finish off */
    start_rendering(&img);
    if (stat.julia_browsing) {
        gtk_widget_hide(j_pre_window);
        gtk_widget_show(j_pre_window);
        start_rendering(&j_pre);
    }
}

static void fracset_attr_ok_cmd(GtkWidget* w, fracset_dialog* dl)
{
    fracset_attr_apply_cmd(w, dl);
    gtk_widget_destroy(dl->dialog);
}

static void fracset_attr_apply_cmd(GtkWidget* w,fracset_dialog* dl)
{
    fractal_settings frs;
    gboolean update_ups = FALSE;
    gboolean popup_ups = FALSE;
    int in, out;
    /* check if a user parameter set is already in use */
    if (get_param_set_used(img.frac_settings.in_par)  >= 0 ||
        get_param_set_used(img.frac_settings.out_par) >= 0)
        update_ups = TRUE; /* yes */

    /* do the regular fractal settings stuff */
    get_fractal_settings(&frs, dl);
    set_fracset_info(&img, &frs);
    set_fracset_info(&j_pre, &frs);

    /* back to trying to figure out user parameters */
    in = get_param_set_used(frs.in_par);
    out = get_param_set_used(frs.out_par);
    /* update_frs_used regardless, not worth figuring logic */
    update_frs_used(&img.frac_settings, &img.user_settings);
    if (user_attr_dlg == NULL) {
        popup_ups = FALSE;
        if (in != -1)
            if (img.user_settings.params[in][0]  == PAR_OFF)
                popup_ups = TRUE;
        if (out != -1)
            if (img.user_settings.params[out][0] == PAR_OFF)
                popup_ups = TRUE;
        if (popup_ups == TRUE)
            do_userpar_dialog();
    }
    else if (update_ups == TRUE || in != -1 || out != -1)
        user_dlg_update(user_attr_dlg);
    /* finish off */
    start_rendering(&img);
    if (stat.julia_browsing) {
        gtk_widget_hide(j_pre_window);
        gtk_widget_show(j_pre_window);
        start_rendering(&j_pre);
    }
}

static void userpar_attr_ok_cmd(GtkWidget* w, user_attr_dialog* dl)
{
    userpar_attr_apply_cmd(w, dl);
    gtk_widget_destroy(dl->dialog);
}

static void userpar_attr_apply_cmd(GtkWidget* w,user_attr_dialog* dl)
{
    user_params ups;
    get_user_params(&ups, dl);
    set_userpar_info(&img, &ups);
    set_userpar_info(&j_pre, &ups);
    start_rendering(&img);
    if (stat.julia_browsing) {
        gtk_widget_hide(j_pre_window);
        gtk_widget_show(j_pre_window);
        start_rendering(&j_pre);
    }
}

/* change preview window to reflect possible new aspect ratio */
void resize_preview(void)
{
    int xw,yw;
    
    if (JPRE_SIZE/img.ratio < JPRE_SIZE) {
        xw = JPRE_SIZE;
        yw = JPRE_SIZE/img.ratio;
        if (yw == 0)
            yw = 1;
    }
    else {
        xw = JPRE_SIZE*img.ratio;
        if (xw == 0)
            xw = 1;
        yw = JPRE_SIZE;
    }
    
    set_image_info(&j_pre, xw, yw, JPRE_AAFACTOR);
    gtk_widget_set_size_request(j_pre.drawing_area, xw, yw);
}


void do_attr_dialog(void)
{
    if (img_attr_dlg)
        return;

    attr_dlg_new(&img_attr_dlg, &img);
    g_signal_connect(GTK_OBJECT(img_attr_dlg->ok_button), "clicked",
                       G_CALLBACK(image_attr_ok_cmd),
                       img_attr_dlg);
    g_signal_connect(GTK_OBJECT(img_attr_dlg->apply_button), "clicked",
                       G_CALLBACK(image_attr_apply_cmd),
                       img_attr_dlg);
}

static void do_reset_zoom(void)
{
    img.depth = 300;
    img.xmin = -2.21;
    img.xmax = 1.0;
    img.ymax = 1.2;

    j_pre.depth = 300;
    j_pre.xmin = -2.0;
    j_pre.xmax = 1.5;
    j_pre.ymax = 1.25;
    start_rendering(&img);
    if (stat.julia_browsing) {
        gtk_widget_hide(j_pre_window);
        gtk_widget_show(j_pre_window);
        start_rendering(&j_pre);
    }
}

static void do_autolay_dialog(void)
{
    if (autolay_attr_dlg) return;
    autolayer_dlg_new(&autolay_attr_dlg, &img);
    g_signal_connect(GTK_OBJECT(autolay_attr_dlg->ok_button), "clicked",
                       G_CALLBACK(autolay_attr_ok_cmd),
                       autolay_attr_dlg);
    g_signal_connect(GTK_OBJECT(autolay_attr_dlg->apply_button),
                     "clicked", G_CALLBACK(autolay_attr_apply_cmd),
                       autolay_attr_dlg);
}

static void do_fracset_dialog(void)
{
    if (fracset_attr_dlg) return;
    fracset_dlg_new(&fracset_attr_dlg, &img);
    g_signal_connect(GTK_OBJECT(fracset_attr_dlg->ok_button), "clicked",
                       G_CALLBACK(fracset_attr_ok_cmd),
                       fracset_attr_dlg);
    g_signal_connect(GTK_OBJECT(fracset_attr_dlg->apply_button),
                     "clicked", G_CALLBACK(fracset_attr_apply_cmd),
                       fracset_attr_dlg);
}

static void do_userpar_dialog(void)
{
    if (user_attr_dlg) return;
    user_dlg_new(&user_attr_dlg, &img);
    g_signal_connect(GTK_OBJECT(user_attr_dlg->ok_button), "clicked",
                       G_CALLBACK(userpar_attr_ok_cmd),
                       user_attr_dlg);
    g_signal_connect(GTK_OBJECT(user_attr_dlg->apply_button),
                     "clicked", G_CALLBACK(userpar_attr_apply_cmd),
                       user_attr_dlg);
}

void do_pal_rot_dialog(void)
{
    if (pal_rot_dlg)
        return;
    palette_rot_dlg_new(&pal_rot_dlg, &img);
}

void do_pal_rnd_dialog(void)
{
    if (pal_rnd_dlg)
        return;
    palette_rnd_dlg_new(&pal_rnd_dlg, &img);
}

void do_pal_fun_dialog(void)
{
    if (pal_fun_dlg)
        return;
    palette_fun_dlg_new(&pal_fun_dlg, &fun_palette);
}

gint child_reaper(gpointer nothing)
{
    /* wait until all dead child processes are cleaned up */
    while (waitpid(-1, NULL, WNOHANG) > 0)
        ;
    
    return TRUE;
}

void duplicate(void)
{
    char fname[] = "/tmp/gkIIXXXXXX";
    int fd;
    FILE* fp;
    pid_t result;
    
    fd = mkstemp(fname);
    if (fd == -1) {
        perror("Can't create temp file");
        exit(1);
    }
    
    fp = fdopen(fd, "w+");
    if (fp == NULL) {
        perror("Can't create temp file");
        exit(1);
    }

    my_fwrite(&img, sizeof(image_info), fp);
    my_fwrite(palette, 256*4, fp);
    my_fwrite(&rnd_palette, sizeof(random_palette), fp);

    if (fclose(fp) != 0) {
        perror("Error writing temp file");
        exit(1);
    }

    result = fork();
    
    if (result == 0) {
        close(ConnectionNumber(gdk_display));
        execl(program_name, program_name, DUMP_COMMAND,
              fname, NULL);
        perror("Error while exec'ing program");
        exit(1);
    }
    else if (result < 0) {
        perror("Could not fork");
        exit(1);
    }
}

void get_coords(long double* x, long double* y)
{
    if (y != NULL)
        *y = img.ymax - ((img.xmax - img.xmin)
            / (long double)img.user_width)
            * (*y);
    if (x != NULL)
        *x = ((*x)/(long double)img.user_width) *
            (img.xmax - img.xmin) + img.xmin;
}

GtkWidget* menu_add(GtkWidget* menu, char* name, void (*func)(void))
{
    GtkWidget* item;

    if (name != NULL) {
        item = gtk_menu_item_new_with_label(name);
        g_signal_connect_object(GTK_OBJECT(item), "activate",
                        G_CALLBACK(func), NULL, G_CONNECT_SWAPPED);
    } else
        /* just add a separator line to the menu */
        item = gtk_menu_item_new();

    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    gtk_widget_show(item);

    return item;
}

void menu_bar_add(GtkWidget* menubar, GtkWidget* submenu, char* name)
{
    GtkWidget* temp = gtk_menu_item_new_with_label(name);
    gtk_widget_show(temp);
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(temp), submenu);
    gtk_menu_shell_append(GTK_MENU_SHELL(menubar), temp);
}

void create_menus(GtkWidget* vbox)
{
    GtkWidget* menu_bar;
    GtkWidget* menu;

    menu_bar = gtk_menu_bar_new();
    gtk_box_pack_start(GTK_BOX(vbox), menu_bar, FALSE, FALSE, 0);
    gtk_widget_show(menu_bar);

    menu = gtk_menu_new();
    menu_add(menu, "Save as PNG", save_cmd);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Open gkII file...", do_opengkIIfile_dialog);
    menu_add(menu, "Save as gkII file...", do_savegkIIfile_dialog);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Duplicate", duplicate);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Exit", quit);
    menu_bar_add(menu_bar, menu, "File");
    
    menu = gtk_menu_new();
    menu_add(menu, "Attributes...", do_attr_dialog);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Fractal...",    do_fracset_dialog);
    menu_add(menu, "Auto Layer...", do_autolay_dialog);
    menu_add(menu, "User Parameters...", do_userpar_dialog);
    menu_add(menu, NULL, NULL);
    switch_menu_cmd = menu_add(menu, "Switch fractal type",
                             switch_fractal_type);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Reset Zoom", do_reset_zoom);

    zoom_new_win =
        gtk_check_menu_item_new_with_label("Zoom in new window");
    gtk_check_menu_item_set_active(
        GTK_CHECK_MENU_ITEM(zoom_new_win), img.zoom_new_win);
    g_signal_connect(GTK_OBJECT(zoom_new_win), "toggled",
        G_CALLBACK(toggle_zoom_new_win), NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), zoom_new_win);
    gtk_widget_show(zoom_new_win);

    menu_bar_add(menu_bar, menu, "Image");

    menu = gtk_menu_new();
    palette_ip = gtk_check_menu_item_new_with_label("Interpolate");
    gtk_check_menu_item_set_active(
        GTK_CHECK_MENU_ITEM(palette_ip), img.palette_ip);
    g_signal_connect(GTK_OBJECT(palette_ip), "toggled",
        G_CALLBACK(toggle_palette_ip), NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), palette_ip);
    gtk_widget_show(palette_ip);

    menu_add(menu, NULL, NULL);
    menu_add(menu, "Load", load_palette_cmd);
    menu_add(menu, NULL, NULL);
/*    menu_add(menu, "Invert", invert);*/
    menu_add(menu, "Cycle...", do_pal_rot_dialog);
    menu_add(menu, NULL, NULL);
    menu_add(menu, "Randomize", do_pal_rnd_dialog);
    menu_add(menu, "Functions", do_pal_fun_dialog);
    menu_bar_add(menu_bar, menu, "Palette");

    menu = gtk_hseparator_new();
    gtk_widget_show(menu);
    gtk_box_pack_start(GTK_BOX(vbox), menu, FALSE, FALSE, 0);
}

GtkWidget* create_pixmap(GtkWidget* widget, char** xpm_data)
{
    GdkPixmap* pixmap;
    GdkBitmap* mask;
    GtkWidget* pwid;

    pixmap = gdk_pixmap_create_from_xpm_d(widget->window, &mask,
             &(gtk_widget_get_style(widget)->bg[GTK_STATE_NORMAL]),
             (gchar**)xpm_data);
    g_assert(pixmap != NULL);
    pwid = gtk_image_new_from_pixmap(pixmap, mask);
    gtk_widget_show(pwid);

    return pwid;
}

void start_rendering(image_info* img)
{
    img->lines_done = 0;
    stop_rendering(img);

    if (!img->j_pre) {
        gtk_spin_button_update(GTK_SPIN_BUTTON(depth_spin));
        img->depth =
            gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(depth_spin));

        gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(pbar), 
                               (gfloat)1.0);/*img->real_height);*/
        gtk_widget_show(pbar);
        gtk_label_set_text(GTK_LABEL(recalc_button_label), TEXT_STOP);
        timer_start(&timing_info);
    }
    img->idle_id = g_idle_add((GtkFunction)idle_callback, img);
}

void stop_rendering(image_info* img)
{
    if (img->idle_id != -1) {
        timer_stop(&timing_info);
        if (opts.timing && (img->lines_done == img->real_height))
            printf("Image rendering took %.3f seconds.\n",
                   timer_get_elapsed(&timing_info) / (double)1e6);
        g_source_remove(img->idle_id);
        img->idle_id = -1;
        if (!img->j_pre) {
            gtk_widget_hide(pbar);
            gtk_label_set_text(
                GTK_LABEL(recalc_button_label), TEXT_RECALC);
        }
    }
}

void start_julia_browsing(void)
{
    stat.julia_browsing = TRUE;
    gtk_widget_show(j_pre_window);
    gtk_widget_set_sensitive(switch_menu_cmd, FALSE);
    gtk_widget_set_sensitive(zoom_in_button, FALSE);
    gtk_widget_set_sensitive(zoom_out_button, FALSE);
    gtk_widget_set_sensitive(reposition_button, FALSE);
}

void stop_julia_browsing(void)
{
    stop_rendering(&j_pre);
    stat.julia_browsing = FALSE;
    gtk_widget_hide(j_pre_window);
    gtk_widget_set_sensitive(switch_menu_cmd, TRUE);
    gtk_widget_set_sensitive(zoom_in_button, TRUE);
    gtk_widget_set_sensitive(zoom_out_button, TRUE);
    gtk_widget_set_sensitive(reposition_button, TRUE);
}

gint j_pre_delete(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    stop_julia_browsing();
    return TRUE;
}

void draw_center_lines(int draw_cl)
{
    int x = 0, y = 0;
    GdkLineStyle linestyle = GDK_LINE_ON_OFF_DASH;
    if (!draw_cl) {
        x = stat.fp_x;
        y = stat.fp_y;
    } else {
        x = stat.cl_x;
        y = stat.cl_y;
        if (stat.second_point) {
            linestyle = GDK_LINE_SOLID;
        }
    }
    gdk_gc_set_function(drawing_area->style->white_gc, GDK_XOR);
    gdk_gc_set_line_attributes(drawing_area->style->white_gc,
            1, linestyle, GDK_CAP_BUTT, GDK_JOIN_MITER);
    gdk_draw_line(drawing_area->window, drawing_area->style->white_gc,
            0, y, img.user_width, y);
    gdk_draw_line(drawing_area->window, drawing_area->style->white_gc,
            x, 0, x, img.user_height);
    gdk_gc_set_function(drawing_area->style->white_gc, GDK_COPY);
}

void start_reposition(void)
{
    stat.repositioning = TRUE;
    stat.second_point = FALSE;
    gtk_widget_set_sensitive(switch_menu_cmd, FALSE);
    gtk_widget_set_sensitive(zoom_in_button, FALSE);
    gtk_widget_set_sensitive(zoom_out_button, FALSE);
    draw_center_lines(TRUE);
}

void stop_reposition(void)
{
    stat.repositioning = FALSE;
    stat.second_point = FALSE;
    gtk_widget_set_sensitive(switch_menu_cmd, TRUE);
    gtk_widget_set_sensitive(zoom_in_button, TRUE);
    gtk_widget_set_sensitive(zoom_out_button, TRUE);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(reposition_button),
                                FALSE);
}

void draw_zoom_box(void)
{
    gdk_gc_set_function(drawing_area->style->white_gc, GDK_XOR);

    gdk_draw_rectangle(
        drawing_area->window, drawing_area->style->white_gc,
        FALSE, stat.z_x, stat.z_y, stat.z_width, stat.z_height);
    
    gdk_gc_set_function(drawing_area->style->white_gc, GDK_COPY);
}

void start_zooming(void)
{
    stat.z_x = 0;
    stat.z_y = 0;
    stat.z_width = ZOOM_BOX_WIDTH * img.user_width;
    stat.z_height = stat.z_width/img.ratio;
    stat.zooming = TRUE;
    gtk_widget_set_sensitive(switch_menu_cmd, FALSE);
    gtk_widget_set_sensitive(reposition_button, FALSE);
    gdk_gc_set_line_attributes(drawing_area->style->white_gc,
            1, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);
    draw_zoom_box();
}

void stop_zooming(void)
{
    stat.zooming = FALSE;
    draw_zoom_box();
    kill_zoom_timers();
    gtk_widget_set_sensitive(switch_menu_cmd, TRUE);
    gtk_widget_set_sensitive(reposition_button, TRUE);
}

void zoom_in(void)
{
    long double xmin,xmax,ymax;

    stat.zooming = FALSE;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(zoom_in_button),
                                FALSE);

    ymax = (long double) stat.z_y;
    xmin = (long double) stat.z_x;
    xmax = (long double) (stat.z_x + stat.z_width - 1);

    get_coords(&xmin, &ymax);
    get_coords(&xmax, NULL);

    if (img.zoom_new_win == TRUE) {
        long double ixmin = img.xmin;
        long double ixmax = img.xmax;
        long double iymax = img.ymax;
        img.ymax = ymax;
        img.xmin = xmin;
        img.xmax = xmax;
        duplicate();
        img.ymax = iymax;
        img.xmin = ixmin;
        img.xmax = ixmax;
    }
    else {
        img.ymax = ymax;
        img.xmin = xmin;
        img.xmax = xmax;
        start_rendering(&img);
    }
}

void zoom_resize(int arg)
{
    stat.z_width += arg * 4;
    stat.z_height = stat.z_width / img.ratio;
}

int zoom_is_valid_size(void)
{
    if ((stat.z_width < 4) || (stat.z_width > (img.user_width - 16))
       || (stat.z_height < 4) || (stat.z_height > (img.user_height - 16)))
        return FALSE;
    else
        return TRUE;
}

gint zoom_callback(int arg)
{
    draw_zoom_box();

    zoom_resize(arg);
    if (!zoom_is_valid_size())
        zoom_resize(-1*arg);

    draw_zoom_box();

    return TRUE;
}

void zoom_in_func(GtkWidget* widget)
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
        start_zooming();
    } else {
        stop_zooming();
    }
}

void reposition_func(GtkWidget* widget)
{
    if (GTK_TOGGLE_BUTTON(widget)->active) {
        start_reposition();
    } else {
        draw_center_lines(TRUE);
        if (stat.second_point)
            draw_center_lines(FALSE);
        stop_reposition();
    }
}


void zoom_out_func(GtkWidget* widget)
{
    long double ymin,half_w,half_h;
        
    ymin = img.ymax - ((img.xmax - img.xmin)/(long double)img.real_width)
        * (long double)(img.real_height-1);
    half_w = (img.xmax-img.xmin)/2.0;
    half_h = (img.ymax-ymin)/2.0;
    img.ymax += half_h;
    img.xmin -= half_w;
    img.xmax += half_w;
    start_rendering(&img);
}

void method_func(GtkWidget* widget, image_info* img)
{
    auto_layer*         als = &img->auto_settings;
    fractal_settings*   frs = &img->frac_settings;
    img->method = gtk_toggle_button_get_active(
                                    GTK_TOGGLE_BUTTON(widget));
    if (frs->bail != BT_MAND || als->bail != BT_MAND) {
        if (frs->bail == BT_MAND && (frs->out_par != PAR_OFF 
                                    || frs->in_par != PAR_OFF)) {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), 0);
        }
        else if (als->bail == BT_MAND && (als->out_par != PAR_OFF 
                                        || als->in_par != PAR_OFF)) {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), 0);
        }
        else if (als->out_par == PAR_OFF && als->in_par == PAR_OFF
                && frs->out_par == PAR_OFF && frs->in_par == PAR_OFF) {
            gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), 0);
        }
        else start_rendering(img);
    }
    else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(widget), 0);
}

void recalc_button(GtkWidget* widget)
{
    if (img.idle_id == -1)
        start_rendering(&img);
    else
        stop_rendering(&img);
}

void toggle_palette_ip(GtkWidget* widget)
{
    img.palette_ip = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(palette_ip));
    reapply_palette();
}

void toggle_zoom_new_win(GtkWidget* widget)
{
    img.zoom_new_win = gtk_check_menu_item_get_active(
        GTK_CHECK_MENU_ITEM(zoom_new_win));
}

gint button_press_event(GtkWidget* widget, GdkEventButton* event)
{
    long double cx = 0, cy = 0, cxd = 0, cyd = 0;
    long double fpx = 0, fpy = 0;
    /* ignore double and triple clicks */
    if ( (event->type == GDK_2BUTTON_PRESS) ||
         (event->type == GDK_3BUTTON_PRESS) )
        return TRUE;

    /* don't react to pressing the other button if we're
       zooming in or out */
    if ((zoom_timer != -1) &&
        ((event->button == 1) || (event->button == 3)) )
        return TRUE;

    if (stat.zooming) {
        draw_zoom_box();
        if (event->button == 1) {
            zoom_resize(1);
            if (!zoom_is_valid_size())
                zoom_resize(-1);
            else
                zoom_timer = g_timeout_add(ZOOM_INTERVAL,
                                          (GtkFunction)zoom_callback,
                                          (gpointer)2);
        }
        else if (event->button == 2) {
            zoom_in();
        }
        else if (event->button == 3) {
            zoom_resize(-1);
            if (!zoom_is_valid_size())
                zoom_resize(1);
            else
                zoom_timer = g_timeout_add(ZOOM_INTERVAL,
                                          (GtkFunction)zoom_callback,
                                          (gpointer)-2);
        }
        draw_zoom_box();
    }
    else if (stat.julia_browsing) {
        if (event->button == 1) {
            img.u.julia.c_re = event->x;
            img.u.julia.c_im = event->y;
            get_coords(&img.u.julia.c_re, &img.u.julia.c_im);
            stop_julia_browsing();
            /* save old coordinates */
            img.old_xmin = img.xmin;
            img.old_xmax = img.xmax;
            img.old_ymax = img.ymax;
            img.xmin = j_pre.xmin;
            img.xmax = j_pre.xmax;
            img.ymax = j_pre.ymax;
            img.fr_type = JULIA;
            start_rendering(&img);
        }
    }
    else if (stat.repositioning) {
        if (event->button == 1) {
            if (!stat.second_point) {
                stat.second_point = TRUE;
                draw_center_lines(TRUE);
                stat.fp_x = event->x;
                stat.fp_y = event->y;
            }
            else {
                cx = stat.cl_x;
                cy = stat.cl_y;
                get_coords(&cx, &cy);
                fpx = stat.fp_x;
                fpy = stat.fp_y;
                get_coords(&fpx, &fpy);
                cxd = (img.xmax - img.xmin) / 2;
                cyd = cxd / img.ratio;
                img.xmin += (fpx - cx);
                img.xmax = img.xmin + cxd * 2;
                img.ymax += (fpy - cy);
                draw_center_lines(TRUE);
                draw_center_lines(FALSE);
                stop_reposition();
                draw_center_lines(TRUE);
                start_rendering(&img);
            }
        }
        else if (event->button == 2) {
            cx = stat.cl_x;
            cy = stat.cl_y;
            get_coords(&cx, &cy);
            cxd = (img.xmax - img.xmin) / 2;
            cyd = cxd / img.ratio;
            img.xmin = cx - cxd;
            img.xmax = img.xmin + cxd * 2;
            img.ymax = cyd + cy;
            if (stat.second_point) {
                draw_center_lines(FALSE);
                draw_center_lines(TRUE);
                stat.second_point=FALSE;
                draw_center_lines(TRUE);
            }
            stop_reposition();
            start_rendering(&img);
        }
        else if (event->button == 3) {
            if (stat.second_point) {
                draw_center_lines(FALSE);
                draw_center_lines(TRUE);
                stat.second_point=FALSE;
                draw_center_lines(TRUE);
            }
            stop_reposition();
        }
    }
    return TRUE;
}

void kill_zoom_timers(void)
{
    if (zoom_timer != -1) {
        g_source_remove(zoom_timer);
    }
    zoom_timer = -1;
}

gint button_release_event(GtkWidget* widget, GdkEventButton* event)
{
    kill_zoom_timers();
    
    return TRUE;
}

gint motion_event(GtkWidget* widget, GdkEventMotion* event)
{
    if ((!stat.zooming) &&
        (!stat.julia_browsing) &&
        (!stat.repositioning)) return TRUE;

    if (stat.julia_browsing) {
        j_pre.u.julia.c_re = event->x;
        j_pre.u.julia.c_im = event->y;
        get_coords(&j_pre.u.julia.c_re, &j_pre.u.julia.c_im);
        start_rendering(&j_pre);
    }
    else if (stat.zooming) {
        draw_zoom_box();
        stat.z_x = event->x;
        stat.z_y = event->y;
        draw_zoom_box();
    }
    else if (stat.repositioning) {
        draw_center_lines(TRUE);
        stat.cl_x = event->x;
        stat.cl_y = event->y;
        draw_center_lines(TRUE);
    }
    return TRUE;
}

gint expose_event(GtkWidget* widget, GdkEventExpose* event,
                  image_info* img)
{
    GdkRectangle pic_area, padding_area,tmp;
    
    /* sometimes gtk gives us invalid exposes when we're changing
       image size, and when that happens, draw_image below segfaults
       when it's trying to access rgb_data at offsets way past the
       current image size.

       so check for that and don't do anything on invalid exposes. */
    if ((event->area.y + event->area.height - 1) >
        (img->user_height - 1)) {
        return TRUE;
    }

    tmp.x = 0;
    tmp.width = img->user_width;
    pic_area = horiz_intersect(&event->area, &tmp);

    tmp.x = img->user_width;
    tmp.width = USHRT_MAX;
    padding_area = horiz_intersect(&event->area, &tmp);

    if (pic_area.width != 0) {
        gdk_draw_rgb_32_image(widget->window,
                              widget->style->white_gc,
                              pic_area.x, pic_area.y,
                              pic_area.width, pic_area.height,
                              GDK_RGB_DITHER_NONE,
                              (guchar*)&(img->rgb_data[pic_area.y*
                                                      img->user_width
                                                      + pic_area.x]),
                              img->user_width*4);
    }

    if (padding_area.width != 0) {
        gdk_draw_rectangle(widget->window,
                           widget->style->white_gc,
                           TRUE,
                           padding_area.x, padding_area.y,
                           padding_area.width, padding_area.height);
    }
    
    return TRUE;
}

gint idle_callback(image_info* img)
{
    int y_offset;

    if (img->aa_factor == 1) {
        fractal_next_line(img);
        palette_apply(img, 0, img->lines_done-1, img->real_width, 1);
    } else {
        int i;

        for (i=0; i < img->aa_factor; i++)
            fractal_next_line(img);
        do_anti_aliasing(img, 0, img->lines_done/img->aa_factor-1,
                         img->user_width, 1);
    }

    y_offset = img->lines_done/img->aa_factor-1;
    gdk_draw_rgb_32_image(img->drawing_area->window,
                          img->drawing_area->style->white_gc,
                          0, y_offset, img->user_width, 1,
                          GDK_RGB_DITHER_NONE,
                          (guchar*)
                          (&(img->rgb_data[img->user_width*y_offset])),
                          img->user_width*4);

    if (!img->j_pre)

        gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(pbar), 
                (gfloat)img->lines_done / (gfloat)img->real_height);

    if (img->lines_done == img->real_height) {
        stop_rendering(img);

        return FALSE;
    }
    else
        return TRUE;
}

void quit(void)
{
    gtk_main_quit();
}

int main(int argc, char** argv)
{
    GtkWidget* vbox;
    GtkWidget* hbox;
    GtkWidget* button;
    GtkWidget* tmp;
    GtkObject* adj;

    srand(time(0));

    program_name = argv[0];
    gtk_init(&argc, &argv);

    img.filesel = NULL;

    /* init random palette move to here from init_misc() */
    rnd_palette.r_strength = 0.85;
    rnd_palette.r_bands = 0.05;
    rnd_palette.g_strength = 0.79;
    rnd_palette.g_bands = 0.08;
    rnd_palette.b_strength = 0.83;
    rnd_palette.b_bands = 0.2;
    rnd_palette.offset = 0.0;
    rnd_palette.stripe = 1.0;
    rnd_palette.spread = 1.0;

    palette = g_malloc(256*4);
    if (palette_load(DEFAULT_PALETTE_FILE) == FALSE ||
        palette_load(DEFAULT_PALETTE_FILE2) == FALSE)
    {   /* initial palette loading or generation must take place
           here, otherwise palette copying when duplicating, breaks. */
        rnd_palette.r_strength = 1.0;
        rnd_palette.g_strength = 1.0;
        rnd_palette.b_strength = 1.0;
        palette_randomize(&rnd_palette);
    }

    init_misc();
    process_args(argc, argv);
    g_timeout_add(10*1000, child_reaper, NULL);
    set_image_info(&img, img.user_width, img.user_height, img.aa_factor);
    set_image_info(&j_pre, JPRE_SIZE, JPRE_SIZE/img.ratio, JPRE_AAFACTOR);


    /* main window */
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_widget_realize(window);

    /* preview window */
    j_pre_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    g_signal_connect(GTK_OBJECT(j_pre_window), "delete_event",
                       G_CALLBACK(j_pre_delete), NULL);
    gtk_window_set_title(GTK_WINDOW(j_pre_window), "Preview");
    gtk_window_set_resizable(GTK_WINDOW(j_pre_window), FALSE);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), vbox);
    gtk_widget_show (vbox);

    create_menus(vbox);
    g_signal_connect (GTK_OBJECT (window), "destroy",
                        G_CALLBACK (quit), NULL);

    /* toolbar stuff */

    hbox = gtk_hbox_new(FALSE, 5);
    gtk_widget_show(hbox);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

    /* zoom in */
    zoom_in_button = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(zoom_in_button),
                      create_pixmap(window, zoom_in_xpm));
    g_signal_connect(GTK_OBJECT(zoom_in_button), "toggled",
                       G_CALLBACK(zoom_in_func), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), zoom_in_button, FALSE, FALSE,
                       0);
    gtk_button_set_relief(GTK_BUTTON(zoom_in_button),
                          GTK_RELIEF_NONE);
    GTK_WIDGET_UNSET_FLAGS(zoom_in_button, GTK_CAN_FOCUS);
    gtk_widget_show(zoom_in_button);

    /* zoom out */
    zoom_out_button = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(zoom_out_button),
                      create_pixmap(window, zoom_out_xpm));
    g_signal_connect(GTK_OBJECT(zoom_out_button), "clicked",
                       G_CALLBACK(zoom_out_func), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), zoom_out_button, FALSE, FALSE,
                       0);
    gtk_button_set_relief(GTK_BUTTON(zoom_out_button),
                          GTK_RELIEF_NONE);
    GTK_WIDGET_UNSET_FLAGS(zoom_out_button, GTK_CAN_FOCUS);
    gtk_widget_show(zoom_out_button);

    /* reposition */
    reposition_button = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(reposition_button),
                      create_pixmap(window, reposition_xpm));
    g_signal_connect(GTK_OBJECT(reposition_button), "clicked",
                       G_CALLBACK(reposition_func), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), reposition_button, FALSE, FALSE,
                       0);
    gtk_button_set_relief(GTK_BUTTON(reposition_button),
                          GTK_RELIEF_NONE);
    GTK_WIDGET_UNSET_FLAGS(reposition_button, GTK_CAN_FOCUS);
    gtk_widget_show(reposition_button);

    /* depth label */
    button = gtk_label_new("Depth");
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);

    /* depth spin-button */
    adj = gtk_adjustment_new((gfloat)img.depth, 1.0,
                                2147483647.0, 1, 1, 0.0);
    depth_spin = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(hbox), depth_spin, FALSE, FALSE, 0);
    gtk_widget_show(depth_spin);

    /* method switch toggle button */
    method_toggle = gtk_toggle_button_new_with_label("M");
    gtk_toggle_button_set_active(
                    GTK_TOGGLE_BUTTON(method_toggle), img.method);
    gtk_box_pack_start(GTK_BOX(hbox), method_toggle, FALSE, FALSE, 0);
    g_signal_connect(GTK_OBJECT(method_toggle), "toggled",
                    G_CALLBACK(method_func), &img);
    gtk_widget_show(method_toggle);

    /* recalc button */
    button = gtk_button_new();
    recalc_button_label = gtk_label_new(TEXT_STOP);
    gtk_misc_set_alignment(GTK_MISC(recalc_button_label),
                           0.5, 0.5);
    gtk_container_add(GTK_CONTAINER(button), recalc_button_label);
    gtk_widget_show(recalc_button_label);
    g_signal_connect(GTK_OBJECT(button), "clicked",
                       G_CALLBACK(recalc_button), NULL);
    gtk_box_pack_end(GTK_BOX(hbox), button, FALSE, FALSE, 0);
    gtk_widget_show(button);
    
    /* progress bar */
    pbar = gtk_progress_bar_new();
    gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(pbar),
                                     GTK_PROGRESS_LEFT_TO_RIGHT);
    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(pbar), 
                               (gfloat)1.0 ); /* img.real_height); */
    gtk_widget_set_size_request(pbar, 75, 0);
    gtk_box_pack_end(GTK_BOX(hbox), pbar, FALSE, FALSE, 0);

    /* separator */
    button = gtk_hseparator_new();
    gtk_widget_show(button);
    gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

    /* main window drawing area */
    tmp = gtk_drawing_area_new();
    gtk_widget_set_events (tmp, GDK_BUTTON_PRESS_MASK
                           | GDK_BUTTON_RELEASE_MASK
                           | GDK_POINTER_MOTION_MASK
                           | GDK_EXPOSURE_MASK);
    gtk_widget_set_size_request(tmp, 
                        (img.user_width >= MIN_WINDOW_WIDTH)
                        ? img.user_width : MIN_WINDOW_WIDTH,
                        img.user_height);
    gtk_box_pack_start(GTK_BOX(vbox), tmp, TRUE, TRUE, 0);
    gtk_widget_show(tmp);
    
    g_signal_connect(GTK_OBJECT(tmp), "button_press_event",
                        G_CALLBACK(button_press_event), NULL);
    g_signal_connect(GTK_OBJECT(tmp), "button_release_event",
                        G_CALLBACK(button_release_event), NULL);
    g_signal_connect(GTK_OBJECT(tmp), "expose_event",
                       G_CALLBACK(expose_event), &img);
    g_signal_connect(GTK_OBJECT(tmp), "motion_notify_event",
                       G_CALLBACK(motion_event), NULL);
    img.drawing_area = drawing_area = tmp;

    /* preview window drawing area */
    tmp = gtk_drawing_area_new();
    gtk_widget_set_events (tmp, GDK_EXPOSURE_MASK);
    gtk_widget_set_size_request(tmp, j_pre.user_width, j_pre.user_height);
    gtk_container_add(GTK_CONTAINER(j_pre_window), tmp);
    g_signal_connect(GTK_OBJECT(tmp), "expose_event",
                       G_CALLBACK(expose_event), &j_pre);
    gtk_widget_show(tmp);
    j_pre.drawing_area = tmp;

    start_rendering(&img);
    gtk_widget_show(window);

    gtk_main();

    return 0;
}
