/* GAdmin-SSHD - An easy to use GTK+ frontend for the openssh server.
 * Copyright (C) 2009 - 2010 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "reread_conf.h"
#include "show_info.h"
#include "functions.h"
#include "apply_server_settings.h"
#include "allocate.h"
#include "get_option_pos.h"
#include "populate_server_settings.h"
#include "populate_conf_tab.h"


extern char global_server_address[1024];
extern char global_server_port[1024];
extern char global_server_name[1024];
extern char global_server_type[1024];

extern int use_tls;

extern int activated;



int has_value(gchar *input)
{
    int have_value = 0;

    if( input!=NULL && strlen(input) > 1 )
        have_value = 1;

    return have_value;
}



/* Removes all lines that contains matches for "rmatch1" to "rmatch3".
   All supplied rmatches must match or the line is not removed. */
void lowercase_remove_sshd_option(gchar *rmatch1, gchar *rmatch2, gchar *rmatch3)
{
    FILE *fp;
    long file_size = 0;
    char *line, *new_conf;
    gchar *info;

    if( ! has_value(rmatch1) )
    {
        printf("Error: No value supplied for rmatch1\n");
        return;
    }

    if((fp=fopen(SSHD_CONF, "r"))==NULL)
    {
        info = g_strdup_printf(_("Can not read sshd configuration file here: \n%s\n"), SSHD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line     = allocate(file_size+1);
    new_conf = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( has_value(rmatch1) )
        {
            if( ! cmplowercase(line, rmatch1) )
            {
                strcat(new_conf, line);
                continue;
            }
        }
	
        if( has_value(rmatch2) )
        {
            if( ! cmplowercase(line, rmatch2) )
            {
                strcat(new_conf, line);
                continue;
            }
        }

        if( has_value(rmatch3) )
        {
            if( ! cmplowercase(line, rmatch3) )
            {
                strcat(new_conf, line);
                continue;
            }
        }

        /* The supplied options matches this line, dump it */
    }
    free(line);
    fclose(fp);

    /* Write the new conf */
    if((fp=fopen(SSHD_CONF, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Can not write sshd configuration file here: \n%s\n"), SSHD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);
}


/* Append lines to sshd configuration */
void append_sshd_option(gchar *option_line)
{
    FILE *fp;
    gchar *info;

    if((fp=fopen(SSHD_CONF, "a+"))==NULL)
    {
        info = g_strdup_printf(_("Can not append sshd configuration file here: \n%s\n"), SSHD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fputs(option_line, fp);
    fclose(fp);
}

/* Prepend lines to sshd configuration */
void prepend_sshd_option(gchar *option_line)
{
    FILE *fp;
    long file_size = 0;
    char *line, *new_conf;
    gchar *info;

    if((fp=fopen(SSHD_CONF, "r"))==NULL)
    {
        info = g_strdup_printf(_("Can not prepend sshd configuration file here: \n%s\n"), SSHD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
            
    line = allocate(file_size);
    new_conf = allocate(file_size+1024);

    snprintf(new_conf, 1000, "%s", option_line);
                    
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        strcat(new_conf, line);
    }
    fclose(fp);
    free(line);

    if((fp=fopen(SSHD_CONF, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Can not prepend sshd configuration file here: \n%s\n"), SSHD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);
}


void apply_server_settings(struct w *widgets)
{
    gint active_index;
    gchar *new_line, *opt_line;
    char *tmp;

    G_CONST_RETURN gchar *ipv4_addresses;
    G_CONST_RETURN gchar *ipv6_addresses;
    G_CONST_RETURN gchar *server_port;

    ipv4_addresses   = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[0]));
    ipv6_addresses   = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[1]));
    server_port = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[0]));

    /* Remove sshd configuration options that we handle... */

    /* ListenAddress (IPv4 and IPv6) */
    lowercase_remove_sshd_option("listenaddress", "", "");

    /* Port */
    lowercase_remove_sshd_option("port ", "", "");
    
    /* PasswordAuthentication */
    lowercase_remove_sshd_option("passwordauthentication ", "", "");

    /* X11Forwarding */
    lowercase_remove_sshd_option("x11forwarding ", "", "");

    /* ChallengeResponseAuthentication */
    lowercase_remove_sshd_option("challengeresponseauthentication ", "", "");

    /* PermitRootLogin */
    lowercase_remove_sshd_option("permitrootlogin ", "", "");



    /* Listen to IPv4 addresses: */
    if( ipv4_addresses!=NULL && strlen(ipv4_addresses) > 1 )
    {
        new_line = g_strdup_printf(" %s ", ipv4_addresses);
        tmp = strtok(new_line, " ");
        while( tmp != NULL )
        {
            opt_line = g_strdup_printf("ListenAddress %s\n", tmp);
            prepend_sshd_option(opt_line);
            g_free(opt_line);

            tmp = strtok(NULL, " ");
        }
        g_free(new_line);
    }

    /* Listen to IPv6 addresses: */
    if( ipv6_addresses!=NULL && strlen(ipv6_addresses) > 1 )
    {
        new_line = g_strdup_printf(" %s ", ipv6_addresses);
        tmp = strtok(new_line, " ");
        while( tmp != NULL )
        {
            opt_line = g_strdup_printf("ListenAddress %s\n", tmp);
            prepend_sshd_option(opt_line);
            g_free(opt_line);

            tmp = strtok(NULL, " ");
        }
        g_free(new_line);
    }

    /* Server port must be located before the Listen options
       so ipv4 and ipv6 plus port are prepended to the conf. */
    if( server_port!=NULL )
    {
        opt_line = g_strdup_printf("port %s\n", server_port);
        prepend_sshd_option(opt_line);
        g_free(opt_line);
    }
    else
    {
        opt_line = g_strdup_printf("port 22\n");
        prepend_sshd_option(opt_line);
        g_free(opt_line);
    }

    /* End of prepending options, go ahead and append... */


    /* Require password */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[0]));
    if( active_index == 0 )
    {
        opt_line = g_strdup_printf("PasswordAuthentication yes\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }
    if( active_index == 1 )
    {
        opt_line = g_strdup_printf("PasswordAuthentication no\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }

    /* X-forwarding */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[1]));
    if( active_index == 0 )
    {
        opt_line = g_strdup_printf("X11Forwarding yes\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }
    if( active_index == 1 )
    {
        opt_line = g_strdup_printf("X11Forwarding no\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }
    
    /* Challenge auth */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[2]));
    if( active_index == 0 )
    {
        opt_line = g_strdup_printf("ChallengeResponseAuthentication yes\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }
    if( active_index == 1 )
    {
        opt_line = g_strdup_printf("ChallengeResponseAuthentication no\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }

    /* Permit root login */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[3]));
    if( active_index == 0 )
    {
        opt_line = g_strdup_printf("PermitRootLogin yes\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }
    if( active_index == 1 )
    {
        opt_line = g_strdup_printf("PermitRootLogin no\n");
        append_sshd_option(opt_line);
        g_free(opt_line);
    }

    /* Populate the server settings */    
    populate_server_settings(widgets);

    /* Populate the configuration tab */
    populate_conf_tab(widgets);

    if( activated )
        reread_conf(widgets);
}
