/* GADMIN-HTTPD - An easy to use GTK+ frontend for Apache HTTPD webserver.
 * Copyright (C) 2006 - 2011 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "allocate.h"
#include "commands.h"
#include "show_info.h"
#include "user_handling.h"
#include "functions.h"
#include "populate_htaccess_tab.h"
#include "select_first_user.h"

extern char global_virtual_user_name[1024];
extern char global_htacc_user_name[1024];



void add_virtual_user(struct w *widgets)
{
    /* Adds a new virtual user */
    FILE *fp;
    int length  = 0;
    gchar *utf8 = NULL;
    gchar *info, *user_pass, *htpasswd_file;
    G_CONST_RETURN gchar *username;
    G_CONST_RETURN gchar *password;
    char *encrypted_pass;

    username = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[0]));
    password = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[1]));

    /* If the username field is empty inform that this cant be done. */
    length = strlen(username);
    if( length == 0 ) 
    {
        info = g_strdup_printf(_("You must write a username first.\n"));
        show_info(info);
        g_free(info);
        return;
    }

    if( username[0]=='r' && username[1]=='o' && username[2]=='o' && username[3]=='t' && strlen(username) == 4 ) 
    {
        info = g_strdup_printf(_("Failed adding user: %s\nThe root user can not be added for security reasons.\n"), username);
        show_info(info);
        g_free(info);
        return;
    }

    if( strstr((char *)username, "<") || strstr((char *)username, ">") ) 
    {
        info = g_strdup_printf(_("Failed adding user: %s\nchars \"<\" and \">\" arent allowed.\n"), username);
        show_info(info);
        g_free(info);
        return;
    }

    /* If the password field has less then 3 chars we inform that this cant be done */
    length = strlen(password); /* This is a virtual user so 3 chars is allowed */
    if( length < 3 )
    {
        info = g_strdup_printf(_("Failed adding user: %s\nA minimum password length of 3 chars is required.\n"), username);
        show_info(info);
        g_free(info);
        return;
    }


    /* Remove the user if it exists */
    if( virtual_user_exists(username) )
      del_virtual_user(widgets);


    /* Add the user */
    htpasswd_file = g_strdup_printf("%s/passwords/passwords", HTTPD_ROOT_PATH);
    if((fp=fopen(htpasswd_file, "a"))==NULL)
    {
        info = g_strdup_printf(_("User not added. The main htpasswd file is missing:\n%s\n"), htpasswd_file);
        show_info(info);
        g_free(info);
        g_free(htpasswd_file);
        return;
    }
    
    encrypted_pass = encrypt_password(password);                            

    user_pass = g_strdup_printf("%s:%s\n", username, encrypted_pass);

    fputs(user_pass, fp);
    fclose(fp);
    
    g_free(htpasswd_file);

    if( user_pass!=NULL )
      g_free(user_pass);

    /* Update the virtual user list */
    populate_virtual_users(widgets);

    if( utf8!=NULL )
      g_free(utf8);
}


void del_virtual_user(struct w *widgets)
{
    FILE *fp;
    char *line, *tmp;
    long file_size = 0;
    int i = 0;
    char *new_file;
    gchar *htpasswd_file;

    htpasswd_file = g_strdup_printf("%s/passwords/passwords", HTTPD_ROOT_PATH);
    if((fp=fopen(htpasswd_file, "r"))==NULL)
    {
        g_free(htpasswd_file);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    tmp  = allocate(file_size+1);
    new_file = allocate(file_size+1);

    /* Collect all users but not the one we are deleting */
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        for(i=0; line[i]!='\0'; i++)
          if( line[i]==':' )
            break;

        snprintf(tmp, file_size, "%s", line);
        tmp[i]='\0';

        if( ! strcmp((char *)global_virtual_user_name, tmp)
        ||    strlen(line) < 5 )
        {
            /* Skip the user or empty lines */
        }
        else
          strcat(new_file, line);
    }
    fclose(fp);    
    free(tmp);
    free(line);


    /* Save the new file */
    if((fp=fopen(htpasswd_file, "w+"))==NULL)
    {
        g_free(htpasswd_file);
        return;
    }
    fputs(new_file, fp);
    fclose(fp);
    free(new_file);
    g_free(htpasswd_file);

    /* Update the virtual user list */
    populate_virtual_users(widgets);
}


int virtual_user_exists(G_CONST_RETURN gchar *username)
{
    /* Checks if the system user exists */
    FILE *fp;
    long file_size;
    int x, user_exists = 0;
    char tempname[4096]="";
    char *line;

    gchar *htpasswd_file = g_strdup_printf("%s/passwords/passwords", HTTPD_ROOT_PATH);

    /* Checks if the user exists in passwd */
    if((fp=fopen(htpasswd_file, "r"))==NULL)
    {
        g_free(htpasswd_file);
        return 0;
    }

    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        for(x=0; line[x]; x++)
        {
            if(line[x]==':')
            {
                snprintf(tempname, 4000, "%s", line);
                tempname[x]='\0';
                if( strcmp(username, tempname) == 0 )
                {
                    user_exists = 1;
                    break;
                }
            }
            if( line[x]=='\0' || line[x]=='\n' )
              break;
        }
    }
    free(line);
    g_free(htpasswd_file);
    fclose(fp);

    return user_exists;
}


int htacc_user_exists(gchar *path)
{
    FILE *fp;
    long file_size = 0;
    char *line;
    gchar * username;
    int ret = 0;
    
    username = g_strdup_printf("%s", global_virtual_user_name);
    
    if((fp=fopen(path, "r"))==NULL)
    {
        printf("Error reading .htaccess file here:\n%s\n", path);
        return ret;
    }

    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
        
    line = allocate(file_size);
            
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( cmplowercase(line, "require")
        &&  cmplowercase(line, "user")
        &&  cmplowercase(line, username))
        {
            ret = 1;
            break;
        }
    }                        
    fclose(fp);
    free(line);

    if( username!=NULL )
      g_free(username);

    return ret;
}


void add_htacc_user(struct w *widgets)
{
    FILE *fp;
    long file_size = 0;
    char *line, *new_conf;
    gchar *user_line, *file_path, *conf, *info;
    G_CONST_RETURN gchar *dir_path;

    if( global_virtual_user_name == NULL ||
	strlen(global_virtual_user_name) < 1 )
    {
        info = g_strdup_printf(_("Add and select a virtual user first.\n"));
        show_info(info);
        g_free(info);
        return;
    }
    
    dir_path = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[2]));

    if( dir_path == NULL || strlen(dir_path) < 5 )
    {
        printf("Directory path too short.\n");
        return;
    }
    if( dir_path[0]!='/' )
    {
        printf("Invalid directory path.\n");
        return;
    }

    file_path = g_strdup_printf("%s/.htaccess", dir_path);

    /* Return if the user exists */
    if( htacc_user_exists(file_path) )
        return;

    if( ! file_exists(file_path) )
    {
        /* A .htaccess file doesnt exist, add one. */
        conf = g_strconcat("\n",
        "AuthUserFile ", HTTPD_ROOT_PATH, "/passwords/passwords\n",
        "AuthName \"Protected Area\"\n",
        "AuthType Basic\n",
        "\n",
        "<Limit GET POST PUT DELETE>\n",
        "</Limit>\n",
        NULL);

        if((fp=fopen(file_path, "w+"))==NULL)
        {
            printf("Error writing .htaccess file here:\n%s\n", file_path);
            g_free(conf);
            g_free(file_path);
            return;
        }
        fputs(conf, fp);
        fclose(fp);

        g_free(conf);
    }

    /* Add "require user UserName" */
    if((fp=fopen(file_path, "r"))==NULL)
    {
        printf("Error reading .htaccess file here:\n%s\n", file_path);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
        
    line = allocate(file_size+1);
    new_conf = allocate(file_size+2048);
    user_line = g_strdup_printf("require user %s\n", global_virtual_user_name);
            
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        strcat(new_conf, line);

        if( cmplowercase(line, "<limit ") )
          strcat(new_conf, user_line);	  
    }
    fclose(fp);
    free(line);

    /* Write the new .htaccess file */
    if((fp=fopen(file_path, "w+"))==NULL)
    {
        printf("Error writing .htaccess file here:\n%s\n", file_path);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);

    free(new_conf);

    if( file_path!=NULL )    
      g_free(file_path);
    
    if( user_line!=NULL )
      g_free(user_line);

    /* Update the htaccess user list */
    populate_htaccess_users(widgets);
}


void del_htacc_user(struct w *widgets)
{
    FILE *fp;
    long file_size = 0;
    int have_other_users = 0;
    char *line, *new_conf;
    gchar *user_line, *file_path, *info;
    G_CONST_RETURN gchar *dir_path;

    if( global_htacc_user_name == NULL ||
        strlen(global_htacc_user_name) < 1 )
    {
        info = g_strdup_printf(_("Select a limited user to delete first.\n"));
        show_info(info);
        g_free(info);
        return;
    }
    
    dir_path = gtk_entry_get_text(GTK_ENTRY(widgets->user_set_entry[2]));

    if( dir_path == NULL || strlen(dir_path) < 5 )
    {
        printf("Directory path too short.\n");
        return;
    }
    if( dir_path[0]!='/' )
    {
        printf("Invalid directory path.\n");
        return;
    }

    file_path = g_strdup_printf("%s/.htaccess", dir_path);
    if( ! file_exists(file_path) )
    {
        printf("htaccess file doesnt exist\n");
        g_free(file_path);
        return;
    }

    /* Delete "require user UserName" */
    if((fp=fopen(file_path, "r"))==NULL)
    {
        printf("Error reading .htaccess file here:\n%s\n", file_path);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
        
    line = allocate(file_size+1);
    new_conf = allocate(file_size+2048);

    user_line = g_strdup_printf("require user %s\n", global_htacc_user_name);
            
    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	if( cmplowercase(line, user_line) )
	{
//	    printf("deleted user\n");	
	}
	else
	  strcat(new_conf, line);

	/* Record if there are any other users */
	if( cmplowercase(line, "require user") && ! cmplowercase(line, user_line) )
	    have_other_users = 1;
    }
    fclose(fp);
    free(line);

    /* Write the new .htaccess file */
    if((fp=fopen(file_path, "w+"))==NULL)
    {
        printf("Error writing .htaccess file here:\n%s\n", file_path);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);

    free(new_conf);

    /* If there are no other users then delete the .htaccess file */
    if( ! have_other_users )
      unlink(file_path);

    if( file_path!=NULL )    
      g_free(file_path);
    if( user_line!=NULL )
      g_free(user_line);

    /* Update the htaccess user list */
    populate_htaccess_users(widgets);
}


/* Show htaccess directory selection */
void show_htacc_dir_sel(struct w *widgets)
{
    /* Show htaccess directory selection */
    widgets->htacc_dir_sel = gtk_file_selection_new("Select directory to limit access to.");
    gtk_widget_show(widgets->htacc_dir_sel);

    /* Connect the ok button to the dir selection ok function */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->htacc_dir_sel)->ok_button), "clicked",
                                                                     G_CALLBACK(htacc_dir_sel_ok), widgets);

    /* Connect the destroy signal to the cancel button */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->htacc_dir_sel)->cancel_button), "clicked",
                                                     G_CALLBACK(gtk_widget_destroy), widgets->htacc_dir_sel);
}
                                                                                                                                                                                                             

/* htaccess selection ok clicked */
void htacc_dir_sel_ok(struct w *widgets)
{
    /* Adds destination directory to the treeview */
    gchar *utf8 = NULL;
        
    G_CONST_RETURN gchar *path = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widgets->htacc_dir_sel));
    utf8 = g_locale_to_utf8(path, strlen(path), NULL, NULL, NULL);
    
    gtk_entry_set_text(GTK_ENTRY(widgets->user_set_entry[2]), utf8);

    gtk_widget_destroy(widgets->htacc_dir_sel);

    populate_htaccess_users(widgets);

    if( utf8!=NULL )
      g_free(utf8);
}

