/* GADMIN-HTTPD - An easy to use GTK+ frontend for Apache HTTPD webserver.
 * Copyright (C) 2006 - 2011 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <dirent.h>
#include "allocate.h"
#include <gtk/gtk.h>
#include "gettext.h"
#include "widgets.h"
#include "functions.h"
#include "commands.h"
#include "commented.h"
#include "get_option_pos.h"
#include "standard_conf.h"
#include "populate_gadmin_httpd.h"
#include "show_info.h"

extern int httpd_mayor_version;



void backup_httpd_conf(struct w *widgets)
{
    gchar *cmd, *info;

    if( ! file_exists(HTTPD_CONF_BACKUP) && file_exists(HTTPD_CONF) )
    {
        cmd = g_strdup_printf("cp %s %s", HTTPD_CONF, HTTPD_CONF_BACKUP);
        if( ! run_command(cmd) )
        {
            info = g_strdup_printf(_("Error: Could not backup the current configuration:\n%s\n"), HTTPD_CONF);
            show_info(info);
        }
        else
        {
            info = g_strdup_printf(_("The configuration was backed up here for safety:\n%s\n"), HTTPD_CONF_BACKUP);
            show_info(info);
        }
        g_free(cmd);
        g_free(info);
    }
}


/* Checks if the configuration includes other configuration files */
int conf_needs_merge(gchar *file_path)
{
    FILE *fp;
    long file_size;
    char *line;
    int retval = 0;

    if((fp=fopen(file_path, "r"))==NULL)
        return retval;

    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( commented(line) )
            continue;

        /* The "include" statements are a problem. We must
           merge all these files and remove the statements */
        if( cmplowercase(line, "include ") )
            retval = 1;
    }
    fclose(fp);
    free(line);

    return retval;
}


/* Appends included file to the real conf */
void append_include_file(char *include_path, char *tmp_file)
{
    FILE *fp, *new_fp;
    long file_size  = 0;
    int write_error = 0;
    char *line;

    if((fp=fopen(include_path, "r"))==NULL)
    {
        printf("Include file doesnt exist: %s\n", include_path);
        free(include_path);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( commented(line) )
            continue;

        /* Append temp file with the include file */
        if((new_fp=fopen(tmp_file, "a+"))==NULL)
        {
            printf("Error appending include temp: %s\n", tmp_file);
            write_error = 1;
            break;
        }
        fputs(line, new_fp);
        fclose(new_fp);
    }
    fclose(fp);
    free(line);
}


/* Handle include lines with directories or wildcards */
void append_multi_includes(char *include_path, char *tmp_file)
{
    char *dirpath, *string, *include_file;
    struct dirent **filelist;
    long x = 0, num = 0;
    int  i = 0, wildcard_path = 0;

    dirpath = allocate(strlen(include_path)+1);
    string  = allocate(strlen(include_path)+1);

    /* Get the wildcard filename */
    if( strstr(include_path, "*") )
    {
        wildcard_path = 1;

        for(i=strlen(include_path)-1; i>0; i--)
        if( include_path[i]=='*' )
        {
            snprintf(string, strlen(include_path), "%s", &include_path[i+1]);
            break;
        }

        for(i=strlen(include_path)-1; i>0; i--)
        if( include_path[i]=='/' )
        {
            include_path[i]='\0';
            snprintf(dirpath, strlen(include_path)+1, "%s", include_path);
            break;
        }
    }
    else /* Regular directory path. */
        snprintf(dirpath, strlen(include_path)+1, "%s", include_path);

    num = scandir(dirpath, &filelist, 0, alphasort);
    if( num < 0 )
    {
        perror("scandir");
        free(dirpath);
        free(string);
        return;
    }

    /* List the files in this directory */
    for(x=0; x<num; x++)
    {
        /* Dont operate on "." or ".." */
        if( strcmp(filelist[x]->d_name, ".")  == 0
        ||  strcmp(filelist[x]->d_name, "..") == 0 )
        {
            free(filelist[x]);
            continue;
        }
        /* Wildcarding does not match the file. */
        if( wildcard_path && ! strstr(filelist[x]->d_name, string) )
        {
            free(filelist[x]);
            continue;
        }

    	/* Create the include filepath */
    	include_file = allocate(strlen(include_path)+strlen(filelist[x]->d_name)+3);

        snprintf(include_file,
        strlen(include_path)+strlen(filelist[x]->d_name)+3, "%s/%s",
        dirpath, filelist[x]->d_name);

        append_include_file(include_file, tmp_file);

        free(include_file);
        free(filelist[x]);
    }
    free(dirpath);
    free(string);
}


/* Set default configured USER GROUP and PIDFILE
   if any of them has a "$" char in the beginning
   of the name after merging */
void fix_user_group_pidfile(struct w *widgets)
{
    FILE *fp;
    char *line, *new_conf, *string;
    long file_size = 0, opt_pos = 0;

    if((fp=fopen(HTTPD_CONF, "r"))==NULL)
    {
        printf("Error correcting user, group and pidfile.\nCan not read: %s\n", HTTPD_CONF);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    new_conf = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	    if( commented(line) )
	      continue;

	    if( cmplowercase(line, "user ")
	    ||  cmplowercase(line, "group ")
	    ||  cmplowercase(line, "pidfile ") )
	    {
	        /* Get the second options position */
	        opt_pos = get_option_pos(line, 1);

	        if( line[opt_pos]=='$' )
	        {
                printf("Correcting user, group or pid line:\n%s\n", line);

                string = allocate(2048);

                if( cmplowercase(line, "user ") )
                  snprintf(string, 2000, "User %s\n", SERVER_USER);
                if( cmplowercase(line, "group ") )
                  snprintf(string, 2000, "Group %s\n", SERVER_GROUP);
                if( cmplowercase(line, "pidfile ") )
                  snprintf(string, 2000, "PidFile %s\n", SERVER_PIDFILE);

                strcat(new_conf, string);
                free(string);
	        }
	        else
	          strcat(new_conf, line);
	    }
	    else
	      strcat(new_conf, line);
    }
    fclose(fp);
    free(line);

    if((fp=fopen(HTTPD_CONF, "w+"))==NULL)
    {
        printf("Error correcting user, group and pidfile.\nCan not write: %s\n", HTTPD_CONF);
        return;
    }
    fputs(new_conf, fp);
    fclose(fp);
    free(new_conf);
}


/* Merge all include files contents into the main configuration file */
void merge_include_files(struct w *widgets)
{
    FILE *fp, *new_fp;
    DIR *dirp;
    G_CONST_RETURN gchar *server_root;
    long file_size, opt_pos;
    char *line, *inc_file_path;
    int write_error = 0;
    int path_is_dir = 0;
    gchar *info;
    char tmp_file[]="/tmp/gadmin-httpd-XXXXXX";
    int tmpfd = -1;

    /* Create a temp file */
    tmpfd = mkstemp(tmp_file);
    if( tmpfd < 0 )
    {
        printf("Could not create temporary file: %s\n", tmp_file);
        return;
    }
    close(tmpfd);

    if( strlen(HTTPD_CONF) < 5 )
    {
        info = g_strdup_printf(_("Merge include files, file path too short:\n%s\n"), HTTPD_CONF);
        show_info(info);
        g_free(info);
        unlink(tmp_file);
        return;
    }

    if((fp=fopen(HTTPD_CONF, "r"))==NULL)
    {
        unlink(tmp_file);
        return;
    }

    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
	    if( commented(line) )
	      continue;

	    /* Write a new temporary file with the merged configurations */
	    if( cmplowercase(line, "include ") )
	    {
	        /* First get the file path to merge */
	        opt_pos = get_option_pos(line, 1);

	        inc_file_path = allocate(2048);

	        if( line[opt_pos]=='/' )
	        {
                snprintf(inc_file_path, 1000, "%s", &line[opt_pos]);
	        }
	        else
	        {
	            /* Include conf.d/\*.conf
	               Change to: Include HTTPD_ROOT_PATH/conf.d/\*.conf */
		        server_root = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[6]));
		        if( strlen(server_root) < 4 )
		          snprintf(inc_file_path, 2000, "%s/%s", HTTPD_ROOT_PATH, &line[opt_pos]);
	        }

	        if( inc_file_path[strlen(inc_file_path)-1]=='\n' )
	          inc_file_path[strlen(inc_file_path)-1]='\0';

	        /* Handle different configuration styles...
	           Include /etc/apache2/mods-enabled/\*.load (Many .load files)
	           Include /etc/apache2/mods-enabled/\*.conf (Many .conf files)
	           Include /etc/apache2/sites-enabled/\*     (All files in dir)
	           Include /etc/apache2/sites-enabled        (All files in dir)
	           Include /etc/apache2/mods/Filename        (Just this file  )
	           Include conf.d/\*.conf       HTTPD_ROOT_PATH/conf.d/\*.conf)
	         */

            /* Check if its a directory. */
            path_is_dir = 0;
            dirp = opendir(inc_file_path);
            if( dirp != NULL )
            {
                path_is_dir = 1;
                closedir(dirp);
            }

            if( path_is_dir || strstr(inc_file_path, "*") )
            {
                append_multi_includes(inc_file_path, tmp_file);
            }
            else /* Append this regular include file. */
                append_include_file(inc_file_path, tmp_file);

            free(inc_file_path);
    	}
    	else
        {
            /* Append original conf line to temp file */
            if((new_fp=fopen(tmp_file, "a+"))==NULL)
            {
                printf("Error appending include temp: %s\n", tmp_file);
                write_error = 1;
                break;
            }
            fputs(line, new_fp);
            fclose(new_fp);
        }
    }
    fclose(fp);
    free(line);

    /* Write error, close the dialog and populate the gui */
    if( write_error )
    {
        if( widgets->default_conf_question_window != NULL )
            gtk_widget_destroy(widgets->default_conf_question_window);

        info = g_strdup_printf(_("Error writing new configuration with merged include files.\n"));
        show_info(info);
        g_free(info);

        unlink(tmp_file);

        populate_gadmin_httpd(widgets);
        return;
    }

    /* Overwrite httpd.conf with the newly merged temp file */
    if((fp=fopen(tmp_file, "r"))==NULL)
    {
        printf("Error reading newly created temp file: %s\n", tmp_file);
        unlink(tmp_file);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);
    line = allocate(file_size+1);

    if( file_size > 50 )
    {
        unlink(HTTPD_CONF);
        if((new_fp=fopen(HTTPD_CONF, "a+"))==NULL)
        {
            printf("Error writing merged temp file to the old location: %s\n", HTTPD_CONF);
            free(line);
            unlink(tmp_file);
            return;
        }
        while(fgets(line, file_size, fp)!=NULL)
        {
            fputs(line, new_fp);
        }
        fclose(new_fp);
    }
    fclose(fp);

    unlink(tmp_file);

    if( widgets->default_conf_question_window != NULL )
        gtk_widget_destroy(widgets->default_conf_question_window);

    /* Set default configured USER GROUP AND PIDFILE
       if any of them have a "$" char in the name */
    fix_user_group_pidfile(widgets);

    populate_gadmin_httpd(widgets);
}


int conf_ok(gchar *file_path)
{
    FILE *fp;
    long file_size;
    char *line;
    int retval   = 0;
    int new_conf = 0;
    gchar *info;

    if((fp=fopen(file_path, "r"))==NULL)
    {
        info = g_strdup_printf(_("Apache httpd's configuration file not found here:\n%s\n"), HTTPD_CONF);
        show_info(info);
        g_free(info);
        return retval;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    /* Conf too small to be ok */
    if( file_size < 30 )
    {
        fclose(fp);
        return retval;
    }

    line = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( commented(line) )
          continue;
    }
    fclose(fp);
    free(line);

    if( ! new_conf )
        retval = 1;

    return retval;
}


/* Different configurations are used with
   different mayor versions of the server */
void add_standard_conf(struct w *widgets)
{
    /* Adds default httpd.conf configurations */
    FILE *fp;
    gchar *info, *conf;

    if((fp=fopen(HTTPD_CONF, "w+"))==NULL)
    {
        info = g_strdup_printf(_("Cant write a new httpd.conf here:\n%s\n"), HTTPD_CONF);
        show_info(info);
        g_free(info);
        return;
    }

    /* For testing httpd 1x 2x configurations */
    /* httpd_mayor_version = 1; */

/* HTTPD Mayor version 2 and up... */
if( httpd_mayor_version >= 2 ) 
{
    conf = g_strconcat("",
    "### Apache httpd.conf version >= 2x ###\n",
    "ServerName www.example.org\n",
    "Listen 0.0.0.0:80\n",
    "ServerAdmin webmaster@www.example.org\n",
    "ServerTokens OS\n",
    "Timeout 120\n",
    "KeepAlive Off\n",
    "MaxKeepAliveRequests 100\n",
    "KeepAliveTimeout 15\n",
    "ExtendedStatus On\n",
    "UseCanonicalName On\n",
    "DocumentRoot \"",HTTPD_DOCUMENT_PATH,"\"\n",
    "ServerRoot \"",HTTPD_ROOT_PATH,"\"\n",
    "User ",SERVER_USER,"\n",
    "Group ",SERVER_GROUP,"\n",
    "PidFile ",SERVER_PIDFILE,"\n",
    "TypesConfig /etc/mime.types\n",
    "Alias /icons/ \"/var/www/icons/\"\n",
    "ScriptAlias /cgi-bin/ \"/var/www/cgi-bin/\"\n",
    "Alias /error/ \"/var/www/error/\"\n",
    "AccessFileName .htaccess\n",
    "DirectoryIndex index.html index.html.var index.php\n",
    "HostnameLookups Off\n",
    "ServerSignature On\n",
    "EnableMMAP off\n",
    "EnableSendfile off\n",
    "DefaultIcon /icons/unknown.gif\n",
    "ReadmeName README.html\n",
    "HeaderName HEADER.html\n",
    "IndexIgnore .??* *~ *# HEADER* README* RCS CVS *,v *,t\n",
    "DefaultLanguage en\n",
    "LanguagePriority en ca cs da de el eo es et fr he hr it ja ko ltz nl nn no pl pt pt-BR ru sv zh-CN zh-TW\n",
    "ForceLanguagePriority Prefer Fallback\n",
    "AddDefaultCharset Off\n",
    "ErrorLog logs/error_log\n",
    "DefaultType text/plain\n",
    "IndexOptions FancyIndexing VersionSort NameWidth=* HTMLTable\n",
    "LogLevel warn\n",
    "LogFormat \"%h %l %u %t \\\"%r\\\" %>s %b "
                "\\\"%{Referer}i\\\" \\\"%{User-Agent}i\\\"\""
                " combined\n",
    "LogFormat \"%h %l %u %t \\\"%r\\\" %>s %b\" common\n",
    "LogFormat \"%{Referer}i -> %U\" referer\n",
    "LogFormat \"%{User-agent}i\" agent\n",
    "CustomLog logs/access_log combined\n",
    "BrowserMatch \"Mozilla/2\" nokeepalive\n",
    "BrowserMatch \"MSIE 4\\.0b2;\" nokeepalive downgrade-1.0 force-response-1.0\n",
    "BrowserMatch \"RealPlayer 4\\.0\" force-response-1.0\n",
    "BrowserMatch \"Java/1\\.0\" force-response-1.0\n",
    "BrowserMatch \"JDK/1\\.0\" force-response-1.0\n",
    "BrowserMatch \"Microsoft Data Access Internet Publishing Provider\" redirect-carefully\n",
    "BrowserMatch \"MS FrontPage\" redirect-carefully\n",
    "BrowserMatch \"^WebDrive\" redirect-carefully\n",
    "BrowserMatch \"^WebDAVFS/1.[0123]\" redirect-carefully\n",
    "BrowserMatch \"^gnome-vfs/1.0\" redirect-carefully\n",
    "BrowserMatch \"^XML Spy\" redirect-carefully\n",
    "BrowserMatch \"^Dreamweaver-WebDAV-SCM1\" redirect-carefully\n",

    /* This is a trouble section since some modules might not exist... */
    "LoadModule auth_basic_module modules/mod_auth_basic.so\n",
    "LoadModule auth_digest_module modules/mod_auth_digest.so\n",
    "LoadModule authn_file_module modules/mod_authn_file.so\n",
    "LoadModule authn_alias_module modules/mod_authn_alias.so\n",
    "LoadModule authn_anon_module modules/mod_authn_anon.so\n",
    "LoadModule authn_dbm_module modules/mod_authn_dbm.so\n",
    "LoadModule authn_default_module modules/mod_authn_default.so\n",
    "LoadModule authz_host_module modules/mod_authz_host.so\n",
    "LoadModule authz_user_module modules/mod_authz_user.so\n",
    "LoadModule authz_owner_module modules/mod_authz_owner.so\n",
    "LoadModule authz_groupfile_module modules/mod_authz_groupfile.so\n",
    "LoadModule authz_dbm_module modules/mod_authz_dbm.so\n",
    "LoadModule authz_default_module modules/mod_authz_default.so\n",
    "LoadModule ldap_module modules/mod_ldap.so\n",
    "LoadModule authnz_ldap_module modules/mod_authnz_ldap.so\n",
    "LoadModule include_module modules/mod_include.so\n",
    "LoadModule log_config_module modules/mod_log_config.so\n",
    "LoadModule logio_module modules/mod_logio.so\n",
    "LoadModule env_module modules/mod_env.so\n",
    "LoadModule ext_filter_module modules/mod_ext_filter.so\n",
    "LoadModule mime_magic_module modules/mod_mime_magic.so\n",
    "LoadModule expires_module modules/mod_expires.so\n",
    "LoadModule deflate_module modules/mod_deflate.so\n",
    "LoadModule headers_module modules/mod_headers.so\n",
    "LoadModule usertrack_module modules/mod_usertrack.so\n",
    "LoadModule setenvif_module modules/mod_setenvif.so\n",
    "LoadModule mime_module modules/mod_mime.so\n",
    "LoadModule dav_module modules/mod_dav.so\n",
    "LoadModule status_module modules/mod_status.so\n",
    "LoadModule autoindex_module modules/mod_autoindex.so\n",
    "LoadModule info_module modules/mod_info.so\n",
    "LoadModule dav_fs_module modules/mod_dav_fs.so\n",
    "LoadModule vhost_alias_module modules/mod_vhost_alias.so\n",
    "LoadModule negotiation_module modules/mod_negotiation.so\n",
    "LoadModule dir_module modules/mod_dir.so\n",
    "LoadModule actions_module modules/mod_actions.so\n",
    "LoadModule speling_module modules/mod_speling.so\n",
    "LoadModule userdir_module modules/mod_userdir.so\n",
    "LoadModule alias_module modules/mod_alias.so\n",
    "LoadModule rewrite_module modules/mod_rewrite.so\n",
    "LoadModule proxy_module modules/mod_proxy.so\n",
    "LoadModule proxy_balancer_module modules/mod_proxy_balancer.so\n",
    "LoadModule proxy_ftp_module modules/mod_proxy_ftp.so\n",
    "LoadModule proxy_http_module modules/mod_proxy_http.so\n",
    "LoadModule proxy_connect_module modules/mod_proxy_connect.so\n",
    "LoadModule cache_module modules/mod_cache.so\n",
    "LoadModule suexec_module modules/mod_suexec.so\n",
    "LoadModule disk_cache_module modules/mod_disk_cache.so\n",
    "LoadModule cgi_module modules/mod_cgi.so\n",

    "AddIconByEncoding (CMP,/icons/compressed.gif) x-compress x-gzip\n",

    "AddIconByType (TXT,/icons/text.gif) text/*\n",
    "AddIconByType (IMG,/icons/image2.gif) image/*\n",
    "AddIconByType (SND,/icons/sound2.gif) audio/*\n",
    "AddIconByType (VID,/icons/movie.gif) video/*\n",

    "AddIcon /icons/binary.gif .bin .exe\n",
    "AddIcon /icons/binhex.gif .hqx\n",
    "AddIcon /icons/tar.gif .tar\n",
    "AddIcon /icons/world2.gif .wrl .wrl.gz .vrml .vrm .iv\n",
    "AddIcon /icons/compressed.gif .Z .z .tgz .gz .zip\n",
    "AddIcon /icons/a.gif .ps .ai .eps\n",
    "AddIcon /icons/layout.gif .html .shtml .htm .pdf\n",
    "AddIcon /icons/text.gif .txt\n",
    "AddIcon /icons/c.gif .c\n",
    "AddIcon /icons/p.gif .pl .py\n",
    "AddIcon /icons/f.gif .for\n",
    "AddIcon /icons/dvi.gif .dvi\n",
    "AddIcon /icons/uuencoded.gif .uu\n",
    "AddIcon /icons/script.gif .conf .sh .shar .csh .ksh .tcl\n",
    "AddIcon /icons/tex.gif .tex\n",
    "AddIcon /icons/bomb.gif core\n",
    "AddIcon /icons/back.gif ..\n",
    "AddIcon /icons/hand.right.gif README\n",
    "AddIcon /icons/folder.gif ^^DIRECTORY^^\n",
    "AddIcon /icons/blank.gif ^^BLANKICON^^\n",

    "AddLanguage ca .ca\n",
    "AddLanguage cs .cz .cs\n",
    "AddLanguage da .dk\n",
    "AddLanguage de .de\n",
    "AddLanguage el .el\n",
    "AddLanguage en .en\n",
    "AddLanguage eo .eo\n",
    "AddLanguage es .es\n",
    "AddLanguage et .et\n",
    "AddLanguage fr .fr\n",
    "AddLanguage he .he\n",
    "AddLanguage hr .hr\n",
    "AddLanguage it .it\n",
    "AddLanguage ja .ja\n",
    "AddLanguage ko .ko\n",
    "AddLanguage ltz .ltz\n",
    "AddLanguage nl .nl\n",
    "AddLanguage nn .nn\n",
    "AddLanguage no .no\n",
    "AddLanguage pl .po\n",
    "AddLanguage pt .pt\n",
    "AddLanguage pt-BR .pt-br\n",
    "AddLanguage ru .ru\n",
    "AddLanguage sv .sv\n",
    "AddLanguage zh-CN .zh-cn\n",
    "AddLanguage zh-TW .zh-tw\n",

    "AddType text/html .shtml\n",
    "AddType application/x-httpd-php .php .php4 .php3 .phtml .wml\n",
    "AddType application/x-tar .tgz .tar.gz\n",

    "AddEncoding x-compress .Z\n",
    "AddEncoding x-gzip .gz .tgz .tar.gz\n",

    "AddHandler type-map var\n",
    "AddHandler cgi-script .cgi\n",
    "AddHandler send-as-is asis\n",

    "AddDescription \"GZIP compressed document\" .gz\n",
    "AddDescription \"tar archive\" .tar\n",
    "AddDescription \"GZIP compressed tar archive\" .tgz .tar.gz\n",

    "AddOutputFilter INCLUDES .shtml\n\n",

    "<IfModule prefork.c>\n",
    "   StartServers         8\n",
    "   MinSpareServers      5\n",
    "   MaxSpareServers      20\n",
    "   ServerLimit          256\n",
    "   MaxClients           256\n",
    "   MaxRequestsPerChild  4000\n",
    "</IfModule>\n\n",

    "<IfModule worker.c>\n",
    "   StartServers        2\n",
    "   MaxClients          150\n",
    "   MinSpareThreads     25\n",
    "   MaxSpareThreads     75\n", 
    "   ThreadsPerChild     25\n",
    "   MaxRequestsPerChild 10\n",
    "</IfModule>\n\n",

    "<Directory />\n",
    "   Options FollowSymLinks\n",
    "   AllowOverride AuthConfig\n",
    "</Directory>\n\n",

    "<Directory \"", HTTPD_DOCUMENT_PATH, "\">\n",
    "   Options Indexes FollowSymLinks\n",
    "   AllowOverride AuthConfig\n",
    "   Order allow,deny\n",
    "   Allow from all\n",
    "</Directory>\n\n",

    "<IfModule mod_userdir.c>\n",
    "    UserDir disable\n",
    "    #UserDir public_html\n",
    "</IfModule>\n\n",

    "<Files ~ \"^\\.ht\">\n",
    "    Order allow,deny\n",
    "    Deny from all\n",
    "</Files>\n\n",

    "<IfModule mod_mime_magic.c>\n",
    "    # MIMEMagicFile /usr/share/magic.mime\n",
    "    MIMEMagicFile conf/magic\n",
    "</IfModule>\n\n",

    "<Directory \"/var/www/icons\">\n",
    "    Options Indexes MultiViews\n",
    "    AllowOverride None\n",
    "    Order allow,deny\n",
    "    Allow from all\n",
    "</Directory>\n\n",

    "<IfModule mod_dav_fs.c>\n",
    "    # Location of the WebDAV lock database.\n",
    "    DAVLockDB /var/lib/dav/lockdb\n",
    "</IfModule>\n",

    "<Directory \"/var/www/cgi-bin\">\n",
    "    AllowOverride None\n",
    "    Options None\n",
    "    Order allow,deny\n",
    "    Allow from all\n",
    "</Directory>\n",

    "<IfModule mod_negotiation.c>\n",
    "    <IfModule mod_include.c>\n",
    "      <Directory \"/var/www/error\">\n",
    "        AllowOverride None\n",
    "        Options IncludesNoExec\n",
    "        AddOutputFilter Includes html\n",
    "        AddHandler type-map var\n",
    "        Order allow,deny\n",
    "        Allow from all\n",
    "        LanguagePriority en es de fr\n",
    "        ForceLanguagePriority Prefer Fallback\n",
    "      </Directory>\n",
    "    </IfModule>\n",
    "</IfModule>\n\n",

    "<IfModule mod_proxy.c>\n",
    "   ProxyRequests Off\n",
    "   ProxyVia Off\n",

    "   <Proxy *>\n",
    "      Order deny,allow\n",
    "      Deny from all\n",
    "      Allow from .example.com\n",
    "   </Proxy>\n\n",

    "   <IfModule mod_disk_cache.c>\n",
    "      CacheEnable disk /\n",
    "      CacheRoot \"/var/cache/mod_proxy\"\n",
    "   </IfModule>\n",
    "</IfModule>\n\n", 

    "<IfModule mod_status.c>\n",
    "   <Location /server-status>\n",
    "      SetHandler server-status\n",
    "      Order allow,deny\n",
    "      Allow from localhost localhost.localdomain 127.0.0.1\n",
    "      Deny from all\n",
    "   </Location>\n",
    "</IfModule>\n", 
    NULL);
} /* Mayor version 2 and up end */
else
{
    /* Begin Default configuration for apache httpd 1x (Debian) */

    conf = g_strconcat("",
    "### Apache httpd.conf version 1x ###\n",
    "ServerType standalone\n",
    "ServerName www.example.org\n",
    "Listen 0.0.0.0:80\n",
    "ServerAdmin webmaster@www.example.org\n",
    "ServerTokens OS\n",
    "Timeout 120\n",
    "LockFile /var/lock/apache.lock\n",
    "ScoreBoardFile /var/run/apache.scoreboard\n",
    "KeepAlive On\n",
    "MaxKeepAliveRequests 100\n",
    "KeepAliveTimeout 15\n",
    "MinSpareServers 5\n",
    "MaxSpareServers 10\n",
    "StartServers 5\n",
    "MaxClients 150\n",
    "MaxRequestsPerChild 100\n",
    "ServerSignature On\n\n",
    "DocumentRoot \"",HTTPD_DOCUMENT_PATH,"\"\n",
    "ServerRoot \"",HTTPD_ROOT_PATH,"\"\n",
    "User ",SERVER_USER,"\n",
    "Group ",SERVER_GROUP,"\n",
    "PidFile ",SERVER_PIDFILE,"\n",
    "\n",
    "<IfModule mod_status.c>\n",
    "    ExtendedStatus On\n",
    "   <Location /server-status>\n",
    "      SetHandler server-status\n",
    "      Order allow,deny\n",
    "      Allow from localhost localhost.localdomain 127.0.0.1\n",
    "      Deny from all\n",
    "   </Location>\n",
    "</IfModule>\n\n",

    "<Directory />\n",
    "	Options SymLinksIfOwnerMatch\n",
    "	AllowOverride AuthConfig\n",
    "</Directory>\n\n",

    "<Directory /var/www/>\n",
    "	Options Indexes Includes FollowSymLinks MultiViews\n",
    "	AllowOverride AuthConfig\n",
    "	Order allow,deny\n",
    "	Allow from all\n",
    "</Directory>\n\n",

    "<IfModule mod_userdir.c>\n",
    "	UserDir public_html\n",
    "	<Directory /home/*/public_html>\n",
    "	    AllowOverride FileInfo AuthConfig Limit\n",
    "	    Options MultiViews Indexes SymLinksIfOwnerMatch IncludesNoExec\n",
    "	    <Limit GET POST OPTIONS PROPFIND>\n",
    "    	Order allow,deny\n",
    "    	Allow from all\n",
    "	    </Limit>\n",
    "	    <Limit PUT DELETE PATCH PROPPATCH MKCOL COPY MOVE LOCK UNLOCK>\n",
    "    	Order deny,allow\n",
    "    	Deny from all\n",
    "	    </Limit>\n",
    "	</Directory>\n",
    "</IfModule>\n\n",

    "<IfModule mod_dir.c>\n",
    "	DirectoryIndex index.html index.htm index.shtml index.cgi index.php\n",
    "</IfModule>\n\n",

    "AccessFileName .htaccess\n",

    "<Files ~ \"^\\.ht\">\n",
    "	Order allow,deny\n",
    "	Deny from all\n",
    "</Files>\n\n",

    "UseCanonicalName Off\n",
    "TypesConfig /etc/mime.types\n",
    "DefaultType text/plain\n",

    "<IfModule mod_mime_magic.c>\n",
    "	MIMEMagicFile /usr/share/file/magic.mime\n",
    "</IfModule>\n",

    "HostnameLookups Off\n",
    "ErrorLog /var/log/apache/error.log\n",
    "LogLevel warn\n",

    "LogFormat \"%h %l %u %t \\\"%r\\\" %>s %b "
                "\\\"%{Referer}i\\\" \\\"%{User-Agent}i\\\"\""
                " combined\n",

    "LogFormat \"%h %l %u %t \\\"%r\\\" %>s %b\" common\n",

    "LogFormat \"%{Referer}i -> %U\" referer\n",
    "LogFormat \"%{User-agent}i\" agent\n",

    "CustomLog /var/log/apache/access.log combined\n",

    "<IfModule mod_log_forensic.c>\n",
    "	ForensicLog /var/log/apache/forensic.log\n",
    "</IfModule>\n\n",

    "<IfModule mod_backtrace.c>\n",
    "	EnableExceptionHook On\n",
    "</IfModule>\n\n",

    "<IfModule mod_whatkilledus.c>\n",
    "	EnableExceptionHook On\n",
    "</IfModule>\n\n",

    "<IfModule mod_alias.c>\n",
    "	Alias /icons/ /usr/share/apache/icons/\n",
    "	<Directory /usr/share/apache/icons>\n",
    "        Options Indexes MultiViews\n",
    "        AllowOverride None\n",
    "        Order allow,deny\n",
    "        Allow from all\n",
    "	</Directory>\n\n",
    "	Alias /images/ /usr/share/images/\n",
    "	<Directory /usr/share/images>\n",
    "     Options MultiViews\n",
    "     AllowOverride None\n",
    "     Order allow,deny\n",
    "     Allow from all\n",
    "	</Directory>\n\n",
    "</IfModule>\n\n",

    "<IfModule mod_alias.c>\n",
    "	ScriptAlias /cgi-bin/ /usr/lib/cgi-bin/\n",
    "	<Directory /usr/lib/cgi-bin/>\n",
    "      AllowOverride None\n",
    "  	   Options ExecCGI -MultiViews +SymLinksIfOwnerMatch\n",
    "  	   Order allow,deny\n",
    "  	   Allow from all\n",
    "	</Directory>\n",
    "</IfModule>\n",

    "<IfModule mod_autoindex.c>\n",
    "	IndexOptions FancyIndexing NameWidth=*\n",
    "	AddIconByEncoding (CMP,/icons/compressed.gif) x-compress x-gzip\n\n",

    "	AddIconByType (TXT,/icons/text.gif) text/*\n",
    "	AddIconByType (IMG,/icons/image2.gif) image/*\n",
    "	AddIconByType (SND,/icons/sound2.gif) audio/*\n",
    "	AddIconByType (VID,/icons/movie.gif) video/*\n",

    "	AddIcon /icons/binary.gif .bin .exe\n",
    "	AddIcon /icons/binhex.gif .hqx\n",
    "	AddIcon /icons/tar.gif .tar\n",
    "	AddIcon /icons/world2.gif .wrl .wrl.gz .vrml .vrm .iv\n",
    "	AddIcon /icons/compressed.gif .Z .z .tgz .gz .zip\n",
    "	AddIcon /icons/a.gif .ps .ai .eps\n",
    "	AddIcon /icons/layout.gif .html .shtml .htm .pdf\n",
    "	AddIcon /icons/text.gif .txt\n",
    "	AddIcon /icons/c.gif .c\n",
    "	AddIcon /icons/p.gif .pl .py\n",
    "	AddIcon /icons/f.gif .for\n",
    "	AddIcon /icons/dvi.gif .dvi\n",
    "	AddIcon /icons/uuencoded.gif .uu\n",
    "	AddIcon /icons/script.gif .conf .sh .shar .csh .ksh .tcl\n",
    "	AddIcon /icons/tex.gif .tex\n",
    "	AddIcon /icons/bomb.gif core\n",
    "	AddIcon /icons/deb.gif .deb\n",
    "	AddIcon /icons/back.gif ..\n",
    "	AddIcon /icons/hand.right.gif README\n",
    "	AddIcon /icons/folder.gif ^^DIRECTORY^^\n",
    "	AddIcon /icons/blank.gif ^^BLANKICON^^\n",
    "	DefaultIcon /icons/unknown.gif\n\n",

    "	AddDescription \"GZIP compressed document\" .gz\n",
    "	AddDescription \"tar archive\" .tar\n",
    "	AddDescription \"GZIP compressed tar archive\" .tgz\n",

    "	ReadmeName README.html\n",
    "	HeaderName HEADER.html\n",
    "	IndexIgnore .??* *~ *# HEADER.html HEADER.txt RCS CVS *,v *,t\n",
    "</IfModule>\n\n",

    "<IfModule mod_mime.c>\n",
    "	AddEncoding x-compress Z\n",
    "	AddEncoding x-gzip gz tgz\n",
    "	AddLanguage da .dk\n",
    "	AddLanguage nl .nl\n",
    "	AddLanguage en .en\n",
    "	AddLanguage et .ee\n",
    "	AddLanguage fr .fr\n",
    "	AddLanguage de .de\n",
    "	AddLanguage el .el\n",
    "	AddLanguage it .it\n",
    "	AddLanguage ja .ja\n",
    "	AddCharset ISO-2022-JP .jis\n",
    "	AddLanguage pl .po\n",
    "	AddCharset ISO-8859-2 .iso-pl\n",
    "	AddLanguage pt .pt\n",
    "	AddLanguage pt-br .pt-br\n",
    "	AddLanguage lb .lu\n",
    "	AddLanguage ca .ca\n",
    "	AddLanguage es .es\n",
    "	AddLanguage sv .se\n",
    "	AddLanguage cs .cz\n\n",

    "	<IfModule mod_negotiation.c>\n",
    "	    LanguagePriority en da nl et fr de el it ja pl pt pt-br lb ca es sv\n",
    "	</IfModule>\n",

    "	AddType application/x-httpd-php .php\n",
    "	AddType application/x-httpd-php-source .phps\n",
    "	AddType application/x-tar .tgz\n",
    "	AddType image/bmp .bmp\n",
    "	AddType text/x-hdml .hdml\n",
    "	AddHandler cgi-script .cgi .sh .pl\n\n",
	
    "	<IfModule mod_include.c>\n",
    "	 AddType text/html .shtml\n",
    "	 AddHandler server-parsed .shtml\n",
    "	</IfModule>\n\n",

    "	AddHandler type-map var\n",
    "</IfModule>\n",

    "<IfModule mod_setenvif.c>\n",
    "	BrowserMatch \"Mozilla/2\" nokeepalive\n",
    "	BrowserMatch \"MSIE 4\\.0b2;\" nokeepalive downgrade-1.0 force-response-1.0\n",
    "	BrowserMatch \"RealPlayer 4\\.0\" force-response-1.0\n",
    "	BrowserMatch \"Java/1\\.0\" force-response-1.0\n",
    "	BrowserMatch \"JDK/1\\.0\" force-response-1.0\n",
    "	BrowserMatch \"Microsoft Data Access Internet Publishing Provider\" redirect-carefully\n",
    "	BrowserMatch \"MS FrontPage\" redirect-carefully\n",
    "	BrowserMatch \"^WebDrive\" redirect-carefully\n",
    "	BrowserMatch \"^WebDAVFS/1.[0123]\" redirect-carefully\n",
    "	BrowserMatch \"^gnome-vfs/1.0\" redirect-carefully\n",
    "	BrowserMatch \"^XML Spy\" redirect-carefully\n",
    "	BrowserMatch \"^Dreamweaver-WebDAV-SCM1\" redirect-carefully\n",
    "</IfModule>\n\n",

    "<IfModule mod_perl.c>\n",
    "  <IfModule mod_alias.c>\n",
    "     Alias /perl/ /var/www/perl/\n",
    "  </IfModule>\n",
    "  <Location /perl>\n",
    "     SetHandler perl-script\n",
    "     PerlHandler Apache::Registry\n",
    "     Options +ExecCGI\n",
    "  </Location>\n",
    "</IfModule>\n\n",

    "<IfModule mod_alias.c>\n",
    "	Alias /doc/ /usr/share/doc/\n",
    "</IfModule>\n\n",

    "<Location /doc>\n",
    "	order deny,allow\n",
    "	deny from all\n",
    "	allow from 127.0.0.0/255.0.0.0\n",
    "	Options Indexes FollowSymLinks MultiViews\n",
    "</Location>\n",

    "<IfModule mod_proxy.c>\n",
    "	ProxyRequests Off\n",
    "	ProxyVia Off\n",

    "	<Directory proxy:*>\n",
    "      Order deny,allow\n",
    "      Deny from all\n",
    "      Allow from .your_domain.com\n",
    "	</Directory>\n",

    "	CacheRoot \"/var/cache/apache\"\n",
    "	CacheSize 5\n",
    "	CacheGcInterval 4\n",
    "	CacheMaxExpire 24\n",
    "	CacheLastModifiedFactor 0.1\n",
    "	CacheDefaultExpire 1\n",
    "	NoCache a_domain.com another_domain.edu joes.garage_sale.com\n",
    "</IfModule>\n",
    NULL);
    /* End Default configuration for apache httpd 1x and Debian */
}

    fputs(conf, fp);
    fclose(fp);

    g_free(conf);

    if( widgets->default_conf_question_window != NULL )
      gtk_widget_destroy(widgets->default_conf_question_window);

    populate_gadmin_httpd(widgets);
}


/* This should probably also write a ~"dont_ask_again=1" to the settings file */
void dont_add_standard_conf(struct w *widgets)
{
    gtk_widget_destroy(widgets->default_conf_question_window);
    populate_gadmin_httpd(widgets);
}


/* If how is '1' then only merge include files */
void create_standard_conf_question(struct w *widgets, int how)
{
  GtkWidget *vbox18, *label182;
  GtkWidget *scrolledwindow18;
  GtkWidget *default_question_textview;
  GtkWidget *hbuttonbox11;
  GtkWidget *yes_default_question_button;
  GtkWidget *alignment44, *hbox98;
  GtkWidget *image44, *label184;
  GtkWidget *no_default_question_button;
  GtkWidget *alignment45, *hbox99;
  GtkWidget *image45, *label185;

  widgets->default_conf_question_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_size_request(widgets->default_conf_question_window, 500, 340);
  gtk_window_set_title(GTK_WINDOW(widgets->default_conf_question_window), _("GADMIN-HTTPD Question"));
  gtk_window_set_position(GTK_WINDOW(widgets->default_conf_question_window), GTK_WIN_POS_CENTER);

  vbox18 = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(widgets->default_conf_question_window), vbox18);

  label182 = gtk_label_new ("");
  gtk_box_pack_start(GTK_BOX(vbox18), label182, FALSE, FALSE, 0);
  gtk_widget_set_size_request(label182, -1, 20);
  gtk_label_set_justify(GTK_LABEL(label182), GTK_JUSTIFY_LEFT);

  scrolledwindow18 = gtk_scrolled_window_new(NULL, NULL);
  gtk_box_pack_start(GTK_BOX(vbox18), scrolledwindow18, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow18), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  default_question_textview = gtk_text_view_new();
  gtk_container_add(GTK_CONTAINER(scrolledwindow18), default_question_textview);
  gtk_widget_set_size_request(default_question_textview, 156, 260);
  gtk_text_view_set_editable(GTK_TEXT_VIEW(default_question_textview), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(default_question_textview), FALSE);
  gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(default_question_textview), 3);
  gtk_text_view_set_left_margin(GTK_TEXT_VIEW(default_question_textview), 30);

  if( how == 1 )
  {
     gtk_text_buffer_set_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(default_question_textview)),
     _("GAdmin HTTPD needs to merge all include files into\nthe main configuration file.\n\nDo you want to do this ?\n\n\t\t\t(If you dont know then press yes)\n"), -1);
  }
  else
  {
     gtk_text_buffer_set_text(gtk_text_view_get_buffer(GTK_TEXT_VIEW(default_question_textview)),
     _("GAdmin HTTPD has detected that your current configuration file\nis lacking required options or is missing.\n\nDo you want to write a new configuration ?\n\n\t\t\t(If you dont know then press yes)\n"), -1);
  }
  hbuttonbox11 = gtk_hbutton_box_new();
  gtk_box_pack_start(GTK_BOX(vbox18), hbuttonbox11, FALSE, FALSE, 0);
  gtk_widget_set_size_request(hbuttonbox11, -1, 40);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox11), GTK_BUTTONBOX_SPREAD);

  yes_default_question_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(hbuttonbox11), yes_default_question_button);
  GTK_WIDGET_SET_FLAGS(yes_default_question_button, GTK_CAN_DEFAULT);

  alignment44 = gtk_alignment_new(0.5, 0.5, 0, 0);
  gtk_container_add(GTK_CONTAINER(yes_default_question_button), alignment44);

  hbox98 = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(alignment44), hbox98);

  image44 = gtk_image_new_from_stock("gtk-yes", GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(hbox98), image44, FALSE, FALSE, 0);

  label184 = gtk_label_new_with_mnemonic(_("Yes"));
  gtk_box_pack_start(GTK_BOX(hbox98), label184, FALSE, FALSE, 0);
  gtk_label_set_justify(GTK_LABEL(label184), GTK_JUSTIFY_LEFT);

  no_default_question_button = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(hbuttonbox11), no_default_question_button);
  GTK_WIDGET_SET_FLAGS(no_default_question_button, GTK_CAN_DEFAULT);

  alignment45 = gtk_alignment_new(0.5, 0.5, 0, 0);
  gtk_container_add(GTK_CONTAINER(no_default_question_button), alignment45);

  hbox99 = gtk_hbox_new(FALSE, 2);
  gtk_container_add(GTK_CONTAINER(alignment45), hbox99);

  image45 = gtk_image_new_from_stock("gtk-no", GTK_ICON_SIZE_BUTTON);
  gtk_box_pack_start(GTK_BOX(hbox99), image45, FALSE, FALSE, 0);

  label185 = gtk_label_new_with_mnemonic(_("No"));
  gtk_box_pack_start(GTK_BOX(hbox99), label185, FALSE, FALSE, 0);
  gtk_label_set_justify(GTK_LABEL(label185), GTK_JUSTIFY_LEFT);

  if( how == 1 )
  {
     g_signal_connect_swapped(G_OBJECT(yes_default_question_button), "clicked",
                              G_CALLBACK(merge_include_files), widgets);
  }
  else
  {
     g_signal_connect_swapped(G_OBJECT(yes_default_question_button), "clicked",
                              G_CALLBACK(add_standard_conf), widgets);
  }

  g_signal_connect_swapped(G_OBJECT(no_default_question_button), "clicked",
                           G_CALLBACK(dont_add_standard_conf), widgets);

  gtk_widget_show_all(widgets->default_conf_question_window);
}
