/* GADMIN-HTTPD - An easy to use GTK+ frontend for Apache HTTPD webserver.
 * Copyright (C) 2006 - 2011 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <gtk/gtk.h>
#include "gettext.h"
#include "support.h"
#include "create_main_window.h"
#include "gadmin_httpd.h"
#include "widgets.h"
#include "commands.h"
#include "functions.h"
#include "status_update.h"
#include "set_version.h"
#include "standard_conf.h"

/* For getuid */
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>

#include "create_server_tab.h"
#include "create_server_settings.h"
#include "create_module_tab.h"
#include "create_htaccess_tab.h"
#include "create_transfer_tab.h"
#include "create_disc_tab.h"
#include "create_access_tab.h"
#include "create_error_tab.h"
#include "create_conf_tab.h"
#include "add_missing_opts.h"
#include "populate_gadmin_httpd.h"
#include "show_info.h"

int MAX_READ_POPEN = 16384;
int activated = 0;
int use_tls = 0;

int info_window_exit_main = 0;
int httpd_mayor_version = 0;

char global_server_address[1024]="";
char global_server_port[1024]="";
char global_server_name[1024]="";
char global_server_type[1024]="";
char global_virtual_user_name[1024]="";
char global_htacc_user_name[1024]="";



int main(int argc, char *argv[])
{
    int i = 0;
    gchar *cmd, *opt, *path, *info;

#ifdef ENABLE_NLS
    bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
    bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    textdomain(GETTEXT_PACKAGE);
#endif

    gtk_set_locale();
    gtk_init(&argc, &argv);

    wid* widgets = g_malloc(sizeof(wid));

    widgets->default_conf_question_window = NULL;

    /* Non root usage */
    if( ! getuid() == 0 )
    {
        /* For setting a different exit method in the info window */
        info_window_exit_main = 1;

        info = g_strdup_printf("You must be root to run: %s\nThis window will close in 10 seconds.\n", PACKAGE);
        show_info(info);
        g_free(info);

        /* Users can close the info window earlier then the timeout */
        for(i=0; i<10; i++)
        {
            while(gtk_events_pending())
                  gtk_main_iteration();

            /* Set when close info window is clicked */
            if( info_window_exit_main == 2 )
                break;

	            usleep(100000*10);
        }

        g_free(widgets);
        return 0;
    }

    
    /* Create the main window */
    create_main_window(widgets);
    
    /* Set HTTPD Version */
    set_version(widgets);

    /* If httpd has the mod_tls module we 
       create, show and use the TLS widgets */
    use_tls = using_tls();

    /* Create the server tab */
    create_server_tab(widgets);

    /* Create the server settings */
    create_server_settings(widgets);

    /* Create the module tab */
    create_module_tab(widgets);

    /* Backup the httpd configuration */
    backup_httpd_conf(widgets);

    /* Create the global passwords file and directory if they dont exist */
    path = g_strdup_printf("%s/passwords", HTTPD_ROOT_PATH);
    if( ! file_exists(path) )
      make_dir_chmod(path, "755");
    if( path!=NULL )
      g_free(path);

    path = g_strdup_printf("%s/passwords/passwords", HTTPD_ROOT_PATH);
    if( ! file_exists(path) )
    {
        cmd = g_strdup_printf("touch %s && chmod 600 %s && chown %s:%s %s", 
    				     path, path, SERVER_USER, SERVER_GROUP, path);
        if( ! run_command(cmd) )
            printf("Running command: %s\nhas failed.\n", cmd);
        if( cmd!=NULL )
            g_free(cmd);
    }
    if( path!=NULL )
        g_free(path);

    /* Create the htaccess tab (directory access) */
    create_htaccess_tab(widgets);

    /* Create the transfer tab */
    create_transfer_tab(widgets);

    /* Create the disc tab */
    create_disc_tab(widgets);

    /* Create the access tab */
    create_access_tab(widgets);

    /* Create the error tab */
    create_error_tab(widgets);

    /* Create the conf tab */
    create_conf_tab(widgets);


    path = g_strdup_printf("%s", HTTPD_CONF);

    /* If theres no httpd.conf, add one */
    if( ! file_exists(path) )
      add_standard_conf(widgets);
    else
    /* If httpd.conf is bad, ask to add a new one */
    if( ! conf_ok(path) )
      create_standard_conf_question(widgets, 0); /* 0 = new conf */
    else /* Include files needs to be merged with the main configuration */
    if( conf_needs_merge(path) )
      create_standard_conf_question(widgets, 1); /* 1 = merge include files */

    g_free(path);

    /* Add missing options */
    add_missing_opts(widgets);

    /* Add server-status directives if missing */
    if( ! have_line("<location /server-status>")
    &&  have_module("status_module")  )
    {
	    opt = g_strconcat("",
	    "<Location /server-status>\n",
	    "    SetHandler server-status\n",
	    "    Order allow,deny\n",
	    "    Allow from localhost localhost.localdomain 127.0.0.1\n",
	    "</Location>\n",
	    NULL);

	    add_module_data(opt);
	    g_free(opt);
    }

    /* The gui must be repopulated
       incase something wad added */
    populate_gadmin_httpd(widgets);

    /* Window close button (x) */   
    g_signal_connect(GTK_OBJECT(widgets->main_window), "destroy",
	             GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

    gtk_timeout_add(1000, (GtkFunction) status_update, widgets);

    gtk_main();

    g_free(widgets);


    return 0;    
}
