/* GADMIN-HTTPD - An easy to use GTK+ frontend for Apache HTTPD webserver.
 * Copyright (C) 2006 - 2011 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "reread_conf.h"
#include "show_info.h"
#include "functions.h"
#include "apply_server_settings.h"
#include "allocate.h"
#include "commented.h"
#include "get_option_pos.h"
#include "populate_servers.h"
#include "populate_server_settings.h"
#include "select_first_server.h"
#include "populate_module_tab.h"
#include "populate_conf_tab.h"


extern char global_server_address[1024];
extern char global_server_port[1024];
extern char global_server_name[1024];
extern char global_server_type[1024];

extern int use_tls;

extern int activated;

/* The module type to add to the modules_buffer */
char module_type[1024]="";
char *modules_buffer;


int add_module_to_buffer(GtkTreeModel *model, GtkTreePath *path,
	        	      GtkTreeIter *iter, struct w *widgets)
{
    gchar *conf_line;
    gchar *option1, *option2;

    gtk_tree_model_get(model, iter, 0, &option1, -1);
    gtk_tree_model_get(model, iter, 1, &option2, -1);

    /* The input must be complete. */
    if( strlen(option1) < 2 || strlen(option2) < 2 )
    {
        printf("Missing value for module tab option [%s].\n", module_type);
        return FALSE;
    }

    conf_line = g_strdup_printf("%s %s %s\n", module_type, option1, option2);

    /* Add to global buffer */
    strcat(modules_buffer, conf_line);

    g_free(conf_line);
    g_free(option1);
    g_free(option2);

    /* Return false to continue iteration */
    return FALSE;
}

void apply_server_settings(struct w *widgets)
{
    /* Change the selected servers configuration. */
    FILE *fp;
    long file_size=0, old_pos=0, opt_pos=0;
    gint active_index;
    gchar *new_line, *address_line, *name_line;
    gchar *saved_vhost_line, *info;
    int listen_one_time = 0, vhost_found = 0;
    char *line, *tmp_line, *config;

    /* For the modules handling */
    GtkTreeModel *model;
    int add_modules_once = 0;

    G_CONST_RETURN gchar *server_name;
    G_CONST_RETURN gchar *server_address;
    G_CONST_RETURN gchar *server_port;
    G_CONST_RETURN gchar *admin_email;
    G_CONST_RETURN gchar *server_user;
    G_CONST_RETURN gchar *server_group;
    G_CONST_RETURN gchar *document_root;
    G_CONST_RETURN gchar *server_root;
    G_CONST_RETURN gchar *directory_index;
    G_CONST_RETURN gchar *default_doc_type;
    G_CONST_RETURN gchar *htaccess_file;
//    G_CONST_RETURN gchar *mimetype_file;
    G_CONST_RETURN gchar *default_language;
    G_CONST_RETURN gchar *language_priority;
    G_CONST_RETURN gchar *default_charset;

    G_CONST_RETURN gchar *connection_timeout;
    G_CONST_RETURN gchar *max_keepalive_requests;
    G_CONST_RETURN gchar *keepalive_timeout;

    /* TLS Entries */
//    G_CONST_RETURN gchar *tls_protocols = NULL;
//    G_CONST_RETURN gchar *tls_logfile = NULL;
//    G_CONST_RETURN gchar *tls_certfile = NULL;


    /* Get values from the entries */
    server_name = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[0]));
    server_address = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[1]));
    admin_email = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[2]));
    server_user  = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[3]));
    server_group = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[4]));
    document_root  = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[5]));
    server_root = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[6]));
    directory_index = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[7]));
    default_doc_type = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[8]));
    htaccess_file = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[9]));
    default_language = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[10]));
    language_priority = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[11]));
    default_charset = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[12]));

    /* Get values from the spinbuttons */
    server_port = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[0]));
    connection_timeout = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[1]));
    max_keepalive_requests = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[2]));
    keepalive_timeout = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[3]));


    /* Change the configuration for the selected server */
    if((fp=fopen(HTTPD_CONF, "r"))==NULL)
    {
        info = g_strdup_printf(_("Cant open httpd.conf here: \n%s\n"), HTTPD_CONF);
        show_info(info);
        g_free(info);
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    config = allocate(file_size+16384); /* Additional bytes */
    line = allocate(file_size+1);


    /* Put all modules into the global modules_buffer */
    modules_buffer = allocate(file_size+10); /* For added newlines */

    strcat(modules_buffer, "\n");

    /* Call foreach on the module treeview */
    strcpy(module_type, "LoadModule");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->module_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the language treeview */
    strcpy(module_type, "AddType");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addtype_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addhandler treeview */
    strcpy(module_type, "AddHandler");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addhandler_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addencoding treeview */
    strcpy(module_type, "AddEncoding");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addencoding_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addlanguage treeview */
    strcpy(module_type, "AddLanguage");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addlanguage_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the adddescription treeview */
    strcpy(module_type, "AddDescription");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->adddescription_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addicon treeview */
    strcpy(module_type, "AddIcon");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addicon_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addiconbytype treeview */
    strcpy(module_type, "AddIconByType");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addiconbytype_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the addiconbyencoding treeview */
    strcpy(module_type, "AddIconByEncoding");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->addiconbyencoding_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the alias treeview */
    strcpy(module_type, "Alias");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->alias_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    /* Call foreach on the alias treeview */
    strcpy(module_type, "ScriptAlias");
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(widgets->scriptalias_treeview));
    gtk_tree_model_foreach(GTK_TREE_MODEL(model),
                          (GtkTreeModelForeachFunc) &add_module_to_buffer, widgets);

    strcat(modules_buffer, "\n");

    /* All modules have been put into the modules_buffer */



/* ------------- Change the standard server ----------- */
    if( strstr("Default server", (char *)global_server_type) )
    {
	    if( file_size > 1 )
	    while(fgets(line, file_size, fp)!=NULL)
	    {
            if( commented(line) )
	            continue;

            if( strlen(line) > 4000 )
            {
	            info = g_strdup_printf(_("A line with over 4000 chars is not valid in: %s\n"), HTTPD_CONF);
	            show_info(info);
	            g_free(info);
	            free(line);
	            free(config);
	            free(modules_buffer);
	            fclose(fp);
	            return;
            }

	        /* End of the standard server, gather everything else */
	        /* Scroll past any virtual hosts */
            if( cmplowercase(line, "<virtualhost ") )
            {
	            strcat(config, line);
	            while(fgets(line, file_size, fp)!=NULL)
                {
                    /* This line will be gathered later
                       so break before catting it. */
        	        if( cmplowercase(line, "</virtualhost") )
                        break;

                    strcat(config, line);
                }
            }

            /* Remove old and add new modules languages Addicons etc */
            if( cmplowercase(line, "loadmodule ")
            ||  cmplowercase(line, "addtype ")
            ||  cmplowercase(line, "addhandler ")
            ||  cmplowercase(line, "addencoding ")
            ||  cmplowercase(line, "addlanguage ")
            ||  cmplowercase(line, "adddescription ")
            ||  cmplowercase(line, "addicon ")
            ||  cmplowercase(line, "addiconbytype ")
            ||  cmplowercase(line, "addiconbyencoding ")
            ||  cmplowercase(line, "alias ")
            ||  cmplowercase(line, "scriptalias ")      )
            {
                if( ! add_modules_once )
        	    {
                    add_modules_once = 1;
                    strcat(config, modules_buffer);
                    free(modules_buffer);
                }
                continue;
            }
            else /* Change AllowOverride so that .htaccess files work */
            if( cmplowercase(line, "allowoverride none") )
            {
                new_line = g_strdup_printf("    AllowOverride AuthConfig\n");
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "servername ") )
       	    {
                new_line = g_strdup_printf("ServerName %s\n", server_name);
                strcat(config, new_line);
                g_free(new_line);
            }
            else  /* Server address and port */
            if( cmplowercase(line, "listen ") )
            {
                /* Only change the first occurance of Listen with an address. */
                if( ! listen_one_time && cmplowercase(line, ":") )
                {
                    listen_one_time = 1;
            	    new_line = g_strdup_printf("Listen %s:%s\n", server_address, server_port);
            	    strcat(config, new_line);
                    g_free(new_line);
                }
                else
                {
                    /* Gather all other Listen lines. */
       	            strcat(config, line);
                }
            }
            else
            if( cmplowercase(line, "serversignature ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[0]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("ServerSignature On\n");
                else
                  new_line = g_strdup_printf("ServerSignature Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "serveradmin ") )
            {
       	        new_line = g_strdup_printf("ServerAdmin %s\n", admin_email);
       	        strcat(config, new_line);
       	        g_free(new_line);
            }
            else
            if( cmplowercase(line, "hostnamelookups ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[1]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("HostnameLookups On\n");
                else
                  new_line = g_strdup_printf("HostnameLookups Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "usecanonicalname ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[2]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("UseCanonicalName On\n");
                else
                  new_line = g_strdup_printf("UseCanonicalName Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "timeout ") && ! cmplowercase(line, "keepalivetimeout") )
            {
                new_line = g_strdup_printf("Timeout %s\n", connection_timeout);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "keepalive ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[3]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("KeepAlive On\n");
                else
                  new_line = g_strdup_printf("Keepalive Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "maxkeepaliverequests ") )
            {
                new_line = g_strdup_printf("MaxKeepAliveRequests %s\n", max_keepalive_requests);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "keepalivetimeout ") )
            {
                new_line = g_strdup_printf("KeepAliveTimeout %s\n", keepalive_timeout);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "user ") )
            {
                new_line = g_strdup_printf("User %s\n", server_user);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "group ") )
            {
                new_line = g_strdup_printf("Group %s\n", server_group);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "servertokens ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[4]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("ServerTokens OS\n");
                else
                  new_line = g_strdup_printf("ServerTokens Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "extendedstatus ") )
            {
                /* It must be set to On to get a client lising for the transfer tab */
                new_line = g_strdup_printf("ExtendedStatus On\n");
        		strcat(config, new_line);
        		g_free(new_line);
            }
            else
            if( cmplowercase(line, "documentroot ") )
            {
                new_line = g_strdup_printf("DocumentRoot \"%s\"\n", document_root);
                strcat(config, new_line);
                g_free(new_line);

                /* Create new document root directories */
                if( document_root != NULL
                &&  ! file_exists((char *)document_root)
                &&  strlen(document_root) > 6 )
                {
                    make_dir_chmod((gchar *)document_root, "0755");
                }
            }
            else
            if( cmplowercase(line, "serverroot ") )
            {
                new_line = g_strdup_printf("ServerRoot \"%s\"\n", server_root);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "directoryindex ") )
            {
                new_line = g_strdup_printf("DirectoryIndex %s\n", directory_index);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "defaulttype ") )
            {
                new_line = g_strdup_printf("DefaultType %s\n", default_doc_type);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "accessfilename ") )
            {
                new_line = g_strdup_printf("AccessFileName %s\n", htaccess_file);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "defaultlanguage ") )
            {
                new_line = g_strdup_printf("DefaultLanguage %s\n", default_language);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "languagepriority ")
            &&  ! cmplowercase(line, "forcelanguagepriority") )
            {
                new_line = g_strdup_printf("LanguagePriority %s\n", language_priority);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "adddefaultcharset ") )
            {
                new_line = g_strdup_printf("AddDefaultCharset %s\n", default_charset);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
        	      strcat(config, line); /* Gather all other lines */
        }
       	free(line);
       	fclose(fp);

        if( ! add_modules_once )
            free(modules_buffer);

        /* Save the standard server */
        if((fp=fopen(HTTPD_CONF, "w+"))==NULL)
        {
            free(config);
            return;
        }
        fputs(config, fp);
        fclose(fp);
        free(config);

	    /* Only repopulate the server list if its
        	   name, address or port has changed      */
	    if( strcmp(server_address, (char *)global_server_address)
	    ||  strcmp(server_name, (char *)global_server_name) 
	    ||  strcmp(server_port, (char *)global_server_port) )
	    {
	        /* Populate the server treeview */
	        populate_servers(widgets);

	        /* Select the first server and set global server values */
	        select_first_server(widgets);
	    }

        /* Populate the server settings */
        populate_server_settings(widgets);

        /* Populate modules */
        populate_module_tab(widgets);

        populate_conf_tab(widgets);

        reread_conf(widgets);

        return;
    }
/* ----------------- End of default server changes ------------------ */



/* ------------ Change a virtualhost --------------- */
    if( strstr("Virtualhost", (char *)global_server_type) )
    {
        /* Setup the address:port and ServerName lines to identify the server with */
        address_line = g_strdup_printf("<VirtualHost %s:%s>\n", global_server_address, global_server_port);
        name_line = g_strdup_printf("ServerName %s\n", global_server_name);

        tmp_line = allocate(1024);

        /* Scan for the vhost address and server name */
        if( file_size > 1 )
        while(fgets(line, file_size, fp)!=NULL)
        {
    	    if( commented(line) )
    	        continue;

            opt_pos = get_option_pos(line, 0);
            snprintf(tmp_line, 1000, "%s", &line[opt_pos]);
        
            if( strcmp(tmp_line, address_line) == 0 )
            {
                /* We have a matching vhost line, save its position */
                old_pos = ftell(fp);
                saved_vhost_line = g_strdup_printf("%s", line);

                while(fgets(line, file_size, fp)!=NULL)
                {
                    if( commented(line) )
                        continue;

                    opt_pos = get_option_pos(line, 0);
                    snprintf(tmp_line, 1000, "%s", &line[opt_pos]);

                    /* If the vhost has no ServerName then its found */
                    if( strlen((char *)global_server_name) < 5
                    &&  cmplowercase((char *)global_server_name, "none") )
                    {
                        vhost_found = 1;
                        break;
                    }

                    /* If it has "ServerName" and it matches the selected
                       server name then the vhost is found */
                    if( strcmp(tmp_line, name_line) == 0 )
                    {
                        /* IE: ServerName www.example.org found */
                        vhost_found = 1;
                        break;
                    }
		
                    if( cmplowercase(line, "</virtualhost>") )
                        break;
                }

                /* Scroll back after testing for a match */
                fseek(fp, old_pos, SEEK_SET);

                if( vhost_found )
                {
                    g_free(saved_vhost_line);
                    break;
                }
                else
                {
                    /* Put back the saved <VirtualHost line */
                    strcat(config, "\n");
                    strcat(config, saved_vhost_line);
                    g_free(saved_vhost_line);
                }

                /* Not the correct vhost, scroll back and collect it */
                if( ! vhost_found )
                {
                    while(fgets(line, file_size, fp)!=NULL)
                    {
                        if( commented(line) )
                            continue;

                        strcat(config, line);

                        if( cmplowercase(line, "</virtualhost") )
                            break;
                    }
                }
            }
            else /* Gather the other lines if they have any content */
            if( strlen(line) > 3 )
                strcat(config, line);
        }
        g_free(address_line);
        g_free(name_line);
        free(tmp_line);

        /* Couldnt find the server */
        if( ! vhost_found )
        {
            printf("The virtualhost was not found, no changes where made\n");
            free(line);
            free(config);
            free(modules_buffer);
            fclose(fp);
            return;
        }

        /* Add the <VirtualHost line for the changed virtual host */
        new_line = g_strdup_printf("\n<VirtualHost %s:%s>\n", server_address, server_port);
        strcat(config, new_line);
        g_free(new_line);

        /* The vhost has been found, change it */
        if( file_size > 1 )
        while(fgets(line, file_size, fp)!=NULL)
        {
            if( commented(line) )
              continue;

            /* Remove old and add new modules languages Addicons etc */
            if( cmplowercase(line, "loadmodule ")
            ||  cmplowercase(line, "addtype ")
            ||  cmplowercase(line, "addhandler ")
            ||  cmplowercase(line, "addencoding ")
            ||  cmplowercase(line, "addlanguage ")
            ||  cmplowercase(line, "adddescription ")
            ||  cmplowercase(line, "addicon ")
            ||  cmplowercase(line, "addiconbytype ")
            ||  cmplowercase(line, "addiconbyencoding ")
            ||  cmplowercase(line, "alias ")
            ||  cmplowercase(line, "scriptalias ")      )
            {
                if( ! add_modules_once )
                {
                    add_modules_once = 1;
                    strcat(config, modules_buffer);
                    free(modules_buffer);
                }
                continue;
            }
            else
            if( cmplowercase(line, "servername ") )
       	    {
                new_line = g_strdup_printf("  ServerName %s\n", server_name);
                strcat(config, new_line);
                g_free(new_line);
            }
            else  /* Server address and port */
            if( cmplowercase(line, "listen ") )
       	    {
                new_line = g_strdup_printf("  Listen %s:%s\n", server_address, server_port);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "  serversignature ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[0]));
                if( active_index == 0 )
                    new_line = g_strdup_printf("  ServerSignature On\n");
                else
                    new_line = g_strdup_printf("  ServerSignature Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "serveradmin ") )
	        {
       	        new_line = g_strdup_printf("  ServerAdmin %s\n", admin_email);
       	        strcat(config, new_line);
       	        g_free(new_line);
            }
            else
            if( cmplowercase(line, "hostnamelookups ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[1]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("  HostnameLookups On\n");
                else
                  new_line = g_strdup_printf("  HostnameLookups Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "usecanonicalname ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[2]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("  UseCanonicalName On\n");
                else
                  new_line = g_strdup_printf("  UseCanonicalName Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "timeout ") && ! cmplowercase(line, "keepalivetimeout") )
            {
                new_line = g_strdup_printf("Timeout %s\n", connection_timeout);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "keepalive ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[3]));
                if( active_index == 0 )
                  new_line = g_strdup_printf("  KeepAlive On\n");
                else
                  new_line = g_strdup_printf("  Keepalive Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "maxkeepaliverequests ") )
            {
                new_line = g_strdup_printf("MaxKeepAliveRequests %s\n", max_keepalive_requests);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "keepalivetimeout ") )
            {
                new_line = g_strdup_printf("  KeepAliveTimeout %s\n", keepalive_timeout);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "user ") )
            {
                new_line = g_strdup_printf("  User %s\n", server_user);
                strcat(config, new_line);
                g_free(new_line);
            }
            else  /* User and group cant be in a vhost */
            if( cmplowercase(line, "group ") )
            {
                new_line = g_strdup_printf("  Group %s\n", server_group);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "servertokens ") )
            {
                active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[4]));
                if( active_index == 0 )
                    new_line = g_strdup_printf("  ServerTokens OS\n");
                else
                    new_line = g_strdup_printf("  ServerTokens Off\n");

                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "extendedstatus ") )
            {
                /* Always set extended status to On so that server-status can be accessed */
                new_line = g_strdup_printf("  ExtendedStatus On\n");
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "documentroot ") )
            {
                new_line = g_strdup_printf("  DocumentRoot \"%s\"\n", document_root);
                strcat(config, new_line);
                g_free(new_line);

                /* Create new document root directories */
                if( document_root != NULL
                &&  ! file_exists((char *)document_root)
                &&  strlen(document_root) > 6 )
                {
                    make_dir_chmod((gchar *)document_root, "0755");
                }
            }
            else
            if( cmplowercase(line, "serverroot ") )
            {
                new_line = g_strdup_printf("  ServerRoot \"%s\"\n", server_root);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "directoryindex ") )
            {
                new_line = g_strdup_printf("  DirectoryIndex %s\n", directory_index);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "defaulttype ") )
            {
                new_line = g_strdup_printf("  DefaultType %s\n", default_doc_type);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "accessfilename ") )
            {
                new_line = g_strdup_printf("  AccessFileName %s\n", htaccess_file);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "defaultlanguage ") )
            {
                new_line = g_strdup_printf("  DefaultLanguage %s\n", default_language);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "languagepriority ")
            && ! cmplowercase(line, "forcelanguagepriority") )
            {
                new_line = g_strdup_printf("  LanguagePriority %s\n", language_priority);
                strcat(config, new_line);
                g_free(new_line);
            }
            else
            if( cmplowercase(line, "adddefaultcharset ") )
            {
                new_line = g_strdup_printf("  AddDefaultCharset %s\n", default_charset);
                strcat(config, new_line);
                g_free(new_line);
            }
            else  /* Break at end of vhost */
            if( cmplowercase(line, "</virtualhost") )
            {
                strcat(config, "</VirtualHost>\n");
                break;
            }
            else
            if( strlen(line) > 3 )
            {
                strcat(config, line); /* Gather all other lines */
            }
        }

        /* Collect any other values */
        if( file_size > 1 )
        while(fgets(line, file_size, fp)!=NULL)
        {
            if( commented(line) )
                continue;

            strcat(config, line);
        }

        free(line);
        fclose(fp);
    }   /* If it was a vhost */


    if( ! add_modules_once )
        free(modules_buffer);

    /* Write the changes */
    if((fp=fopen(HTTPD_CONF, "w+"))==NULL)
    {
        free(config);
        return;
    }
    fputs(config, fp);
    fclose(fp);
    free(config);

    /* Only repopulate the server list if its
       name, address or port has changed      */
    if( strcmp(server_address, (char *)global_server_address)
    ||  strcmp(server_name, (char *)global_server_name) 
    ||  strcmp(server_port, (char *)global_server_port) )
    {
        /* Populate the server treeview */
        populate_servers(widgets);

        /* Select the first server and set global server values */
        select_first_server(widgets);
    }

    /* Populate the server settings */
    populate_server_settings(widgets);

    /* Populate modules */
    populate_module_tab(widgets);

    populate_conf_tab(widgets);

    if( activated )
        reread_conf(widgets);
}
