/* GADMIN-DHCPD - An easy to use GTK+ frontend for ISC DHCPD.
 * Copyright (C) 2004 - 2011 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "widgets.h"
#include "gettext.h"
#include "show_info.h"
#include "allocate.h"
#include "commented.h"

extern char LEASE_FILE_BUF[1024];

#define MAX_LEASE_LINE 1024

/* Only the last occurance of a host declaration in
   dhcpd.leases should be listed in the treeview */

struct node {
    char *line;
    struct node *next;
};

struct node *add_node(struct node *list, char *line);
int node_exists(struct node *list, char *line);
void free_list(struct node *list);


struct node *add_node(struct node *list, char *line)
{
    struct node *ptr = (struct node *)malloc(sizeof(struct node));
    /* Add line */
    ptr->line = strdup(line);
    ptr->next = list;

    return ptr;
}

int node_exists(struct node *list, char *line)
{
    struct node *current = list;
    int ret = 0;

    while( current != NULL )
    {
	    if( strcmp(current->line, line) == 0 )
	    {
	        ret = 1;
	        break;
	    }
	    current = current->next;
    }

    return ret;
}

void free_list(struct node *list)
{
    struct node *current = list;
  
    while( current != NULL )
    {
	    /* Free ip's */
	    if( current->line != NULL )
	        free(current->line);

	    current = current->next;
    }
    free(list);
}

void add_to_treeview(struct w *widgets, char *line)
{
    int i = 0, col = 0, insert = 1;
    gchar *utf8;
    char *buf = allocate(strlen(line)+100);

    /* Snip some unwanted stuff */
    for(i=0; i<strlen(line); i++)
        if( line[i]=='\n' || line[i]==';' )
            line[i]='\0';

    /* Binding state */
    if( strstr(line, "binding state ") && ! strstr(line, "next ") )
    {
        col = 0;
	    if( strstr(line, "free") )
	        strcpy(buf, _("Free"));
	    if( strstr(line, "active") )
	        strcpy(buf, _("Active"));
    }
    else
    /* IP address */
    if( strstr(line, "lease ") && strstr(line, "{") )
    {
        col = 1;
	    sscanf(line, "%*s %s", buf);
    }
    else
    /* Hardware type and address */
    if( strstr(line, "hardware ") )
    {
        /* Hardware type */
        col = 2;
	    sscanf(line, "%*s %s", buf);
        utf8 = g_locale_to_utf8(buf, strlen(buf), NULL, NULL, NULL);
        gtk_list_store_set(GTK_LIST_STORE(widgets->lease_store), &widgets->lease_iter, col, utf8, -1);
        if( utf8!=NULL )
            g_free(utf8);
        /* Hardware address */
        col = 3;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* Client-hostname */
    if( strstr(line, "client-hostname ") )
    {
        col = 4;
	    sscanf(line, "%*s %s", buf);
	    /* Snip the first and last chars: "Hostname" */
	    buf[0]=' ';
	    buf[strlen(buf)-1]='\0';
    }
    else
    /* Starts */
    if( strstr(line, "starts ") )
    {
        col = 5;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* Ends */
    if( strstr(line, "ends ") )
    {
        col = 6;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* Next binding state */
    if( strstr(line, "next binding state ") )
    {
        col = 7;
	    if( strstr(line, "free") )
	        strcpy(buf, _("Free"));
	    if( strstr(line, "active") )
	        strcpy(buf, _("Active"));
    }
    else
    /* CLTT */
    if( strstr(line, "cltt ") )
    {
        col = 8;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* TSTP */
    if( strstr(line, "tstp ") )
    {
        col = 9;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* TSFP */
    if( strstr(line, "tsfp ") )
    {
        col = 10;
	    sscanf(line, "%*s %*s %s", buf);
    }
    else
    /* UUID */
    if( strstr(line, "uuid ") )
    {
        col = 11;
	    sscanf(line, "%*s %s", buf);
    }
    else
    {
        insert = 0;
	    printf("Unhandled lease file option: %s\n", buf);
    }

    if( insert )
    {
        utf8 = g_locale_to_utf8(buf, strlen(buf), NULL, NULL, NULL);
        gtk_list_store_set(GTK_LIST_STORE(widgets->lease_store), &widgets->lease_iter, col, utf8, -1);
        if( utf8!=NULL )
            g_free(utf8);
    }
    if( buf!=NULL )
        free(buf);
}

void trim_end(char *line)
{
    if( line[strlen(line)-1]=='\n' )
        line[strlen(line)-1]='\0';
}

void populate_leases(struct w *widgets)
{
    FILE *fp;
    long file_size = 0;
    long file_pos  = 0;
    long orig_file_pos = 0;
    char *line, *buf;
    struct node *list = NULL;

    gtk_list_store_clear(widgets->lease_store);

    if((fp=fopen(LEASE_FILE_BUF, "r"))==NULL)
    {
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    buf  = allocate(MAX_LEASE_LINE+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( commented(line) || strlen(line) > MAX_LEASE_LINE )
	        continue;

        trim_end(line);

	    /* Continue until we find a lease. */
        if( ! (strstr(line, "lease ") && strstr(line, "{")) )
            continue;

        /* Dont add duplicate nodes */
        if( node_exists(list, line) )
            continue;

	    /* Record file pos */
        orig_file_pos = ftell(fp);
	    file_pos = ftell(fp);
	
	    snprintf(buf, MAX_LEASE_LINE, "%s", line);

	    /* Search for buf... */	    
	    while(fgets(line, file_size, fp)!=NULL)
	    {
		    if( commented(line) || strlen(line) > MAX_LEASE_LINE )
		        continue;

            trim_end(line);
            trim_end(buf);

		    /* Last lease for this host will be found here */
		    if( strcmp(line, buf) == 0 )
		    {
		        file_pos = ftell(fp);
		    }
        }

        /* We are at the last lease for this host.
           Use -1 because newlines are trimmed. */
		fseek(fp, file_pos-strlen(buf)-1, SEEK_SET);
	    
		gtk_list_store_append(GTK_LIST_STORE(widgets->lease_store), &widgets->lease_iter);

	    /* Parse and insert selected lease data */
        while(fgets(line, file_size, fp)!=NULL)
	    {
		    if( commented(line) || strlen(line) > MAX_LEASE_LINE )
		        continue;

            trim_end(line);

            if( strlen(line) > 1 )
	            add_to_treeview(widgets, line);

		    if( strstr(line, "lease ") && strstr(line, "{") )
            {
                /* Record what leases we have added */
                list = add_node(list, line);
            }
            /* Lease end */
		    if( line[0]=='}' )
		        break;
        }
	    /* Rewind to file_pos and scan for the next lease */
	    fseek(fp, orig_file_pos, SEEK_SET);
    }
    fclose(fp);
    free(line);
    free(buf);
    free_list(list);
}

/* Show lease info not shown in the treeview */
void show_extended_leases(struct w *widgets)
{
    FILE *fp;
    char *line, *buf;
    long file_size = 0;

    if((fp=fopen(LEASE_FILE_BUF, "r"))==NULL)
    {
        return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size+1);
    buf  = allocate(file_size+1);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( commented(line) )
            continue;

        /* Skip host leases */
        if( strstr(line, "lease ") && strstr(line, "{") )
	{
	    while(fgets(line, file_size, fp)!=NULL)
	    {
        	if( line[0]=='}' )
            	    break;    
    	    }
        }
        else
        if( strlen(line) > 1 )
            strcat(buf, line);
    }
    show_info(buf);
    free(buf);
}

