/*
 * File:    cloopreader.c 
 * Purpose: cloop file reader for fusecloop.
 * Copyright (c)  2007 Vitaly  "_Vi" Shukela
 * Copyright 1999-2003 by Paul `Rusty' Russell & Klaus Knopper.
 *
 * Contact Email: public_vi@tut.by
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "debug.h"
#include "cloopreader.h"

int read_all(int fh, void* block, size_t size){
    dfuncinfo("fh=%d block=0x%lx size=0x%lx",
	    fh,(ulong)block,(ulong)size);
    char* bl=(char*)block;
    int ret;
    for(;size;){
	for(;;){
	    dprintf("invoking read(%d,0x%lx,0x%lx)\n",
		    fh,(ulong)bl,(ulong)size);
	    ret=read(fh,bl,size);
	    dprintf("    returned %d, errno=%d\n",ret,errno);
	    if(ret==-1&&errno==EINTR)continue;
	    if(ret==-1||ret==0)return -1;
	    break;
	}
	size-=ret;
        bl+=ret;
    }
    return 0;
}

int cloop_init(struct cloop_data *c, int fh){
    dfuncinfo("fh=%d",fh);
    c->fh=fh;
    struct cloop_head head;
    OP(read_all(c->fh,&head,sizeof head));   /* read Header */

    c->numblocks=ntohl(head.num_blocks);
    c->blocksize=ntohl(head.block_size);

    dprintf("block_size=%lx  num_blocks=%x\n", c->blocksize, c->numblocks);

    ALLOC(c->pblock,c->blocksize);

    c->tocsize=sizeof *c->toc * (c->numblocks+1); /* One extra address is position of EOF */
    ALLOC(c->toc,c->tocsize);

    OP(read_all(c->fh,c->toc,c->tocsize));  /* read Data Index */
    c->cblocksizecur=0;
    c->curblock=-1;
    return 0;
};

int cloop_swap(struct cloop_data *c,ulong page){
    dfuncinfo("page=0x%lx",page);
    ulong destlen; 

    if(page==c->curblock){
	dprintf("No use to switch page\n");
	return 0;
    }
    if(page>=c->numblocks){errno=EFAULT;return -1;}
    c->curblock=page;

    dprintf("Seeking to 0x%Lx\n",btc(c->toc[page]));
    OP(lseek(c->fh,btc(c->toc[page]), SEEK_SET)); 

    c->cblocksize=btc(c->toc[page+1]) - btc(c->toc[page]);
    dprintf("Compressed size=%lu\n",c->cblocksize);
    if(c->cblocksize > c->cblocksizecur){
	if(c->cblocksizecur)free(c->cblock);
	c->cblocksizecur=(c->cblocksize&~0xFF)+0x100;
	dprintf("allocating 0x%lx bytes\n",c->cblocksizecur);
	ALLOC(c->cblock,c->cblocksizecur);
	OPA(c->cblock,ZER);
    }

    OP(read_all(c->fh, c->cblock, c->cblocksize));

    destlen=c->blocksize;
    dprintf(
	    "pblock=0x%lx &destlen=0x%lx cblock=0x%lx cblocksize=%lu\n",
	    (ulong)c->pblock,(ulong)&destlen,(ulong)c->cblock,c->cblocksize
	    );
    switch(uncompress(c->pblock,&destlen,c->cblock,c->cblocksize)){
	case Z_OK: break;
	#define entry(x)\
	case x: dprintf( #x"\n"); break; 
	entry(Z_MEM_ERROR)
	entry(Z_BUF_ERROR)
	entry(Z_DATA_ERROR)
	#undef entry
	default: dprintf("Z_UNKNOWN_ERROR\n");
    }
    if(destlen!=c->blocksize)dprintf("Size mismatch\n");
    return 0;
}

int cloop_read(struct cloop_data* c, off_t offset,void* buf,ulong size){
    dfuncinfo("offset=0x%llx size=0x%lx buf=0x%lx",offset,size,(ulong)buf);
    int page;
    if(!size)return 0;
    page = offset / c->blocksize;
    if(page >= c->numblocks){
	// End of file;
	return 0;
    }
    offset %= c->blocksize;
    dprintf("page=%d offset=0x%llx size=0x%lx\n",page,offset,size);
    if(cloop_swap(c,page))return -1;
    dprintf("size=0x%lx c->blocksize=0x%lx\n",size,c->blocksize);
    if(offset+size > c->blocksize){
	size=c->blocksize-offset; /* chop to size of block */
	dprintf("size updated: size=0x%lx\n",size);
    }
    memcpy(buf,c->pblock+offset,size);
    return size;
}

int cloop_read_all(struct cloop_data* c, off_t offset,void* buf,ulong size){
    dfuncinfo("offset=0x%llx size=0x%lx buf=0x%lx sizeof(off_t)=%d",offset,size,(ulong)buf,sizeof(off_t));
    char* bl=(char*)buf;
    int ret;
    int siz=size;
    for(;siz;){
	ret=cloop_read(c,offset,bl,siz);
	if(ret==-1||ret==0)return -1;
	siz-=ret;
        bl+=ret;
	offset+=ret;
    }
    return size;
}

