const Ci = Components.interfaces;

const CLASS_ID = Components.ID("6224daa1-71a2-4d1a-ad90-01ca1c08e323");
const CLASS_NAME = "Simple AutoComplete";
const CONTRACT_ID = "@mozilla.org/autocomplete/search;1?name=simple-autocomplete";

// Implements nsIAutoCompleteResult
function SimpleAutoCompleteResult(searchString, searchResult,
                                  defaultIndex, errorDescription,
                                  results, storage) {
  this._searchString = searchString;
  this._searchResult = searchResult;
  this._defaultIndex = defaultIndex;
  this._errorDescription = errorDescription;
  this._results = results;
  this._storage = storage;
}

SimpleAutoCompleteResult.prototype = {
  _searchString: "",
  _searchResult: 0,
  _defaultIndex: 0,
  _errorDescription: "",
  _results: [],

  /**
   * The original search string
   */
  get searchString() {
    return this._searchString;
  },

  /**
   * The result code of this result object, either:
   *         RESULT_IGNORED   (invalid searchString)
   *         RESULT_FAILURE   (failure)
   *         RESULT_NOMATCH   (no matches found)
   *         RESULT_SUCCESS   (matches found)
   */
  get searchResult() {
    return this._searchResult;
  },

  /**
   * Index of the default item that should be entered if none is selected
   */
  get defaultIndex() {
    return this._defaultIndex;
  },

  /**
   * A string describing the cause of a search failure
   */
  get errorDescription() {
    return this._errorDescription;
  },

  /**
   * The number of matches
   */
  get matchCount() {
    return this._results.length;
  },

  /**
   * Get the value of the result at the given index
   */
  getValueAt: function(index) {
    return this._results[index];
  },

  /**
   * Get the comment of the result at the given index
   */
  getCommentAt: function(index) {
    return null;
  },

  /**
   * Get the style hint for the result at the given index
   */
  getStyleAt: function(index) {
    return "";
  },

  getImageAt: function(index) {
    return null;
  },

  /**
   * Remove the value at the given index from the autocomplete results.
   * If removeFromDb is set to true, the value should be removed from
   * persistent storage as well.
   */
  removeValueAt: function(index, removeFromDb) {
    if (removeFromDb) {
      this._storage.deleteAddress(this._results[index].id);
    }
    this._results.splice(index, 1);
  },

  QueryInterface: function(aIID) {
    if (!aIID.equals(Ci.nsIAutoCompleteResult) && !aIID.equals(Ci.nsISupports))
        throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

function Address(id, name, email) {
  this.id = id;
  this.name = name;
  this.email = email;

  this.toString = function() {
    return this.name + (this.name && this.email ? " " : "") +
           (this.email ? "<" + this.email + ">" : "");
  }
}

function Storage() {
  var file = getDataDirectory();
  file.append("simplemail.sqlite");
  var storageService = Components.classes["@mozilla.org/storage/service;1"]
                       .getService(Ci.mozIStorageService);
  var connection = storageService.openDatabase(file);

  function getDataDirectory() {
    var prefs = Components.classes["@mozilla.org/preferences-service;1"]
                .getService(Components.interfaces.nsIPrefService)
                .getBranch("extensions.simplemail.");
    var file;
    try {
      file = prefs.getComplexValue("dataDirectory",
                                   Components.interfaces.nsIRelativeFilePref).file;
    }
    catch(e) {
      file = prefs.getComplexValue("dataDirectory",
                                   Components.interfaces.nsILocalFile);
    }
    if (!file.exists()) file.create(Components.interfaces.nsIFile.DIRECTORY_TYPE, 0777);
    return file;
  }

  var text = "text: ";

  function bindString(statement, id, value) {
    if (value && !isNaN(Number(value))) value = text + value;
    statement.bindStringParameter(id, value);
  }

  function getString(statement, id) {
    var value = statement.getString(id);
    return value && value.substr(0, text.length) == text ? value.substr(text.length) : value;
  }

  this.findAddresses = function(text) {
    var sql = "SELECT id, name, email FROM addressBook "+
              "WHERE name LIKE ?1 OR email LIKE ?1 ORDER BY name, email";
    var statement = connection.createStatement(sql);
    try {
      var addresses = new Array();
      bindString(statement, 0, text.replace(/\s+|$/g, function($0) { return "%" + $0; }));
      while(statement.executeStep()) {
        addresses.push(new Address(statement.getInt32(0),
                                   getString(statement, 1),
                                   getString(statement, 2)));
      }
      return addresses;
    }
    finally {
      statement.reset();
    }
  }

  this.deleteAddress = function(id) {
    var statement = connection.createStatement("DELETE FROM addressBook WHERE id=?1");
    statement.bindInt32Parameter(0, id);
    statement.execute();
  }
}

// Implements nsIAutoCompleteSearch
function SimpleAutoCompleteSearch() {
  this.storage = new Storage();
}

SimpleAutoCompleteSearch.prototype = {
  /*
   * Search for a given string and notify a listener (either synchronously
   * or asynchronously) of the result
   *
   * @param searchString - The string to search for
   * @param searchParam - An extra parameter
   * @param previousResult - A previous result to use for faster searchinig
   * @param listener - A listener to notify when the search is complete
   */
  startSearch: function(searchString, searchParam, result, listener) {
    var addresses = this.storage.findAddresses(searchParam);
    var newResult = new SimpleAutoCompleteResult(searchString, Ci.nsIAutoCompleteResult.RESULT_SUCCESS,
                                                 0, "", addresses, this.storage);
    listener.onSearchResult(this, newResult);
  },

  /*
   * Stop an asynchronous search that is in progress
   */
  stopSearch: function() {
  },
    
  QueryInterface: function(aIID) {
    if (!aIID.equals(Ci.nsIAutoCompleteSearch) && !aIID.equals(Ci.nsISupports))
        throw Components.results.NS_ERROR_NO_INTERFACE;
    return this;
  }
};

// Factory
var SimpleAutoCompleteSearchFactory = {
  singleton: null,
  createInstance: function (aOuter, aIID) {
    if (aOuter != null)
      throw Components.results.NS_ERROR_NO_AGGREGATION;
    if (this.singleton == null)
      this.singleton = new SimpleAutoCompleteSearch();
    return this.singleton.QueryInterface(aIID);
  }
};

// Module
var SimpleAutoCompleteSearchModule = {
  registerSelf: function(aCompMgr, aFileSpec, aLocation, aType) {
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.registerFactoryLocation(CLASS_ID, CLASS_NAME, CONTRACT_ID, aFileSpec, aLocation, aType);
  },

  unregisterSelf: function(aCompMgr, aLocation, aType) {
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.unregisterFactoryLocation(CLASS_ID, aLocation);        
  },
  
  getClassObject: function(aCompMgr, aCID, aIID) {
    if (!aIID.equals(Ci.nsIFactory))
      throw Components.results.NS_ERROR_NOT_IMPLEMENTED;

    if (aCID.equals(CLASS_ID))
      return SimpleAutoCompleteSearchFactory;

    throw Components.results.NS_ERROR_NO_INTERFACE;
  },

  canUnload: function(aCompMgr) { return true; }
};

// Module initialization
function NSGetModule(aCompMgr, aFileSpec) { return SimpleAutoCompleteSearchModule; }
