const COLORS = ["", "red", "orange", "green", "blue", "violet"];

function SmAddress(id, name, email, group, favorite) {
  this.id = id;
  this.name = name;
  this.email = email;
  this.group = group;
  this.favorite = favorite;

  this.getLogin = function() {
    return /(.*)@/.exec(this.email)[1];
  }
  this.getHost = function() {
    return /@(.*)/.exec(this.email)[1];
  }
  this.toString = function() {
    return this.name + (this.name && this.email ? " " : "") +
           (this.email ? "<" + this.email + ">" : "");
  }
  this.toHtml = function() {
    return this.name + (this.name && this.email ? " " : "") +
           (this.email ? "&lt;<a href='mailto:" + this.email + "'>" +
                         this.email + "</a>&gt;" : "");
  }
}

SmAddress.parse = function(text) {
  if (!text) return;

  var name, email;
  text = text.replace(/[^'"\s<>]+@[^'"\s<>]+/, function($0) {
    email = $0;
    return "";
  });
  name = text.replace(/^['"\s<>]*|['"\s<>]*$/g, "");
  if (name || email) {
    return new SmAddress(null, name, email);
  }
}

SmAddress.parseList = function(text) {
  var addresses = new Array();
  if (text) {
    var list = ("" + text).split(/[,;]/);
    for(var i in list) {
      var address = SmAddress.parse(list[i]);
      if (address) addresses.push(address);
    }
  }
  return addresses;
}

SmAddress.create = function() {
  return new SmAddress(null, null, null, null);
}

function SmAccount(id, index, name, email, replyTo, refresh, deleteMode,
                   color, subject, signature, isImap, pop3auth,
                   pop3host, pop3port, pop3ssl, pop3login,
                   smtpHost, smtpPort, smtpSsl, smtpLogin) {
  this.id = id;
  this.index = index;
  this.name = name;
  this.email = email;
  this.replyTo = replyTo;
  this.refresh = refresh;
  this.deleteMode = deleteMode;
  this.color = color;
  this.subject = subject;
  this.signature = signature;

  this.pop3 = {
    isImap: isImap,
    auth:   pop3auth,
    host:   pop3host,
    port:   pop3port,
    ssl:    pop3ssl,
    login:  pop3login
  }
  this.pop3.getPassword = function() {
    return SmPasswordManager.getPassword(id + "/pop3/");
  }
  this.pop3.setPassword = function(password) {
    return SmPasswordManager.setPassword(id + "/pop3/", password);
  }
  this.smtp = {
    host:  smtpHost,
    port:  smtpPort,
    ssl:   smtpSsl,
    login: smtpLogin
  }
  this.smtp.getPassword = function() {
    return SmPasswordManager.getPassword(id + "/smtp/");
  }
  this.smtp.setPassword = function(password) {
    return SmPasswordManager.setPassword(id + "/smtp/", password);
  }
  this.toString = function() {
    return (this.name ? this.name + " " : "") + "<" + this.email + ">";
  }
}

SmAccount.create = function() {
  return new SmAccount(null,
                       null,
                       null,
                       null,
                       null,
                       60,
                       null,
                       null,
                       null,
                       null,
                       null,
                       0,
                       null, 
                       110,
                       null,
                       null,
                       null,
                       25,
                       null,
                       null);
}

SmAccount.ASK = "ask";
SmAccount.NEVER = "never";
SmAccount.ALWAYS = "always";
SmAccount.ON_RECEIVE = "onReceive";

function SmMessage(id, accountId, uid, from, to, cc, bcc, subject, date, size, charset,
                   read, deleted, attachmentsDir, attachmentsCount, html, folderId,
                   partial, template, color, returnReceiptTo, replyTo) {
  this.id = id;
  this.accountId = accountId;
  this.uid = uid;
  this.from = from;
  this.to = to || "";
  this.cc = cc || "";
  this.bcc = bcc || "";
  this.subject = subject;
  this.date = date;
  this.size = size;
  this.charset = charset;
  this.read = read;
  this.deleted = deleted;
  this.attachmentsDir = attachmentsDir || SmUtils.getUniqueId();
  this.attachmentsCount = attachmentsCount || 0;
  this.html = html;
  this.folderId = folderId;
  this.partial = partial;
  this.template = template;
  this.color = color;
  this.returnReceiptTo = returnReceiptTo;
  this.replyTo = replyTo;

  this.contentType;
  this.source;

  this.getSize = function() {
    if (this.size == null) {
      this.size = this.toString().length;
    }
    return this.size;
  }

  this.toString = function() {
    if (!this.source) {
      this.source = SmMessageEncoder.encode(this);
    }
    return this.source;
  }

  this.toHtml = function() {
    function mailList2html(value) {
      var addresses = SmAddress.parseList(value);
      var result = "";
      for(var i in addresses) {
        result += (result ? ", " : "") + addresses[i].toHtml();
      }
      return result;
    }
    var from = this.from ? SmAddress.parse(this.from).toHtml() : "";
    return "<span style='color: gray;'>" +
           "From: " + from + "<br>" +
           "To: " + mailList2html(this.to) + "<br>" + 
           (this.cc ? "Cc: " + mailList2html(this.cc) + "<br>" : "") +
           "Date: " + SmDate.toString(this.date) + "<br>" +
           "Subject: " + this.subject + "<br>" +
           "</span><br>" + this.html;
  }

  this.createReply = function(all) {
    var reply = new SmMessage();
    reply.accountId = this.accountId;
    reply.charset = this.charset;
    reply.to = this.replyTo || this.from;
    if (all) {
      if (this.to) reply.to += ", " + this.to;
      reply.cc = this.cc;
      reply.bcc = this.bcc;
    }
    reply.subject = this.subject.replace(/^(Re: )?/i, "Re: ");
    // Remove attachments
    reply.html = SmText.replaceURLs(this.html,
      function(text, url) {
        return url.match(/^simplemail:/)
           ? "[Attachment: " + SmFile.getFileName(url) + "]" : text;
      });

    reply.html = "<br><br><span style='color: gray;'>" +
                 SmPrefs.getString("reply")
                        .replace(/{author}/gi, SmAddress.parse(this.from).toHtml())
                        .replace(/{date}/gi, SmDate.toString(this.date))
                        .replace(/{date-gmt}/gi, SmDate.toGMT(this.date))
                        .replace(/{date-locale}/gi, new Date(this.date).toLocaleString())
                        .replace(/{subject}/gi, this.subject) +
                 "</span><br>" +
                 "<blockquote style='border-left: 1px solid rgb(204, 204, 204);" +
                                    "margin: 0 0 0 0.8ex;" +
                                    "padding: 1ex 0 0 1ex;'>" + reply.html + "</blockquote>";
    return reply;
  }

  this.createForward = function() {
    var forward = new SmMessage();
    forward.accountId = this.accountId;
    forward.charset = this.charset;
    forward.subject = this.subject.replace(/^(Fwd?: )?/i, "Fwd: ");
    forward.html = "<br><br><hr>" + this.toHtml();
    return forward;
  }

  this.createReturnReceipt = function(recipient) {
    var receipt = new SmMessage();
    receipt.accountId = this.accountId;
    receipt.from = recipient;
    receipt.to = this.returnReceiptTo;
    receipt.charset = "utf-8";
    receipt.subject = "Rcpt: " + this.subject;
    receipt.html = SmBundle.getString("messageReceived");
    if (receipt.html != "Message received") {
      receipt.html = "Message received / " + receipt.html;
    }
    return receipt;
  }

  var fields = ["from", "to", "cc", "bcc", "subject", "charset",
                "html", "returnReceiptTo"];

  this.equals = function(message) {
    for(var i in fields) {
      var field = fields[i];
      if (this[field] != message[field]) return false;
    }
    return true;
  }

  this.copyTo = function(message) {
    for(var i in fields) {
      var field = fields[i];
      message[field] = this[field];
    }
  }

  this.recode = function(charset) {
    try {
      var message = new SmMessage();
      for(var i in fields) {
        var field = fields[i];
        message[field] = SmText.fromUTF8(this[field], this.charset);
        message[field] = SmText.toUTF8(message[field], charset);
      }
      message.charset = charset;
      // If no exceptions, update original message
      message.copyTo(this);
    }
    catch(e) {}
  }
}

SmMessage.parse = function(source) {
  return SmMessageEncoder.decode(source);
}

SmMessage.parseMailto = function(mailto) {
  function getUrlParameter(name) {
    var regexp = new RegExp("[?&]" + name + "=([^&]*)", "i");
    var result = regexp.exec(mailto);
    return result ? decodeURIComponent(result[1]) : null;
  }
  var message = new SmMessage();
  message.to = decodeURIComponent(/mailto:([^?&]*)/.exec(mailto)[1]);
  message.cc = getUrlParameter("cc");
  message.bcc = getUrlParameter("bcc");
  message.subject = getUrlParameter("subject");
  message.html = getUrlParameter("body");
  return message;
}

function SmFolder(id, parentId, name, open) {
  this.id = id;
  this.parentId = parentId;
  this.name = name;
  this.open = open;

  this.getName = function() {
    return SmBundle.getString(this.name) || this.name;
  }
}

SmFolder.DEFAULT_NAMES = ["inbox", "sent", "drafts", "trash"];
SmFolder.INBOX = 1;
SmFolder.SENT = 2;
SmFolder.DRAFTS = 3;
SmFolder.TRASH = 4;

function SmRule(id, filterId, property, not, condition, param) {
  this.id = id;
  this.filterId = filterId;
  this.property = property;
  this.not = not;
  this.condition = condition;
  this.param = param;

  function matchAll(string, patterns) {
    string = String.toLowerCase(string);
    for(var i = 0; i < patterns.length; i++) {
      var pattern = SmText.trim(patterns[i]).toLowerCase();
      if (string.indexOf(pattern) == -1) return false;
    }
    return true;
  }

  this.test = function(message, storage, mail) {
    var value = message[property];
    var result;

    switch(this.condition) {
      case "equals":
          result = value == this.param;
          break;
      case "contains":
          // "," separator means "or", "+" means "and"
          var any = this.param.split(/,/);
          for(var i = 0; !result && i < any.length; i++) {
            var all = any[i].split(/\+/);
            result |= matchAll(value, all);
          }
          break;
      case "inAddressBook":
          var address = SmAddress.parse(value);
          result = address && storage.getAddressByEmail(address.email);
          break;
      case "greaterThan":
          result = parseInt(value) > parseInt(this.param);
          break;
      default:
          SmUtils.error("Unknown conditon: " + this.condition);
          break;
    }
    return this.not ? !result : result;
  }
}

SmRule.getProperties = function() {
 return ["accountId", "from", "to", "subject", "html", "size", "source"];
}

SmRule.getConditions = function(property) {
  switch(property) {
    case "accountId":
        return ["equals"];
    case "from":
        return ["contains", "inAddressBook"];
    case "size":
        return ["greaterThan"];
    default:
        return ["contains"];
  }
}

SmRule.create = function() {
  return new SmRule(null, null, null, null, null, null);
}

function SmAction(id, filterId, name, param) {
  this.id = id;
  this.filterId = filterId;
  this.name = name;
  this.param = param;

  this.execute = function(message, storage, mail) {
    switch(this.name) {
      case "sound":
          SmSound.play(param);
          break
      case "color":
          message.color = param;
          break;
      case "folder":
          if (storage.getFolder(param)) {
            message.folderId = param;
          }
          break;
      case "delete":
          mail.delete(message);
          return;
      case "download":
          if (message.partial) {
            mail.getBody(message);
            return;
          }
          break;
      case "charset":
          message.recode(param);
          break;
      case "popup":
      case "counter":
          break; // Obsolete
      default:
          SmUtils.error("Unknown action: " + this.name);
          break;
    }
    return message;
  }
}

SmAction.getNames = function() {
  return [ "sound", "color", "folder", "delete", "download", "charset" ];
}

SmAction.create = function() {
  return new SmAction(null, null, null, null);
}

function SmFilter(id, index, name, type, rules, actions) {
  this.id = id;
  this.index = index;
  this.name = name;
  this.type = type || SmFilter.INBOX;
  this.rules = rules;
  this.actions = actions;

  this.getName = function() {
    return SmBundle.getString(this.name) || this.name;
  }

  this.execute = function(message, storage, mail) {
    for(var i = 0; i < this.rules.length; i++) {
      if (!this.rules[i].test(message, storage, mail)) return message;
    }
    for(var i = 0; message && i < this.actions.length; i++) {
      message = this.actions[i].execute(message, storage, mail);
    }
    return message;
  }
}

SmFilter.INBOX = 1;
SmFilter.SENT = 2;

SmFilter.create = function() {
  return new SmFilter(null, null, null, SmFilter.INBOX, new Array(), new Array());
}

function SmFilterManager(storage, mail) {
  this.storage = storage;
  this.mail = mail;

  var filters = storage.getFilters();

  this.execute = function(message, type) {
    for(var i = 0; message && i < filters.length; i++) {
      var filter = filters[i];
      if (type == filter.type) {
        message = filter.execute(message, this.storage, this.mail);
      }
    }
    return message;
  }
}
  