function displayInbox() {
  var browserWindow = SmUtils.getBrowserWindow();
  browserWindow.focus();

  var gBrowser = browserWindow.getBrowser();
  gBrowser.selectedTab = SmUtils.getDocumentTab(document);

  if (folderTree.currentIndex != 0) {
    folderTree.view.selection.select(0);
  }
  else selectFirstMessage();
  messageTree.focus();
}

function getCurrentFolder() {
  return getFolder(folderTree.currentIndex);
}

function getFolder(row) {
  var index = 0, result;
  forEachOpenFolder(0, function(folder) {
                         if (index++ == row) result = folder;
                       });
  return result;
}

function forEachOpenFolder(parentId, onFolder, level) {
  if (!level) level = 0;
  var children = folders[parentId];
  for(var i in children) {
    var child = children[i];
    onFolder(child, level);
    if (child.open) {
      forEachOpenFolder(child.id, onFolder, level + 1);
    }
  }
}

function getFolderRow(folderId) {
  var index = 0, row;
  forEachOpenFolder(0, function(f) {
                         if (f.id == folderId) row = index;
                         index++;
                       });
  return row;
}

function getChildCount(folderId) {
  var count = 0;
  forEachOpenFolder(folderId, function() { count++; });
  return count;
}

var folderTreeView = {
  get rowCount() {
    return getChildCount(0);
  },
  getCellText: function(row, column) {
    var folder = getFolder(row);
    var name = column.id.replace(/folder-/, "");
    var count = getFolderCount(folder.id, name);
    switch(name) {
      case "name":   return folder.getName();
      case "size":   return SmUtils.size2string(count);
      default:       return count;
    }
  },
  getCellProperties: function(row, column, props) {
    var folder = getFolder(row);
    var name = column.id.replace(/folder-/, "");
    switch(name) {
      case "name":
        var type = (folder.id == SmFolder.TRASH) ? "trash" : "folder";
        var atom = folderTreeView.atomService.getAtom(type);
        props.AppendElement(atom);
        if (folder.id == SmFolder.TRASH && isTrashEmpty()) {
          var atom = folderTreeView.atomService.getAtom("empty");
          props.AppendElement(atom);
        }
        if (getFolderCount(folder.id, "unread") ||
            (getFolderCount(folder.id, "childUnread") && !folder.open))
        {
          var atom = folderTreeView.atomService.getAtom("unread");
          props.AppendElement(atom);
        }
        break;
      case "unread":
        if (getFolderCount(folder.id, "unread")) {
          var atom = folderTreeView.atomService.getAtom("unread");
          props.AppendElement(atom);
        }
        break;
    }
  },
  isContainer: function(row) {
    return true;
  },
  isContainerOpen: function(row) {
    return getFolder(row).open;
  },
  isContainerEmpty: function(row) {
    var folder = getFolder(row);
    return !folders[folder.id] || folders[folder.id].length == 0;
  },
  getParentIndex: function(row) {
    var folder = getFolder(row);
    return getFolderRow(folder.parentId) || -1;
  },
  getLevel: function(row) {
    var index = 0, result;
    forEachOpenFolder(0, function(folder, level) {
                           if (index++ == row) result = level;
                         });
    return result;
  },
  hasNextSibling: function(row, afterRow) {
    var folder = getFolder(row);
    var index, result;
    forEachOpenFolder(folder.parentId,
                      function(f, level) {
                        if (f.id == folder.id) index = 0;
                        if (level == 0 &&
                            index > afterRow - row) result = true;
                        index++;
                      });
    return result;
  },
  toggleOpenState: function(row) {
    var folder = getFolder(row);
    if (!folder.open && folderTreeView.isContainerEmpty(row)) return;

    folder.open = !folder.open;
    folderTreeView.treebox.rowCountChanged(row + 1,
      (folder.open ? 1 : -1) * getChildCount(folder.id));

    storage.saveFolder(folder);
  },
  canDrop: function(row, orientation) {
    if (orientation == folderTreeView.DROP_ON) {
      if (dragObserver.source == messageTree) return true;

      if (dragObserver.source == folderTree) {
        // Deny setting self or child as a parent
        var folder = getCurrentFolder();
        var newParent = getFolder(row);

        var result = folder.id != newParent.id;
        forEachOpenFolder(folder.id, function(f) {
                                       if (f.id == newParent.id) result = false;
                                     });
        return result;
      }
    }
    if (orientation == folderTreeView.DROP_AFTER &&
        dragObserver.source == folderTree &&
        row == folderTreeView.rowCount - 1) return true;
  },
  drop: function(row, orientation) {
    var target = getFolder(row);

    if (dragObserver.source == folderTree) {
      var folder = getCurrentFolder();
      var parentId = (orientation != folderTreeView.DROP_AFTER) ? target.id : 0;
      setFolderParent(folder, parentId);
    }
    else if (dragObserver.source == messageTree) {
      moveToFolder(target.id);
    }
    dragObserver.source = null;
  }
};
DefaultTreeView.extend(folderTreeView);

// Drag-n-Drop
var dragObserver = {
  source: null,
  onDragStart: function(event, transferData, action) {
    if (event.target.nodeName != "treechildren") return;

    if (event.target.parentNode == folderTree) {
      var folder = getCurrentFolder();
      if (folder.id <= SmFolder.TRASH) return false;
    }
    transferData.data = new TransferData();
    transferData.data.addDataForFlavour("text/unicode", "foo");
    dragObserver.source = event.target.parentNode;
  },
  getSupportedFlavours : function () {
    var flavours = new FlavourSet();
    flavours.appendFlavour("text/unicode");
    return flavours;
  },
  onDrop: function(event, transferData, session) {}
}

var contextFolder;

function populateFolderMenu(event) {
  var row = {}, column = {}, child = {};
  folderTreeView.treebox.getCellAt(event.clientX, event.clientY, row,
                                   column, child);
  contextFolder = (row.value != -1) ? getFolder(row.value) : null;
  var isCustomFolder = contextFolder && contextFolder.id > SmFolder.TRASH;
  SmDom.get("folder-menu-rename").hidden = !isCustomFolder;
  SmDom.get("folder-menu-delete").hidden = !isCustomFolder;
  SmDom.get("folder-menu-import").hidden = !contextFolder;
  SmDom.get("folder-menu-export").hidden = !contextFolder;
  SmDom.get("folder-menu-empty").hidden = !contextFolder ||
                                          contextFolder.id != SmFolder.TRASH;
}

function setFolderParent(folder, parentId) {
  if (folder.parentId == parentId) return;

  folderTreeView.treebox.beginUpdateBatch();
  if (folder.parentId != null) {
    folderTreeView.treebox.rowCountChanged(folderTree.currentIndex - 1, -1);
    var siblings = folders[folder.parentId];
    for(var i in siblings) {
      if (siblings[i].id == folder.id) {
        siblings.splice(i, 1);
      }
    }
  }

  folder.parentId = parentId;
  if (!folders[parentId]) {
    folders[parentId] = new Array();
  }
  folders[parentId].push(folder);

  var parentRow = (parentId == 0) ? parentRow = folderTreeView.rowCount - 1
                                  : getFolderRow(parentId);
  if (parentRow) {
    folderTreeView.treebox.rowCountChanged(parentRow + 1, 1);
  }
  folderTreeView.treebox.endUpdateBatch();
  storage.saveFolder(folder);

  updateFolderCounts();
}

function addFolder() {
  var name = {};
  if (SmUtils.prompt("folderName", name) && name.value) {
    var parentId = contextFolder ? contextFolder.id : 0;
    var folder = new SmFolder(null, null, name.value);
    setFolderParent(folder, parentId);
  }
}

function renameFolder() {
  var name = { value: contextFolder.getName() };
  if (SmUtils.prompt("folderName", name) && name.value) {
    contextFolder.name = name.value;
    folderTreeView.treebox.invalidateRow(folderTree.currentIndex);
    storage.saveFolder(contextFolder);
  }
}

function deleteFolder() {
  if (contextFolder.parentId != SmFolder.TRASH &&
      SmPrefs.getBool("moveToTrash"))
  {
    return setFolderParent(contextFolder, SmFolder.TRASH);
  }
  var name = contextFolder.getName();
  var confirmText = SmBundle.getString("deleteFolder", [name]);
  askDeleteFolder(contextFolder, confirmText);
}

function emptyTrash() {
  if (!isTrashEmpty()) {
    var confirmText = SmBundle.getString("emptyTrash");
    askDeleteFolder(contextFolder, confirmText, true);
  }
}

function forEachFolder(folder, onFolder) {
  var children = folders[folder.id];
  for(var i in children) {
    forEachFolder(children[i], onFolder);
  }
  onFolder(folder);
}

function askDeleteFolder(folder, confirmText, childrenOnly) {

  function doDeleteFolder() {
    storage.deleteFolder(folder, childrenOnly);

    // Update tree: delete child folders
    var folderRow = getFolderRow(folder.id);
    var children = folderRow && folder.open ? getChildCount(folder.id) : 0;
    forEachFolder(folder, function(f) {
      delete folders[f.id];
    });
    folderTreeView.treebox.rowCountChanged(folderRow, -children);

    if (childrenOnly) {
      if (folderRow == folderTree.currentIndex) displayFolder();
      return;
    }

    // Update tree: delete folder itself
    var siblings = folders[folder.parentId];
    for(var i in siblings) {
      if (siblings[i].id == folder.id) {
        siblings.splice(i, 1);
        folderTreeView.treebox.rowCountChanged(folderRow - 1, -1);
      }
    }
  }

  // All messages including subfolders

  function getFolderMessages() {
    var msgs = new Array();
    forEachFolder(folder, function(f) {
      msgs = msgs.concat(storage.getFolderMessages(f.id));
    });
    return msgs;
  }

  var msgs = getFolderMessages();
  if (msgs.length) {
    SimpleMail.askDeleteMail(msgs, confirmText, window, function() {
      if (getFolderMessages().length == 0) doDeleteFolder();
    });
  }
  else if (SmUtils.confirm(confirmText)) doDeleteFolder();
}

function exportFolder() {
  var folder = getCurrentFolder();
  var file = SmFile.chooseFile("save", [], folder.getName());
  if (!file) return;

  var out = SmFile.createOutputStream(file);
  var msgs = storage.getFolderMessages(folder.id);

  for(var i = msgs.length - 1; i >= 0; i--) {
    var source = SmFile.getAttachmentsDirectory(msgs[i].attachmentsDir);
    source.append(SmFile.MESSAGE_SOURCE);

    out.write("From - " + new Date(msgs[i].date) + "\r\n");
    out.write(SmFile.readFile(source) + "\r\n\r\n");
  }
  out.close();
}

function importFolder() {
  var folder = getCurrentFolder();
  var files = SmFile.chooseFile("multiple");
  if (!files) return;

  for(var i in files) {
    var file = files[i];
    var contents = SmFile.readFile(file);
    var chunks;

    if (file.leafName.match(/\.eml$/)) {
      chunks = [ contents ];
    } else {
      chunks = contents.split(/^From .*\r\n|\r?\nFrom .*\r\n/);
      chunks.shift();
    }

    for(var j = 0; j < chunks.length; j++) {
      var chunk = chunks[j].replace(/\r\n$/, "");
      var message = SmMessage.parse(chunk);
      message.folderId = folder.id;
      SimpleMail.addMessage(message);
    }
  }
}

var folderCounts;

function getFolderCount(folderId, name) {
  if (!folderCounts) folderCounts = storage.getFolderCounts();
  var counts = folderCounts && folderCounts[folderId];
  return counts && counts[name] || 0;
}

function updateFolderCounts() {
  folderCounts = null;
  folderTreeView.treebox.invalidate();
  SimpleMail.updateIcons();
}

function isTrashEmpty() {
  var children = folders[SmFolder.TRASH];
  return (!children || !children.length) && !getFolderCount(SmFolder.TRASH, "total");
}
