var SmPrefs = {
  prefs: Components.classes["@mozilla.org/preferences-service;1"]
         .getService(Components.interfaces.nsIPrefService)
         .getBranch("extensions.simplemail."),

  getInt: function(name) {
    try {
      return this.prefs.getIntPref(name);
    } catch(e) {
      return 0;
    }
  },

  setInt: function(name, value) {
    this.prefs.setIntPref(name, value);
  },

  getString: function(name) {
    try {
      return this.prefs.getComplexValue(name,
        Components.interfaces.nsISupportsString).data;
    } catch(e) {
      return null;
    }
  },

  setString: function(name, value) {
    var str = Components.classes["@mozilla.org/supports-string;1"]
              .createInstance(Components.interfaces.nsISupportsString);
    str.data = value;
    this.prefs.setComplexValue(name,
      Components.interfaces.nsISupportsString, str);
  },

  getBool: function(name) {
    try {
      return this.prefs.getBoolPref(name);
    } catch(e) {
      return false;
    }
  },

  setBool: function(name, value) {
    this.prefs.setBoolPref(name, value);
  },

  getObject: function(name) {
    return eval(this.getString(name));
  },

  setObject: function(name, value) {
    this.setString(name, SmUtils.toJSON(value));
  },

  delete: function(name) {
    try {
      if (name) this.prefs.deleteBranch(name);
    } catch(e) {}
  },

  clear: function(name) {
    try {
      if (name) this.prefs.clearUserPref(name);
    } catch(e) {}
  }
}

var SmFile = {
  MESSAGE_SOURCE: "message.txt",

  getDataDirectory: function() {
    var dir;
    try {
      dir = SmPrefs.prefs.getComplexValue("dataDirectory",
              Components.interfaces.nsIRelativeFilePref).file;
    }
    catch(e) {
      dir = SmPrefs.prefs.getComplexValue("dataDirectory",
              Components.interfaces.nsILocalFile);
    }
    if (!dir.exists()) dir.create(dir.DIRECTORY_TYPE, 0777);
    return dir;
  },

  getExtensionDirectory: function() {
    var id = "simplemail@telega.phpnet.us";
    return Components.classes["@mozilla.org/extensions/manager;1"]
           .getService(Components.interfaces.nsIExtensionManager)
           .getInstallLocation(id)
           .getItemLocation(id);
  },

  getSmilesDirectory: function() {
    var dir = SmFile.getExtensionDirectory();
    dir.append("chrome");
    dir.append("skin");
    dir.append("smiles");
    return dir;
  },

  getDirectory: function() {
    var dir = SmFile.getDataDirectory();
    for(var i = 0; i < arguments.length; i++) {
      dir.append(arguments[i]);
      if (!dir.exists()) dir.create(dir.DIRECTORY_TYPE, 0777);
    }
    return dir;
  },

  encodeURI: function(url) {
    return encodeURI(url).replace(/'/g, "%27");
  },

  getURL: function() {
    var url = "";
    for(var i = 0; i < arguments.length; i++) {
      url += (url ? "/" : "") + arguments[i];
    }
    return SmFile.encodeURI("simplemail://" + url);
  },

  getAttachmentsDirectory: function(directory) {
    return SmFile.getDirectory("attachments", directory);
  },

  saveAttachment: function(directory, name, contents) {
    var file = SmFile.getAttachmentsDirectory(directory);
    file.append(name);
    SmFile.writeFile(file, contents);
  },

  deleteAttachments: function(directory) {
    if (!directory) return;

    var file = SmFile.getAttachmentsDirectory(directory);
    if (file.exists()) {
      try { file.remove(true); } catch(e) {}
    }
  },

  getAttachmentURL: function(directory, name) {
    return SmFile.getURL("attachments", directory, name);
  },

  getImagesDirectory: function() {
    return SmFile.getDirectory("images");
  },

  deleteImage: function(name) {
    var file = SmFile.getImagesDirectory();
    file.append(name);
    try { file.remove(false); } catch(e) {}
  },

  saveURL: function(url, file) {
    var io = Components.classes["@mozilla.org/network/io-service;1"]
                       .getService(Components.interfaces.nsIIOService);
    var source = io.newURI(url, "UTF-8", null);
      
    var persist = Components.classes["@mozilla.org/embedding/browser/nsWebBrowserPersist;1"]
                            .createInstance(Components.interfaces.nsIWebBrowserPersist);
    
    persist.persistFlags = persist.PERSIST_FLAGS_REPLACE_EXISTING_FILES;
    persist.persistFlags |= persist.PERSIST_FLAGS_AUTODETECT_APPLY_CONVERSION;
    
    persist.saveURI(source, null, null, null, null, file);
  },

  saveCanvas: function(canvas, fileName) {
    var file = SmFile.getImagesDirectory();
    file.append(fileName);
    SmFile.saveURL(canvas.toDataURL("image/png"), file);
  },

  createOutputStream: function(file) {
    var stream = Components.classes["@mozilla.org/network/file-output-stream;1"]
                 .createInstance(Components.interfaces.nsIFileOutputStream);
    stream.init(file, 0x04 | 0x08 | 0x20, 0666, 0); // read & write, create, truncate
    var buffered = Components.classes["@mozilla.org/network/buffered-output-stream;1"]
                   .createInstance(Components.interfaces.nsIBufferedOutputStream);
    buffered.init(stream, 1*1024*1024);
    return { stream: stream,
             buffered: buffered,
             write: function(data) {
               this.buffered.write(data, data.length);
             },
             close: function() {
               this.buffered.close();
               this.stream.close();
             }
           };
  },

  createInputStream: function(file) {
    var stream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                 .createInstance(Components.interfaces.nsIFileInputStream);
    stream.init(file, -1, 0, 0);
    var binary = Components.classes["@mozilla.org/binaryinputstream;1"]
                 .createInstance(Components.interfaces.nsIBinaryInputStream);
    binary.setInputStream(stream);
    return { stream: stream,
             binary: binary,
             readFully: function() {
               return this.binary.readBytes(this.binary.available());
             },
             close: function() {
               this.binary.close();
               this.stream.close();
             }
           };
  },

  getNsiFile: function(file) {
    if (file instanceof Components.interfaces.nsIFile) return file;
    else {
      var nsiFile = Components.classes["@mozilla.org/file/local;1"]
                .createInstance(Components.interfaces.nsILocalFile);
      nsiFile.initWithPath(file);
      return nsiFile;
    }
  },

  readFile: function(file, unichar) {
    var nsiFile = SmFile.getNsiFile(file);
    var inn = unichar ? SmFile.createUnicharInputStream(nsiFile) 
                      : SmFile.createInputStream(nsiFile);
    try {
      return inn.readFully();
    }
    finally {
      inn.close();
    }
  },

  writeFile: function(file, data, unichar) {
    var nsiFile = SmFile.getNsiFile(file);
    var out = unichar ? SmFile.createUnicharOutputStream(nsiFile)
                      : SmFile.createOutputStream(nsiFile);
    try {
      out.write(data);
    }
    finally {
      out.close();
    }
  },

  createTemporaryFile: function() {
    var file = Components.classes["@mozilla.org/file/directory_service;1"]
               .getService(Components.interfaces.nsIProperties)
               .get("TmpD", Components.interfaces.nsIFile);
    file.append("simplemail.tmp");
    file.createUnique(file.NORMAL_FILE_TYPE, 0666);

    var out = SmFile.createOutputStream(file);
    var inn = SmFile.createInputStream(file);

    return { write: function(data) { out.write(data); },
             readFully: function() {
               out.buffered.flush();
               return inn.readFully();
             },
             remove: function() {
               out.close();
               inn.close();
               file.remove(false);
             }
           };
  },

  createUnicharOutputStream: function(file) {
    var stream = Components.classes["@mozilla.org/network/file-output-stream;1"]
                 .createInstance(Components.interfaces.nsIFileOutputStream);
    stream.init(file, 0x04 | 0x08 | 0x20, 0666, 0); // read & write, create, truncate
    var unichar = Components.classes["@mozilla.org/intl/converter-output-stream;1"]
                  .createInstance(Components.interfaces.nsIConverterOutputStream);
    unichar.init(stream, "utf-8", 0, 0xFFFD);

    return { stream: stream,
             unichar: unichar,
             write: function(data) {
               this.unichar.writeString(data);
             },
             close: function() {
               this.unichar.close();
               this.stream.close();
             }
           };
  },

  createUnicharInputStream: function(file) {
    var stream = Components.classes["@mozilla.org/network/file-input-stream;1"]
                 .createInstance(Components.interfaces.nsIFileInputStream);
    stream.init(file, -1, 0, 0);
    var unichar = Components.classes["@mozilla.org/intl/converter-input-stream;1"]
                  .createInstance(Components.interfaces.nsIConverterInputStream);
    unichar.init(stream, "utf-8", stream.available(), 0xFFFD);
    return { stream: stream,
             unichar: unichar,
             readFully: function() {
               var str = {};
               this.unichar.readString(this.stream.available(), str);
               return str.value;
             },
             close: function() {
               this.unichar.close();
               this.stream.close();
             }
           };
  },

  getStatusIconURL: function(status) {
    var image = (status == "loading") ? "loading.gif" : status + ".png";
    return "chrome://simplemail/skin/status/" + image;
  },

  getEnvelopeIconURL: function(name) {
    return "chrome://simplemail/skin/envelope/" + name;
  },

  getFileURL: function(file) {
    var ios = Components.classes["@mozilla.org/network/io-service;1"]
              .getService(Components.interfaces.nsIIOService);
    var fileHandler = ios.getProtocolHandler("file")
                     .QueryInterface(Components.interfaces.nsIFileProtocolHandler);
    return fileHandler.getURLSpecFromFile(file);
  },

  getMimeType: function(uriOrFile) {
    try {
      var mimeService = Components.classes["@mozilla.org/mime;1"]
                        .getService(Components.interfaces.nsIMIMEService);
      if (uriOrFile instanceof Components.interfaces.nsIFile) {
        return mimeService.getTypeFromFile(uriOrFile);
      } else {
        var url = SmFile.getNsiURL(uriOrFile);
        return mimeService.getTypeFromURI(url);
      }
    } catch(e) {
      return "application/octet-stream";
    }
  },

  isImage: function(url) {
    var mimeType = SmFile.getMimeType(url);
    return mimeType.match(/^image/i);
  },

  readURL: function(url) {
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                    .getService(Components.interfaces.nsIIOService);
    var bstream = Components.classes["@mozilla.org/binaryinputstream;1"]
                  .createInstance(Components.interfaces.nsIBinaryInputStream);
    var channel = ioService.newChannel(url, null, null);
    var input = channel.open();
    bstream.setInputStream(input);

    var size, data = "";
    while(size = bstream.available()) {
      data += bstream.readBytes(size);
    }
    bstream.close();
    input.close();
    return data;
  },

  getFileName: function(url) {
    var nsiUrl = (nsiUrl instanceof Components.interfaces.nsIURL)
                   ? url : SmFile.getNsiURL(url);
    return decodeURIComponent(nsiUrl.fileName);
  },

  getNsiURL: function(url) {
    var nsiUrl = Components.classes["@mozilla.org/network/standard-url;1"]
               .createInstance(Components.interfaces.nsIURL);
    nsiUrl.spec = url;
    return nsiUrl;
  },

  isLocalURL: function(url) {
    return ("" + url).match(/^(file:|chrome:|simplemail:)/i);
  },

  chooseFile: function(mode, filters, name) {
    var fp = Components.classes["@mozilla.org/filepicker;1"]
             .createInstance(Components.interfaces.nsIFilePicker);
    fp.init(window, null, mode == "save" ? fp.modeSave :
                          mode == "folder" ? fp.modeGetFolder :
                          mode == "multiple" ? fp.modeOpenMultiple : fp.modeOpen);
    for(var i in filters) {
      switch(filters[i]) {
        case "images": fp.appendFilters(fp.filterImages); break;
        case "html":   fp.appendFilters(fp.filterHTML); break;
        default:       fp.appendFilter(filters[i], filters[i]); break;
      }
    }
    fp.appendFilters(fp.filterAll);
    fp.defaultString = name;

    var result = fp.show();
    if (result == fp.returnOK || result == fp.returnReplace) {
      if (mode != "multiple") return fp.file;
      var enum = fp.files;
      var files = [];
      while(enum.hasMoreElements()) {
        var file = enum.getNext();
        file.QueryInterface(Components.interfaces.nsIFile);
        files.push(file);
      }
      return files;
    }
  },

  forEachFile: function(dir, onFile) {
    var files = dir.directoryEntries;
    while(files.hasMoreElements()) {
      var file = files.getNext();
      file.QueryInterface(Components.interfaces.nsIFile);
      onFile(file);
    }
  }
}

var SmSound = {
  sound: Components.classes["@mozilla.org/sound;1"]
         .getService(Components.interfaces.nsISound),

  isPlaying: false,

  getURL: function(path) {
    var ioService = Components.classes["@mozilla.org/network/io-service;1"]
                    .getService(Components.interfaces.nsIIOService);
    return ioService.newURI("file://" + path, null, null);
  },

  getTime: function(path) {
    if (!path) return 0;

    var file = SmFile.getNsiFile(path);
    var inn = SmFile.createInputStream(file);
    var header = inn.binary.readBytes(100);
    inn.close();

    function getInt(regexp) {
      var value = regexp.exec(header)[1];
      return value.charCodeAt(0) +
             value.charCodeAt(1) * 0x100 +
             value.charCodeAt(2) * 0x10000 +
             value.charCodeAt(3) * 0x1000000;
    }

    if (header.match(/^RIFF/)) {
      var bytesPerSec = getInt(/fmt .{12}(.{4})/);
      var bytesTotal = getInt(/data(.{4})/);
      return bytesTotal / bytesPerSec;
    }
  },

  play: function(path) {
    if (SmSound.isPlaying) return;
    SmSound.isPlaying = true;

    try {
      path ? SmSound.sound.play(SmSound.getURL(path))
           : SmSound.sound.playSystemSound("_moz_mailbeep");

      setTimeout(function() { SmSound.isPlaying = false; },
                 SmSound.getTime(path) * 1000);
    }
    catch(e) {
      SmUtils.error("cantPlaySound", [path]);
    }
  }
}
