var SmMessageEncoder = {
  createAttachmentHtml: function(url) {
    return SmFile.isImage(url) ? "<img src='" + url + "'>"
                               : "<a class='attachment' href='" + url + "'>" +
                                 SmFile.getFileName(url) + "</a>";
  },

  encode: function(message) {
    // Base64 encode header
    function encodeHeader(string) {
      if (string && message.charset) {
        for(var i in string) {
          // Do not encode string if it contains only English characters
          if (string.charCodeAt(i) > 127) {
            return "=?" + message.charset + "?B?" +
                   btoa(SmText.fromUTF8(string, message.charset)) + "?=";
          }
        }
      }
      return string;
    }

    function addHeader(name, value) {
      headers += name + ": " + value + "\r\n";
    }

    function addAddressHeader(name, value) {
      if (value && message.charset) {
        var addresses = SmAddress.parseList(value);
        for(var i in addresses) {
          addresses[i].name = encodeHeader(addresses[i].name);
        }
        value = addresses.join(", ");
      }
      if (value) addHeader(name, value);
    }

    // String concatenation on large (10 Mb) strings is time consuming,
    // it's better to write chunks to a temporary file and then read it back into memory

    function addCRLFs(string) {
      const CHUNK_LEN = 75;

      var tmpFile = SmFile.createTemporaryFile();
      var index = 0;
      var chunk;

      while(chunk = string.substr(index, CHUNK_LEN)) {
        tmpFile.write(chunk, chunk.length);
        tmpFile.write("\r\n", 2);
        index += CHUNK_LEN;
      }
      string = tmpFile.readFully();
      tmpFile.remove();
      return string;
    }

    // Some SMTP servers don't allow lines longer than 1000 bytes (not characters).
    // If line.length > MAX_LEN then insert line break beside nearest white-space.
    // If line has no white-spaces, leave it as it is.

    function splitLongLines(string) {
      const MAX_LEN = 75;

      var result = "";
      var lines = string.split("\r\n");

      for(var i = 0; i < lines.length; i++) {
        var line = lines[i];
        if (!line) continue; // skip empty lines
        do {
          var index = line.length > MAX_LEN ? line.lastIndexOf(" ", MAX_LEN) : -1;
          result += (index != -1 ? line.substr(0, index) : line) + "\r\n";
        }
        while(index != -1 && (line = line.substr(index + 1)));
      }
      return result;
    }

    function encodeBody() {
      var attachments = new Array();
      message.html = SmText.replaceURLs(message.html,
        function(text, url, image) {
          if (!image && !SmFile.isLocalURL(url)) return text;

          url = url.replace(/&amp;/gi, "&"); // Decode html entities
          message.attachmentsCount++;

          var name = SmFile.getFileName(url);
          attachments[name] = {
            url: url,
            cid: SmUtils.getUniqueId() + "@simplemail"
          };

          var newUrl = SmFile.getAttachmentURL(message.attachmentsDir, name);
          return SmMessageEncoder.createAttachmentHtml(newUrl);
        });

      var html = "<html>" + message.html + "</html>";

      html = SmText.replaceURLs(html,
        function(text, url) {
          if (!url.match(/^simplemail:/)) return text;

          var name = SmFile.getFileName(url);
          var regexp = url.replace(/[\/\(\)\[\]\{\}\.\+\?]/g, function($0) { return "\\" + $0; });
          return text.replace(new RegExp(regexp, "g"), "cid:" + attachments[name].cid);
        });

      //Replace "bare LFs", some mail servers don't accept them
      html = html.replace(/(^|[^\r])\n+/g, function($0) {
                                             return $0.replace(/\n/g, "\r\n");
                                           });
      // Replace <br> with <br>\r\n to beautify result html
      html = html.replace(/(<br.*?>)(?!\r)/g, "$1\r\n");

      html = "--" + boundary + "\r\n" +
             "Content-Type: text/html; charset=" + message.charset + "\r\n" +
             "Content-Transfer-Encoding: 8bit\r\n" +
             "\r\n" + SmText.fromUTF8(splitLongLines(html), message.charset);

      for(var name in attachments) {
        var attachment = attachments[name]
        var contents = SmFile.readURL(attachment.url);
        SmFile.saveAttachment(message.attachmentsDir, name, contents);
        var disposition = SmFile.isImage(attachment.url) ? "inline" : "attachment";
        name = encodeHeader(name);

        html += "--" + boundary + "\r\n" +
                "Content-Type: " + SmFile.getMimeType(attachment.url) +";" +
                " name=\"" + name + "\"\r\n" +
                "Content-Transfer-Encoding: base64\r\n" +
                "Content-Disposition: " + disposition + "\r\n" +
                "Content-ID: <" + attachment.cid + ">\r\n" + "\r\n" +
                addCRLFs(btoa(contents));
      }
      return html + "--" + boundary + "--\r\n";
    }

    var headers = "";
    addHeader("MIME-Version", "1.0");
    addAddressHeader("Disposition-Notification-To", message.returnReceiptTo);
    addAddressHeader("From", message.from);
    addAddressHeader("Reply-To", message.replyTo);
    addAddressHeader("To", message.to);
    addAddressHeader("Cc", message.cc);
    addAddressHeader("Bcc", message.bcc);
    addHeader("Subject", encodeHeader(message.subject));
    message.date = new Date().getTime();
    addHeader("Date", SmDate.toMIME(message.date));
    addHeader("User-Agent", "Simple Mail " + SmUtils.getVersion());
    var boundary = "----------simplemail" + SmUtils.getUniqueId();
    addHeader("Content-Type", "multipart/related; boundary=" + boundary);
    return headers + "\r\n" + encodeBody();
  },

  parse: function(source, attachments, cidNames) {
    function Part(source) {
      var self = this;
      var headers = getHeaders();

      function getHeaders() {
        var end = source.indexOf("\r\n\r\n");
        return (end != -1) ? source.substr(0, end + 2) : source;
      }

      this.getHeader = function(name) {
        // Headers may be multiline (RFC 2822: "folding")
        var regexp = new RegExp("(^|\\r\\n)" + name +
                                " *: *((.|\\r\\n\\s)*) *\\r\\n", "i");
        var result = regexp.exec(headers);
        if (result) {
          result = result[2].replace(/\r\n(?= )/g, ""); // Unfold (RFC 2822)
          return decodeHeader(result);
        }
        return "";
      }

      // Removes redundant quotes, etc.
      function cleanAddressList(list) {
        var addresses = SmAddress.parseList(list);
        return addresses.join(", ");
      }

      this.getAddressHeader = function(name) {
        return cleanAddressList(self.getHeader(name));
      }

      // Decode Quoted Printable string
      function qp(string) {
        return string.replace(/=\r\n|=(..)/g, function($0, $1) {
          return ($0 != "=\r\n") ? String.fromCharCode("0x" + $1) : "";
        });
      }

      function atob_fixed(str) {
        try {
          return str ? atob(str.replace(/=+$/, "")) : str;
        }
        catch(e) {
          return SmUtils.error("messageParseError", [e]);
        }
      }

      // Decode Base64 / Quoted Printable encoded header
      function decodeHeader(string) {
        if (string) {
          var charset = message.charset;
          var regexp = /=\?(.*?)\?([BQ])\?(.*?)\?=/gi;
          string = string.replace(regexp, function($0, $1, $2, $3) {
            charset = $1;
            return $2 == 'B' || $2 == 'b' ? atob_fixed($3)
                                          : qp($3).replace(/_/g, " ");
          });
          try {
            string = SmText.toUTF8(string, charset);
          }
          catch(e) {}
        }
        return string;
      }

      function getByRegexp(string, regexp) {
        var result = regexp.exec(string);
        return result ? result[1] : "";
      }

      function getContentType() {
        var contentType = self.getHeader("Content-Type");
        return getByRegexp(contentType, /([^;\s]*)/);
      }

      function getBoundary() {
        var contentType = self.getHeader("Content-Type");
        return getByRegexp(contentType, /boundary *= *['"]?([^;\r'"]*)/i);
      }

      this.getCharset = function() {
        var contentType = self.getHeader("Content-Type");
        return getByRegexp(contentType, /charset *= *['"]?([^;\s'"]*)/i);
      }

      function getFileName() {
        var contentDisposition = self.getHeader("Content-Disposition");
        var contentType = self.getHeader("Content-Type");
        var regexp = /(file)?name *= *['"]?([^;\r'"]*)/i;
        var result = regexp.exec(contentDisposition) || regexp.exec(contentType);
        return result ? result[2].replace(/[\\\/]/g, "_") : "";
      }

      function getContentId() {
        var cid = self.getHeader("Content-ID");
        if (cid) return cid.replace(/^<|>$/g, "");
      }

      function isAttachment() {
        var contentDisposition = self.getHeader("Content-Disposition");
        var value = getByRegexp(contentDisposition, /([^;\s]*)/);
        return value.toLowerCase() == "attachment";
      }

      // String.replace(/\r\n/g, "") on large (10 Mb) strings is time consuming,
      // it's better to split() string, write chunks to a temporary file and then read it back into memory

      function stripCRLFs(string) {
        var tmpFile = SmFile.createTemporaryFile();
        var chunks = string.split(/\r?\n/);
        for(var i in chunks) {
          tmpFile.write(chunks[i], chunks[i].length);
        }
        string = tmpFile.readFully();
        tmpFile.remove();
        return string;
      }

      function getBody() {
        var start = source.indexOf("\r\n\r\n");
        if (start == -1) return "";

        var body = source.substr(start + 4);
        var encoding = self.getHeader("Content-Transfer-Encoding");

        if (encoding.match(/base64/i)) {
          var end = body.lastIndexOf("=");
          if (end != -1) body = body.substr(0, end + 1);
          body = atob_fixed(stripCRLFs(body));
        }
        else if (encoding.match(/quoted-printable/i)) {
          body = qp(body);
        }

        var charset = self.getCharset();
        try {
          body = SmText.toUTF8(body, charset);
          if (!message.charset) message.charset = charset;
        }
        catch(e) {}

        return body;
      }

      function parseMultipart() {
        var html = "";
        var parts = source.split("--" + getBoundary());

        var count = parts.length;
        if (parts[count - 1].match(/^--/)) count--;

        // For multipart/alternative take the latest alternate part
        var i = getContentType().match(/multipart\/alternative/i) ? count - 1 : 1;

        for(; i >= 1 && i < count; i++) {
          html += new Part(parts[i]).toHtml();
        }
        return html;
      }

      function parseAttachment(body) {
        var name = getFileName() || SmUtils.getUniqueId();
        attachments[name] = body;

        var cid = getContentId();
        if (cid) cidNames[cid] = name;

        return "";
      }

      // Some mail servers split lines longer than 1000 bytes by inserting "\r\n\s".
      // When such string is inserted inside html attribute, html gets broken.

      function repairHtml(html) {
        return html.replace(/<[^>]+>/g, function(tag) {
          return tag.replace(/["'][^'"]+["']/g, function(attribute) {
            return attribute.replace(/\r\n\s/g, "");
          });
        });
      }

      this.toHtml = function() {
        var contentType = getContentType();
        if (contentType.match(/^multipart/i)) return parseMultipart();

        var body = getBody();

        if (!isAttachment()) {
          if (!contentType || contentType.match(/text\/plain/i)) {
            return SmText.toHtml(body);
          }
          else if (contentType.match(/text\/html/i)) {
            return repairHtml(body);
          }
          else if (contentType.match(/message\/rfc822/i)) {
            var msg = SmMessageEncoder.parse(body, attachments, cidNames);
            return "<hr>" + msg.toHtml();
          }
        }
        return parseAttachment(body);
      }
    }

    var message = new SmMessage();
    var part = new Part(source);
    message.html = part.toHtml();
    message.source = source;
    message.size = source.length;
    message.date = SmDate.parse(part.getHeader("Date"));
    message.subject = part.getHeader("Subject");
    message.from = part.getAddressHeader("From");
    message.to = part.getAddressHeader("To");
    message.cc = part.getAddressHeader("Cc");
    message.bcc = part.getAddressHeader("Bcc");
    message.replyTo = part.getAddressHeader("Reply-To");
    message.returnReceiptTo = part.getAddressHeader("Disposition-Notification-To") ||
                              part.getAddressHeader("Return-Receipt-To");
    return message;
  },

  decode: function(source) {
    var attachments = new Array();
    var cidNames = new Array();
    var message = SmMessageEncoder.parse(source, attachments, cidNames);

    var linked = new Array();
    message.html = SmText.replaceURLs(message.html,
      function(text, url) {
        return SmFile.isLocalURL(url) ? "" : text;
      });

    message.html = message.html.replace(/cid:([^'"]*)/gi,
      function(text, cid) {
        var name = cidNames[cid];
        if (!name) return text;

        linked[name] = true;
        return SmFile.getAttachmentURL(message.attachmentsDir, name);
      });

    for(var name in attachments) {
      message.attachmentsCount++;
      try {
        SmFile.saveAttachment(message.attachmentsDir, name, attachments[name]);
      }
      catch(e) {
        SmUtils.error("messageParseError", [e]);
      }
      if (!linked[name]) {
        var newUrl = SmFile.getAttachmentURL(message.attachmentsDir, name);
        message.html += SmMessageEncoder.createAttachmentHtml(newUrl);
      }
    }
    return message;
  }
}
