/* $Id: e2_devkit.c 1958 2009-10-05 07:57:11Z tpgww $

Copyright (C) 2009 tooar <tooar@emelfm2.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file src/filesystem/e2_devkit.c
@brief functions related to system drive/disk hardware. Supersedes HAL.
*/

/*
TODO
confirm that mount-point and -options are effectively hardcoded within devkit mount opration
*/

#include "emelfm2.h"

#ifdef E2_DEVKIT

#include <dbus/dbus-glib.h>
#include "e2_fs.h"

#define DEVKIT_IFACE_NAME "org.freedesktop.DeviceKit.Disks"
#define DEVKIT_PATH_ROOT "/org/freedesktop/DeviceKit/Disks/devices/"

//subset of all possible device-properties
typedef struct _DeviceProperties
{
	gchar	*device_file; 		//e.g. /dev/sda
//	gchar  **device_file_by_id; //NULL terminated array of unique target(s) linked to corresponding device(s)
								//e.g. /dev/disk/by-id/ata-ST910021AS_3MH05AVA-part1 >> /dev/sda1
								//AND ALSO uid(s) linked to device(s)
								//e.g. /dev/disk/by-uuid/4165-968D >> /dev/sda1
//	gchar  **device_file_by_path; //NULL terminated array of port/partition(s) linked to corresponding device(s)
								//e.g. /dev/disk/by-path/pci-0000:00:1f.2-scsi-0:0:1:0 >> /dev/sda2
								//i.e. can get a count of mountpoints for the device_file
	gchar	*device_mount_path;	//where the device is mounted, valid only if device-is-mounted is TRUE
//	gchar	*device_presentation_name; //device name for UI use
	gchar	*id_usage;	//filesystem; crypto; partitiontable; raid; other; blank=(unknown)
	gchar	*id_type;	//per id-usage: ext3, vfat etc swap blank=(unknown)
	gchar	*drive_media;	//flash* or floppy* or optical*, ? blank or "flash" if unknown
//	guint64 device_media_detection_time;
	gboolean device_is_removable;
	gboolean device_is_media_available;
	gboolean device_is_media_change_detected;
	gboolean device_is_media_change_detection_inhibited;
	gboolean device_is_drive;
	gboolean device_is_optical_disc;
	gboolean device_is_mounted;
//	gboolean device_is_busy;
	gboolean drive_is_media_ejectable;
//	gboolean drive_requires_eject; not application-controllable ?
	gboolean optical_disc_is_blank;
	gboolean ignored;	//flag for device not relevant, not a devkit property
} DeviceProperties;

typedef struct _DeviceData
{
	gchar *object_path; //DEVKIT_PATH_ROOT"<native-path-cleaned-basename>" i.e. with any '-' >> '_'
	DBusGProxy *devproxy;	//not the same as the session-proxy, per-device to support async usage of the proxy
	DeviceProperties props;
	guint timer_id;		//timer to allow DE etc to do [un]mount before we do so
} DeviceData;

typedef struct _SessionDevices
{
	DBusGConnection *bus;	//connection to system bus
	DBusGProxy *proxy;		//abstracted proxy for interface exported by a connection on bus
	GHashTable *logged_devices; /* known removable devices, with
		key = (const) device object path, value = the corresponding DeviceData */
} SessionDevices;

static SessionDevices devicesdata;

static void _e2_devkit_device_properties_clear (DeviceProperties *props);
static void _e2_devkit_device_changed_cb (DBusGProxy *proxy,
	const gchar *object_path, gpointer user_data);
static void _e2_devkit_mount_async_cb
	(DBusGProxy *proxy, DBusGProxyCall *call, gpointer user_data);
static void _e2_devkit_unmount_async_cb
	(DBusGProxy *proxy, DBusGProxyCall *call, gpointer user_data);


/**
@brief cleanup allocated properties data for a device
@param props the data struct to be processed
@return
*/
static void _e2_devkit_device_properties_clear (DeviceProperties *props)
{
	g_free (props->device_file);
//	g_strfreev (props->device_file_by_id);
//	g_strfreev (props->device_file_by_path);
	g_free (props->device_mount_path);
//	g_free (props->device_presentation_name);
	g_free (props->id_usage);
	g_free (props->id_type);
	g_free (props->drive_media);
}
/**
@brief populate relevant properties for a device represented by @a object_path, if it's one of interest
If @a object_path is not for not a removable drive, props->ignored is set TRUE
The property-names conform to the ones used in gnome-disk-utility 2.28
like "DeviceIsDrive", replacing ones like "device-is-drive"
@param bus connection to system bus for this session
@param object_path DEVKIT_PATH_ROOT etc
@param props pointer to empty (0'd) properties struct to be populated
@return TRUE if process completes successfully, FALSE upon error
*/
static gboolean _e2_devkit_device_properties_get (DBusGConnection *bus,
	const gchar *object_path, DeviceProperties *props)
{
	gboolean retval;
	DBusGProxy *prop_proxy;
	GValue value1 = {0};
	GError *error;
	GHashTable *hash_table;

	memset (props, 0, sizeof(DeviceProperties));	//start with clear data

	prop_proxy = dbus_g_proxy_new_for_name (bus, DEVKIT_IFACE_NAME,
		object_path, "org.freedesktop.DBus.Properties");
	if (prop_proxy == NULL)
	{
//		_e2_devkit_advise (_("message"), NULL);
		printd (WARN, "Unable to connect to devicekit daemon");
		return FALSE;
	}

	error = NULL;
	//no point in doing async interrogations here
	//since most devices are not relevant, best to do a separate, minmimal,
	//check before getting full set of device properties
	if (dbus_g_proxy_call (prop_proxy,
			"Get",
			&error,
			G_TYPE_STRING,
			DEVKIT_IFACE_NAME".Device",
			G_TYPE_STRING,
			"DeviceIsDrive",
			G_TYPE_INVALID,
			G_TYPE_VALUE,
			&value1,
			G_TYPE_INVALID))
	{
		if (g_value_get_boolean (&value1))
			props->device_is_drive = TRUE;
		else
		{
abort:
			g_value_unset (&value1);
			props->ignored = TRUE;
			g_object_unref (G_OBJECT(prop_proxy));
			return TRUE;
		}
	}
	else
	{
		printd (DEBUG,"Get() failed to get properties for %s: %s",
			object_path, error->message);
		g_error_free (error);
		return FALSE;
	}

	g_value_unset (&value1);

	if (dbus_g_proxy_call (prop_proxy,
			"Get",
			&error,
			G_TYPE_STRING,
			DEVKIT_IFACE_NAME".Device",
			G_TYPE_STRING,
			"DeviceIsRemovable",
			G_TYPE_INVALID,
			G_TYPE_VALUE,
			&value1,
			G_TYPE_INVALID))
	{
		if (g_value_get_boolean (&value1))
			props->device_is_removable = TRUE;
		else
			goto abort;
	}

	g_value_unset (&value1);

	//populate other properties the slow and lazy way (get all properties then filter them)
	if (dbus_g_proxy_call (prop_proxy,
			"GetAll",
			&error,
			G_TYPE_STRING,
			DEVKIT_IFACE_NAME".Device",
			G_TYPE_INVALID,
			dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE),
			&hash_table,
			G_TYPE_INVALID))
	{
		GValue *propvalue;

		propvalue = g_hash_table_lookup (hash_table, "DeviceFile");
		if (propvalue != NULL)
			props->device_file = g_strdup (g_value_get_string (propvalue));
//		propvalue = g_hash_table_lookup (hash_table, "DeviceFileById");
//		if (propvalue != NULL)
//			props->device_file_by_id = g_strdupv (g_value_get_boxed (propvalue));
//		propvalue = g_hash_table_lookup (hash_table, "DeviceFileByPath");
//		if (propvalue != NULL)
//			props->device_file_by_path = g_strdupv (g_value_get_boxed (propvalue));
		propvalue = g_hash_table_lookup (hash_table, "DeviceMountPath");
		if (propvalue != NULL)
			props->device_mount_path = g_strdup (g_value_get_string (propvalue));
//		propvalue = g_hash_table_lookup (hash_table, "DevicePresentationName");
//		if (propvalue != NULL)
//			props->device_presentation_name = g_strdup (g_value_get_string (propvalue));
		propvalue = g_hash_table_lookup (hash_table, "IdUsage");
		if (propvalue != NULL)
			props->id_usage = g_strdup (g_value_get_string (propvalue));
		propvalue = g_hash_table_lookup (hash_table, "IdType");
		if (propvalue != NULL)
			props->id_type = g_strdup (g_value_get_string (propvalue));
		propvalue = g_hash_table_lookup (hash_table, "DriveMedia");
		if (propvalue != NULL)
			props->drive_media = g_strdup (g_value_get_string (propvalue));
//		propvalue = g_hash_table_lookup (hash_table, "DeviceMediaDetectionTime");
//		if (propvalue != NULL)
//			props->device_media_detection_time = g_value_get_uint64 (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "DeviceIsMediaAvailable");
		if (propvalue != NULL)
			props->device_is_media_available = g_value_get_boolean (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "DeviceIsMediaChangeDetected");
		if (propvalue != NULL)
			props->device_is_media_change_detected = g_value_get_boolean (propvalue);

		propvalue = g_hash_table_lookup (hash_table, "DeviceIsMediaChangeDetectionInhibited");
		if (propvalue != NULL)
			props->device_is_media_change_detection_inhibited = g_value_get_boolean (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "DeviceIsOpticalDisc");
		if (propvalue != NULL)
			props->device_is_optical_disc = g_value_get_boolean (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "DeviceIsMounted");
		if (propvalue != NULL)
			props->device_is_mounted = g_value_get_boolean (propvalue);
//		propvalue = g_hash_table_lookup (hash_table, "device-is-busy");
//		if (propvalue != NULL)
//			props->device_is_busy = g_value_get_boolean (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "DriveIsMediaEjectable");
		if (propvalue != NULL)
			props->drive_is_media_ejectable = g_value_get_boolean (propvalue);
//		propvalue = g_hash_table_lookup (hash_table, "DriveRequiresEject");
//		if (propvalue != NULL)
//			props->drive_requires_eject = g_value_get_boolean (propvalue);
		propvalue = g_hash_table_lookup (hash_table, "OpticalDiscIsBlank");
		if (propvalue != NULL)
			props->optical_disc_is_blank = g_value_get_boolean (propvalue);

		g_hash_table_destroy (hash_table);
		retval = TRUE;
	}
	else
	{
		printd (DEBUG,"GetAll() failed to get properties for %s: %s",
			object_path, error->message);
		g_error_free (error);
		retval = FALSE;
	}
	g_object_unref (prop_proxy);

	return retval;
}
/**
@brief cleanup device-data
@param data data for a device
@return
*/
static void _e2_devkit_device_data_clear (DeviceData *data)
{
	if (data->timer_id > 0)	//clobber any timeout now in operation
	{
		g_source_remove (data->timer_id);
//redundant unless some race-risk exists		data->timer_id = 0;
	}
	g_free (data->object_path);
	_e2_devkit_device_properties_clear (&data->props);
	g_object_unref(G_OBJECT(data->devproxy));

	DEALLOCATE (DeviceData, data);
}
/**
@brief zero most device-data
@param data data for a device
@return
*/
static void _e2_devkit_device_data_freshen (DeviceData *data)
{
	//NOTE: ensure this cleanup conforms to the contents of data->props
	g_free (data->props.device_mount_path);
	data->props.device_mount_path = NULL;

	g_free (data->props.id_usage);
 	data->props.id_usage = NULL;

	g_free (data->props.id_type);
 	data->props.id_type = NULL;

	g_free (data->props.drive_media);
 	data->props.drive_media = NULL;

	data->props.device_is_mounted = FALSE;
	data->props.device_is_media_available = FALSE;
}
/**
@brief determine whether @a object_path represents a device we're interested in,
 and if so, set up data for it.
@param pool
@param object_path DEVKIT_PATH_ROOT etc
@return pointer to device data, or NULL if not wanted or error
*/
static DeviceData *_e2_devkit_device_get (SessionDevices *pool,
	const gchar *object_path)
{
	DeviceProperties props;
	if (!_e2_devkit_device_properties_get (pool->bus, object_path, &props))
	{
		printd (DEBUG, "Devkit: Cannot determine properties for device %s", object_path);
		return NULL;
	}

	if (props.ignored)
	{
//		printd (DEBUG, "Devkit: Not interested in device %s", object_path);
		return NULL;
	}
	else
	{
		//this is a device of interest
		DeviceData *device;
		DBusGProxy *dev_proxy; //CHECKME needed ?

		device = ALLOCATE0 (DeviceData);
		CHECKALLOCATEDWARN (device, _e2_devkit_device_properties_clear (&props);return NULL;)

		dev_proxy = dbus_g_proxy_new_for_name (pool->bus,
			DEVKIT_IFACE_NAME,
			object_path,
			DEVKIT_IFACE_NAME".Device");

		if (dev_proxy == NULL)
		{
//			_e2_devkit_advise (_("message"), NULL);
			printd (WARN, "Unable to connect to devicekit daemon");
			_e2_devkit_device_properties_clear (&props);
			DEALLOCATE (DeviceData, device);
			return NULL;
		}

		device->object_path = g_strdup (object_path);
		memcpy (&device->props, &props, sizeof (DeviceProperties));
		device->devproxy = dev_proxy;

		dbus_g_proxy_set_default_timeout (dev_proxy, INT_MAX);
		//get idle-time reports if one or more device-properties change
//		dbus_g_proxy_add_signal (dev_proxy, "Changed", G_TYPE_INVALID);
//		dbus_g_proxy_connect_signal (dev_proxy, "Changed",
//			G_CALLBACK (_e2_devkit_device_property_changed_cb), device, NULL);

		return device;
	}
}
/**
@brief log current removable devices
@param pool pointer to devkit data struct
@return TRUE if enumeration succeeded
*/
static gboolean _e2_devkit_detect_current_removables (SessionDevices *pool)
{
	guint i;
	GPtrArray *devices;
	GError *error;

//	printd (DEBUG, "_e2_devkit_detect_current_removables");
	error = NULL;
	//CHECKME synchronous process ok ?
	if (!dbus_g_proxy_call (pool->proxy, "EnumerateDevices", &error, G_TYPE_INVALID,
			dbus_g_type_get_collection ("GPtrArray", DBUS_TYPE_G_OBJECT_PATH),
			&devices, G_TYPE_INVALID))
	{
		printd (WARN, "Cannot get devices data: %s", error->message);
		g_error_free (error);
		return FALSE;
	}

//	printd (DEBUG, "Devkit: %d devices reported", devices->len);
	for (i = 0; i < devices->len; i++)
	{
		DeviceData *device = _e2_devkit_device_get (pool, devices->pdata[i]);
		if (device != NULL)
		{
			printd (DEBUG, "Devkit: Logging device %s", devices->pdata[i]);
			g_hash_table_insert (pool->logged_devices,
				(gpointer) device->object_path,	//not another copy, not truncated
				(gpointer) device);
		}
//		else
//			printd (DEBUG, "Devkit: Skipping device %s @ %x", devices->pdata[i], device);

		g_free (devices->pdata[i]); //CHECKME does the array-data self-destruct (glib >= 2.22)
		devices->pdata[i] = NULL;	//in case of double-free when array destroyed
	}

	g_ptr_array_free (devices, TRUE);
	return TRUE;
}
/**
@brief automount a volume
This is called from within timer callback i.e. BGL is off
@param pool
@param device

@return TRUE if the mount is performed successfully or is not needed
*/
static gboolean _e2_devkit_mount_volume (SessionDevices *pool, DeviceData *device)
{
	gboolean optical;
	gchar *fstype;

	optical = (device->props.drive_media != NULL
				&& g_str_has_prefix (device->props.drive_media, "optical"));

	if (device->props.device_is_mounted
		|| !device->props.device_is_media_available
		|| (optical && device->props.optical_disc_is_blank)
		)
	{
		printd (DEBUG, "Devkit: device does not need to be mounted");
		return TRUE;
	}

	fstype = device->props.id_usage;
	if (fstype == NULL || strcmp (fstype, "filesystem"))	//CHECKME "crypto", "raid", "other"
	{
		printd (DEBUG, "Devkit: mounting this device-type is not supported");
		return TRUE;	//some unmountable sort of device
	}

	fstype = device->props.id_type;
	if (fstype == NULL || *fstype == '\0')
	{
		printd (DEBUG, "Devkit: device filesystem-type is not recognised");
		return TRUE;	//not a data-medium
	}

#if 0
	//determine a suitable mountpoint ... apparently unusable if devkit does the mounting

#define FIRSTOPTIC 10
	const gchar *disktypes[] =
	{
		//for DeviceKit-Disks 004
		"flash",
		"flash_cf",
		"flash_ms",
		"flash_sm",		//aka "smart"

		"flash_sd",
		"flash_sdhc",
		"flash_mmc",	//aka "sd_mmc"

		"floppy",
		"floppy_zip",	//aka "zip"
		"floppy_jaz",	//aka "jaz"

		"optical_cd",	//"cd", index 10
		"optical_cd_r",	//"cdrom"
		"optical_cd_rw", //"cd" writable, index 12
		"optical_dvd",	//"dvd"
		"optical_dvd_r", //"dvdrom"
		"optical_dvd_rw", //"dvd" writable 15
		"optical_dvd_ram", //writable 16
		"optical_dvd_plus_r",
		"optical_dvd_plus_rw", //writable 18
		"optical_dvd_plus_r_dl",
		"optical_dvd_plus_rw_dl", //writable 20
		"optical_bd",
		"optical_bd_r",
		"optical_bd_re",
		"optical_hddvd", //24
		"optical_hddvd_r",
		"optical_hddvd_rw", //writable 26
		"optical_mo",
		"optical_mrw",
		"optical_mrw_w", //writable 29
	};

	const gchar *last;
	gboolean readonly;
	gint i;

/*	check for an available mountpoint
	mountpoint from lsb spec:
		/media
			/floppy	Floppy drive (optional)
			/cdrom	CD-ROM drive (optional)
			/cdrecorder	CD writer (optional)
			/zip	Zip drive (optional)
	Where more than one device exists for mounting a certain type of media, mount
	directories can be created by appending a digit to the name, starting with
	'0', but the unqualified name must also exist

	for device names:
	Partitions or volumes will appear as device-appended numbers, like
	   sda1, sda2 ....
	Some devices e.g. cdrom are themselves treated as a volume, and will then
	have no partition number
	Devices to be mounted may already exist in /dev
*/
	if (optical)
	{	//this volume is, or is part of, an optical disk
		gint count = sizeof (disktypes) / sizeof (const gchar *);
		if (device->props.drive_media == NULL || *device->props.drive_media == '\0')
			i = count;	//set defaults
		else
		{
			for (i = FIRSTOPTIC; i < count; i++)
			{
				if (!strcmp (disktypes[i], device->props.drive_media))
				{
					switch (i)
					{
						case 12:
						case 15:
						case 16:
						case 18:
						case 20:
						case 26:
						case 29:
							if (i <= 12)
								last = "cdrecorder";
							else if (i <= 20 || i == 24 || i == 25 || i == 26)
								last = "dvdrecorder";
							else
								last = "recorder";
							readonly = FALSE;
							break;
						default:
							if (i <= 12)
								last = "cdrom";
							else if (i <= 20 || i == 24 || i == 25 || i == 26)
								last = "dvdrom";
							else
								last = "disk";
							readonly = TRUE;
							break;
					}
					break;
				}
			}
		}
		if (i == count)
		{	//defaults
			last = "cdrom";
			readonly = TRUE;
		}
	}
	else //not an optical disk
	{
		if (device->props.drive_media == NULL || *device->props.drive_media == '\0')
			i = FIRSTOPTIC;	//set defaults
		else
		{
			for (i = 0; i < FIRSTOPTIC; i++)
			{
				last = disktypes[i];
				if (!strcmp (last, device->props.drive_media))
				{
					switch (i)
					{
						default:
							last = "disk";	//FIXME
							break;
						case 7:
							last = "floppy";
							break;
						case 8:
							last = "zip";
							break;
						case 9:
							last = "jaz";
							break;
					}
					readonly = FALSE;
					break;
				}
			}
		}
		if (i == FIRSTOPTIC)
		{ 	//defaults
			last = "disk";
			readonly = FALSE;
		}
	}

	GList *mounts_now, *member;
	gchar *str, *utf, *mountpoint;

	mounts_now = e2_fs_mount_get_mounts_list ();
	mountpoint = g_build_filename (E2_MOUNTPLACE, last, NULL);
	str = g_strdup (mountpoint);
	utf = F_FILENAME_FROM_LOCALE (str);	//not strictly necessary, the default path is all ASCII
	guint count = 0;
retry:
	for (member = mounts_now; member != NULL; member = member->next)
	{
		if (!strcmp ((gchar*)member->data, utf))
		{
			g_free (str);
			F_FREE (utf, str);
			str = g_strdup_printf ("%s%d", mountpoint, count++);
			utf = F_FILENAME_FROM_LOCALE (str);
			goto retry;
		}
	}

	g_free (mountpoint);
	F_FREE (utf, str);
	e2_list_free_with_data (&mounts_now);
	mountpoint = str;

	//determine mount options from the available ones ... apparently unusable if devkit does the mounting

	gboolean readonly_valid = FALSE;
	gint freeindex = -1;

	GPtrArray *extra_options = g_ptr_array_sized_new (6);
	gboolean FIXME; //get suitable options
	gchar **mountopts = NULL;
	if (mountopts != NULL)
	{
		gchar **thisopt;
		for (thisopt = mountopts; *thisopt != NULL; thisopt++)
		{
		//CHECKME which mount options are most suitable ?
		//e.g. ro,sync,dirsync,noatime,nodiratime,noexec,quiet,remount,exec,utf8
		//shortname=,codepage=,iocharset=,umask=,dmask=,fmask=,uid=,flush
			if (strcmp (*thisopt, "exec") == 0)
			{
				g_ptr_array_add (extra_options, *thisopt);
			}
			else if (strncmp (*thisopt, "umask=", 6) == 0)
			{
				g_ptr_array_add (extra_options, g_strconcat (*thisopt, "0", NULL));
				freeindex = extra_options->len - 1; //for leak fix
			}
			else if (strcmp (*thisopt, "sync") == 0)
			{
				g_ptr_array_add (extra_options, *thisopt);
			}
			else if (strcmp (*thisopt, "ro") == 0)
			{
				readonly_valid = TRUE;
				if (readonly)
					g_ptr_array_add (extra_options, *thisopt);
			}
			else if (readonly)
			{
				readonly_valid = TRUE;
					g_ptr_array_add (extra_options, "ro");
			}

		}
	}
	g_ptr_array_add (extra_options, NULL);	//terminate the array
#endif //0

	gboolean retval;

	//do it
# if 0 //direct
	gint res = run command
	retval = (res == 0);
/*	if (retval)
	{
		//"refresh" the filelist(s) if needed
		gchar *utf = F_FILENAME_FROM_LOCALE (mountpoint);
		if (g_str_has_prefix (curr_view->dir, utf))
			do something
		if (g_str_has_prefix (other_view->dir, utf))
			do something
		F_FREE (utf, mountpoint);
	}
*/
# else //via devkit
	gpointer cb_data = device;
#if 1
	gchar *options[2];

	options[0] = NULL;
#endif

	retval =
		(dbus_g_proxy_begin_call (device->devproxy, "FilesystemMount",
		_e2_devkit_mount_async_cb,
		cb_data, (GDestroyNotify) NULL,
		G_TYPE_STRING, fstype,
#  if 0
		G_TYPE_STRV, extra_options->pdata,
#  else
		G_TYPE_STRV, options,  //no mount options
#  endif
		G_TYPE_INVALID) != NULL);
# endif
#if 0
	//cleanup
	g_free (mountpoint);
	//CHECKME free all options ? does the array-data self-destruct (glib >= 2.22)
	if (freeindex != -1)
		g_free (g_ptr_array_index (extra_options, freeindex));
	g_ptr_array_free (extra_options, TRUE);
#endif
	return retval;
}

/**
@brief auto-unmount a volume
This is called from within timer callback i.e. BGL is off
@param pool pointer to devkit data struct
@param device pointer to data for the device

@return TRUE if the unmount is performed successfully or is not needed
*/
static gboolean _e2_devkit_unmount_volume (SessionDevices *pool, DeviceData *device)
{
	//CHECKME upstream check for still-need-unmount is too racy?
	if (!device->props.device_is_mounted)
	{
		printd (DEBUG, "_e2_devkit_unmount_volume() - no need to do anything");
		return TRUE;
	}

	const gchar *mountpoint = device->props.device_mount_path;
	if (mountpoint != NULL	//should never happen
#ifdef E2_VFSTMP
	//FIXME dir when not mounted local
#else
		&& g_str_has_prefix (curr_view->dir, mountpoint))
#endif
	{	//if possible, change CWD out of mountpoint so we don't ourself block the unmount
		gchar *elsewhere;
		const gchar *s = strrchr (mountpoint, G_DIR_SEPARATOR);	//assumes no trailer on mountpoint string
		if (s > mountpoint + 1) //not doing root dir
			elsewhere = g_strndup (mountpoint, s - mountpoint);
		else //can't cd
			elsewhere = g_strdup (G_DIR_SEPARATOR_S);

		e2_fs_get_valid_path (&elsewhere, TRUE E2_ERR_NONE());
		e2_pane_change_dir (NULL, elsewhere);
		g_free (elsewhere);
	}

	//do it
# if 0 //direct
	gint res = run command
	retval = (res == 0);
	//do stuff if successful
	if (retval)
		_e2_devkit_device_data_freshen (device);
	return retval;
# else //via devkit
	gpointer cb_data = device;
	gchar *options[2];

#ifdef DEBUG_MESSAGES
	gboolean CHECKME;
#endif
	options[0] = NULL;

	return
		(dbus_g_proxy_begin_call (device->devproxy, "FilesystemUnmount",
		_e2_devkit_unmount_async_cb,
		cb_data, (GDestroyNotify) NULL,
		G_TYPE_STRV, options, G_TYPE_INVALID) != NULL);
# endif
}

  /*********************/
 /***** callbacks *****/
/*********************/

/**
@brief callback for completion of async mount operation
@param proxy
@param call
@param user_data pointer to device data specified when cb was connected (do not free)
@return
*/
static void _e2_devkit_mount_async_cb
	(DBusGProxy *proxy, DBusGProxyCall *call, gpointer user_data)
{
	GError *error = NULL;
	gchar *mount_path;

	dbus_g_proxy_end_call (proxy, call, &error, G_TYPE_STRING, &mount_path,
		G_TYPE_INVALID);
	if (error == NULL)
	{
		printd (DEBUG, "Device mounted at %s", mount_path);
		//do stuff ?
/*		//"refresh" the filelist(s) if needed
		gchar *utf = F_FILENAME_FROM_LOCALE (mount_path);
		if (g_str_has_prefix (curr_view->dir, utf))
			do something
		if (g_str_has_prefix (other_view->dir, utf))
			do something
		F_FREE (utf, mount_path);
*/
		g_free (mount_path);
	}
	else
	{
		gchar *msg = g_strdup_printf (_("Error mounting device: %s"), error->message);
		e2_output_print_error (msg, TRUE);
		g_error_free (error);
	}
}
/**
@brief callback for completion of async unmount operation
@param proxy
@param call
@param user_data pointer to device data specified when cb was connected (do not free)
@return
*/
static void _e2_devkit_unmount_async_cb
	(DBusGProxy *proxy, DBusGProxyCall *call, gpointer user_data)
{
	GError *error = NULL;
	dbus_g_proxy_end_call (proxy, call, &error, G_TYPE_INVALID);

	if (error == NULL)
	{
		//do stuff ?
		DeviceData *device = (DeviceData *)user_data;
		printd (DEBUG, "Device %s unmounted", device->object_path);
		_e2_devkit_device_data_freshen (device);
	}
	else
	{
		gchar *msg = g_strdup_printf (_("Error unmounting device: %s"), error->message);
		e2_output_print_error (msg, TRUE);
		g_error_free (error);
	}
}
/**
@brief timer callback after a device-add event
If the device's volume(s) are not yet mounted (by the OS etc), mount them now.
Any unmount that happens before this timeout expired will have aborted this callback.
@param device pointer to data for the device

@return FALSE (to cancel the timer) unless mount-operation is needed but it failed
*/
static gboolean _e2_devkit_check_mount (DeviceData *device)
{
	device->timer_id = 0;	//ASAP indicate to any other process
							//but we don't remove this source in case mount fails
	printd (DEBUG, "Devkit: mount timer callback, device %s", device->props.device_file);

	if (!device->props.device_is_mounted
		//&& non-blank media is present
		)
	{
		printd (DEBUG, "device needs to be mounted");
#ifdef DEBUG_MESSAGES
		gboolean retval = !_e2_devkit_mount_volume (&devicesdata, device);
		if (retval)
			printd (DEBUG, "mount operation failed");
		return retval;	//try again in case of failure
#else
		return (!_e2_devkit_mount_volume (&devicesdata, device));
#endif
	}

	return FALSE;
}
/**
@brief callback for a device-add event
@param proxy connection to devkit daemon
@param object_path DEVKIT_PATH_ROOT etc
@param user_data pointer to device monitor data struct specified when callback was connected
@return
*/
static void _e2_devkit_device_added_cb (DBusGProxy *proxy,
	const gchar *object_path, gpointer user_data)
{
	SessionDevices *pool;
	DeviceData *device;

	pool = (SessionDevices *)user_data;
	device = g_hash_table_lookup (pool->logged_devices, object_path);

	if (G_LIKELY(device == NULL))
	{
		device = _e2_devkit_device_get (pool, object_path);
		if (device != NULL)
		{
			//new device is one we want
			printd (DEBUG, "Devkit: Logging device %s @ %x", object_path, device);
			g_hash_table_insert (pool->logged_devices,
				(gpointer) device->object_path,	//not another copy, not truncated
				(gpointer) device);
			//CHECKME is 2-sec delay long enough ?
			device->timer_id =
#ifdef USE_GLIB2_14
				g_timeout_add_seconds (2,
#else
				g_timeout_add (2000,
#endif
				(GSourceFunc) _e2_devkit_check_mount, device);
		}
		else
			printd (DEBUG,"Not monitoring device %s", object_path);
	}
	else
	{
		//clobber any unmount callback
		if (device->timer_id > 0)
		{
			g_source_remove (device->timer_id);
			device->timer_id = 0;
		}
		printd (WARN, "Devkit: add-event for previously added device %s, treat as change",
			object_path);
		_e2_devkit_device_changed_cb (proxy, object_path, user_data);
	}
}
/**
@brief timer callback after a device-remove event
If the device is not unmounted yet (by the OS etc), unmount it now
Also, always cleanup the mount data
@param device pointer to data struct for the affected device

@return FALSE (to cancel the timer) if unmount succeeded or not needed
*/
static gboolean _e2_devkit_check_unmount (DeviceData *device)
{
	gboolean retval;

	if (device->timer_id > 0)
	{
		g_source_remove (device->timer_id);	//the source may now be unrelated to this callback
		device->timer_id = 0;	//ASAP indicate to any other process
	}

	printd (DEBUG, "Devkit: unmount timer callback");

	if (device->props.device_is_mounted)
	{
			printd (DEBUG, "device needs to be unmounted");
			retval = !_e2_devkit_unmount_volume (&devicesdata, device);
#ifdef DEBUG_MESSAGES
			if (retval)
				printd (DEBUG, "but unmount operation failed");
#endif
	}
	else
	{
		printd (DEBUG, "device is already unmounted");
		retval = FALSE;
		_e2_devkit_device_data_freshen (device);
	}

	return retval;
}
/**
@brief callback for a device-remove event
@param proxy
@param object_path DEVKIT_PATH_ROOT etc
@param user_data pointer to device monitor data struct specified when callback was connected
@return
*/
static void _e2_devkit_device_removed_cb (DBusGProxy *proxy,
	const gchar *object_path, gpointer user_data)
{
	SessionDevices *pool;
	DeviceData *device;

	pool = (SessionDevices *)user_data;
	if (pool->logged_devices == NULL)
		return;
	device = g_hash_table_lookup (pool->logged_devices, object_path);

	if (G_LIKELY(device != NULL))
	{
		printd (DEBUG,"Remove logged device %s", object_path);
#ifdef DEBUG_MESSAGES
		gboolean CHECKME;	//do other stuff ?
#endif
		//wait awhile (longer than the mount-timer interval) to check for unmount and then cleanup
		device->timer_id =
#ifdef USE_GLIB2_14
			g_timeout_add_seconds (5,
#else
			g_timeout_add (5000,
#endif
				(GSourceFunc) _e2_devkit_check_unmount, device);
	}
	else
		printd (WARN,"No logged device to remove for %s", object_path);
}
/**
@brief callback for a property-change for a logged (removable) device
This will be called twice when a device is added/removed. The 2nd call does nothing.
@param proxy connection to devkit daemon
@param object_path DEVKIT_PATH_ROOT/devicename (e.g. sr0, but with any '-' converted to '_')
@param user_data pointer to session-devices data struct specified when callback was connected
@return
*/
static void _e2_devkit_device_changed_cb (DBusGProxy *proxy,
	const gchar *object_path, gpointer user_data)
{
	SessionDevices *pool;
	DeviceData *device;

	printd (DEBUG,"_e2_devkit_device_changed_cb: device %s", object_path);
	pool = (SessionDevices *)user_data;
	if (pool->logged_devices == NULL)
	{
#ifdef DEBUG_MESSAGES
unknown:
#endif
		printd (DEBUG,"Ignoring change event on un-logged device");
		return;
	}
	device = g_hash_table_lookup (pool->logged_devices, object_path);

	if (G_LIKELY(device != NULL))
	{
		DeviceProperties new_properties;

		if (device->timer_id > 0)
		{
			g_source_remove (device->timer_id);	//don't need any current timeout
			device->timer_id = 0;
		}

		if (_e2_devkit_device_properties_get (pool->bus, object_path, &new_properties))
		{
			if (G_LIKELY(!new_properties.ignored))
			{
				gboolean differ1;	//, differ2;

				differ1 = (new_properties.device_is_media_available != device->props.device_is_media_available);
//				printd (DEBUG, "Media availability is %s", (differ1)? "different now":"unchanged");
//				differ2 = (new_properties.device_is_media_change_detected != device->props.device_is_media_change_detected);

				//replace all, in case other property(ies) also changed
				_e2_devkit_device_properties_clear (&device->props);
				memcpy (&device->props, &new_properties, sizeof (DeviceProperties));
				if (differ1)
				{
#ifdef DEBUG_MESSAGES
					gboolean CHECKME2;	//do other stuff ?
#endif
					if (new_properties.device_is_media_available)
					{
						device->timer_id =
#ifdef USE_GLIB2_14
							g_timeout_add_seconds (2,
#else
							g_timeout_add (2000,
#endif
							(GSourceFunc) _e2_devkit_check_mount, device);
					}
					else
					{
						device->timer_id =
#ifdef USE_GLIB2_14
							g_timeout_add_seconds (5,
#else
							g_timeout_add (5000,
#endif
							(GSourceFunc) _e2_devkit_check_unmount, device);
					}
				}
			}
			else
			{
				printd (WARN,"device %s is no longer suitable for monitoring", object_path);
				//FIXME unmount? cleanup
			}
		}
	}
#ifdef DEBUG_MESSAGES
	else
		goto unknown;
#endif
}
/* *
@brief callback for any changed property(ies) for a removable device
This callback happens only at idle-time, and before any device-changed callback.
There is no info on what properties have changed, so we must poll.
@param object UNUSED DEVKIT_DISKS_DEVICE object
@param user_data pointer to data struct for device that's changed
@return
*/
/*static void _e2_devkit_device_property_changed_cb (GObject *obj,
	gpointer user_data)
{
	DeviceData *device;

	printd (DEBUG,"_e2_devkit_device__property_changed_cb: device %x", user_data);
	device = (DeviceData*)user_data;
	gboolean TODO;
}
*/

  /********************/
 /****** public ******/
/********************/

/**
@brief helper func for finding a logged removable device with a specific device-path
@param key UNUSED key for a hashtable entry
@param value value for a hashtable entry
@param devpath the device we're looking for
@return TRUE when a match is found
*/
static gboolean _e2_devkit_check_device_path (gpointer *key, gpointer value,
	const gchar *devpath)
{
	if (((DeviceData*)value)->props.device_file == NULL)
	{	//some setup race/problem happened
		printd (DEBUG, "Device %s has a NULL device-file", devpath);
		return FALSE;
	}
	return (!strcmp(((DeviceData*)value)->props.device_file, devpath));
}
/**
@brief check whether native @a devpath resides on a removable device
@param devpath /dev/<whatever>
@return TRUE if it's removable
*/
gboolean e2_devkit_device_is_removable (const gchar *devpath)
{
	if (devicesdata.logged_devices != NULL)
	{
		return (g_hash_table_find (devicesdata.logged_devices,
				(GHRFunc) _e2_devkit_check_device_path,
				(gpointer)devpath) != NULL);
	}
	return FALSE;
}
/**
@brief check whether native @a devpath resides on an ejectable device
@param devpath /dev/<whatever>
@return TRUE if it's ejectable
*/
gboolean e2_devkit_device_is_ejectable (const gchar *devpath)
{
	if (devicesdata.logged_devices != NULL)
	{
		DeviceData *dev;
		dev = (DeviceData *)g_hash_table_find (devicesdata.logged_devices,
				(GHRFunc) _e2_devkit_check_device_path,
				(gpointer)devpath);
		if (dev != NULL)
			return (dev->props.drive_is_media_ejectable
					//? && dev->props.device_is_media_available
					);
	}
	return FALSE;
}
/**
@brief initialise connection to devicekit daemon, and log current removables
@return
*/
void e2_devkit_init (void)
{
	SessionDevices *pool;
	GError *error;

	pool = &devicesdata;
	error = NULL;

	pool->bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (pool->bus == NULL)
	{
//		_e2_devkit_advise (_("message"), NULL);
		printd (WARN, "Unable to connect to system bus: %s", error->message);
		g_error_free (error);
		return;
	}
	pool->proxy = dbus_g_proxy_new_for_name (pool->bus,
											DEVKIT_IFACE_NAME,
											"/org/freedesktop/DeviceKit/Disks",
											DEVKIT_IFACE_NAME);
	//CHECKME specifically confirm daemon presence ?
	if (pool->proxy == NULL)
	{
//		_e2_devkit_advise (_("message"), NULL);
		printd (WARN, "Unable to connect to devicekit daemon");
		dbus_g_connection_unref (pool->bus);
		return;
	}

	//table with key = D-Bus object path, value = the corresponding device data struct
	pool->logged_devices = g_hash_table_new_full (
		g_str_hash, g_str_equal, NULL,
		(GDestroyNotify)_e2_devkit_device_data_clear);

	if (_e2_devkit_detect_current_removables (pool))
	{
		const gchar *signame;

		signame = "DeviceAdded";
		dbus_g_proxy_add_signal (pool->proxy, signame, DBUS_TYPE_G_OBJECT_PATH,
			G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (pool->proxy, signame,
			 G_CALLBACK (_e2_devkit_device_added_cb), pool, NULL);
		signame = "DeviceRemoved";
		dbus_g_proxy_add_signal (pool->proxy, signame, DBUS_TYPE_G_OBJECT_PATH,
			G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (pool->proxy, signame,
			 G_CALLBACK (_e2_devkit_device_removed_cb), pool, NULL);
		signame = "DeviceChanged";
		dbus_g_proxy_add_signal (pool->proxy, signame, DBUS_TYPE_G_OBJECT_PATH,
			G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (pool->proxy, signame,
			 G_CALLBACK (_e2_devkit_device_changed_cb), pool, NULL);
	}
	else
	{
//		_e2_devkit_advise (_("message"), NULL);
		dbus_g_connection_unref (pool->bus);
		g_object_unref (G_OBJECT(pool->proxy));
		g_hash_table_destroy (pool->logged_devices);
		memset (pool, 0, sizeof (SessionDevices));
	}
}
/**
@brief end devicekit connection and related cleanup
@return
*/
void e2_devkit_disconnect (void)
{
	SessionDevices *pool;

	pool = &devicesdata;
	if (pool->bus != NULL)
		dbus_g_connection_unref (pool->bus);
	if (pool->proxy != NULL)
		g_object_unref (G_OBJECT(pool->proxy));
	if (pool->logged_devices != NULL)
		g_hash_table_destroy (pool->logged_devices);
}
#endif //def E2_DEVKIT
