/* $Id: e2p_selmatch.c 1835 2009-09-17 00:35:44Z tpgww $

Copyright (C) 2009 tooar <tooar@emelfm2.net>

This file is part of emelFM2.
emelFM2 is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3, or (at your option)
any later version.

emelFM2 is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emelFM2; see the file GPL. If not, contact the Free Software
Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

/**
@file plugins/e2p_selmatch.c
@brief plugin for selecting active[-pane items which are selected in inactive pane
*/

#include "emelfm2.h"
#include "e2_plugins.h"
#include "e2_fileview.h"
#include "e2_filelist.h"

/**
@brief iterate over active pane file list to check for matches
@param from the button, menu item etc which was activated
@param art action runtime data

@return TRUE if action completed successfully, else FALSE
*/
static gboolean _e2p_select_same (gpointer from, E2_ActionRuntime *art)
{
	E2_PaneRuntime *rt;
	ViewInfo *selview;
	GtkTreeModel *models;
	GtkTreeIter iters;

	rt = e2_pane_get_runtime (from, art->data, NULL);
	selview = (ViewInfo *)rt;
	WAIT_FOR_REFRESH(selview)
	models = selview->model;
	if (gtk_tree_model_get_iter_first (models, &iters))
	{	//it's not empty
		ViewInfo *othview;
		GtkTreeModel *modelo;
		GtkTreeSelection *sel;
		GList *selpaths, *rowpath;
		GHashTable *selitems;
		FileInfo *info;

		e2_filelist_disable_refresh ();

		othview = (rt == curr_pane) ? &other_pane->view : &curr_pane->view;
		WAIT_FOR_REFRESH(othview)
		selpaths = gtk_tree_selection_get_selected_rows (othview->selection, &modelo);
		if (selpaths == NULL)
		{
			e2_filelist_enable_refresh ();
			return FALSE;
		}
		// Log the selected items, for quick lookup
		selitems = g_hash_table_new (g_str_hash, g_str_equal);
		for (rowpath = selpaths; rowpath != NULL; rowpath = rowpath->next)
		{
			GtkTreePath *path;
			GtkTreeIter itero;
			path = (GtkTreePath *) rowpath->data;
			if (gtk_tree_model_get_iter (modelo, &itero, path))
			{
				gtk_tree_model_get (modelo, &itero, FINFO, &info, -1);
				g_hash_table_insert (selitems, info->filename, info); //no key dup, info must persist
			}
			gtk_tree_path_free (path);
		}
		g_list_free (selpaths);

		sel = selview->selection;
		gtk_tree_selection_unselect_all (sel);	//start with clean slate
		do
		{
			gtk_tree_model_get (models, &iters, FINFO, &info, -1);
			//We only check for name, ignore other statbuf parameters
			if (g_hash_table_lookup(selitems, info->filename) != NULL)
				gtk_tree_selection_select_iter (sel, &iters);
		} while (gtk_tree_model_iter_next (models, &iters));

		e2_filelist_enable_refresh ();
		g_hash_table_destroy (selitems);

		return TRUE;
	}
	return FALSE;
}

//aname must be confined to this module
static gchar *aname;
/**
@brief plugin initialization function, called by main program

@param p ptr to plugin data struct

@return TRUE if the initialization succeeds, else FALSE
*/
gboolean init_plugin (Plugin *p)
{
#define ANAME "selmatch"
  aname = _("selmatch");

  p->signature = ANAME VERSION;
  p->menu_name = _("_Select same");
  p->description = _("Select items whose name matches a selected item in the other pane");
  p->icon = "plugin_"ANAME E2ICONTB;  //prepend path if appropriate

  if (p->action == NULL)
 {
	//don't free name string here
	E2_Action plugact =
	{g_strconcat (_A(7),".",aname,NULL),_e2p_select_same,FALSE,E2_ACTION_TYPE_ITEM,0,NULL,NULL}; // not _A(6) !
	p->action = e2_plugins_action_register (&plugact);
	if G_LIKELY((p->action != NULL))
		return TRUE;
	g_free (plugact.name);
  }
  return FALSE;
}
/**
@brief cleanup transient things for this plugin

@param p pointer to data struct for the plugin

@return TRUE if all cleanups were completed
*/
gboolean clean_plugin (Plugin *p)
{
  gchar *action_name = g_strconcat (_A(7),".",aname,NULL);
  gboolean ret = e2_plugins_action_unregister (action_name);
  g_free (action_name);
  return ret;
}
