#include <Elementary.h>
#include "elm_priv.h"

typedef struct _Widget_Data Widget_Data;
struct _Widget_Data 
{
   Evas_Object *scr, *bx;
   Elm_Panel_Orient orient;
   Eina_Bool hidden : 1;
};

static void _del_hook(Evas_Object *obj);
static void _theme_hook(Evas_Object *obj);
static void _sizing_eval(Evas_Object *obj);
static void _resize(void *data, Evas *evas, Evas_Object *obj, void *event);
static void _layout(Evas_Object *o, Evas_Object_Box_Data *priv, void *data);
static void _toggle_panel(void *data, Evas_Object *obj, const char *emission, const char *source);

static void 
_del_hook(Evas_Object *obj) 
{
   Widget_Data *wd = elm_widget_data_get(obj);

   free(wd);
}

static void 
_theme_hook(Evas_Object *obj) 
{
   Widget_Data *wd = elm_widget_data_get(obj);
   int scale = 0;

   scale = (elm_widget_scale_get(obj) * _elm_config->scale);
   edje_object_scale_set(wd->scr, scale);
   _sizing_eval(obj);
}

static void 
_sizing_eval(Evas_Object *obj) 
{
   Widget_Data *wd = elm_widget_data_get(obj);
   Evas_Coord mw = -1, mh = -1;
   Evas_Coord vw = 0, vh = 0;
   Evas_Coord w, h;

   evas_object_smart_calculate(wd->bx);
   edje_object_size_min_calc(elm_smart_scroller_edje_object_get(wd->scr), 
                             &mw, &mh);
   evas_object_geometry_get(obj, NULL, NULL, &w, &h);
   if (w < mw) w = mw;
   if (h < mh) h = mh;
   evas_object_resize(wd->scr, w, h);

   evas_object_size_hint_min_get(wd->bx, &mw, &mh);
   if (w > mw) mw = w;
   if (h > mh) mh = h;
   evas_object_resize(wd->bx, mw, mh);

   elm_smart_scroller_child_viewport_size_get(wd->scr, &vw, &vh);
   mw = mw + (w - vw);
   mh = mh + (h - vh);
   evas_object_size_hint_min_set(obj, mw, mh);
   evas_object_size_hint_max_set(obj, -1, -1);
}

static void 
_resize(void *data, Evas *evas, Evas_Object *obj, void *event) 
{
   Widget_Data *wd = elm_widget_data_get(data);
   Evas_Coord mw, mh, vw, vh, w, h;

   elm_smart_scroller_child_viewport_size_get(wd->scr, &vw, &vh);
   evas_object_size_hint_min_get(wd->bx, &mw, &mh);
   evas_object_geometry_get(wd->bx, NULL, NULL, &w, &h);
   if ((vw >= mw) || (vh >= mh))
     {
        if ((w != vw) || (h != vh)) evas_object_resize(wd->bx, vw, vh);
     }
}

static void 
_layout(Evas_Object *o, Evas_Object_Box_Data *priv, void *data) 
{
   Widget_Data *wd = data;

   _els_box_layout(o, priv, EINA_TRUE, EINA_FALSE);
}

static void 
_toggle_panel(void *data, Evas_Object *obj, const char *emission, const char *source) 
{
   Widget_Data *wd = elm_widget_data_get(data);

   if (wd->hidden) 
     {
        edje_object_signal_emit(elm_smart_scroller_edje_object_get(wd->scr), 
                                "elm,action,show", "elm");
        wd->hidden = EINA_FALSE;
     }
   else
     {
        edje_object_signal_emit(elm_smart_scroller_edje_object_get(wd->scr), 
                                "elm,action,hide", "elm");
        wd->hidden = EINA_TRUE;
     }
}

EAPI Evas_Object *
elm_panel_add(Evas_Object *parent) 
{
   Evas_Object *obj;
   Evas *evas;
   Widget_Data *wd;

   wd = ELM_NEW(Widget_Data);
   evas = evas_object_evas_get(parent);
   obj = elm_widget_add(evas);
   elm_widget_type_set(obj, "panel");
   elm_widget_sub_object_add(parent, obj);
   elm_widget_data_set(obj, wd);
   elm_widget_del_hook_set(obj, _del_hook);
   elm_widget_theme_hook_set(obj, _theme_hook);
   elm_widget_can_focus_set(obj, 0);

   wd->scr = elm_smart_scroller_add(evas);
   elm_smart_scroller_bounce_allow_set(wd->scr, 0, 0);
   elm_smart_scroller_theme_set(wd->scr, "panel", "base", "left");
   elm_widget_resize_object_set(obj, wd->scr);
   elm_smart_scroller_policy_set(wd->scr, ELM_SMART_SCROLLER_POLICY_OFF, 
                                 ELM_SMART_SCROLLER_POLICY_OFF);

   wd->hidden = FALSE;
   wd->orient = ELM_PANEL_ORIENT_LEFT;

   wd->bx = evas_object_box_add(evas);
   evas_object_size_hint_align_set(wd->bx, 0.5, 0.5);
   evas_object_box_layout_set(wd->bx, _layout, wd, NULL);
   elm_widget_sub_object_add(obj, wd->bx);
   elm_smart_scroller_child_set(wd->scr, wd->bx);
   evas_object_show(wd->bx);

   edje_object_signal_callback_add(elm_smart_scroller_edje_object_get(wd->scr), 
                                   "elm,action,panel,toggle", "*", 
                                   _toggle_panel, obj);

   evas_object_event_callback_add(wd->scr, EVAS_CALLBACK_RESIZE, _resize, obj);

   _sizing_eval(obj);
   return obj;
}

EAPI void 
elm_panel_orient_set(Evas_Object *obj, Elm_Panel_Orient orient) 
{
   Widget_Data *wd = elm_widget_data_get(obj);

   wd->orient = orient;
   switch (orient) 
     {
      case ELM_PANEL_ORIENT_TOP:
      case ELM_PANEL_ORIENT_BOTTOM:
        break;
      case ELM_PANEL_ORIENT_LEFT:
        elm_smart_scroller_theme_set(wd->scr, "panel", "base", "left");
      case ELM_PANEL_ORIENT_RIGHT:
        break;
      default:
        break;
     }
   _sizing_eval(obj);
}

EAPI void 
elm_panel_content_set(Evas_Object *obj, Evas_Object *content) 
{
   Widget_Data *wd = elm_widget_data_get(obj);

   evas_object_box_remove_all(wd->bx, EINA_TRUE);
   if (!content) return;
   evas_object_box_append(wd->bx, content);
   evas_object_show(content);
   _sizing_eval(obj);
}
