/*
 * xvmalloc.c
 *
 * Copyright (C) 2008, 2009  Nitin Gupta
 *
 * This code is released using a dual license strategy: GPL/LGPL
 * You can choose the licence that better fits your requirements.
 *
 * Released under the terms of the GNU General Public License Version 2.0
 * Released under the terms of the GNU Lesser General Public License Version 2.1
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/bitops.h>
#include <linux/errno.h>
#include <linux/highmem.h>
#include <linux/string.h>
#include <linux/slab.h>

#include "xvmalloc.h"

#ifdef CONFIG_XV_STATS
static void statInc(u64 *value)
{
	*value = *value + 1;
}

static void statDec(u64 *value)
{
	*value = *value - 1;
}
#else
#define statInc(x) do { } while(0)
#define statDec(x) do { } while(0)
#endif

static void SetBit(u32 *value, u32 bitIdx)
{
	*value |= (u32)(1 << bitIdx);
}

static void ClearBit(u32 *value, u32 bitIdx)
{
	*value &= (u32)(~(1 << bitIdx));
}

static u32 IsBlockFree(BlockHeader *block)
{
	return (block->prev & BLOCK_FREE);
}

static u32 IsPrevBlockFree(BlockHeader *block)
{
	return (block->prev & PREV_FREE);
}

static void SetBlockFree(BlockHeader *block)
{
	block->prev |= BLOCK_FREE;
}

static void SetBlockPrevFree(BlockHeader *block)
{
	block->prev |= PREV_FREE;
}

static void SetBlockUsed(BlockHeader *block)
{
	block->prev &= ~BLOCK_FREE;
}

static void SetBlockPrevUsed(BlockHeader *block)
{
	block->prev &= ~PREV_FREE;
}

static u16 GetBlockPrev(BlockHeader *block)
{
	return (block->prev & PREV_MASK);
}

static void SetBlockPrev(BlockHeader *block, u16 newOffset)
{
	block->prev = newOffset | (block->prev & FLAGS_MASK);
}

// Get index of free list containing blocks of
// maximum size which is <= given size
static u32 GetIndexForInsert(u32 size)
{
	size = size > XV_MAX_ALLOC_SIZE ? XV_MAX_ALLOC_SIZE : size;
	size &= ~FL_DELTA_MASK;
	return ((size - XV_MIN_ALLOC_SIZE) >> FL_DELTA_SHIFT);
}

// Get index of free list having blocks of size >= requested size
static u32 GetIndex(u32 size)
{
	size = (size + FL_DELTA_MASK) & ~FL_DELTA_MASK; 
	return ((size - XV_MIN_ALLOC_SIZE) >> FL_DELTA_SHIFT);
}

static void *GetPtrAtomic(u32 pageNum, u16 offset, enum km_type type)
{
	unsigned char *page_ptr;
	page_ptr = kmap_atomic(pfn_to_page(pageNum), type);
	return (page_ptr + offset);
}

static void PutPtrAtomic(void *ptr, enum km_type type)
{
	kunmap_atomic(ptr, type);
}

static u32 AllocPage(void)
{
	struct page *page;

	page = alloc_page(GFP_NOIO | __GFP_HIGHMEM);

	if (unlikely(!page)) {
		return INVALID_PGNUM;
	}

	return page_to_pfn(page);
}

static void FreePage(u32 pageNum)
{
	__free_page(pfn_to_page(pageNum));
}

static u32 FindBlock(Pool *pool, u32 size, u32 *pageNum, u32 *offset)
{
	u32 flBitmap, slBitmap;
	u32 flIndex, slIndex, slBitStart;

	if (!pool->flBitmap) {
		return 0;
	}

	if (unlikely(size < XV_MIN_ALLOC_SIZE)) {
		size = XV_MIN_ALLOC_SIZE;
	}

	slIndex = GetIndex(size);
	slBitmap = pool->slBitmap[slIndex >> BITMAP_SHIFT];
	slBitStart = slIndex & BITMAP_MASK;
	
	if (slBitmap & (1 << slBitStart)) {
		*pageNum = pool->FreeList[slIndex].pageNum;
		*offset = pool->FreeList[slIndex].offset;
		return slIndex;
	}

	slBitStart++;
	slBitmap >>= slBitStart;
	
	if ((slBitStart != BITMAP_BITS) && slBitmap) {
		slIndex += ffs(slBitmap);
		*pageNum = pool->FreeList[slIndex].pageNum;
		*offset = pool->FreeList[slIndex].offset;
		return slIndex;
	}
	
	flIndex = slIndex >> BITMAP_SHIFT;
	
	flBitmap = (pool->flBitmap) >> (flIndex + 1);	
	if (!flBitmap) {
		return 0;
	}
	flIndex += ffs(flBitmap);
	slBitmap = pool->slBitmap[flIndex];
	slIndex = (flIndex << BITMAP_SHIFT) + ffs(slBitmap) - 1;
	*pageNum = pool->FreeList[slIndex].pageNum;
	*offset = pool->FreeList[slIndex].offset;

	return slIndex;
}

static void InsertBlock(Pool *pool, u32 pageNum, u32 offset,
			BlockHeader *block)
{
	u32 flIndex, slIndex;
	BlockHeader *nextBlock;
	
	slIndex = GetIndexForInsert(block->size);
	flIndex = slIndex >> BITMAP_SHIFT;
	
	block->link.prevPageNum = INVALID_PGNUM;
	block->link.prevOffset = 0;
	block->link.nextPageNum = pool->FreeList[slIndex].pageNum;
	block->link.nextOffset = pool->FreeList[slIndex].offset;
	pool->FreeList[slIndex].pageNum = pageNum;
	pool->FreeList[slIndex].offset = offset;
	
	if (block->link.nextPageNum != INVALID_PGNUM) {
		nextBlock = (BlockHeader *)GetPtrAtomic(
				block->link.nextPageNum,
				block->link.nextOffset, KM_USER1
			);
		nextBlock->link.prevPageNum = pageNum;
		nextBlock->link.prevOffset = offset;
		PutPtrAtomic(nextBlock, KM_USER1);
	}
	
	SetBit(&pool->slBitmap[flIndex], slIndex & BITMAP_MASK);
	SetBit(&pool->flBitmap, flIndex);
}

static void RemoveBlockHead(Pool *pool, BlockHeader *block,
			    u32 slIndex)
{
	BlockHeader *tmpBlock;
	u32 flIndex = slIndex >> BITMAP_SHIFT;
	
	pool->FreeList[slIndex].pageNum = block->link.nextPageNum;
	pool->FreeList[slIndex].offset = block->link.nextOffset;
	block->link.prevPageNum = INVALID_PGNUM;
	block->link.prevOffset = 0;
	
	if (pool->FreeList[slIndex].pageNum == INVALID_PGNUM) {
		ClearBit(&pool->slBitmap[flIndex], slIndex & BITMAP_MASK);
		if (!pool->slBitmap[flIndex]) {
			ClearBit(&pool->flBitmap, flIndex);
		}
	} else {
		// Debug
		tmpBlock = (BlockHeader *)GetPtrAtomic(
				pool->FreeList[slIndex].pageNum,
				pool->FreeList[slIndex].offset, KM_USER1
			);
		tmpBlock->link.prevPageNum = INVALID_PGNUM;
		tmpBlock->link.prevOffset = 0;
		PutPtrAtomic(tmpBlock, KM_USER1);
	}
}

static void RemoveBlock(Pool *pool, u32 pageNum, u32 offset,
			BlockHeader *block, u32 slIndex)
{
	u32 flIndex;
	BlockHeader *tmpBlock;
	
	if (pool->FreeList[slIndex].pageNum == pageNum
	   && pool->FreeList[slIndex].offset == offset) {
		RemoveBlockHead(pool, block, slIndex);
		return;
	}
	
	flIndex = slIndex >> BITMAP_SHIFT;
	
	if (block->link.prevPageNum != INVALID_PGNUM) {
		tmpBlock = (BlockHeader *)GetPtrAtomic(
				block->link.prevPageNum,
				block->link.prevOffset, KM_USER1
			);
		tmpBlock->link.nextPageNum = block->link.nextPageNum;
		tmpBlock->link.nextOffset = block->link.nextOffset;
		PutPtrAtomic(tmpBlock, KM_USER1);
	}
	
	if (block->link.nextPageNum != INVALID_PGNUM) {
		tmpBlock = (BlockHeader *)GetPtrAtomic(
				block->link.nextPageNum,
				block->link.nextOffset, KM_USER1
			);
		tmpBlock->link.prevPageNum = block->link.prevPageNum;
		tmpBlock->link.prevOffset = block->link.prevOffset;
		PutPtrAtomic(tmpBlock, KM_USER1);
	}
	
	return;
}

static int GrowPool(Pool *pool)
{
	u32 pageNum;
	BlockHeader *block;

	pageNum = AllocPage();
	if (unlikely(pageNum == INVALID_PGNUM)) {
		return -ENOMEM;
	}
	statInc(&pool->totalPages);
	
	spin_lock(&pool->lock);
	block = GetPtrAtomic(pageNum, 0, KM_USER0);
	
	block->size = PAGE_SIZE - XV_ALIGN;
	SetBlockFree(block);
	SetBlockPrevUsed(block);
	SetBlockPrev(block, 0);

	InsertBlock(pool, pageNum, 0, block);

	PutPtrAtomic(block, KM_USER0);
	spin_unlock(&pool->lock);
	
	return 0;
}

PoolID xvCreateMemPool(void)
{
	int i;
	void *ovhdMem;
	Pool *pool;
	u32 poolOvhd;

	poolOvhd = ROUNDUP(sizeof(Pool), PAGE_SIZE);
	ovhdMem = kmalloc(poolOvhd, GFP_KERNEL);
	if (!ovhdMem) {
		return INVALID_POOL_ID;
	}
	pool = ovhdMem;
	memset(pool, 0, poolOvhd);

	for (i = 0; i < NUM_FREE_LISTS; i++) {
		pool->FreeList[i].pageNum = INVALID_PGNUM;
	}

	spin_lock_init(&pool->lock);

	return pool;
}
EXPORT_SYMBOL_GPL(xvCreateMemPool);

void xvDestroyMemPool(PoolID pool)
{
	kfree(pool);
}
EXPORT_SYMBOL_GPL(xvDestroyMemPool);

int xvMalloc(PoolID pool, u32 size, u32 *pageNum, u32 *offset)
{
	int error;
	u32 index, tmpSize, origSize, tmpOffset;
	BlockHeader *block, *tmpBlock = NULL;
	
	*pageNum = INVALID_PGNUM;
	*offset = 0;
	origSize = size;

	if (unlikely(!size || size > XV_MAX_ALLOC_SIZE)) {
		return -ENOMEM;
	}

	if (unlikely(size < XV_MIN_ALLOC_SIZE)) {
		size = XV_MIN_ALLOC_SIZE;
	} else {
		size = ROUNDUP_ALIGN(size);
	}

	spin_lock(&pool->lock);
	
	index = FindBlock(pool, size, pageNum, offset);
	
	if (*pageNum == INVALID_PGNUM) {
		spin_unlock(&pool->lock);
		error = GrowPool(pool);
		if (unlikely(error)) {
			return -ENOMEM;
		}
		spin_lock(&pool->lock);
		index = FindBlock(pool, size, pageNum, offset);
	}

	if (*pageNum == INVALID_PGNUM) {
		spin_unlock(&pool->lock);
		return -ENOMEM;
	}

	block = (BlockHeader *)GetPtrAtomic(*pageNum, *offset, KM_USER0);

	RemoveBlockHead(pool, block, index);

	// Split the block if required
	tmpOffset = *offset + size + XV_ALIGN;
	tmpSize = block->size - size;
	tmpBlock = (BlockHeader *)((char *)block + size + XV_ALIGN);
	if (tmpSize) {
		tmpBlock->size = tmpSize - XV_ALIGN;
		SetBlockFree(tmpBlock);
		SetBlockPrevUsed(tmpBlock);
		
		SetBlockPrev(tmpBlock, *offset);
		if (tmpBlock->size >= XV_MIN_ALLOC_SIZE) {
			InsertBlock(pool, *pageNum, tmpOffset, tmpBlock);
		}

		if (tmpOffset + XV_ALIGN + tmpBlock->size < PAGE_SIZE) {
			tmpBlock = (BlockHeader *)((char *)tmpBlock
						+ tmpBlock->size + XV_ALIGN);

			SetBlockPrev(tmpBlock, tmpOffset);
		}
	} else {
		// this block is exact fit
		if (tmpOffset < PAGE_SIZE) {
			SetBlockPrevUsed(tmpBlock);
		}
	}

	block->size = origSize;
	SetBlockUsed(block);

	PutPtrAtomic(block, KM_USER0);
	spin_unlock(&pool->lock);

	*offset += XV_ALIGN;

	return 0;
}
EXPORT_SYMBOL_GPL(xvMalloc);

void xvFree(PoolID pool, u32 pageNum, u32 offset)
{
	void *page;
	BlockHeader *block, *tmpBlock;

	offset -= XV_ALIGN;

	spin_lock(&pool->lock);

	page = GetPtrAtomic(pageNum, 0, KM_USER0);
	block = (BlockHeader *)((char *)page + offset);
	
	if (unlikely(block->size < XV_MIN_ALLOC_SIZE)) {
		block->size = XV_MIN_ALLOC_SIZE;
	} else {
		block->size = ROUNDUP_ALIGN(block->size);
	}
	
	tmpBlock = (BlockHeader *)((char *)block + block->size + XV_ALIGN);
	if (offset + block->size + XV_ALIGN == PAGE_SIZE) {
		tmpBlock = NULL;
	}

	if (tmpBlock && IsBlockFree(tmpBlock)) {
		// we do not keep blocks smaller than this in any free list
		if (tmpBlock->size >= XV_MIN_ALLOC_SIZE) {
			RemoveBlock(pool, pageNum,
				    offset + block->size + XV_ALIGN, tmpBlock,
				    GetIndexForInsert(tmpBlock->size));
		}
		block->size += tmpBlock->size + XV_ALIGN;
	}

	if (IsPrevBlockFree(block)) {
		tmpBlock = (BlockHeader *)((char *)(page) +
				GetBlockPrev(block));
		offset = offset - tmpBlock->size - XV_ALIGN;
		
		if (tmpBlock->size >= XV_MIN_ALLOC_SIZE) {
			RemoveBlock(pool, pageNum, offset, tmpBlock,
				    GetIndexForInsert(tmpBlock->size));
		}
		tmpBlock->size += block->size + XV_ALIGN;
		block = tmpBlock;
	}

	if (block->size == PAGE_SIZE - XV_ALIGN) {
		PutPtrAtomic(page, KM_USER0);
		spin_unlock(&pool->lock);

		FreePage(pageNum);
		statDec(&pool->totalPages);
		return;
	}

	SetBlockFree(block);
	InsertBlock(pool, pageNum, offset, block);

	if (offset + block->size < PAGE_SIZE - XV_ALIGN) {
		tmpBlock = (BlockHeader *)((char *)(block) + block->size +
								XV_ALIGN);
		SetBlockPrevFree(tmpBlock);
		SetBlockPrev(tmpBlock, offset);
	}

	PutPtrAtomic(page, KM_USER0);
	spin_unlock(&pool->lock);

	return;
}
EXPORT_SYMBOL_GPL(xvFree);

u32 xvGetObjectSize(void *obj)
{
	BlockHeader *blk;

	blk = (BlockHeader *)((char *)(obj) - XV_ALIGN);
	return blk->size;
}
EXPORT_SYMBOL_GPL(xvGetObjectSize);

// Returns total memory used by allocator (userdata + metadata)
u64 xvGetTotalSizeBytes(PoolID pool)
{
#ifdef CONFIG_XV_STATS
	return (pool->totalPages << PAGE_SHIFT);
#else
	return 0;
#endif
}
EXPORT_SYMBOL_GPL(xvGetTotalSizeBytes);

static int __init xvMallocInit(void)
{
	return 0;
}

static void __exit xvMallocExit(void)
{
	return;
}

module_init(xvMallocInit);
module_exit(xvMallocExit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Nitin Gupta <nitingupta910@gmail.com>");
MODULE_DESCRIPTION("xvMalloc Memory Allocator");

