# Copyright (C) 2005, 2006, 2007, 2011 Aaron Bentley <aaron@aaronbentley.com>
# Copyright (C) 2005, 2006, 2011 Canonical Limited.
# Copyright (C) 2006 Michael Ellerman.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import bzrlib

from bzrlib.lazy_import import lazy_import
lazy_import(globals(), """
from bzrlib import help, urlutils
import shelf
""")

from bzrlib.plugins import bzrtools
from command import BzrToolsCommand
from errors import CommandError
from patchsource import BzrPatchSource

import bzrlib.commands
from bzrlib.branch import Branch
from bzrlib.commands import get_cmd_object
from bzrlib.errors import BzrCommandError
from bzrlib.option import Option, RegistryOption


class cmd_graph_ancestry(BzrToolsCommand):
    """Produce ancestry graphs using dot.

    Output format is detected according to file extension.  Some of the more
    common output formats are html, png, gif, svg, ps.  An extension of '.dot'
    will cause a dot graph file to be produced.  HTML output has mouseovers
    that show the commit message.

    Branches are labeled r?, where ? is the revno.  If they have no revno,
    with the last 5 characters of their revision identifier are used instead.

    The value starting with d is "(maximum) distance from the null revision".

    If --merge-branch is specified, the two branches are compared and a merge
    base is selected.

    Legend:
    white    normal revision
    yellow   THIS  history
    red      OTHER history
    orange   COMMON history
    blue     COMMON non-history ancestor
    green    Merge base (COMMON ancestor farthest from the null revision)
    dotted   Ghost revision (missing from branch storage)

    Ancestry is usually collapsed by skipping revisions with a single parent
    and descendant.  The number of skipped revisions is shown on the arrow.
    This feature can be disabled with --no-collapse.

    By default, revisions are ordered by distance from root, but they can be
    clustered instead using --cluster.

    If available, rsvg is used to antialias PNG and JPEG output, but this can
    be disabled with --no-antialias.
    """
    takes_args = ['file', 'merge_branch?']
    takes_options = [Option('no-collapse', help="Do not skip simple nodes."),
                     Option('no-antialias',
                     help="Do not use rsvg to produce antialiased output."),
                     Option('merge-branch', type=str,
                     help="Use this branch to calcuate a merge base."),
                     Option('cluster', help="Use clustered output."),
                     Option('max-distance',
                            help="Show no nodes farther than this.", type=int),
                     Option('directory',
                            help='Source branch to use (default is current'
                            ' directory).',
                            short_name='d',
                            type=unicode),
                    ]
    def run(self, file, merge_branch=None, no_collapse=False,
            no_antialias=False, cluster=False, max_distance=100,
            directory='.'):
        if max_distance == -1:
            max_distance = None
        import graph
        if cluster:
            ranking = "cluster"
        else:
            ranking = "forced"
        graph.write_ancestry_file(directory, file, not no_collapse,
                                  not no_antialias, merge_branch, ranking,
                                  max_distance=max_distance)


class cmd_fetch_ghosts(BzrToolsCommand):
    """Attempt to retrieve ghosts from another branch.
    If the other branch is not supplied, the last-pulled branch is used.
    """
    aliases = ['fetch-missing']
    takes_args = ['branch?']
    takes_options = [Option('no-fix', help="Skip additional synchonization.")]
    def run(self, branch=None, no_fix=False):
        from fetch_ghosts import fetch_ghosts
        fetch_ghosts(branch, do_reconcile=not no_fix)

strip_help="""Strip the smallest prefix containing num leading slashes  from \
each file name found in the patch file."""


class cmd_patch(BzrToolsCommand):
    """Apply a named patch to the current tree.
    """
    takes_args = ['filename?']
    takes_options = [Option('strip', type=int, short_name='p',
                            help=strip_help),
                     Option('silent', help='Suppress chatter.')]
    def run(self, filename=None, strip=None, silent=False):
        from patch import patch
        from bzrlib.workingtree import WorkingTree
        wt = WorkingTree.open_containing('.')[0]
        if strip is None:
            strip = 0
        return patch(wt, filename, strip, silent)


class cmd_shelve1(BzrToolsCommand):
    """Temporarily set aside some changes from the current tree.

    Shelve allows you to temporarily put changes you've made "on the shelf",
    ie. out of the way, until a later time when you can bring them back from
    the shelf with the 'unshelve1' command.

    Shelve is intended to help separate several sets of text changes that have
    been inappropriately mingled.  If you just want to get rid of all changes
    (text and otherwise) and you don't need to restore them later, use revert.
    If you want to shelve all text changes at once, use shelve1 --all.

    By default shelve1 asks you what you want to shelve, press '?' at the
    prompt to get help. To shelve everything run shelve1 --all.

    If filenames are specified, only the changes to those files will be
    shelved, other files will be left untouched.

    If a revision is specified, changes since that revision will be shelved.

    You can put multiple items on the shelf. Normally each time you run
    unshelve1 the most recently shelved changes will be reinstated. However,
    you can also unshelve changes in a different order by explicitly
    specifiying which changes to unshelve1. This works best when the changes
    don't depend on each other.

    While you have patches on the shelf you can view and manipulate them with
    the 'shelf1' command. Run 'bzr shelf1 -h' for more info.
    """

    takes_args = ['file*']
    takes_options = [Option('message',
            help='A message to associate with the shelved changes.',
            short_name='m', type=unicode),
            'revision',
            Option('all', help='Shelve all changes without prompting.'),
            Option('no-color', help='Never display changes in color.')]

    def run(self, all=False, file_list=None, message=None, revision=None,
            no_color=False):
        if revision is not None and revision:
            if len(revision) == 1:
                revision = revision[0]
            else:
                raise CommandError("shelve only accepts a single revision "
                                  "parameter.")

        source = BzrPatchSource(revision, file_list)
        s = shelf.Shelf(source.base)
        s.shelve(source, all, message, no_color)
        return 0


# The following classes are only used as subcommands for 'shelf1', they're
# not to be registered directly with bzr.

class cmd_shelf_list(bzrlib.commands.Command):
    """List the patches on the current shelf."""
    aliases = ['list', 'ls']
    def run(self):
        self.shelf.list()


class cmd_shelf_delete(bzrlib.commands.Command):
    """Delete the patch from the current shelf."""
    aliases = ['delete', 'del']
    takes_args = ['patch']
    def run(self, patch):
        self.shelf.delete(patch)


class cmd_shelf_switch(bzrlib.commands.Command):
    """Switch to the other shelf, create it if necessary."""
    aliases = ['switch']
    takes_args = ['othershelf']
    def run(self, othershelf):
        s = shelf.Shelf(self.shelf.base, othershelf)
        s.make_default()


class cmd_shelf_show(bzrlib.commands.Command):
    """Show the contents of the specified or topmost patch."""
    aliases = ['show', 'cat', 'display']
    takes_args = ['patch?']
    def run(self, patch=None):
        self.shelf.display(patch)


class cmd_shelf_upgrade(bzrlib.commands.Command):
    """Upgrade old format shelves."""
    aliases = ['upgrade']
    def run(self):
        self.shelf.upgrade()


class cmd_shelf1(BzrToolsCommand):
    """Perform various operations on your shelved patches. See also shelve1."""
    takes_args = ['subcommand', 'args*']

    subcommands = [cmd_shelf_list, cmd_shelf_delete, cmd_shelf_switch,
        cmd_shelf_show, cmd_shelf_upgrade]

    def run(self, subcommand, args_list):
        import sys

        if args_list is None:
            args_list = []
        cmd = self._get_cmd_object(subcommand)
        source = BzrPatchSource()
        s = shelf.Shelf(source.base)
        cmd.shelf = s

        if args_list is None:
            args_list = []
        return cmd.run_argv_aliases(args_list)

    def _get_cmd_object(self, cmd_name):
        for cmd_class in self.subcommands:
            for alias in cmd_class.aliases:
                if alias == cmd_name:
                    return cmd_class()
        raise CommandError("Unknown shelf subcommand '%s'" % cmd_name)

    def help(self):
        text = ["%s\n\nSubcommands:\n" % self.__doc__]

        for cmd_class in self.subcommands:
            text.extend(self.sub_help(cmd_class) + ['\n'])

        return ''.join(text)

    def sub_help(self, cmd_class):
        text = []
        cmd_obj = cmd_class()
        indent = 2 * ' '

        usage = cmd_obj._usage()
        usage = usage.replace('bzr shelf-', '')
        text.append('%s%s\n' % (indent, usage))

        text.append('%s%s\n' % (2 * indent, cmd_class.__doc__))

        # Somewhat copied from bzrlib.help.help_on_command_options
        option_help = []
        for option_name, option in sorted(cmd_obj.options().items()):
            if option_name == 'help':
                continue
            option_help.append('%s--%s' % (3 * indent, option_name))
            if option.type is not None:
                option_help.append(' %s' % option.argname.upper())
            if option.short_name():
                option_help.append(', -%s' % option.short_name())
            option_help.append('%s%s\n' % (2 * indent, option.help))

        if len(option_help) > 0:
            text.append('%soptions:\n' % (2 * indent))
            text.extend(option_help)

        return text


class cmd_unshelve1(BzrToolsCommand):
    """Restore shelved changes.

    By default the most recently shelved changes are restored. However if you
    specify a patch by name those changes will be restored instead.

    See 'shelve1' for more information.
    """
    takes_options = [
            Option('all', help='Unshelve all changes without prompting.'),
            Option('force', help='Force unshelving even if errors occur.'),
            Option('no-color', help='Never display changes in color.')
        ]
    takes_args = ['patch?']
    def run(self, patch=None, all=False, force=False, no_color=False):
        source = BzrPatchSource()
        s = shelf.Shelf(source.base)
        s.unshelve(source, patch, all, force, no_color)
        return 0


class cmd_shell(BzrToolsCommand):
    """Begin an interactive shell tailored for bzr.
    Bzr commands can be used without typing bzr first, and will be run natively
    when possible.  Tab completion is tailored for bzr.  The shell prompt shows
    the branch nick, revno, and path.

    If it encounters any moderately complicated shell command, it will punt to
    the system shell.

    Example:
    $ bzr shell
    bzr bzrtools:287/> status
    modified:
      __init__.py
    bzr bzrtools:287/> status --[TAB][TAB]
    --all        --help       --revision   --show-ids
    bzr bzrtools:287/> status --
    """
    takes_options = [
        Option('directory',
            help='Branch in which to start the shell, '
                 'rather than the one containing the working directory.',
            short_name='d',
            type=unicode,
            ),
        ]
    def run(self, directory=None):
        import shell
        return shell.run_shell(directory)


class cmd_branch_history(BzrToolsCommand):
    """\
    Display the development history of a branch.

    Each different committer or branch nick is considered a different line of
    development.  Committers are treated as the same if they have the same
    name, or if they have the same email address.
    """
    takes_args = ["branch?"]
    def run(self, branch=None):
        from branchhistory import branch_history
        return branch_history(branch)


class cmd_zap(BzrToolsCommand):
    """\
    Remove a lightweight checkout, if it can be done safely.

    This command will remove a lightweight checkout without losing data.  That
    means it only removes lightweight checkouts, and only if they have no
    uncommitted changes.

    If --branch is specified, the branch will be deleted too, but only if the
    the branch has no new commits (relative to its parent).

    If bzr-pipeline is also installed, the --store option will store changes
    in the branch before deleting the tree.  To restore the changes, do::

      bzr checkout --lightweight $BRANCH $CHECKOUT
      bzr switch-pipe -d $CHECKOUT `bzr nick -d $CHECKOUT`
    """
    takes_options = [Option("branch", help="Remove associated branch from"
                                           " repository."),
                     RegistryOption('change_policy',
                                    'How to handle changed files',
                                    lazy_registry =
                                    ('bzrlib.plugins.bzrtools.zap',
                                        'change_policy_registry'),
                                    value_switches=True,
                                    enum_switch=False)]
    takes_args = ["checkout"]
    def run(self, checkout, branch=False, change_policy=None):
        from zap import (
            change_policy_registry,
            StoreChanges,
            zap,
        )
        if change_policy is None:
            change_policy = change_policy_registry.get()
        if change_policy is StoreChanges:
            try:
                import bzrlib.plugins.pipeline
            except ImportError:
                raise BzrCommandError('--store requires bzr-pipeline.')
        return zap(checkout, remove_branch=branch, policy=change_policy)


class cmd_cbranch(BzrToolsCommand):
    """
    Create a new checkout, associated with a new repository branch.

    When you cbranch, bzr looks up a target location in locations.conf, and
    creates the branch there.

    In your locations.conf, add the following lines:
    [/working_directory_root]
    cbranch_target = /branch_root
    cbranch_target:policy = appendpath

    This will mean that if you run "bzr cbranch foo/bar foo/baz" in the
    working directory root, the branch will be created in
    "/branch_root/foo/baz"

    NOTE: cbranch also supports "cbranch_root", but that behaviour is
    deprecated.
    """
    takes_options = [Option("lightweight",
                            help="Create a lightweight checkout."), 'revision',
                     Option('files-from', type=unicode,
                            help='Accelerate checkout using files from this'
                                 ' tree.'),
                     Option('hardlink',
                            help='Hard-link files from source/files-from tree'
                            ' where posible.')]
    takes_args = ["source", "target?"]
    def run(self, source, target=None, lightweight=False, revision=None,
            files_from=None, hardlink=False):
        from cbranch import cbranch
        return cbranch(source, target, lightweight=lightweight,
                       revision=revision, files_from=files_from,
                       hardlink=hardlink)


class cmd_list_branches(BzrToolsCommand):
    """Scan a location for branches"""
    @property
    def aliases(self):
        from bzrlib import commands
        return commands.plugin_cmds.get_info('list-branches').aliases

    takes_args = ["location?"]
    def run(self, location=None):
        from branches import branches
        return branches(location)

class cmd_trees(BzrToolsCommand):
    """Scan a location for trees"""
    takes_args = ['location?']
    def run(self, location='.'):
        from bzrlib.workingtree import WorkingTree
        from bzrlib.transport import get_transport
        t = get_transport(location)
        for tree in WorkingTree.find_trees(location):
            self.outf.write('%s\n' % t.relpath(
                tree.bzrdir.root_transport.base))

class cmd_multi_pull(BzrToolsCommand):
    """Pull all the branches under a location, e.g. a repository.

    Both branches present in the directory and the branches of checkouts are
    pulled.
    """
    takes_args = ["location?"]
    def run(self, location=None):
        from bzrlib.transport import get_transport
        from bzrtools import iter_branch_tree
        if location is None:
            location = '.'
        t = get_transport(location)
        possible_transports = []
        if not t.listable():
            print "Can't list this type of location."
            return 3
        for branch, wt in iter_branch_tree(t):
            if wt is None:
                pullable = branch
            else:
                pullable = wt
            parent = branch.get_parent()
            if parent is None:
                continue
            if wt is not None:
                base = wt.basedir
            else:
                base = branch.base
            if base.startswith(t.base):
                relpath = base[len(t.base):].rstrip('/')
            else:
                relpath = base
            print "Pulling %s from %s" % (relpath, parent)
            try:
                branch_t = get_transport(parent, possible_transports)
                pullable.pull(Branch.open_from_transport(branch_t))
            except Exception, e:
                print e



class cmd_import(BzrToolsCommand):
    """Import sources from a directory, tarball or zip file

    This command will import a directory, tarball or zip file into a bzr
    tree, replacing any versioned files already present.  If a directory is
    specified, it is used as the target.  If the directory does not exist, or
    is not versioned, it is created.

    Tarballs may be gzip or bzip2 compressed.  This is autodetected.

    If the tarball or zip has a single root directory, that directory is
    stripped when extracting the tarball.  This is not done for directories.
    """

    takes_args = ['source', 'tree?']
    def run(self, source, tree=None):
        from upstream_import import do_import
        do_import(source, tree)


class cmd_cdiff(BzrToolsCommand):
    """A color version of bzr's diff"""
    takes_args = property(lambda x: get_cmd_object('diff').takes_args)
    takes_options = list(get_cmd_object('diff').takes_options) + [
        RegistryOption.from_kwargs('color',
            'Color mode to use.',
            title='Color Mode', value_switches=False, enum_switch=True,
            never='Never colorize output.',
            auto='Only colorize output if terminal supports it and STDOUT is a'
            ' TTY.',
            always='Always colorize output (default).'),
        Option('check-style',
            help='Warn if trailing whitespace or spurious changes have been'
                 ' added.')]

    def run(self, color='always', check_style=False, *args, **kwargs):
        from colordiff import colordiff
        colordiff(color, check_style, *args, **kwargs)


class cmd_conflict_diff(BzrToolsCommand):

    """Compare a conflicted file against BASE."""

    encoding_type = 'exact'
    takes_args = ['file*']
    takes_options = [
        RegistryOption.from_kwargs('direction', 'Direction of comparison.',
            value_switches=True, enum_switch=False,
            other='Compare OTHER against common base.',
            this='Compare THIS against common base.')]

    def run(self, file_list, direction='other'):
        from bzrlib.plugins.bzrtools.colordiff import DiffWriter
        from conflict_diff import ConflictDiffer
        dw = DiffWriter(self.outf, check_style=False, color='auto')
        ConflictDiffer().run(dw, file_list, direction)


class cmd_rspush(BzrToolsCommand):
    """Upload this branch to another location using rsync.

    If no location is specified, the last-used location will be used.  To
    prevent dirty trees from being uploaded, rspush will error out if there are
    unknown files or local changes.  It will also error out if the upstream
    directory is non-empty and not an earlier version of the branch.
    """
    takes_args = ['location?']
    takes_options = [Option('overwrite', help='Ignore differences between'
                            ' branches and overwrite unconditionally.'),
                     Option('no-tree', help='Do not push the working tree,'
                            ' just the .bzr.')]

    def run(self, location=None, overwrite=False, no_tree=False):
        from bzrlib import workingtree
        import bzrtools
        cur_branch = workingtree.WorkingTree.open_containing(".")[0]
        bzrtools.rspush(cur_branch, location, overwrite=overwrite,
                      working_tree=not no_tree)


class cmd_link_tree(BzrToolsCommand):
    """Hardlink matching files to another tree.

    Only files with identical content and execute bit will be linked.
    """
    takes_args = ['location']

    def run(self, location):
        from bzrlib import workingtree
        from bzrlib.plugins.bzrtools.link_tree import link_tree
        target_tree = workingtree.WorkingTree.open_containing(".")[0]
        source_tree = workingtree.WorkingTree.open(location)
        target_tree.lock_write()
        try:
            source_tree.lock_read()
            try:
                link_tree(target_tree, source_tree)
            finally:
                source_tree.unlock()
        finally:
            target_tree.unlock()


class cmd_create_mirror(BzrToolsCommand):
    """Create a mirror of another branch.

    This is similar to `bzr branch`, but copies more settings, including the
    submit branch and nickname.

    It sets the public branch and parent of the target to the source location.
    """

    takes_args = ['source', 'target']

    def run(self, source, target):
        source_branch = Branch.open(source)
        from bzrlib.plugins.bzrtools.mirror import create_mirror
        create_mirror(source_branch, target, [])
