/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    カラーウィンドウのサブウィジェット
*/


#include "subwidget_color.h"

#include "CConfig.h"
#include "CPalette.h"
#include "cursor.h"

#include "draw_main.h"

#include "global.h"
#include "strid.h"

#include "AXScrollBar.h"
#include "AXMenu.h"
#include "AXGC.h"
#include "AXFont.h"
#include "AXApp.h"
#include "AXAppRes.h"
#include "AXUtilColor.h"



//***************************************
// CColSub_drawcol : 描画色/背景色
//***************************************


CColSub_drawcol::CColSub_drawcol(AXWindow *pParent,DWORD dwPadding)
    : AXWindow(pParent, 0, LF_FIX_WH, 0, dwPadding)
{
    m_nW = m_nH = 40;
}

//! 描画色部分だけ描画

void CColSub_drawcol::paintOnlyDrawCol()
{
    AXGC gc;

    gc.createColor(g_conf->dwDrawCol);

    drawFillBox(1, 1, m_nW - SPACE - 2, m_nH - SPACE - 2, gc);
}

//! ボタン押し（色入れ替え）

BOOL CColSub_drawcol::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT)
    {
        draw::toggleDrawCol();
        redrawUpdate();

        getNotify()->onNotify(this, 0, 0);
    }

    return TRUE;
}

//! 描画

BOOL CColSub_drawcol::onPaint(AXHD_PAINT *phd)
{
    AXGC gc;

    //背景

    drawFillBox(0, 0, m_nW, m_nH, AXAppRes::FACE);

    //背景色

    gc.createColor(g_conf->dwBackCol);

    drawBox(SPACE, SPACE, m_nW - SPACE, m_nH - SPACE, AXAppRes::BLACK);
    drawFillBox(SPACE + 1, SPACE + 1, m_nW - SPACE - 2, m_nH - SPACE - 2, gc);

    //描画色

    gc.setColorRGB(g_conf->dwDrawCol);

    drawBox(0, 0, m_nW - SPACE, m_nH - SPACE, AXAppRes::BLACK);
    drawFillBox(1, 1, m_nW - SPACE - 2, m_nH - SPACE - 2, gc);

    return TRUE;
}


//******************************************
// CColSub_colmasktype : 色マスクタイプボタン
//******************************************


CColSub_colmasktype::CColSub_colmasktype(AXWindow *pParent)
    : AXWindow(pParent, 0, 0)
{

}

void CColSub_colmasktype::calcDefSize()
{
    m_nTextW[0] = m_pFont->getTextWidth("ON", 2);
    m_nTextW[1] = m_pFont->getTextWidth("REV", 3);

    m_nOneW = ((m_nTextW[0] > m_nTextW[1])? m_nTextW[0]: m_nTextW[1]) + XSPACE * 2;

    m_nDefW = m_nOneW * 2;
    m_nDefH = m_pFont->getHeight() + YSPACE * 2;
}

//!< ボタン押し

BOOL CColSub_colmasktype::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT)
    {
        int n = (phd->x < m_nOneW)? 1: 2;

        if(g_conf->btColMaskType == n)
            g_conf->btColMaskType = 0;
        else
            g_conf->btColMaskType = n;

        redraw();
    }

    return TRUE;
}

//!< 描画

BOOL CColSub_colmasktype::onPaint(AXHD_PAINT *phd)
{
    AXDrawText dt(m_id);
    int i,x,f;

    for(i = 0, x = 0; i < 2; i++, x += m_nOneW)
    {
        f = (g_conf->btColMaskType == i + 1);

        //枠

        if(f)
            drawFrameSunken(x, 0, m_nOneW, m_nH);
        else
            drawFrameRaise(x, 0, m_nOneW, m_nH);

        //背景

        drawFillBox(x + 1, 1, m_nOneW - 2, m_nH - 2, (f)? AXAppRes::BACKREVTEXT: AXAppRes::FACE);

        //テキスト

        dt.draw(*m_pFont, x + (m_nOneW - m_nTextW[i]) / 2, YSPACE,
                (i == 0)? "ON": "REV", 2 + i, (f)? AXAppRes::TC_REV: AXAppRes::TC_NORMAL);
    }

    dt.end();

    return TRUE;
}


//***************************************
// CColSub_colmask : 色マスク
//***************************************


CColSub_colmask::CColSub_colmask(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_FIX_WH)
{
    int i;

    m_nW = ONESIZE * CConfig::COLMASK_NUM + 1;
    m_nH = ONESIZE + 1;

    m_img.create(m_nW, m_nH);

    //枠描画

    m_img.box(0, 0, m_nW, m_nH, 0);

    for(i = 1; i < CConfig::COLMASK_NUM; i++)
        m_img.lineV(i * ONESIZE, 1, ONESIZE - 1, 0);

    draw();
}

//!< 描画

void CColSub_colmask::draw()
{
    int i,x,xx,yy;
    DWORD col;

    for(i = 0, x = 1; i < CConfig::COLMASK_NUM; i++, x += ONESIZE)
    {
        col = g_conf->dwColMask[i];

        if(!(col & 0xff000000))
            //色なし
            m_img.drawCheckTwo(x, 1, ONESIZE - 1);
        else
        {
            m_img.fillBox(x, 1, ONESIZE - 1, ONESIZE - 1, col & 0xffffff);

            //クリアボタン

            xx = x + ONESIZE - 7;
            yy = ONESIZE - 6;

            m_img.box(xx, yy, 7, 7, 0);
            m_img.fillBox(xx + 1, yy + 1, 5, 5, 0xffffff);
            m_img.line(xx + 2, yy + 2, xx + 4, yy + 4, 0);
            m_img.line(xx + 4, yy + 2, xx + 2, yy + 4, 0);
        }
    }

    redraw();
}

//!< 描画

BOOL CColSub_colmask::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//!< ボタン押し

BOOL CColSub_colmask::onButtonDown(AXHD_MOUSE *phd)
{
    int no;
    DWORD col;

    no = phd->x / ONESIZE;
    if(no >= CConfig::COLMASK_NUM) no = CConfig::COLMASK_NUM - 1;

    col = g_conf->dwColMask[no];

    //

    if(phd->button == BUTTON_RIGHT)
    {
        //右ボタンでスポイト

        if(col & 0xff000000)
            getNotify()->onNotify(this, 0, col);
    }
    else if(phd->button == BUTTON_LEFT)
    {
        //左ボタン

        if(phd->state & STATE_CTRL)
            //+Ctrl でクリア
            col = 0;
        else if((col & 0xff000000) && phd->x >= (no + 1) * ONESIZE - 6 && phd->y >= ONESIZE - 6)
            //クリアボタン -> クリア(色がある時のみ)
            col = 0;
        else
            //描画色をセット
            col = g_conf->dwDrawCol | 0xff000000;

        draw::changeColMaskColor(no, col);

        draw();
    }

    return TRUE;
}


//***************************************
// CColSub_grad : グラデーションパレット
//***************************************


CColSub_grad::CColSub_grad(AXWindow *pParent,UINT uID,DWORD col1,DWORD col2)
    : AXWindow(pParent, 0, LF_EXPAND_W | LF_FIX_H, uID, 0)
{
    m_nH = 13;
    m_nMinW = 80;

    m_dwCol1 = col1;
    m_dwCol2 = col2;

    m_fDrag = 0;

    _changeCol();
}

//! 左右色変更時

void CColSub_grad::_changeCol()
{
    m_sr = _GETR(m_dwCol1);
    m_sg = _GETG(m_dwCol1);
    m_sb = _GETB(m_dwCol1);

    m_er = _GETR(m_dwCol2);
    m_eg = _GETG(m_dwCol2);
    m_eb = _GETB(m_dwCol2);
}

//! 描画

void CColSub_grad::_draw()
{
    int size,w,i,x,r,g,b,div;

    size = m_nH;

    //背景

    m_img.clear(axres->colRGB(AXAppRes::FACE));

    //左枠

    m_img.box(0, 0, size, size, 0);
    m_img.fillBox(1, 1, size - 2, size - 2, m_dwCol1);

    //右枠

    m_img.box(m_nW - size, 0, size, size, 0);
    m_img.fillBox(m_nW - size + 1, 1, size - 2, size - 2, m_dwCol2);

    //グラデーション

    w = m_nW - (size + SPACE) * 2;

    m_img.box(size + SPACE, 0, w, size, 0);

    w -= 2;
    x = size + SPACE + 1;
    div = w - 1;

    for(i = 0; i < w; i++, x++)
    {
        r = (m_er - m_sr) * i / div + m_sr;
        g = (m_eg - m_sg) * i / div + m_sg;
        b = (m_eb - m_sb) * i / div + m_sb;

        m_img.lineV(x, 1, size - 2, _RGB(r,g,b));
    }

    //

    redraw();
}

//! グラデーション部分のドラッグ時

void CColSub_grad::_moveGrad(int wx)
{
    int x,w,r,g,b;

    x = wx - m_nH - SPACE - 1;
    w = m_nW - (m_nH + SPACE) * 2 - 2;

    if(x < 0) x = 0; else if(x >= w) x = w - 1;

    //カーソル

    if(m_nDragX >= 0)
        m_img.lineV(m_nDragX + m_nH + SPACE + 1, 0, m_nH, AXImage::COL_XOR);

    m_img.lineV(x + m_nH + SPACE + 1, 0, m_nH, AXImage::COL_XOR);

    redraw();

    m_nDragX = x;

    //色

    w--;

    r = (m_er - m_sr) * x / w + m_sr;
    g = (m_eg - m_sg) * x / w + m_sg;
    b = (m_eb - m_sb) * x / w + m_sb;

    //通知

    getNotify()->onNotify(this, NOTIFY_COLOR, _RGB(r,g,b));
}

//!< サイズ変更時

BOOL CColSub_grad::onSize()
{
    m_img.recreate(m_nW, m_nH, 10, 0);

    _draw();

    return TRUE;
}

//!< 描画

BOOL CColSub_grad::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//!< ボタン押し

BOOL CColSub_grad::onButtonDown(AXHD_MOUSE *phd)
{
    if(!m_fDrag && phd->button == BUTTON_LEFT)
    {
        if(phd->x < m_nH)
        {
            //左枠 -> 描画色セット

            m_dwCol1 = g_conf->dwDrawCol;
            _changeCol();
            _draw();

            getNotify()->onNotify(this, NOTIFY_CHANGE_LEFT, 0);
        }
        else if(phd->x >= m_nW - m_nH)
        {
            //右枠

            m_dwCol2 = g_conf->dwDrawCol;
            _changeCol();
            _draw();

            getNotify()->onNotify(this, NOTIFY_CHANGE_RIGHT, 0);
        }
        else if(phd->x >= m_nH + SPACE + 1 && phd->x < m_nW - m_nH - SPACE)
        {
            //グラデーション部分

            m_nDragX = -1;

            _moveGrad(phd->x);

            m_fDrag = TRUE;
            grabPointer();
        }
    }

    return TRUE;
}

//!< ボタン離し

BOOL CColSub_grad::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDrag)
    {
        //カーソル消去
        m_img.lineV(m_nDragX + m_nH + SPACE + 1, 0, m_nH, AXImage::COL_XOR);
        redraw();

        m_fDrag = FALSE;
        ungrabPointer();
    }

    return TRUE;
}

//!< 移動

BOOL CColSub_grad::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_fDrag)
        _moveGrad(phd->x);

    return TRUE;
}



//***************************************
// CColSub_hlspal : HLSパレット
//***************************************


CColSub_hlspal::CColSub_hlspal(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_FIX_WH)
{
    m_nW = TOP_H_W * 30 + 1;
    m_nH = TOPH + TOPSP + PALH * PALYCNT + 1;

    m_fDrag = 0;

    //イメージ

    m_img.create(m_nW, m_nH);

    _drawFirst();

    drawAll();
}

//! 最初の描画

void CColSub_hlspal::_drawFirst()
{
    int i,n;

    //背景
    m_img.clear(axres->colRGB(AXAppRes::FACE));

    //トップの枠
    m_img.box(0, 0, TOP_H_W * 30 + 1, TOPH, 0);

    //パレットの枠

    for(i = 0, n = 0; i < PALXCNT + 1; i++, n += PALW)
        m_img.lineV(n, PALY, PALH * PALYCNT + 1, 0xcccccc);

    for(i = 0, n = PALY; i < PALYCNT + 1; i++, n += PALH)
        m_img.lineH(0, n, PALW * PALXCNT + 1, 0xcccccc);
}

//!< トップのカーソル描画

void CColSub_hlspal::_drawTopCursor()
{
    int pos;

    //カーソルX位置

    switch(g_conf->nColHLSVal[0])
    {
        case 0:
            pos = g_conf->nColHLSVal[1] / 12 * TOP_H_W + 1 + (TOP_H_W - 1) / 2;
            break;
        case 1:
            pos = (92 - g_conf->nColHLSVal[2]) / 6 * TOP_LS_W + 1 + (TOP_LS_W - 1) / 2;
            break;
        case 2:
            pos = (100 - g_conf->nColHLSVal[3]) / 6 * TOP_LS_W + 1 + (TOP_LS_W - 1) / 2;
            break;
    }

    //描画

    m_img.fillBox(0, TOPH, m_nW, 3, axres->colRGB(AXAppRes::FACE));

    m_img.setPixel(pos, TOPH, 0);
    m_img.lineH(pos - 1, TOPH + 1, 3, 0);
    m_img.lineH(pos - 2, TOPH + 2, 5, 0);
}

//! トップ・パレット・カーソル描画

void CColSub_hlspal::drawAll()
{
    int i,n;

    //トップ

    switch(g_conf->nColHLSVal[0])
    {
        //H:SL
        case 0:
            for(i = 0, n = TOP_H_W; i < 29; i++, n += TOP_H_W)
                m_img.lineV(n, 1, TOPH - 2, 0);

            for(i = 0, n = 1; i < 30; i++, n += TOP_H_W)
                m_img.fillBox(n, 1, TOP_H_W - 1, TOPH - 2, AXHLStoRGB(i * 12, 0.5, 1.0));
            break;
        //L:SH
        case 1:
            for(i = 0, n = TOP_LS_W; i < 15; i++, n += TOP_LS_W)
                m_img.lineV(n, 1, TOPH - 2, 0);

            for(i = 0, n = 1; i < 15; i++, n += TOP_LS_W)
                m_img.fillBox(n, 1, TOP_LS_W - 1, TOPH - 2, AXHLStoRGB(0, 0.92 - i * 0.06, 1.0));
            break;
        //S:LH
        case 2:
            for(i = 0, n = TOP_LS_W; i < 15; i++, n += TOP_LS_W)
                m_img.lineV(n, 1, TOPH - 2, 0);

            for(i = 0, n = 1; i < 15; i++, n += TOP_LS_W)
                m_img.fillBox(n, 1, TOP_LS_W - 1, TOPH - 2, AXHLStoRGB(0, 0.5, 1 - i * 0.06));
            break;
    }

    //カーソル・パレット

    _drawTopCursor();

    drawPalette();

    redraw();
}

//! パレット部分の描画

void CColSub_hlspal::drawPalette()
{
    int ix,iy,x,y;
    double dtmp;

    switch(g_conf->nColHLSVal[0])
    {
        //S-L
        case 0:
            for(iy = 0, y = PALY + 1; iy < PALYCNT; iy++, y += PALH)
                for(ix = 0, x = 1; ix < PALXCNT; ix++, x += PALW)
                    m_img.fillBox(x, y, PALW - 1, PALH - 1, AXHLStoRGB(g_conf->nColHLSVal[1], 0.92 - ix * 0.06, 1 - iy * 0.1));
            break;
        //S-H
        case 1:
            dtmp = g_conf->nColHLSVal[2] * 0.01;

            //上段

            for(iy = 0, y = PALY + 1; iy < PALYCNT / 2; iy++, y += PALH)
                for(ix = 0, x = 1; ix < PALXCNT; ix++, x += PALW)
                    m_img.fillBox(x, y, PALW - 1, PALH - 1, AXHLStoRGB(ix * 12, dtmp, 1 - iy * 0.2));

            //下段

            for(iy = 0, y = PALY + 1 + PALYCNT / 2 * PALH; iy < PALYCNT / 2; iy++, y += PALH)
                for(ix = 0, x = 1; ix < PALXCNT; ix++, x += PALW)
                    m_img.fillBox(x, y, PALW - 1, PALH - 1, AXHLStoRGB(ix * 12 + 180, dtmp, 1 - iy * 0.2));
            break;
        //L-H
        case 2:
            dtmp = g_conf->nColHLSVal[3] * 0.01;

            //上段

            for(iy = 0, y = PALY + 1; iy < PALYCNT / 2; iy++, y += PALH)
                for(ix = 0, x = 1; ix < PALXCNT; ix++, x += PALW)
                    m_img.fillBox(x, y, PALW - 1, PALH - 1, AXHLStoRGB(ix * 12, 0.85 - iy * 0.15, dtmp));

            //下段

            for(iy = 0, y = PALY + 1 + PALYCNT / 2 * PALH; iy < PALYCNT / 2; iy++, y += PALH)
                for(ix = 0, x = 1; ix < PALXCNT; ix++, x += PALW)
                    m_img.fillBox(x, y, PALW - 1, PALH - 1, AXHLStoRGB(ix * 12 + 180, 0.85 - iy * 0.15, dtmp));
            break;
    }

    redraw();
}

//! マウス位置からトップ値変更

void CColSub_hlspal::_moveTop(int winx)
{
    int type,n;

    type = g_conf->nColHLSVal[0];

    //カラー値

    if(type == 0)
    {
        n = (winx - 1) / TOP_H_W;
        if(n < 0) n = 0; else if(n > 29) n = 29;

        n *= 12;
    }
    else
    {
        n = (winx - 1) / TOP_LS_W;
        if(n < 0) n = 0; else if(n > 14) n = 14;

        if(type == 1)
            n = 92 - n * 6;
        else
            n = 100 - n * 6;
    }

    //変更

    if(g_conf->nColHLSVal[type + 1] != n)
    {
        g_conf->nColHLSVal[type + 1] = n;

        _drawTopCursor();
        drawPalette();
    }
}

//! マウス位置からパレットの色選択

void CColSub_hlspal::_movePal(int wx,int wy)
{
    int x,y;
    DWORD col;
    double dtmp;

    //パレット位置

    x = (wx - 1) / PALW;
    y = (wy - PALY - 1) / PALH;

    if(x < 0) x = 0; else if(x >= PALXCNT) x = PALXCNT - 1;
    if(y < 0) y = 0; else if(y >= PALYCNT) y = PALYCNT - 1;

    //前回位置と比較

    if(x != m_nDragX || y != m_nDragY)
    {
        //カーソル

        if(m_nDragX >= 0)
            m_img.box(m_nDragX * PALW, m_nDragY * PALH + PALY, PALW + 1, PALH + 1, 0xcccccc);

        m_img.box(x * PALW, y * PALH + PALY, PALW + 1, PALH + 1, 0);

        redraw();

        //

        m_nDragX = x;
        m_nDragY = y;

        //色取得

        switch(g_conf->nColHLSVal[0])
        {
            case 0:
                col = AXHLStoRGB(g_conf->nColHLSVal[1], 0.92 - x * 0.06, 1 - y * 0.1);
                break;
            case 1:
                dtmp = g_conf->nColHLSVal[2] * 0.01;

                if(y < PALYCNT / 2)
                    col = AXHLStoRGB(x * 12, dtmp, 1 - y * 0.2);
                else
                    col = AXHLStoRGB(x * 12 + 180, dtmp, 1 - (y - PALYCNT/2) * 0.2);
                break;
            case 2:
                dtmp = g_conf->nColHLSVal[3] * 0.01;

                if(y < PALYCNT / 2)
                    col = AXHLStoRGB(x * 12, 0.85 - y * 0.15, dtmp);
                else
                    col = AXHLStoRGB(x * 12 + 180, 0.85 - (y - PALYCNT/2) * 0.15, dtmp);
                break;
        }

        //通知

        getNotify()->onNotify(this, 0, col);
    }
}

//!< 描画

BOOL CColSub_hlspal::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);

    return TRUE;
}

//!< ボタン押し

BOOL CColSub_hlspal::onButtonDown(AXHD_MOUSE *phd)
{
    if(!m_fDrag && phd->button == BUTTON_LEFT)
    {
        if(phd->y < PALY)
        {
            //トップ

            _moveTop(phd->x);

            m_fDrag = DRAG_TOP;
        }
        else
        {
            //パレット部分

            m_nDragX = m_nDragY = -1;

            _movePal(phd->x, phd->y);

            m_fDrag = DRAG_PAL;
        }

        grabPointer();
    }

    return TRUE;
}

//!< ボタン離し

BOOL CColSub_hlspal::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDrag)
    {
        //パレット選択時、カーソル消去

        if(m_fDrag == DRAG_PAL)
        {
            m_img.box(m_nDragX * PALW, m_nDragY * PALH + PALY, PALW + 1, PALH + 1, 0xcccccc);
            redraw();
        }

        //

        m_fDrag = FALSE;
        ungrabPointer();
    }

    return TRUE;
}

//!< 移動

BOOL CColSub_hlspal::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_fDrag == DRAG_TOP)
        _moveTop(phd->x);
    else if(m_fDrag == DRAG_PAL)
        _movePal(phd->x, phd->y);

    return TRUE;
}


//***************************************
// CColSub_paltab : パレット選択タブ
//***************************************


CColSub_paltab::CColSub_paltab(AXWindow *pParent)
    : AXWindow(pParent, 0, 0)
{

}

void CColSub_paltab::calcDefSize()
{
    m_nOneH = m_pFont->getHeight() + YSPACE * 2 + 1;

    m_nDefW = 30;
    m_nDefH = m_nOneH * 4 + 1;
}

//!< ボタン押し

BOOL CColSub_paltab::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT)
    {
        int no = phd->y / m_nOneH;

        if(no >= CPaletteList::CNT) no = CPaletteList::CNT - 1;

        //選択変更

        if(g_conf->btColPalSel != no)
        {
            getNotify()->onNotify(this, 0, no);

            redraw();
        }
    }

    return TRUE;
}

//!< 描画

BOOL CColSub_paltab::onPaint(AXHD_PAINT *phd)
{
    int i,y,w;
    char str[2] = {'1',0};

    //枠

    drawBox(0, 0, m_nW, m_nH, AXAppRes::BLACK);

    for(i = 0, y = m_nOneH; i < 3; i++, y += m_nOneH)
        drawLineH(0, y, m_nW, AXAppRes::BLACK);

    //

    AXDrawText dt(m_id);

    for(i = 0, y = 1; i < 4; i++, y += m_nOneH)
    {
        //背景

        if(i == g_conf->btColPalSel)
            drawFillBox(0, y, m_nW - 1, m_nOneH - 1, AXAppRes::FACELIGHT);
        else
            drawFillBox(1, y, m_nW - 2, m_nOneH - 1, AXAppRes::FACEDARK);

        //テキスト

        w = m_pFont->getTextWidth(str, 1);

        dt.draw(*m_pFont, 1 + (m_nW - 2 - w) / 2, y + YSPACE, str, 1);

        str[0]++;
    }

    dt.end();

    return TRUE;
}


//***************************************
// CColSub_pal : パレット
//***************************************


CColSub_pal::CColSub_pal(AXWindow *pParent,AXScrollBar *pScr)
    : AXWindow(pParent, 0, LF_EXPAND_WH)
{
    m_pScr  = pScr;
    m_fDown = 0;

    m_nOneW = g_conf->wPalOneSize >> 8;
    m_nOneH = g_conf->wPalOneSize & 255;
}

//! 全体更新

void CColSub_pal::changeAll()
{
    _setScroll();
    draw();
}

//! 一つの表示サイズが変わった時

void CColSub_pal::changeOneSize()
{
    m_nOneW = g_conf->wPalOneSize >> 8;
    m_nOneH = g_conf->wPalOneSize & 255;

    m_nXCnt = m_nW / m_nOneW;
    if(m_nXCnt == 0) m_nXCnt = 1;

    changeAll();
}

//!< スクロール範囲セット

void CColSub_pal::_setScroll()
{
    int max,page;

    max = (PALETTE->m_pal[g_conf->btColPalSel].getCnt() + m_nXCnt - 1) / m_nXCnt;

    page = m_nH / m_nOneH;
    if(page <= 0) page = 1;

    m_pScr->setStatus(0, max, page);
}

//! 描画

void CColSub_pal::draw()
{
    int ix,iy,x,y,pcnt,no;
    LPDWORD p;

    //背景

    m_img.clear(0xcccccc);

    //パレット

    no   = m_pScr->getPos() * m_nXCnt;
    p    = PALETTE->m_pal[g_conf->btColPalSel].getBuf() + no;
    pcnt = PALETTE->m_pal[g_conf->btColPalSel].getCnt();

    for(iy = 0, y = 1; 1; iy++, y += m_nOneH)
    {
        for(ix = 0, x = 1; ix < m_nXCnt; ix++, x += m_nOneW, p++, no++)
        {
            if(no >= pcnt) goto END;

            m_img.fillBox(x, y, m_nOneW - 1, m_nOneH - 1, *p & 0xffffff);
        }
    }

END:
    redraw();
}

//! カーソル描画

void CColSub_pal::_drawCursor(BOOL bErase)
{
    m_img.box((m_nDownNo % m_nXCnt) * m_nOneW, (m_nDownNo / m_nXCnt - m_pScr->getPos()) * m_nOneH,
            m_nOneW + 1, m_nOneH + 1, (bErase)? 0xcccccc: 0xff0000);

    redraw();
}

//! D&Dカーソル描画

void CColSub_pal::_drawDDCursor(BOOL bErase)
{
    DWORD col;

    if(!bErase)
        col = 0xff0000;
    else
        col = (m_nMoveNo == m_nDownNo || m_nMoveNo == m_nDownNo + 1)? 0xff0000: 0xcccccc;

    m_img.lineV((m_nMoveNo % m_nXCnt) * m_nOneW, (m_nMoveNo / m_nXCnt - m_pScr->getPos()) * m_nOneH,
                m_nOneH + 1, col);

    redraw();
}

//! マウス位置からパレット番号取得
/*!
    @return -1 で範囲外
*/

int CColSub_pal::_getPalNo(int x,int y)
{
    int no;

    if(x < 0 || y < 0) return -1;

    x = x / m_nOneW;
    y = y / m_nOneH + m_pScr->getPos();

    if(x >= m_nXCnt) return -1;

    no = x + y * m_nXCnt;

    if(no >= PALETTE->m_pal[g_conf->btColPalSel].getCnt()) return -1;

    return no;
}

//! メニュー実行

void CColSub_pal::_runMenu(int x,int y)
{
    AXMenu *pmenu;
    int id;

    _drawCursor(FALSE);

    //

    pmenu = new AXMenu;

    _trgroup(strid::GROUP_COLORWIN);
    pmenu->addTrMul(strid::COLWIN_PALCOL_SET, 3);

    id = pmenu->popup(NULL, x, y, 0);

    delete pmenu;

    //

    if(id == -1)
        _drawCursor(TRUE);
    else
    {
        switch(id)
        {
            case strid::COLWIN_PALCOL_SET:
                PALETTE->m_pal[g_conf->btColPalSel].setCol(m_nDownNo, g_conf->dwDrawCol);
                break;
            case strid::COLWIN_PALCOL_INS:
                PALETTE->m_pal[g_conf->btColPalSel].insertCol(m_nDownNo, g_conf->dwDrawCol);
                break;
            case strid::COLWIN_PALCOL_DEL:
                PALETTE->m_pal[g_conf->btColPalSel].deleteCol(m_nDownNo);
                break;
        }

        changeAll();
    }
}


//---------------


//!< サイズ変更

BOOL CColSub_pal::onSize()
{
    m_img.recreate(m_nW, m_nH, 10, 10);

    m_nXCnt = m_nW / m_nOneW;
    if(m_nXCnt == 0) m_nXCnt = 1;

    changeAll();

    return TRUE;
}

//!< 描画

BOOL CColSub_pal::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);
    return TRUE;
}

//!< ボタン押し

BOOL CColSub_pal::onButtonDown(AXHD_MOUSE *phd)
{
    if(m_fDown) return TRUE;

    //押された位置

    m_nDownNo = _getPalNo(phd->x, phd->y);
    if(m_nDownNo == -1) return TRUE;

    //

    if(phd->button == BUTTON_LEFT)
    {
        //------- 左ボタン

        if(phd->state & STATE_CTRL)
        {
            //+Ctrl : 描画色登録

            PALETTE->m_pal[g_conf->btColPalSel].setCol(m_nDownNo, g_conf->dwDrawCol);

            draw();
            m_fDown = DOWNF_POS;
        }
        else if(phd->state & STATE_SHIFT)
        {
            //+Shift : 描画色挿入

            PALETTE->m_pal[g_conf->btColPalSel].insertCol(m_nDownNo, g_conf->dwDrawCol);

            changeAll();
            m_fDown = DOWNF_POS;
        }
        else if(phd->state & STATE_ALT)
        {
            //+Alt : グラデーション

            m_img.setPixel(phd->x, phd->y, AXImage::COL_XOR);

            m_fDown = DOWNF_GRAD;
        }
        else
        {
            //スポイト

            getNotify()->onNotify(this, 0, PALETTE->m_pal[g_conf->btColPalSel].getCol(m_nDownNo));

            m_fDown = DOWNF_NORMAL;
            m_ptDown.x = phd->x;
            m_ptDown.y = phd->y;
        }
    }
    else if(phd->button == BUTTON_RIGHT)
    {
        //------- 右ボタン

        if(phd->state & STATE_CTRL)
        {
            //+Ctrl : 削除

            PALETTE->m_pal[g_conf->btColPalSel].deleteCol(m_nDownNo);

            changeAll();
            m_fDown = DOWNF_POS;
        }
        else
            //メニュー
            _runMenu(phd->rootx, phd->rooty);
    }

    //カーソル描画

    if(m_fDown == DOWNF_POS || m_fDown == DOWNF_NORMAL)
        _drawCursor(FALSE);

    //グラブ

    if(m_fDown)
    {
        m_ptDown.x = phd->x;
        m_ptDown.y = phd->y;

        m_ptBk = m_ptDown;

        grabPointer();
    }

    return TRUE;
}

//!< ボタン離し

BOOL CColSub_pal::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDown)
    {
        if(m_fDown == DOWNF_POS || m_fDown == DOWNF_NORMAL || m_fDown == DOWNF_MOVE)
            _drawCursor(TRUE);

        if(m_fDown == DOWNF_MOVE)
        {
            //移動

            unsetCursor();

            PALETTE->m_pal[g_conf->btColPalSel].moveCol(m_nDownNo, m_nMoveNo);

            draw();
        }
        else if(m_fDown == DOWNF_GRAD)
        {
            //グラデーション

            m_nMoveNo = _getPalNo(phd->x, phd->y);

            if(m_nMoveNo == -1)
            {
                m_img.line(m_ptDown.x, m_ptDown.y, m_ptBk.x, m_ptBk.y, AXImage::COL_XOR);
                redraw();
            }
            else
            {
                PALETTE->m_pal[g_conf->btColPalSel].gradient(m_nDownNo, m_nMoveNo);
                draw();
            }
        }

        //

        m_fDown = 0;
        ungrabPointer();
    }

    return TRUE;
}

//! 移動

BOOL CColSub_pal::onMouseMove(AXHD_MOUSE *phd)
{
    int no;

    switch(m_fDown)
    {
        //通常押し時、別のパレットに移動したらD&D開始
        case DOWNF_NORMAL:
            //押し位置から一定範囲内は判定しない

            if(phd->x < m_ptDown.x - 8 || phd->x > m_ptDown.x + 8 ||
               phd->y < m_ptDown.y - 8 || phd->y > m_ptDown.y + 8)
            {
                m_nMoveNo = _getPalNo(phd->x, phd->y);

                if(m_nMoveNo != -1 && m_nDownNo != m_nMoveNo)
                {
                    _drawDDCursor(FALSE);

                    setCursor(cursor::getImgDat(cursor::ITEMMOVE));

                    m_fDown = DOWNF_MOVE;
                }
            }
            break;
        //D&D
        case DOWNF_MOVE:
            no = _getPalNo(phd->x, phd->y);

            if(no != -1 && no != m_nMoveNo)
            {
                _drawDDCursor(TRUE);
                m_nMoveNo = no;
                _drawDDCursor(FALSE);
            }
            break;
        //グラデーション
        case DOWNF_GRAD:
            m_img.line(m_ptDown.x, m_ptDown.y, m_ptBk.x, m_ptBk.y, AXImage::COL_XOR);
            m_img.line(m_ptDown.x, m_ptDown.y, phd->x, phd->y, AXImage::COL_XOR);

            redraw();

            m_ptBk.x = phd->x;
            m_ptBk.y = phd->y;
            break;
    }

    return TRUE;
}
