/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    各サブウィジェット

    CPressCurve, CSelImgBar, CImgPrev, CArrowMenuBtt, CScaleRotBar
*/


#include <math.h>

#include "CArrowMenuBtt.h"
#include "CScaleRotBar.h"
#include "CSelImgBar.h"
#include "CImgPrev.h"
#include "CPressCurve.h"

#include "CSelImgDlg.h"

#include "strid.h"

#include "AXFont.h"
#include "AXMenu.h"
#include "AXApp.h"
#include "AXAppRes.h"
#include "AXUtilStr.h"



//***********************************
// CPressCurve - 筆圧カーブ
//***********************************



CPressCurve::CPressCurve(AXWindow *pParent,UINT uItemID,int size)
    : AXWindow(pParent, 0, 0, uItemID, 0)
{
    m_uFlags |= FLAG_REDRAW;
    m_nMinW = m_nMinH = size;

    m_nVal = 100;

    m_img.create(size, size);
}

//! 値セット

void CPressCurve::setVal(int val)
{
    m_nVal = val;

    redrawUpdate();
}

//! 描画

BOOL CPressCurve::onPaint(AXHD_PAINT *phd)
{
    if(m_uFlags & FLAG_REDRAW)
        _draw();

    m_img.put(m_id);

    return TRUE;
}

//! ボタン押し

BOOL CPressCurve::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && !(m_uFlags & FLAG_TEMP1))
    {
        _changePos(phd->x, phd->y);

        m_uFlags |= FLAG_TEMP1;
        grabPointer();
    }

    return TRUE;
}

//! ボタン離し

BOOL CPressCurve::onButtonUp(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && (m_uFlags & FLAG_TEMP1))
    {
        m_uFlags &= ~FLAG_TEMP1;
        ungrabPointer();
    }

    return TRUE;
}

//! 移動

BOOL CPressCurve::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_uFlags & FLAG_TEMP1)
        _changePos(phd->x, phd->y);

    return TRUE;
}

//! マウス位置から値セット

void CPressCurve::_changePos(int x,int y)
{
    int len,val,max;

    x -= m_nW / 2;
    y -= m_nW / 2;

    len = (int)(::sqrt(x * x + y * y) + 0.5);

    max = m_nW - 10;
    if(len > max) len = max;

    //値

    if(x < 0 || y < 0)
        val = 100 - (100 - 1) * len / max;
    else
        val = (600 - 100) * len / max + 100;

    if(val >= 98 && val <= 102) val = 100;

    //変更

    if(m_nVal != val)
    {
        m_nVal = val;

        redrawUpdate();

        getNotify()->onNotify(this, 0, 0);
    }
}

//! 描画

void CPressCurve::_draw()
{
    int size,size2,i,y,by;
    double gamma,val;

    size  = m_nMinW;
    size2 = m_nMinW - 1;

    //背景

    m_img.clear(isEnabled()? 0xffffff: 0xdddddd);

    //基準線

    m_img.line(0, size2, size2, 0, 0x808080);

    //枠

    m_img.box(0, 0, size, size, 0);

    //曲線

    gamma = m_nVal * 0.01;

    by = size2;

    for(i = 1; i < size; i++)
    {
        val = ::pow((double)i / size2, gamma);

        y = size2 - (int)(val * size2 + 0.5);

        m_img.line(i - 1, by, i, y, 0x0000ff);

        by = y;
    }
}


//*****************************************
// CSelImgBar : ブラシ/テクスチャ画像選択バー
//*****************************************



CSelImgBar::CSelImgBar(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags,UINT uItemID,DWORD dwPadding,int type)
    : AXWindow(pParent, uStyle, uLayoutFlags, uItemID, dwPadding)
{
    m_nType = type;
}

//! 名前セット

void CSelImgBar::setName(const AXString &name)
{
    m_strName.empty();
    m_strNameSrc = name;

    //表示用名

    _trgroup(strid::GROUP_TEXBRUSHIMG);

    switch(m_nType)
    {
        case TYPE_OPT_TEXTURE:
            if(name.isEmpty())
                m_strName = _str(strid::TEXBRUSH_NONE);
            break;
        case TYPE_BRUSH_TEXTURE:
            if(name.isEmpty())
                m_strName = _str(strid::TEXBRUSH_NONE_FORCE);
            else if(name == "?")
                m_strName = _str(strid::TEXBRUSH_OPTTEXTURE);
            break;
        case TYPE_BRUSH_BRUSH:
            if(name.isEmpty())
                m_strName = _str(strid::TEXBRUSH_CIRCLE);
            break;
    }

    if(m_strName.isEmpty())
        m_strName = name;

    redraw();
}

//! 標準サイズ計算

void CSelImgBar::calcDefSize()
{
    m_nDefH = m_pFont->getHeight() + 6;
    m_nDefW = m_nDefH + 12;
}

//! 描画

BOOL CSelImgBar::onPaint(AXHD_PAINT *phd)
{
    int bsize = m_nH - 4;

    //枠

    drawBox(0, 0, m_nW, m_nH, AXAppRes::BLACK);

    //背景

    drawFillBox(1, 1, m_nW - 2, m_nH - 2, (isEnabled())? AXAppRes::WHITE: AXAppRes::FACELIGHT);

    //ボタン

    drawBox(m_nW - 2 - bsize, 2, bsize, bsize, AXAppRes::BLACK);
    drawArrowDown(m_nW - 2 - bsize + bsize / 2, 2 + bsize / 2, AXAppRes::BLACK);

    //名前

    AXDrawText dt(m_id);

    dt.setClipRect(3, 0, m_nW - bsize - 3 - 5, m_nH);
    dt.draw(*m_pFont, 3, 3, m_strName, AXAppRes::TC_BLACK);

    dt.end();

    return TRUE;
}

//! ボタン押し

BOOL CSelImgBar::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_RIGHT)
    {
        //右ボタン -> 空文字でセット

        if(m_strNameSrc.isNoEmpty())
        {
            AXString str;

            setName(str);

            getNotify()->onNotify(this, 0, 0);
        }
    }
    else if(phd->button == BUTTON_LEFT)
    {
        //左ボタン

        if(phd->x >= m_nW - 2 - (m_nH - 4))
            _runMenu();
        else
            _dialog();
    }

    return TRUE;
}

//! 画像選択ダイアログ

void CSelImgBar::_dialog()
{
    CSelImgDlg *pdlg;

    pdlg = new CSelImgDlg(m_pTopLevel, m_nType, &m_strNameSrc);

    if(pdlg->runDialog())
    {
        m_strName = m_strNameSrc;
        redraw();

        getNotify()->onNotify(this, 0, 0);
    }
}

//! メニュー実行

void CSelImgBar::_runMenu()
{
    AXMenu *pmenu;
    int i,n;
    AXRectSize rcs;
    AXString str;
    WORD id[3][3] = {
        {strid::TEXBRUSH_MENU_NONE, strid::TEXBRUSH_MENU_SELIMG, 0},
        {strid::TEXBRUSH_MENU_OPTTEXTURE, strid::TEXBRUSH_MENU_NONE_FORCE, strid::TEXBRUSH_MENU_SELIMG},
        {strid::TEXBRUSH_MENU_CIRCLE, strid::TEXBRUSH_MENU_SELIMG, 0}
    };

    _trgroup(strid::GROUP_TEXBRUSHIMG);

    pmenu = new AXMenu;

    for(i = 0; i < 3; i++)
    {
        if(id[m_nType][i])
            pmenu->addTr(id[m_nType][i]);
    }

    getWindowRect(&rcs);

    n = pmenu->popup(NULL, rcs.x + rcs.w, rcs.y + rcs.h, AXMenu::POPF_RIGHT);

    delete pmenu;

    //-------

    if(n == -1) return;

    if(n == strid::TEXBRUSH_MENU_SELIMG)
        _dialog();
    else
    {
        if(n == strid::TEXBRUSH_MENU_OPTTEXTURE)
            str = '?';

        setName(str);
        getNotify()->onNotify(this, 0, 0);
    }
}


//****************************************************
// CImgPrev : AXImage を表示するだけのウィジェット（枠付き）
//****************************************************



CImgPrev::CImgPrev(AXWindow *pParent,UINT uStyle,UINT uLayoutFlags,DWORD dwPadding,int w,int h)
    : AXWindow(pParent, uStyle, uLayoutFlags, 0, dwPadding)
{
    m_nMinH = h + 2;
    m_nMinW = w + 2;

    m_img.create(w, h);
}

void CImgPrev::calcDefSize()
{
    m_nDefW = 10;
    m_nDefH = 5;
}

BOOL CImgPrev::onPaint(AXHD_PAINT *phd)
{
    drawFrameSunken(0, 0, m_nW, m_nH);

    m_img.put(m_id, 1, 1, 0, 0, m_nW - 2, m_nH - 2);

    return TRUE;
}


//*********************************************
// CArrowMenuBtt : 矢印のメニューボタン
//*********************************************
/*
    プレビューウィンドウやイメージビューウィンドウに付く。
    ボタンが押された時、親の onNotify が実行される。
*/


CArrowMenuBtt::CArrowMenuBtt(AXWindow *pParent)
    : AXWindow(pParent, 0, 0)
{
    resize(17, 17);
}

//! 親のサイズにあわせて位置セット

void CArrowMenuBtt::moveParent()
{
    move(m_pParent->getWidth() - m_nW, 0);
}

//! ボタン押し時

BOOL CArrowMenuBtt::onButtonDown(AXHD_MOUSE *phd)
{
    m_pParent->sendNotify(this, 0, 0);

    return TRUE;
}

//! 描画

BOOL CArrowMenuBtt::onPaint(AXHD_PAINT *phd)
{
    //枠

    drawBox(0, 0, m_nW, m_nH, AXAppRes::WHITE);
    drawBox(1, 1, m_nW - 2, m_nH - 2, AXAppRes::BLACK);

    //背景

    drawFillBox(2, 2, m_nW - 4, m_nH - 4, AXAppRes::FACEFOCUS);

    //矢印

    drawArrowDown(m_nW / 2, m_nH / 2, AXAppRes::BLACK);

    return TRUE;
}



//*************************************************
// CScaleRotBar : ツールウィンドウの表示倍率・回転バー
//*************************************************


CScaleRotBar::CScaleRotBar(AXWindow *pParent,UINT uStyle,DWORD dwPadding,int min,int max)
    : AXWindow(pParent, uStyle, LF_EXPAND_W, 0, dwPadding)
{
    m_nMin = min;
    m_nMax = max;
    m_nPos = min;
}

//! 位置セット

void CScaleRotBar::setPos(int pos)
{
    if(pos != m_nPos)
    {
        m_nPos = pos;
        redraw();
    }
}

void CScaleRotBar::calcDefSize()
{
    m_nDefH = m_pFont->getHeight() + 2;
}

//! 位置変更

BOOL CScaleRotBar::_changePos(int x)
{
    int barw,ct,pos;

    x -= 2;

    barw = m_nW - m_nH - 8;
    ct   = barw >> 1;

    if(x < 0) x = 0; else if(x >= barw) x = barw - 1;

    //位置

    if(m_uStyle & STYLE_ROTATE)
    {
        if(x >= ct)
            pos = (int)((x - ct) * 18000.0 / (barw - 1 - ct) + 0.5);
        else
            pos = (int)((x - ct) * 18000.0 / ct - 0.5);
    }
    else
    {
        //100%以上の場合は1.0単位

        if(x >= ct)
        {
            pos = (int)((double)(x - ct) * (m_nMax / 10 - 100) / (barw - 1 - ct) + 0.5) + 100;
            pos *= 10;
        }
        else
            pos = (int)((double)x * (1000 - m_nMin) / ct + 0.5) + m_nMin;
    }

    //セット

    if(pos != m_nPos)
    {
        m_nPos = pos;
        redrawUpdate();
        return TRUE;
    }

    return FALSE;
}

//! 描画

BOOL CScaleRotBar::onPaint(AXHD_PAINT *phd)
{
    int barw,ct,n,len;
    BOOL bRotate;
    char m[32];

    bRotate = m_uStyle & STYLE_ROTATE;
    barw    = m_nW - m_nH - 8;
    ct      = barw >> 1;

    //バー背景

    drawFillBox(0, 0, m_nW - m_nH, m_nH, AXAppRes::FACE);

    //バー上辺

    drawFillBox(2, 0, barw, 2, AXAppRes::FRAMEDARK);

    //中央線

    drawLineV(2 + ct, 2, 7, AXAppRes::FRAMEDARK);

    //ボタン

    n = m_nW - m_nH;

    drawBox(n, 0, m_nH, m_nH, AXAppRes::FRAMEOUTSIDE);
    drawFillBox(n + 1, 1, m_nH - 2, m_nH - 2, AXAppRes::FACELIGHT);
    drawFillBox(n + 3, m_nH / 2 - 1, m_nH - 6, 2, AXAppRes::TEXTNORMAL);

    //------- カーソル

    if(bRotate)
    {
        //回転

        if(m_nPos >= 0)
            n = (int)(ct + m_nPos * (barw - 1 - ct) / 18000.0 + 0.5);
        else
            n = (int)(ct + m_nPos * ct / 18000.0 + 0.5);
    }
    else
    {
        //拡大

        if(m_nPos >= 1000)
            n = (int)(ct + (double)(m_nPos - 1000) * (barw - 1 - ct) / (m_nMax - 1000) + 0.5);
        else
            n = (int)((double)(m_nPos - m_nMin) * ct / (1000 - m_nMin) + 0.5);
    }

    n += 2;

    drawArrowUp(n, 3, AXAppRes::TEXTNORMAL);

    //---------- 数値

    if(bRotate)
        len = AXIntToFloatStr(m, m_nPos, 2);
    else
    {
        len = AXIntToFloatStr(m, m_nPos, 1);
        m[len++] = '%';
        m[len] = 0;
    }

    //X位置

    if((bRotate && m_nPos >= 0) || (!bRotate && m_nPos >= 1000))
        n = 2;
    else
        n = m_nW - m_nH - 6 - m_pFont->getTextWidth(m, len);

    //描画

    AXDrawText dt(m_id);
    dt.draw(*m_pFont, n, 1, m, len);
    dt.end();

    return TRUE;
}

//! ボタン押し時

BOOL CScaleRotBar::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && !(m_uFlags & FLAG_DRAG))
    {
        if(phd->x >= m_nW - m_nH)
        {
            //ボタン

            getNotify()->onNotify(this, NOTIFY_BUTTON, 0);
        }
        else
        {
            //バー

            m_uFlags |= FLAG_DRAG;
            grabPointer();

            _changePos(phd->x);

            getNotify()->onNotify(this, NOTIFY_DOWN, 0);
        }
    }

    return TRUE;
}

//! ボタン離し時

BOOL CScaleRotBar::onButtonUp(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && (m_uFlags & FLAG_DRAG))
    {
        m_uFlags &= ~FLAG_DRAG;
        ungrabPointer();

        delTimerAll();

        getNotify()->onNotify(this, NOTIFY_UP, 0);
    }

    return TRUE;
}

//! マウス移動時

BOOL CScaleRotBar::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_uFlags & FLAG_DRAG)
    {
        if(_changePos(phd->x))
            addTimer(0, 5);
    }

    return TRUE;
}

//! タイマー

BOOL CScaleRotBar::onTimer(UINT uTimerID,ULONG lParam)
{
    delTimer(uTimerID);

    getNotify()->onNotify(this, NOTIFY_MOVE, 0);

    return TRUE;
}
