/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/

#include "CSelImgDlg.h"

#include "CImgPrev.h"
#include "CImage8.h"
#include "CImage32.h"
#include "CConfig.h"

#include "global.h"
#include "strid.h"

#include "AXLayout.h"
#include "AXButton.h"
#include "AXTreeView.h"
#include "AXListBox.h"
#include "AXDir.h"
#include "AXFileStat.h"
#include "AXTreeViewItem.h"
#include "AXListBoxItem.h"
#include "AXApp.h"
#include "AXUtilFile.h"
#include "AXUtilStr.h"
#include "AXTrSysID.h"



//*********************************************
// CSelImgDlg : テクスチャ・ブラシ画像選択ダイアログ
//*********************************************


CSelImgDlg::CSelImgDlg(AXWindow *pOwner,int type,AXString *pstrRet)
    : AXDialog(pOwner, WS_DIALOG_NORMAL | WS_BK_FACE)
{
    AXLayout *plTop,*pl;
    AXWindow *pbt1,*pbt2;

    m_nType   = type;
    m_pstrRet = pstrRet;
    m_bUserDir = FALSE;

    setTitle(_string(strid::GROUP_TEXBRUSHIMG, strid::TEXBRUSH_DLG_TITLE));

    setLayout(plTop = new AXLayoutHorz(0, 8));
    plTop->setSpacing(8);

    //--------------

    //ツリー

    plTop->addItem(m_pTree = new AXTreeView(this, AXScrollView::SVS_HORZVERTSUNKEN,
                    LF_EXPAND_WH, WID_TREE, 0));

    //リスト

    plTop->addItem(m_pList = new AXListBox(this,
                                AXScrollView::SVS_SUNKEN|AXScrollView::SVS_VERT,
                                LF_EXPAND_WH, WID_LIST, 0));

    //プレビュー・ボタン

    plTop->addItem(pl = new AXLayoutVert(LF_EXPAND_H, 0));

    pl->addItem(m_pPrev = new CImgPrev(this, 0, LF_EXPAND_Y, MAKE_DW4(0,0,0,10), 100, 150));

    pl->addItem(pbt1 = new AXButton(this, WS_DISABLE, LF_EXPAND_W, 1, MAKE_DW4(0,0,0,2), TRSYS_GROUPID, TRSYS_OK));
    pl->addItem(pbt2 = new AXButton(this, 0, LF_EXPAND_W, 2, 0, TRSYS_GROUPID, TRSYS_CANCEL));

    pbt1->setMinHeight(25);
    pbt2->setMinHeight(25);

    setDefaultButton(pbt1);

    //---------

    _setInitPath();
    _initTree();

    if(m_strDir.isNoEmpty()) _setFileList();

    m_pPrev->getImg()->clear(0x808080);

    //---------

    calcDefSize();
    resizeSwitch(g_conf->szSelImgDlg.w, g_conf->szSelImgDlg.h, m_nDefW + 220, m_nDefH + 80);
    show();
}

//! 通知

BOOL CSelImgDlg::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    switch(pwin->getItemID())
    {
        //ツリー
        case WID_TREE:
            if(uNotify == AXTreeView::TVN_SELCHANGE)
                _treeSelChange();
            break;
        //リスト
        case WID_LIST:
            if(uNotify == AXListBox::LBN_SELCHANGE)
                _listChangeSel();
            else if(uNotify == AXListBox::LBN_DBLCLK)
                _end(TRUE);
            break;

        //OK
        case 1:
            _end(TRUE);
            break;
        //キャンセル
        case 2:
            _end(FALSE);
            break;
    }

    return TRUE;
}


//==================================
// サブ
//==================================


//! 現在の選択ファイルで終了

void CSelImgDlg::_end(BOOL bOK)
{
    if(bOK)
    {
        AXListBoxItem *p;

        p = m_pList->getSelItem();
        if(!p) return;

        //ファイル名パス

        if(m_bUserDir)
            m_pstrRet->empty();
        else
            *m_pstrRet = '/';

        *m_pstrRet += m_strDirPath;
        m_pstrRet->path_add(p->m_strText);
    }

    //サイズ保存

    g_conf->szSelImgDlg.w = m_nW;
    g_conf->szSelImgDlg.h = m_nH;

    endDialog(bOK);
}

//! 初期パスセット

void CSelImgDlg::_setInitPath()
{
    AXString str;

    str = *m_pstrRet;

    if(str.isNoEmpty() && str != "?")
    {
        //先頭が'/'でデフォルト（システム）ディレクトリ

        if(str[0] == '/')
        {
            str = m_pstrRet->at(1);
            m_bUserDir = FALSE;
        }
        else
            m_bUserDir = TRUE;

        //ディレクトリパス

        m_strDirPath.path_removeFileName(str);

        _getRootDir(&m_strDir, m_bUserDir);
        m_strDir.path_add(m_strDirPath);

        //存在しなければトップ

        if(!AXIsExistFile(m_strDir, TRUE))
        {
            m_strDir.empty();
            m_strDirPath.empty();
        }
    }
}

//! 画像のルートディレクトリ取得

void CSelImgDlg::_getRootDir(AXString *pStr,BOOL bUser)
{
    if(m_nType == TYPE_BRUSH_BRUSH)
    {
        if(bUser)
            *pStr = g_conf->strUserBrushDir;
        else
            axapp->getResourcePath(pStr, "brush");
    }
    else
    {
        if(bUser)
            *pStr = g_conf->strUserTexDir;
        else
            axapp->getResourcePath(pStr, "texture");
    }
}

//! プレビューイメージセット

BOOL CSelImgDlg::_setPrevImg(const AXString &filename)
{
    CImage8 img8;
    CImage32 img32;
    CImage32::LOADINFO info;

    //32bit読み込み

    if(!img32.loadImage(filename, &info))
        return FALSE;

    //8bit変換＆プレビューセット

    if(m_nType == TYPE_BRUSH_BRUSH)
    {
        //ブラシ

        if(!img8.createFrom32bit_brush(img32, info.bAlpha))
            return FALSE;

        img8.drawBrushPrev(m_pPrev->getImg());
    }
    else
    {
        //テクスチャ

        if(!img8.createFrom32bit(img32, info.bAlpha))
            return FALSE;

        img8.drawTexturePrev(m_pPrev->getImg());
    }

    return TRUE;
}


//==================================
// ツリー関連
//==================================


//! ツリーの選択変更

void CSelImgDlg::_treeSelChange()
{
    AXTreeViewItem *pSel,*p;
    AXString str;

    pSel = m_pTree->getFocusItem();
    if(!pSel) return;

    //ディレクトリパス

    m_bUserDir = ((pSel->m_lParam & TREEPF_USER) != 0);

    m_strDirPath.empty();

    for(p = pSel; p && p->parent(); p = p->parent())
    {
        str = m_strDirPath;
        m_strDirPath = p->m_strText;
        m_strDirPath.path_add(str);
    }

    _getRootDir(&m_strDir, m_bUserDir);
    m_strDir.path_add(m_strDirPath);

    //ツリー、このディレクトリ下のディレクトリを追加
    /*
        選択されたらディレクトリの中のディレクトリを検索して追加する。
        一度読み込まれたディレクトリはフラグがONになっている。
    */

    if(!(pSel->m_lParam & TREEPF_READ))
        _addTreeDir(pSel, m_strDir);

    //ファイルリスト

    _setFileList();

    //OKボタンを無効

    enableWidget(1, FALSE);
}

//! ディレクトリツリー初期化

void CSelImgDlg::_initTree()
{
    AXString str,strPath;
    AXTreeViewItem *pDef,*pUser,*pParent,*p;
    LPCUSTR pu;

    //デフォルトディレクトリ

    str = "default /";
    pDef = m_pTree->addItem(NULL, str, -1, AXTreeViewItem::FLAG_EXPAND, 0);

    _getRootDir(&str, FALSE);
    _addTreeDir(pDef, str);

    //ユーザーディレクトリ

    str = "user /";
    pUser = m_pTree->addItem(NULL, str, -1, AXTreeViewItem::FLAG_EXPAND, TREEPF_USER);

    _getRootDir(&str, TRUE);
    _addTreeDir(pUser, str);

    //初期パスのディレクトリ

    if(m_strDir.isNoEmpty())
    {
        str = m_strDirPath;
        str += '/';
        str.replace('/', 0);

        _getRootDir(&strPath, m_bUserDir);

        pParent = (m_bUserDir)? pUser: pDef;

        for(pu = str; *pu; pu += AXUStrLen(pu) + 1)
        {
            //パス名検索

            for(p = pParent->first(); p; p = p->next())
            {
                if(p->m_strText == pu) break;
            }

            if(!p) break;

            pParent = p;
            pParent->expand();

            //追加

            strPath.path_add(pu);
            _addTreeDir(pParent, strPath);
        }

        m_pTree->setFocusItem(pParent);
    }
}

//! ツリーにフォルダ追加

void CSelImgDlg::_addTreeDir(AXTreeViewItem *pParent,const AXString &path)
{
    AXDir dir;
    AXFILESTAT stat;
    ULONG param;

    //読み込み済みフラグセット
    pParent->m_lParam |= TREEPF_READ;

    param = pParent->m_lParam & TREEPF_USER;

    //

    if(!dir.open(path)) return;

    while(dir.read())
    {
        if(dir.isSpecPath()) continue;

        if(dir.getFileStat(&stat))
        {
            if(stat.isDirectory())
                m_pTree->addItem(pParent, dir.getCurName(), -1, 0, param);
        }
    }

    dir.close();
}


//==================================
// ファイルリスト関連
//==================================


//! ファイル選択変更時

void CSelImgDlg::_listChangeSel()
{
    AXListBoxItem *p;
    AXString str;

    p = m_pList->getSelItem();
    if(!p) return;

    str = m_strDir;
    str.path_add(p->m_strText);

    //プレビューイメージ

    if(_setPrevImg(str))
        enableWidget(1, TRUE);
    else
        m_pPrev->getImg()->clear(0x808080);

    m_pPrev->redraw();
}

//! ソート関数

int sortfunc(AXListItem *pitem1,AXListItem *pitem2,ULONG lParam)
{
    return ((AXListBoxItem *)pitem1)->m_strText.compare(((AXListBoxItem *)pitem2)->m_strText);
}

//! 現在のパスのファイルリストセット

void CSelImgDlg::_setFileList()
{
    AXDir dir;
    AXString str,exts;
    AXFILESTAT stat;

    m_pList->deleteItemAll();

    exts = "bmp;png;jpg;jpeg;gif;";
    exts.replace(';', 0);

    //

    if(!dir.open(m_strDir)) return;

    while(dir.read())
    {
        if(!dir.getFileStat(&stat)) continue;

        if(!stat.isDirectory())
        {
            str = dir.getCurName();

            if(str.path_compareExt(exts))
                m_pList->addItem(str);
        }
    }

    dir.close();

    //

    m_pList->sortItem(sortfunc, 0);
}
