/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CMainWin [file] - ファイル関連
*/

#include "CMainWin.h"

#include "drawdat.h"
#include "draw_main.h"
#include "draw_update.h"
#include "draw_file.h"

#include "CConfig.h"
#include "CImageRGB16.h"

#include "CProgressDlg.h"
#include "CNewDlg.h"
#include "CSaveOptDlg.h"

#include "AXFile.h"
#include "AXMenu.h"
#include "AXToolBar.h"
#include "AXFileDialog.h"
#include "AXMessageBox.h"
#include "AXUtilStr.h"
#include "AXUtilImg.h"
#include "AXApp.h"

#include "global.h"
#include "strid.h"


//----------------------

void *thread_loadImage(void *);
void *thread_saveFile(void *);

struct FILESAVE_PROGINFO
{
    AXString    *pfilename;
    int         nFormat,
                nTPCol;
};

//----------------------


//! 新規作成

void CMainWin::newImage()
{
    CNewDlg *pdlg;
    int w,h;
    AXString str;

    pdlg = new CNewDlg(this);

    if(!pdlg->runDialog()) return;

    //履歴の先頭から取得

    w = g_conf->dwNewRecentSize[0] >> 16;
    h = g_conf->dwNewRecentSize[0] & 0xffff;

    //確認

    if(!checkImageUpdate()) return;

    //新規

    if(!draw::newImage(w, h, g_conf->wNewRecentDPI[0], TRUE))
        draw::newImage(300, 300, g_conf->wNewRecentDPI[0], TRUE);

    //更新

    updateNewCanvas(&str);
}


//=============================
// 画像開く
//=============================


//! 開く
/*!
    @param pstrDir 初期ディレクトリ指定。NULL で指定なし。
*/

void CMainWin::openFile(const AXString *pstrDir)
{
    AXString str,filter;

    //ファイル名取得

    filter = "APD/ADW/PSD/BMP/PNG/JPEG/GIF\t*.apd;*.adw;*.psd;*.bmp;*.png;*.jpg;*.jpeg;*.gif\tAzPainter file (*.apd)\t*.apd\tAll files\t*";

    if(!AXFileDialog::openFile(this, filter, 0,
            (pstrDir)? *pstrDir: g_conf->strOpenDir, 0, &str))
        return;

    //保存確認

    if(!checkImageUpdate()) return;

    //読み込み

    loadImage(str);
}

//! 画像ファイル読み込み処理

BOOL CMainWin::loadImage(const AXString &filename)
{
    AXString str;
    CProgressDlg *pdlg;
    int ret,format;

    //画像フォーマット

    format = _getFileFormat(filename);
    if(format == -1) return FALSE;

    //読み込み

    str = filename;

    pdlg = new CProgressDlg(this, thread_loadImage, (LPVOID)&str, &format);

    ret = pdlg->run();

    //

    if(ret == draw::LOADERR_SUCCESS)
    {
        //--------- 成功

        m_nFileFormat = format;

        //フォルダ記録

        g_conf->strOpenDir.path_removeFileName(str);

        //最近使ったファイル

        _addRecentFile(str);

        //ディレクトリ履歴

        AXAddRecentString(g_conf->strRecOpenDir, CConfig::RECDIR_NUM, g_conf->strOpenDir);

        //更新

        updateNewCanvas(&str);

        return TRUE;
    }
    else
    {
        //--------- エラー

        WORD errid;

        //エラーメッセージ

        switch(ret)
        {
            case draw::LOADERR_IMGSIZE: errid = strid::MES_LOADERR_IMGSIZE; break;
            case draw::LOADERR_OPENFILE: errid = strid::MES_LOADERR_OPENFILE; break;
            case draw::LOADERR_FORMAT: errid = strid::MES_LOADERR_FORMAT; break;
            default: errid = strid::MES_FAILED_LOAD; break;
        }

        errMes(strid::GROUP_MESSAGE, errid);

        /* レイヤが一つもなければ新規作成。
           カレントレイヤが指定されていなければ途中まで読み込まれた */

        if(draw::errLoadImage())
            updateNewCanvas(&str);

        return FALSE;
    }
}

//! [スレッド] 画像ファイル読み込み

void *thread_loadImage(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    AXString *pstr;
    int format,ret = 0;

    pstr   = (AXString *)pdlg->m_pParam1;
    format = *((int *)pdlg->m_pParam2);

    switch(format)
    {
        //APD
        case CMainWin::FILEFORMAT_APD:
            ret = draw::loadAPD(*pstr, pdlg);
            break;
        //ADW
        case CMainWin::FILEFORMAT_ADW:
            ret = draw::loadADW(*pstr, pdlg);
            break;
        //PSD
        case CMainWin::FILEFORMAT_PSD:
            ret = draw::loadPSD(*pstr, pdlg);
            ret = draw::errPSDtoLOAD(ret);
            break;

        //画像ファイル
        /* 透過色がある場合、g_draw->nSaveTPCol にセットされる */
        default:
            ret = draw::loadImage(*pstr, pdlg);
            break;
    }

    pdlg->endThread(ret);

    return NULL;
}

//! ファイルからヘッダを読み取り画像フォーマット判定
/*!
    @return -1 でエラー
*/

int CMainWin::_getFileFormat(const AXString &filename)
{
    BYTE d[8];
    int n;

    //先頭8バイト読み込み

    if(!AXFile::readFile(filename, d, 8)) return -1;

    //ヘッダから判別

    if(d[0] == 'A' && d[1] == 'Z' && d[2] == 'P' &&
       d[3] == 'D' && d[4] == 'A' && d[5] == 'T' && d[6] == 'A')
        //APD
        return FILEFORMAT_APD;
    else if(d[0] == 'A' && d[1] == 'Z' && d[2] == 'D' && d[3] == 'W' &&
            d[4] == 'D' && d[5] == 'A' && d[6] == 'T')
        //ADW
        return FILEFORMAT_ADW;
    else if(d[0] == '8' && d[1] == 'B' && d[2] == 'P' && d[3] == 'S')
        //PSD
        return FILEFORMAT_PSD;
    else
    {
        //その他

        n = AXGetImageFileType(d);

        switch(n)
        {
            case 'B': return FILEFORMAT_BMP;
            case 'P': return FILEFORMAT_PNG;
            case 'J': return FILEFORMAT_JPEG;
            case 'G': return FILEFORMAT_GIF;
            default: return -1;
        }
    }
}


//=============================
// 画像保存
//=============================


//! ファイルに保存
/*!
    @param  bRename 別名で保存
    @param  pstrDir 別名保存時、ディレクトリ指定（NULL で通常）
    @return FALSE でキャンセルされた
*/

BOOL CMainWin::saveFile(BOOL bRename,const AXString *pstrDir)
{
    AXString strName,str;
    int format,ret;
    LPCUSTR puDir;
    CSaveOptDlg *pdlgOpt;
    FILESAVE_PROGINFO info;

    /* strName にファイル名、format に保存形式をセットする */

    //-------- 上書き保存時

    if(!bRename && m_strFileName.isNoEmpty())
    {
        ret = _save_overwrite();

        if(ret == 0)
            //キャンセル
            return FALSE;
        else if(ret == 1)
        {
            //上書き保存

            strName     = m_strFileName;
            format      = m_nFileFormat;
            info.nTPCol = g_draw->nSaveTPCol;
        }
        else if(ret == 2)
            //別名保存
            bRename = TRUE;
    }

    //--------- ファイル名取得
    //（別名保存か新規イメージの場合）

    if(bRename || m_strFileName.isEmpty())
    {
        //フィルタ

        _getSaveFilterString(&str, FILEFORMAT_ALL);

        //ファイル名初期

        if(!m_strFileName.isEmpty())
            strName.path_filenameNoExt(m_strFileName);

        //初期ディレクトリ

        if(pstrDir)
            puDir = *pstrDir;
        else
            puDir = g_conf->strSaveDir;

        //ファイル名・フォーマット取得

        if(!AXFileDialog::saveFile(this, str, 0, puDir, 0, &strName, &format))
            return FALSE;

        //拡張子セット

        _setFormatExt(&strName, format);

        //保存設定ダイアログ

        if(format == FILEFORMAT_PNG || format == FILEFORMAT_JPEG || format == FILEFORMAT_GIF)
        {
            pdlgOpt = new CSaveOptDlg(this, format, &info.nTPCol);

            if(!pdlgOpt->runDialog()) return FALSE;
        }
    }

    //------- 保存

    CProgressDlg *pdlg;

    info.pfilename = &strName;
    info.nFormat   = format;

    pdlg = new CProgressDlg(this, thread_saveFile, &info);

    //

    ret = pdlg->run();

    if(ret)
    {
        //失敗

        _errMes_save(ret);
    }
    else
    {
        //--------- 成功

        m_strFileName = strName;
        m_nFileFormat = format;

        g_draw->nSaveTPCol = info.nTPCol;

        //保存ディレクトリ

        g_conf->strSaveDir.path_removeFileName(strName);

        //履歴

        AXAddRecentString(g_conf->strRecSaveDir, CConfig::RECSAVEDIR_NUM, g_conf->strSaveDir);

        //タイトル

        setWinTitle();

        //最近使ったファイル

        _addRecentFile(strName);

        //UNDOイメージ変更フラグOFF

        draw::setUndoChangeOff();
    }

    return TRUE;
}

//! [スレッド] 画像保存
/*
    [戻り値] 0:成功 1:失敗 2:GIF256色以上
*/

void *thread_saveFile(void *pParam)
{
    CProgressDlg *pdlg = (CProgressDlg *)pParam;
    FILESAVE_PROGINFO *pinfo;
    int ret = 1;

    pinfo = (FILESAVE_PROGINFO *)pdlg->m_pParam1;

    switch(pinfo->nFormat)
    {
        //APD
        case CMainWin::FILEFORMAT_APD:
            ret = draw::saveAPD(*(pinfo->pfilename), pdlg);
            break;
        //PSD
        case CMainWin::FILEFORMAT_PSD:
        case CMainWin::FILEFORMAT_PSD_8BITRGB:
        case CMainWin::FILEFORMAT_PSD_GRAYSCALE:
        case CMainWin::FILEFORMAT_PSD_MONOCHROME:
            draw::blendImage();

            g_draw->pimgBlend->toRGB8bit();

            if(pinfo->nFormat == CMainWin::FILEFORMAT_PSD)
                ret = draw::savePSD(*(pinfo->pfilename), pdlg);
            else
                ret = draw::savePSD_nolayer(*(pinfo->pfilename), pinfo->nFormat - CMainWin::FILEFORMAT_PSD_8BITRGB, pdlg);

            draw::updateImage();
            break;
        //アルファ付きPNG
        case CMainWin::FILEFORMAT_PNG_ALPHA:
            ret = draw::savePNG_alpha(*(pinfo->pfilename), pdlg);
            break;

        //画像ファイル
        default:
            //pimgBlend に合成

            draw::blendImage();

            //RGB8bitに変換

            g_draw->pimgBlend->toRGB8bit();

            //保存

            if(pinfo->nFormat == CMainWin::FILEFORMAT_BMP)
                //BMP
                ret = g_draw->pimgBlend->saveBMP(*(pinfo->pfilename), g_draw->nImgDPI, pdlg);
            else if(pinfo->nFormat == CMainWin::FILEFORMAT_PNG)
                //PNG
                ret = g_draw->pimgBlend->savePNG(*(pinfo->pfilename), g_conf->btPNGLevel, pinfo->nTPCol, g_draw->nImgDPI, pdlg);
            else if(pinfo->nFormat == CMainWin::FILEFORMAT_JPEG)
            {
                //JPEG
                ret = g_draw->pimgBlend->saveJPEG(*(pinfo->pfilename),
                        g_conf->uJPEGSave & 127, (g_conf->uJPEGSave >> 7) & 511, g_draw->nImgDPI, pdlg);
            }
            else
                //GIF
                ret = g_draw->pimgBlend->saveGIF(*(pinfo->pfilename), pinfo->nTPCol, pdlg);

            //pimgBlend 戻す

            draw::updateImage();

            break;
    }

    pdlg->endThread(ret);

    return NULL;
}

//! 上書き保存時のメッセージ表示など
/*!
    @return [0]キャンセル [1]上書き保存 [2]別名保存
*/

BOOL CMainWin::_save_overwrite()
{
    UINT ret;

    //ADW 形式での上書き保存は不可 -> 別名保存

    if(m_nFileFormat == FILEFORMAT_ADW)
        return 2;

    //APD 形式以外の場合、APD で保存するか選択

    if(m_nFileFormat != FILEFORMAT_APD && (g_conf->uFlags & CConfig::FLAG_MES_SAVE_APD))
    {
        ret = AXMessageBox::message(this, NULL, _string(strid::GROUP_MESSAGE, strid::MES_SAVE_NOAPD),
                                 AXMessageBox::YES | AXMessageBox::NO | AXMessageBox::NOTSHOW,
                                 AXMessageBox::YES);

        //このメッセージ表示しない

        if(ret & AXMessageBox::NOTSHOW)
            g_conf->uFlags ^= CConfig::FLAG_MES_SAVE_APD;

        //

        if(ret & AXMessageBox::YES)
            return 2;
        else
            return 1;
    }

    //上書き保存確認メッセージ

    if(g_conf->uFlags & CConfig::FLAG_MES_SAVE_OVERWRITE)
    {
        AXString str;

        str.path_filename(m_strFileName);
        str += '\n';
        str += _string(strid::GROUP_MESSAGE, strid::MES_SAVE_OVERWRITE);

        ret = AXMessageBox::message(this, NULL, str,
                                 AXMessageBox::SAVE | AXMessageBox::CANCEL | AXMessageBox::NOTSHOW,
                                 AXMessageBox::SAVE);

        //

        if(ret & AXMessageBox::NOTSHOW)
            g_conf->uFlags ^= CConfig::FLAG_MES_SAVE_OVERWRITE;

        if(!(ret & AXMessageBox::SAVE))
            return 0;
    }

    return 1;
}

//! 保存ダイアログのフィルタ文字列取得

void CMainWin::_getSaveFilterString(AXString *pstr,int format)
{
    LPCSTR pc[] = {
        "AzPainter (*.apd)\t*.apd\t",
        "PhotoShop (*.psd)\t*.psd\t",
        "BMP (*.bmp)\t*.bmp\t",
        "PNG (*.png)\t*.png\t",
        "JPEG (*.jpg)\t*.jpg\t",
        "GIF (*.gif)\t*.gif\t"
    };

    if(format == FILEFORMAT_ALL)
    {
        pstr->empty();

        for(int i = 0; i < 6; i++)
            *pstr += pc[i];
    }
    else
    {
        if(format == FILEFORMAT_PNG_ALPHA)
            *pstr = pc[3];
        else if(format >= FILEFORMAT_PSD_8BITRGB && format <= FILEFORMAT_PSD_MONOCHROME)
            *pstr = pc[1];
        else
            *pstr = pc[format];
    }
}

//! 拡張子をセット

void CMainWin::_setFormatExt(AXString *pstr,int format)
{
    LPCSTR pc,pcExt[] = { "apd", "psd", "bmp", "png", "jpg", "gif" };

    if(format == FILEFORMAT_PNG_ALPHA)
        pc = pcExt[3];
    else if(format >= FILEFORMAT_PSD_8BITRGB && format <= FILEFORMAT_PSD_MONOCHROME)
        pc = pcExt[1];
    else
        pc = pcExt[format];

    pstr->path_setExt(pc);
}

//! 保存時のエラーメッセージ表示

void CMainWin::_errMes_save(int ret)
{
    errMes(strid::GROUP_MESSAGE, (ret == 2)? strid::MES_SAVE_GIF_OVER: strid::MES_FAILED_SAVE);
}


//===============================
// エクスポート
//===============================


//! 画像エクスポート
/*!
    @param format  保存フォーマット
    @param pstrDir 初期ディレクトリ（NULL で指定なし）
*/

void CMainWin::exportFile(int format,const AXString *pstrDir)
{
    CProgressDlg *pProgDlg;
    CSaveOptDlg *pdlgOpt;
    FILESAVE_PROGINFO info;
    AXString strName,filter;
    LPCUSTR puDir;
    int ret;

    //ファイル名取得

    _getSaveFilterString(&filter, format);

    if(pstrDir)
        puDir = *pstrDir;
    else if(g_conf->strExportDir.isEmpty())
        puDir = g_conf->strSaveDir;
    else
        puDir = g_conf->strExportDir;

    if(!AXFileDialog::saveFile(this, filter, 0, puDir, 0, &strName))
        return;

    _setFormatExt(&strName, format);

    //保存設定ダイアログ

    if(format == FILEFORMAT_PNG || format == FILEFORMAT_JPEG || format == FILEFORMAT_GIF)
    {
        pdlgOpt = new CSaveOptDlg(this, format, &info.nTPCol);

        if(!pdlgOpt->runDialog()) return;
    }

    //保存処理

    info.pfilename = &strName;
    info.nFormat   = format;

    pProgDlg = new CProgressDlg(this, thread_saveFile, &info);

    ret = pProgDlg->run();

    if(ret)
        _errMes_save(ret);
    else
    {
        //ディレクトリ保存

        g_conf->strExportDir.path_removeFileName(strName);

        //履歴保存

        AXAddRecentString(g_conf->strRecExpDir, CConfig::RECEXPDIR_NUM, g_conf->strExportDir);
    }
}


//===============================
// サブ
//===============================


//! 最近使ったファイルに追加

void CMainWin::_addRecentFile(const AXString &filename)
{
    AXAddRecentString(g_conf->strRecFile, CConfig::RECFILE_NUM, filename);

    //メニュー
    m_pmenuRecFile->setStrArray(CMDID_MENU_RECENTFILE, g_conf->strRecFile, CConfig::RECFILE_NUM);
}

//! ツールバーからのディレクトリ履歴メニュー実行（共通処理）

int CMainWin::_runRecentDirMenu(AXString *pstrArray,int cnt,int bttid)
{
    AXMenu *pm;
    AXRect rc;
    int no;

    //履歴が一つもない

    if(pstrArray[0].isEmpty()) return -1;

    //メニュー

    pm = new AXMenu;

    pm->setStrArray(0, pstrArray, cnt);

    m_pToolBar->getItemRect(bttid, &rc, TRUE);
    no = pm->popup(NULL, rc.left, rc.bottom + 1, 0);

    delete pm;

    return no;
}

//! 開く・ディレクトリ履歴メニュー実行（ツールバーから）

void CMainWin::_runRecOpenDirMenu()
{
    int no;

    no = _runRecentDirMenu(g_conf->strRecOpenDir, CConfig::RECDIR_NUM, strid::MENU_FILE_OPEN);

    if(no != -1)
        openFile(g_conf->strRecOpenDir + no);
}

//! 保存ディレクトリ履歴メニュー実行（ツールバーから）

void CMainWin::_runRecSaveDirMenu()
{
    int no;

    no = _runRecentDirMenu(g_conf->strRecSaveDir, CConfig::RECSAVEDIR_NUM, strid::MENU_FILE_SAVERE);

    if(no != -1)
        saveFile(TRUE, g_conf->strRecSaveDir + no);
}

//! 指定形式で保存・ツールバーからのメニュー実行

void CMainWin::_runTBExportMenu()
{
    AXMenu *pm,*pmsub;
    AXRect rc;
    int n;

    //----------

    pm = new AXMenu;

    //保存形式

    pmsub = new AXMenu;

    _trgroup(strid::GROUP_MAINMENU);

    for(n = 0; n < FILEFORMAT_EXPORT_NUM; n++)
        pmsub->add(n, AXMenu::MIF_RADIO, _str(strid::MENU_FILE_FORMAT_APD + n));

    pmsub->check(g_conf->btExportType, TRUE);

    pm->add(100, _string(strid::GROUP_TOOLTIP_MAIN, 100), pmsub);

    //履歴

    pm->addSep();

    for(n = 0; n < CConfig::RECEXPDIR_NUM; n++)
    {
        if(g_conf->strRecExpDir[n].isEmpty()) break;

        pm->add(1000 + n, g_conf->strRecExpDir[n]);
    }

    //

    m_pToolBar->getItemRect(strid::MAINEX_EXPORT, &rc, TRUE);
    n = pm->popup(NULL, rc.left, rc.bottom + 1, 0);

    delete pm;

    //-----------

    if(n == -1) return;

    if(n < FILEFORMAT_EXPORT_NUM)
        //保存形式
        g_conf->btExportType = n;
    else if(n >= 1000)
        //履歴ディレクトリから実行
        exportFile(g_conf->btExportType, &g_conf->strRecExpDir[n - 1000]);
}
