/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CLayerWinArea - レイヤウィンドウの、レイヤ一覧部分
*/


#include <stdlib.h>

#include "CLayerWinArea.h"

#include "CMainWin.h"
#include "CLayerItem.h"
#include "CLayerList.h"
#include "CXImage.h"
#include "CConfig.h"

#include "drawdat.h"
#include "draw_layer.h"

#include "global.h"
#include "strid.h"
#include "cursor.h"

#include "AXScrollBar.h"
#include "AXFont.h"
#include "AXMenu.h"
#include "AXGC.h"
#include "AXAppRes.h"
#include "AXApp.h"
#include "AXUtilStr.h"


//---------------------------

#define COL_BKGND           0xaaaaaa
#define COL_BK_SEL          _RGB(255,194,184)
#define COL_FRAME_SEL       0xff0000
#define COL_CK_DISABLE_FG   0xc0c0c0
#define COL_CK_ENABLE_BK    0x555555

//---------------------------



CLayerWinArea::~CLayerWinArea()
{
    delete m_pimg;
}

CLayerWinArea::CLayerWinArea(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_EXPAND_WH)
{
    m_pimg = new CXImage;

    m_fDrag = 0;
}

//! スクロール情報セット

void CLayerWinArea::setScrollInfo()
{
    int n;

    n = m_nH / ITEM_H;
    if(n < 0) n = 1;

    m_pScrV->setStatus(0, g_draw->player->getViewCnt(), n);
}

//! カレントレイヤが表示される位置にスクロール

void CLayerWinArea::updateShowCur()
{
    CLayerItem *p = g_draw->player->getTopItem();
    int y,scr;

    //カレントレイヤのY位置

    for(y = 0; p; p = p->nextExpand(), y += ITEM_H)
    {
        if(p == g_draw->pcurlayer) break;
    }

    //

    scr = m_pScrV->getPos() * ITEM_H;

    if(y - scr < 0)
        //上
        m_pScrV->setPos(y / ITEM_H);
    else if(y - scr > m_nH - ITEM_H)
        //下
        m_pScrV->setPos((y - m_nH + ITEM_H + ITEM_H - 1) / ITEM_H);

    draw();
}


//===========================
// サブ
//===========================


//! 押し時の処理（ダブルクリックと共通）
/*
    pCmd : チェックなどの部分なら0以外、1でカレント変更なし、2でカレント変更あり
*/

CLayerItem *CLayerWinArea::_onDown(int x,int y,UINT state,BOOL bDblClk,int *pCmd)
{
    CLayerItem *p;
    int ty,i,xx,bCmd = 0,bFolder;

    p = _getItem(y, &ty);
    if(!p) return NULL;

    x -= p->getDepth() * DEPTHW;
    y -= ty;

    bFolder = p->isFolder();

    /* ! チェック等コマンドを実行したら bCmd を 1 or 2 に */

    if(x >= CKSHOW_X && x < CKSHOW_X + BTTSIZE && y >= CKSHOW_Y && y < CKSHOW_Y + BTTSIZE)
    {
        //表示/非表示

        draw::layer_revShow(p);
        _drawOne(p, ty, TRUE);

        bCmd = 1;
    }
    if(bFolder && x >= 19 && x <= 41 && y >= 3 && y <= 15)
    {
        //フォルダ展開/閉じる（カレントレイヤ変更する）

        draw::layer_revFolderExpand(p);

        bCmd = 2;
    }
    else if(!bFolder && x >= COLAREA_X && x < COLAREA_X + BTTSIZE && y >= CKUNDER_Y && y < CKUNDER_Y + BTTSIZE)
    {
        //線の色 -> 描画色セット

        if(p->isColTypeOnlyA())
        {
            draw::layer_changeCol(p, g_conf->dwDrawCol);
            bCmd = 1;
        }
    }
    else if(x >= CKUNDER_X && x < CKUNDER_X + (BTTSIZE - 1) * 4 + 1 &&
            y >= CKUNDER_Y && y < CKUNDER_Y + BTTSIZE)
    {
        //-------- 下段チェック

        for(i = 0, xx = CKUNDER_X; i < 4; i++, xx += BTTSIZE - 1)
        {
            if(x >= xx && x < xx + BTTSIZE)
            {
                //フォルダはリンクのみ

                if(bFolder && i) break;

                //

                if(i == 0)
                    draw::layer_revLinkFlag(p);
                else if(i == 1)
                    draw::layer_revPaintFlag(p);
                else if(i == 2)
                    draw::layer_changeLayerMask(p, ((state & STATE_SHIFT) || bDblClk));
                else
                    draw::layer_revCheckFlag(p);

                bCmd = 1;
                break;
            }
        }
    }

    //

    *pCmd = bCmd;

    return p;
}

//! Y位置からアイテム取得

CLayerItem *CLayerWinArea::_getItem(int y,int *pTopY)
{
    CLayerItem *p = g_draw->player->getTopItem();
    int topy;

    topy = m_pScrV->getPos() * -ITEM_H;

    for(; p; p = p->nextExpand(), topy += ITEM_H)
    {
        if(topy <= y && y < topy + ITEM_H)
        {
            if(pTopY) *pTopY = topy;
            return p;
        }
    }

    return NULL;
}

//! アイテムの表示Y位置取得
/*!
    @return マイナスで表示範囲外
*/

int CLayerWinArea::_getItemTopY(CLayerItem *pItem)
{
    CLayerItem *p = g_draw->player->getTopItem();
    int topy;

    topy = m_pScrV->getPos() * -ITEM_H;

    for(; p; p = p->nextExpand(), topy += ITEM_H)
    {
        if(topy >= m_nH) return -1;

        if(p == pItem)
            return topy;
    }

    return -1;
}

//! 右クリックメニュー

void CLayerWinArea::_runRMenu(CLayerItem *pItem,int x,int y)
{
    AXMenu *pmenu;
    int id;
    BOOL bImg;

    bImg = !pItem->isFolder();

    _trgroup(strid::GROUP_LAYERRMENU);

    //

    pmenu = new AXMenu;

    pmenu->addTr(strid::LAYERRMENU_RENAME);
    if(bImg) pmenu->addTr(strid::LAYERRMENU_CHANGECOLTYPE);
    if(bImg && pItem->isColTypeOnlyA()) pmenu->addTr(strid::LAYERRMENU_SETCOL);
    pmenu->addSep();
    pmenu->addTrCheck(strid::LAYERRMENU_LOCK, pItem->isLock());
    pmenu->addTrCheck(strid::LAYERRMENU_LINK, pItem->isLink());
    if(bImg)
    {
        pmenu->addTrCheck(strid::LAYERRMENU_PAINT, pItem->isPaint());
        pmenu->addTrCheck(strid::LAYERRMENU_LAYERMASK, pItem->isLayerMask());
        pmenu->addTrCheck(strid::LAYERRMENU_LAYERMASKUNDER, pItem->isLayerMaskUnder());
        pmenu->addTrCheck(strid::LAYERRMENU_CHECK, pItem->isCheck());
    }

    id = pmenu->popup(NULL, x, y, 0);

    delete pmenu;

    //

    if(id == -1) return;

    switch(id)
    {
        case strid::LAYERRMENU_RENAME:
            MAINWIN->_layerRename(pItem);
            break;
        case strid::LAYERRMENU_CHANGECOLTYPE:
            MAINWIN->_layerChangeColType(pItem);
            break;
        case strid::LAYERRMENU_SETCOL:
            draw::layer_changeCol(pItem, g_conf->dwDrawCol);
            break;
        case strid::LAYERRMENU_LOCK:
            draw::layer_revLockFlag(pItem);
            break;
        case strid::LAYERRMENU_LINK:
            draw::layer_revLinkFlag(pItem);
            break;
        case strid::LAYERRMENU_PAINT:
            draw::layer_revPaintFlag(pItem);
            break;
        case strid::LAYERRMENU_LAYERMASK:
            draw::layer_changeLayerMask(pItem, 2);
            break;
        case strid::LAYERRMENU_LAYERMASKUNDER:
            draw::layer_changeLayerMask(pItem, 1);
            break;
        case strid::LAYERRMENU_CHECK:
            draw::layer_revCheckFlag(pItem);
            break;
    }
}


//===========================
// ハンドラ
//===========================


//! サイズ変更時

BOOL CLayerWinArea::onSize()
{
    m_pimg->recreate((m_nW < MINWIDTH)? MINWIDTH: m_nW, m_nH, 16, ITEM_H);

    setScrollInfo();
    draw();

    return TRUE;
}

//! 描画

BOOL CLayerWinArea::onPaint(AXHD_PAINT *phd)
{
    m_pimg->put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);
    return TRUE;
}

//! マウスホイール（スクロール）

BOOL CLayerWinArea::onMouseWheel(AXHD_MOUSE *phd,BOOL bUp)
{
    if(m_pScrV->movePos((bUp)? -1: 1))
        draw();

    return TRUE;
}

//! ダブルクリック時

BOOL CLayerWinArea::onDblClk(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && !m_fDrag)
    {
        CLayerItem *p;
        int cmd;

        //チェックなど処理

        p = _onDown(phd->x, phd->y, phd->state, TRUE, &cmd);

        //レイヤ名前

        if(p && !cmd)
            MAINWIN->sendCommand(strid::MENU_LAYER_RENAME, 0, 0);
    }

    return TRUE;
}

//! ボタン押し

BOOL CLayerWinArea::onButtonDown(AXHD_MOUSE *phd)
{
    CLayerItem *p;

    if(m_fDrag) return TRUE;

    if(phd->button == BUTTON_RIGHT)
    {
        //------- 右ボタン

        p = _getItem(phd->y);

        if(p)
            _runRMenu(p, phd->rootx, phd->rooty);
    }
    else if(phd->button == BUTTON_LEFT)
    {
        //------- 左ボタン

        int cmd;

        //各チェックなど処理（レイヤ外 or 処理済なら終了）

        p = _onDown(phd->x, phd->y, phd->state, FALSE, &cmd);

        if(!p || cmd == 1) return TRUE;

        //カレントレイヤ変更

        draw::layer_changeCurrent(p);

        //レイヤ移動のD&D判定開始

        if(cmd == 0)
        {
            m_fDrag  = DRAGF_BEGINMOVE;
            m_nDragY = phd->y;

            grabPointer();
        }
    }

    return TRUE;
}

//! ボタン離し時

BOOL CLayerWinArea::onButtonUp(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && m_fDrag)
    {
        ungrabPointer();

        if(m_fDrag == DRAGF_MOVE)
        {
            //D&D、終了

            delTimer(0);

            unsetCursor();

            _drawDND(TRUE);

            //レイヤ移動

            if(m_nDragDstType >= 0)
                draw::layer_moveDND(m_pDragDst, m_nDragDstType);
        }

        m_fDrag = 0;
    }

    return TRUE;
}

//! マウス移動時

BOOL CLayerWinArea::onMouseMove(AXHD_MOUSE *phd)
{
    CLayerItem *p;
    int y,type;

    if(!m_fDrag) return TRUE;

    /*
        m_pDragDst : ドラッグ先。NULLで一番下
        m_nDragDstType: [-1]ドラッグ先なし [-2]範囲外タイマー中 [0]挿入 [1]フォルダ
    */

    //ドラッグ開始判定中

    if(m_fDrag == DRAGF_BEGINMOVE)
    {
        if(::abs(phd->y - m_nDragY) > ITEM_H / 2)
        {
            setCursor(cursor::getImgDat(cursor::ITEMMOVE));

            m_fDrag = DRAGF_MOVE;
            m_pDragDst     = NULL;
            m_nDragDstType = -1;
        }
    }

    //ドラッグ中

    if(m_fDrag == DRAGF_MOVE)
    {
        //一覧外に来たらスクロールタイマ起動

        if(phd->y < 0 || phd->y >= m_nH)
        {
            if(m_nDragDstType != -2)
            {
                addTimer(0, 200, (phd->y >= m_nH));

                _drawDND(TRUE);
                m_nDragDstType = -2;
            }

            return TRUE;
        }

        if(m_nDragDstType == -2)
            delTimer(0);

        //

        p = _getItem(phd->y, &y);

        type = 0;

        if(p == g_draw->pcurlayer)
            type = -1;
        else if(!p)
            //一番下
            y = phd->y / ITEM_H * ITEM_H;
        else if(p->isChild(g_draw->pcurlayer))
            //フォルダをそのフォルダ下へドラッグする場合
            type = -1;
        else if(p->isFolder())
            type = (phd->y - y >= 8);

        //

        if(p != m_pDragDst || type != m_nDragDstType)
        {
            _drawDND(TRUE);

            m_pDragDst = p;
            m_nDragY   = y;
            m_nDragDstType = type;

            _drawDND(FALSE);
        }
    }

    return TRUE;
}

//! タイマ

BOOL CLayerWinArea::onTimer(UINT uTimerID,ULONG lParam)
{
    if(m_pScrV->movePos((lParam)? 1: -1))
        draw();

    return TRUE;
}


//===========================
// 描画
//===========================


//! 全体描画

void CLayerWinArea::draw()
{
    CLayerItem *p;
    int y;

    m_pimg->clear(COL_BKGND);

    p = g_draw->player->getTopItem();
    y = m_pScrV->getPos() * -ITEM_H;

    for(; p && y < m_nH; p = p->nextExpand(), y += ITEM_H)
    {
        if(y >= 0)
            _drawOne(p, y, FALSE);
    }

    redraw();
}

//! 指定アイテムのみ描画

void CLayerWinArea::draw(CLayerItem *p)
{
    int y = _getItemTopY(p);

    if(y >= 0) _drawOne(p, y, TRUE);
}

//! アイテム一つ描画

void CLayerWinArea::_drawOne(CLayerItem *p,int ytop,BOOL bRedraw)
{
    int bSel,bFolder,w,x,y,n,i,xtop;
    DWORD f;
    char m[10];
    LPBYTE pPat;
    DWORD ckflag[4] = {
        CLayerItem::FLAG_LINK, CLayerItem::FLAG_PAINT,
        CLayerItem::FLAG_LAYERMASK|CLayerItem::FLAG_LAYERMASKUNDER, CLayerItem::FLAG_CHECK
    };
    BYTE patCk[5][18] = {
        {0x55,0x00, 0x55,0x00, 0x5d,0x00, 0x53,0x00, 0x49,0x00, 0xa4,0x80, 0xba,0x80, 0xa2,0x80, 0xa2,0x80}, //リンク
        {0x08,0x00, 0x14,0x00, 0x22,0x00, 0x47,0x00, 0x6f,0x80, 0x5f,0x00, 0x4e,0x00, 0x44,0x00, 0xc0,0x00}, //塗りつぶし判定元
        {0xff,0x80, 0xff,0x80, 0xe3,0x80, 0xc1,0x80, 0xc1,0x80, 0xc1,0x80, 0xe3,0x80, 0xff,0x80, 0xff,0x80}, //マスク
        {0x00,0x00, 0x01,0x80, 0x03,0x00, 0x06,0x00, 0xcc,0x00, 0x7c,0x00, 0x38,0x00, 0x18,0x00, 0x00,0x00}, //チェック
        {0xff,0x00, 0xff,0x00, 0xe7,0x00, 0xc3,0x00, 0xc3,0x00, 0xe7,0x00, 0xf0,0x00, 0xf7,0x80, 0x03,0x00}  //マスク・下レイヤ
    };
    BYTE patLock[20] = { 0x00,0x00, 0x0e,0x00, 0x11,0x00, 0x11,0x00, 0x3f,0x80, 0x3f,0x80, 0x3b,0x80, 0x3b,0x80, 0x3f,0x80, 0x00,0x00 };
    BYTE pat_arrowR[9] = { 0x80,0xc0,0xe0,0xf0,0xf8,0xf0,0xe0,0xc0,0x80 };
    BYTE pat_arrowD[10] = { 0xff,0x80, 0x7f,0x00, 0x3e,0x00, 0x1c,0x00, 0x08,0x00 };
    BYTE pat_foler[2] = { 0x78, 0x84 };
    BYTE pat_16[6] = { 0x00, 0x5c, 0x50, 0x5c, 0x54, 0x5c };
    BYTE pat_1[6] = { 0x00, 0x60, 0x20, 0x20, 0x20, 0x70 };
    const char *szBlend[draw::BLENDMODE_NUM] = {
        "NORM", "MUL", "ADD", "SUB", "SCRN", "OVERL", "HARDL", "SOFTL",
        "DODGE", "BURN", "LBURN", "VIVID", "LINRL", "PINL",
        "DARK", "LIGHT", "DIFF"
    };

    //---------

    w    = (m_nW < MINWIDTH)? MINWIDTH: m_nW;
    xtop = p->getDepth() * DEPTHW;

    bSel    = (p == g_draw->pcurlayer);
    bFolder = p->isFolder();

    //背景・下線

    m_pimg->fillBox(xtop, ytop, w - xtop, ITEM_H - 1, (bSel)? COL_BK_SEL: 0xffffff);
    m_pimg->lineH(0, ytop + ITEM_H - 1, w, 0);

    //選択枠

    if(bSel)
        m_pimg->box(xtop, ytop, w - xtop, ITEM_H - 1, COL_FRAME_SEL);

    //-------- 上段

    //表示/非表示チェック

    x = xtop + CKSHOW_X, y = ytop + CKSHOW_Y;

    m_pimg->box(x, y, BTTSIZE, BTTSIZE, 0);
    m_pimg->fillBox(x + 1, y + 1, BTTSIZE - 2, BTTSIZE - 2, 0xffffff);

    if(p->isVisibleFlag())
        m_pimg->fillBox(x + 2, y + 2, BTTSIZE - 4, BTTSIZE - 4, 0x002080);
    else
    {
        m_pimg->line(x + 1, y + 1, x + BTTSIZE - 2, y + BTTSIZE - 2, 0);
        m_pimg->line(x + 1, y + BTTSIZE - 2, x + BTTSIZE - 2, y + 1, 0);
    }

    //フォルダ矢印、アイコン

    if(bFolder)
    {
        //矢印

        if(p->isExpand())
            m_pimg->drawPattern(xtop + 19, ytop + 9, _RGB(0,96,0), pat_arrowD, 9, 5);
        else
            m_pimg->drawPattern(xtop + 23, ytop + 5, _RGB(0,96,0), pat_arrowR, 5, 9);

        //アイコン

        m_pimg->drawPattern(xtop + 31, ytop + 5, 0, pat_foler, 6, 2);
        m_pimg->box(xtop + 31, ytop + 7, 11, 8, 0);
    }

    //名前

    x = xtop + ((bFolder)? 46: 21);

    m_pimg->drawText(*m_pFont, x, ytop + 3, w - x - 3, m_pFont->getHeight(),
                     p->m_strName, p->m_strName.getLen(), (bSel)? _RGB(200,0,0): 0);

    //---------- 下段

    y = ytop + CKUNDER_Y;

    //線の色/カラータイプ

    if(!bFolder)
    {
        n = p->m_nColType;
        x = xtop + COLAREA_X;

        if(n == 1)
            m_pimg->drawAlphabet(x + 1, y + BTTSIZE - 8, 0x666666, "GR");
        else if(n >= 2)
        {
            //A16/A1

            m_pimg->box(x, y, BTTSIZE, BTTSIZE, 0);
            m_pimg->fillBox(x + 1, y + 1, BTTSIZE - 2, BTTSIZE - 2, p->m_dwCol);

            if(n == 2)
                m_pimg->drawPattern(x + BTTSIZE - 6, y + BTTSIZE - 6, 0, 0xffffff, pat_16, 6, 6);
            else
                m_pimg->drawPattern(x + BTTSIZE - 4, y + BTTSIZE - 6, 0, 0xffffff, pat_1, 4, 6);
        }
    }

    //各チェック枠・背景

    x = xtop + CKUNDER_X;
    n = (bFolder)? 1: 4;

    m_pimg->box(x, y, (BTTSIZE - 1) * n + 1, BTTSIZE, 0x808080);

    if(bSel)
        m_pimg->fillBox(x + 1, y + 1, (BTTSIZE - 1) * n + 1 - 2, BTTSIZE - 2, 0xffffff);

    for(i = 0; i < n - 1; i++)
        m_pimg->lineV(x + BTTSIZE - 1 + i * (BTTSIZE - 1), y + 1, BTTSIZE - 2, 0x808080);

    //各チェック

    x = xtop + CKUNDER_X + 1;

    for(i = 0; i < 4; i++, x += BTTSIZE - 1)
    {
        //フォルダはリンクのみ
        if(bFolder && i != 0) continue;

        f = p->m_dwFlags & ckflag[i];

        //背景

        if(f) m_pimg->fillBox(x, y + 1, BTTSIZE - 2, BTTSIZE - 2, COL_CK_ENABLE_BK);

        //柄

        if(i == 2 && p->isLayerMaskUnder())
            pPat = patCk[4];
        else
            pPat = patCk[i];

        m_pimg->drawPattern(x + 1, y + 2, (f)? 0xffffff: COL_CK_DISABLE_FG, pPat, 9, 9);
    }

    //-----------

    //濃度

    n = AXIntToStr(m, (int)(p->m_nOpacity * 100.0 / 128.0 + 0.5));

    m_pimg->drawNumber(xtop + 87 - n * 5, ytop + 23, m, 0x000088);

    //合成モード

    if(!bFolder)
        m_pimg->drawAlphabet(xtop + 92, ytop + 23, _RGB(42,0,180), szBlend[p->m_nBlendMode]);

    //ロック

    if(p->isLock())
    {
        x = m_nW - 14;
        if(x - xtop < 123) x = xtop + 123;

        m_pimg->drawPattern(x, ytop + 22, 0xffffff, _RGB(142,142,35), patLock, 11, 10);
    }

    //更新

    if(bRedraw)
        redraw(0, ytop, m_nW, ytop + ITEM_H);
}

//! アイテムD＆D時の線描画

void CLayerWinArea::_drawDND(BOOL bErase)
{
    DWORD col;
    int x;

    if(m_nDragDstType < 0) return;

    //色

    if(bErase)
        col = (m_pDragDst)? 0xffffff: COL_BKGND;
    else
        col = _RGB(60,200,255);

    //X位置

    if(!m_pDragDst)
        x = 0;
    else
        x = m_pDragDst->getDepth() * DEPTHW;

    //描画

    if(m_nDragDstType == 1)
    {
        //フォルダ
        m_pimg->box(x, m_nDragY, m_nW - x, ITEM_H - 1, col);
        m_pimg->box(x + 1, m_nDragY + 1, m_nW - x - 2, ITEM_H - 3, col);
    }
    else
        m_pimg->fillBox(x, m_nDragY, m_nW - x, 2, col);

    redraw();
}
