/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタダイアログの特殊ウィジェット
*/


#include "CFilterWidget.h"

#include "CTileImage.h"
#include "struct.h"

#include "AXApp.h"
#include "AXAppRes.h"



//***************************************
// CFilterPrev - ダイアログ内プレビュー
//***************************************


CFilterPrev::CFilterPrev(AXWindow *pParent,UINT uLayoutFlags,UINT uID,DWORD dwPadding,
                         int w,int h,CTileImage *pimgSrc)
    : AXWindow(pParent, 0, uLayoutFlags, uID, dwPadding)
{
    m_pimgSrc = pimgSrc;

    //キャンバスサイズ

    m_nCanvasW = CTileImage::m_pinfo->nImgW;
    m_nCanvasH = CTileImage::m_pinfo->nImgH;

    //プレビューサイズがキャンバスサイズより大きい場合

    if(w - 2 > m_nCanvasW) w = m_nCanvasW + 2;
    if(h - 2 > m_nCanvasH) h = m_nCanvasH + 2;

    //

    m_nMinW = w;
    m_nMinH = h;

    //スクロールありか

    m_bScroll = (w - 2 < m_nCanvasW || h - 2 < m_nCanvasH);

    //ソース画像の表示範囲

    m_rcsSrc.set(0, 0, w - 2, h - 2);

    //キャンバスのフレーム枠（プレビューサイズ内に収める）

    m_rcsFrame.set(0, 0, m_nCanvasW, m_nCanvasH);
    m_rcsFrame.inBoxKeepAspect(w - 2, h - 2, TRUE);

    //キャンバスフレーム枠に対する、プレビュー部分サイズ

    m_szAreaBox.w = (int)((double)m_rcsSrc.w * m_rcsFrame.w / m_nCanvasW + 0.5);
    m_szAreaBox.h = (int)((double)m_rcsSrc.h * m_rcsFrame.h / m_nCanvasH + 0.5);

    //イメージ

    if(m_img.create(w, h))
        m_img.box(0, 0, w, h, 0x000080);
}

//! 描画範囲取得

void CFilterPrev::getDrawRect(RECTANDSIZE *prs)
{
    prs->x1 = m_rcsSrc.x;
    prs->y1 = m_rcsSrc.y;
    prs->x2 = m_rcsSrc.x + m_rcsSrc.w - 1;
    prs->y2 = m_rcsSrc.y + m_rcsSrc.h - 1;
    prs->w  = m_rcsSrc.w;
    prs->h  = m_rcsSrc.h;
}

//! イメージ描画

void CFilterPrev::drawImg(CTileImage *pimg)
{
    pimg->drawFilterPrev(&m_img, m_rcsSrc);
    redraw();
}

//! ドラッグ時、m_rcsSrc の位置セット

BOOL CFilterPrev::_setDragPos(int x,int y)
{
    x = x - 1 - m_rcsFrame.x - m_szAreaBox.w / 2;
    y = y - 1 - m_rcsFrame.y - m_szAreaBox.h / 2;

    x = (int)((double)x * m_nCanvasW / m_rcsFrame.w + 0.5);
    y = (int)((double)y * m_nCanvasH / m_rcsFrame.h + 0.5);

    if(x < 0) x = 0;
    else if(x > m_nCanvasW - m_rcsSrc.w) x = m_nCanvasW - m_rcsSrc.w;

    if(y < 0) y = 0;
    else if(y > m_nCanvasH - m_rcsSrc.h) y = m_nCanvasH - m_rcsSrc.h;

    if(x != m_rcsSrc.x || y != m_rcsSrc.y)
    {
        m_rcsSrc.x = x;
        m_rcsSrc.y = y;
        return TRUE;
    }
    else
        return FALSE;
}

//! ドラッグ中イメージ描画

void CFilterPrev::_drawImgDrag()
{
    int x,y;

    //イメージ

    m_pimgSrc->drawFilterPrev(&m_img, m_rcsSrc);

    //キャンバス枠

    m_img.box(m_rcsFrame.x + 1, m_rcsFrame.y + 1, m_rcsFrame.w, m_rcsFrame.h, 0xff0000);

    //プレビュー部分枠

    x = (int)((double)m_rcsSrc.x * m_rcsFrame.w / m_nCanvasW + m_rcsFrame.x + 0.5);
    y = (int)((double)m_rcsSrc.y * m_rcsFrame.h / m_nCanvasH + m_rcsFrame.y + 0.5);

    if(x + m_szAreaBox.w > m_rcsFrame.x + m_rcsFrame.w)
        x = m_rcsFrame.x + m_rcsFrame.w - m_szAreaBox.w;

    if(y + m_szAreaBox.h > m_rcsFrame.y + m_rcsFrame.h)
        y = m_rcsFrame.y + m_rcsFrame.h - m_szAreaBox.h;

    m_img.drawDashBox(x + 1, y + 1, m_szAreaBox.w, m_szAreaBox.h);

    //

    m_img.put(m_id);
}


//============================
// ハンドラ
//============================


//! 描画

BOOL CFilterPrev::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//! ボタン押し時

BOOL CFilterPrev::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && !(m_uFlags & FLAG_TEMP1) && m_bScroll)
    {
        m_uFlags |= FLAG_TEMP1;
        grabPointer();

        _setDragPos(phd->x, phd->y);
        _drawImgDrag();
    }

    return TRUE;
}

//! ボタン離し時

BOOL CFilterPrev::onButtonUp(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && (m_uFlags & FLAG_TEMP1))
    {
        m_uFlags &= ~FLAG_TEMP1;
        ungrabPointer();

        getNotify()->onNotify(this, NOTIFY_UP, 0);
    }

    return TRUE;
}

//! マウス移動時

BOOL CFilterPrev::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_uFlags & FLAG_TEMP1)
    {
        if(_setDragPos(phd->x, phd->y))
            _drawImgDrag();
    }

    return TRUE;
}



//************************************
// CFilterWidgetLevel - レベル補正
//************************************

#define SPACE_CURSOR    3
#define CURSOR_H        8
#define HISTOGRAM_H     150
#define OUTGRAD_H       12
#define SPACE_MIDDLE    8

#define DRAGF_IN    1
#define DRAGF_OUT   2


CFilterWidgetLevel::CFilterWidgetLevel(AXWindow *pParent,UINT uID,DWORD dwPadding)
    : AXWindow(pParent, 0, 0, uID, dwPadding)
{
    m_nMinW = 257 + SPACE_CURSOR * 2;
    m_nMinH = HISTOGRAM_H + CURSOR_H * 2 + SPACE_MIDDLE + OUTGRAD_H;

    m_fDrag = 0;

    m_img.create(m_nMinW, m_nMinH);

    m_nVal[0] = 0;
    m_nVal[1] = 128;
    m_nVal[2] = 256;
    m_nVal[3] = 0;
    m_nVal[4] = 256;
}

//! 値取得

void CFilterWidgetLevel::getVal(int *pVal)
{
    int i;

    for(i = 0; i < 5; i++)
        pVal[i] = m_nVal[i] << 7;
}

//! 全体イメージ描画

void CFilterWidgetLevel::drawAll(LPDWORD pHistogram)
{
    int i,n;
    DWORD peek = 0,peekBk = 0,val;

    m_img.clear(axres->colRGB(AXAppRes::FACE));

    //--------- ヒストグラム

    //ピーク値（2番目に大きい値の 1.2 倍）

    for(i = 0; i <= 256; i++)
    {
        val = pHistogram[i];

        if(val > peek)
        {
            peekBk = peek;
            peek   = val;
        }

        if(val > peekBk && val < peek)
            peekBk = val;
    }

    peekBk = (int)(peekBk * 1.2);
    if(peekBk != 0) peek = peekBk;

    //白背景

    m_img.fillBox(SPACE_CURSOR, 0, 257, HISTOGRAM_H, 0xffffff);

    //中央線

    m_img.lineV(SPACE_CURSOR + 128, 0, HISTOGRAM_H, 0xdddddd);

    //線

    for(i = 0; i <= 256; i++)
    {
        if(pHistogram[i])
        {
            n = (int)((double)pHistogram[i] / peek * HISTOGRAM_H);
            if(n > HISTOGRAM_H) n = HISTOGRAM_H;

            if(n) m_img.lineV(SPACE_CURSOR + i, HISTOGRAM_H - n, n, 0);
        }
    }

    //-------- 出力グラデーション

    for(i = 0; i <= 256; i++)
    {
        n = (i == 256)? 255: i;
        m_img.lineV(SPACE_CURSOR + i, HISTOGRAM_H + CURSOR_H + SPACE_MIDDLE, OUTGRAD_H, _RGB(n, n, n));
    }

    //-------- カーソル

    _drawCursor();
}

//! カーソル描画

void CFilterWidgetLevel::_drawCursor()
{
    int i,x,y;

    //消去

    m_img.fillBox(0, HISTOGRAM_H, m_nMinW, CURSOR_H, axres->colRGB(AXAppRes::FACE));
    m_img.fillBox(0, HISTOGRAM_H + CURSOR_H + SPACE_MIDDLE + OUTGRAD_H, m_nMinW, CURSOR_H, axres->colRGB(AXAppRes::FACE));

    //各カーソル

    y = HISTOGRAM_H;

	for(i = 0; i < 5; i++)
	{
		x = m_nVal[i] + SPACE_CURSOR;

		if(i == 3)
            y = HISTOGRAM_H + CURSOR_H + SPACE_MIDDLE + OUTGRAD_H;

		//

		m_img.line(x, y, x - 3, y + 7, 0);
		m_img.line(x, y, x + 3, y + 7, 0);
		m_img.line(x - 3, y + 7, x + 3, y + 7, 0);
	}
}

//! クリック/移動時

void CFilterWidgetLevel::_changePos(int x,int y)
{
    int top,cnt,pos,i,len[3],n,curno;

    if(m_fDrag == DRAGF_IN)
        top = 0, cnt = 3;
    else
        top = 3, cnt = 2;

    //

    pos = x - SPACE_CURSOR;
    if(pos < 0) pos = 0; else if(pos > 256) pos = 256;

    //各カーソルとの距離

    for(i = 0; i < cnt; i++)
    {
        n = pos - m_nVal[top + i];
        if(n < 0) n = -n;

        len[i] = n;
    }

    //距離の近いカーソル

    for(i = 0, n = 257, curno = top; i < cnt; i++)
    {
        if(len[i] < n)
        {
            n     = len[i];
            curno = top + i;
        }
    }

    //

    m_nVal[curno] = pos;

    //カーソル描画

    _drawCursor();

    redrawUpdate();
}


//==============================
//ハンドラ
//==============================


//! 描画

BOOL CFilterWidgetLevel::onPaint(AXHD_PAINT *phd)
{
    m_img.put(m_id);
    return TRUE;
}

//! ボタン押し時

BOOL CFilterWidgetLevel::onButtonDown(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && !m_fDrag)
    {
        //入力 or 出力

        if(phd->y >= HISTOGRAM_H && phd->y < HISTOGRAM_H + CURSOR_H)
            m_fDrag = DRAGF_IN;
        else if(phd->y >= HISTOGRAM_H + CURSOR_H + SPACE_MIDDLE + OUTGRAD_H && phd->y < m_nMinH)
            m_fDrag = DRAGF_OUT;
        else
            return TRUE;

        //

        grabPointer();

        _changePos(phd->x, phd->y);
    }

    return TRUE;
}

//! ボタン離し時

BOOL CFilterWidgetLevel::onButtonUp(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT && m_fDrag)
    {
        m_fDrag = 0;
        ungrabPointer();

        getNotify()->onNotify(this, 0, 0);
    }

    return TRUE;
}

//! マウス移動時

BOOL CFilterWidgetLevel::onMouseMove(AXHD_MOUSE *phd)
{
    if(m_fDrag)
        _changePos(phd->x, phd->y);

    return TRUE;
}

