/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CColorWin - カラーウィンドウ
*/


#include "CColorWin.h"

#include "subwidget_color.h"

#include "CConfig.h"
#include "CResource.h"
#include "CPalette.h"
#include "CValBar.h"
#include "CValBarSimple.h"
#include "CApp.h"

#include "global.h"
#include "strid.h"
#include "draw_main.h"

#include "AXLayout.h"
#include "AXTab.h"
#include "AXLabel.h"
#include "AXLineEdit.h"
#include "AXCheckButton.h"
#include "AXHSVPicker.h"
#include "AXArrowButton.h"
#include "AXScrollBar.h"
#include "AXMenu.h"
#include "AXStrDialog.h"
#include "AXMessageBox.h"
#include "AXFileDialog.h"
#include "AXUtilColor.h"
#include "AXUtilStr.h"

#include "CColorWin_tab.h"




CColorWin *CColorWin::m_pSelf = NULL;


CColorWin::CColorWin(AXWindow *pOwner,UINT addstyle)
    : AXTopWindow(pOwner,
                  WS_HIDE | WS_TITLE | WS_CLOSE | WS_BORDER | WS_BK_FACE | WS_TABMOVE | addstyle)
{
    AXLayout *plTop,*pl,*pl2,*pl3;
    AXString str;
    const char *tabname[5] = {"BAR","HSV","HLS","GRAD","PAL"};
    int i;

    m_pSelf = this;

    m_pwinTab = NULL;

    AXMemZero(m_nPalScrPos, sizeof(m_nPalScrPos));

    //

    setTitle("color");

    if(g_res->pfontSubWin)
        setFont(g_res->pfontSubWin);

    attachAccel(((AXTopWindow *)m_pOwner)->getAccel());

    //------------

    setLayout(plTop = new AXLayoutVert(LF_EXPAND_WH, 0));

    //描画色

    plTop->addItem(pl = new AXLayoutHorz(0, 12));

    pl->setPadding(6, 6, 6, 8);

    pl->addItem(m_pDrawCol = new CColSub_drawcol(this, 0));

    //色マスク

    pl->addItem(pl2 = new AXLayoutVert(0, 4));

    pl2->addItem(pl3 = new AXLayoutHorz(LF_EXPAND_W, 6));

    pl3->addItem(new AXLabel(this, 0, LF_EXPAND_X|LF_CENTER_Y, 0, _string(strid::GROUP_COLORWIN, strid::COLWIN_COLMASK)));
    pl3->addItem(m_pColMaskType = new CColSub_colmasktype(this));

    pl2->addItem(m_pColMask = new CColSub_colmask(this));

    //タブ

    plTop->addItem(m_ptab = new AXTab(this, AXTab::TABS_TOPTAB, LF_EXPAND_W));

    for(i = 0; i < 5; i++)
    {
        str = tabname[i];
        m_ptab->addItem(str);
    }

    m_ptab->setCurSel(g_conf->btColWinTab);

    //タブ中身

    _setTabWidget(g_conf->btColWinTab, FALSE);
}

//! 表示/非表示 切り替え

void CColorWin::showToggle()
{
    if(g_conf->isColorWin())
        showRestore();
    else
        hide();
}

//! 描画色から各値セット

void CColorWin::setDrawColor()
{
    m_pDrawCol->paintOnlyDrawCol();

    m_pwinTab->setColor(g_conf->dwDrawCol);
}

//! マスク色更新

void CColorWin::updateMaskColor()
{
    m_pColMask->draw();
}

//! 描画色/背景色更新

void CColorWin::updateDrawColor()
{
    m_pDrawCol->redraw();

    m_pwinTab->setColor(g_conf->dwDrawCol);
}

//! 閉じる（非表示）

BOOL CColorWin::onClose()
{
    m_pOwner->onCommand(strid::MENU_VIEW_COLORWIN, 0, 0);

    return TRUE;
}

//! 通知

BOOL CColorWin::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pwinTab || pwin == m_pColMask)
    {
        //描画色が変更された [lParam = 色]

        draw::changeDrawCol((DWORD)lParam);

        m_pDrawCol->paintOnlyDrawCol();
    }
    else if(pwin == m_ptab)
    {
        //タブ選択変更

        if(uNotify == AXTab::TABN_SELCHANGE)
        {
            g_conf->btColWinTab = m_ptab->getCurSel();

            _setTabWidget(g_conf->btColWinTab);
        }
    }
    else if(pwin == m_pDrawCol)
    {
        //描画色/背景色入れ替え時 -> 各タブの値変更

        if(m_pwinTab)
            m_pwinTab->setColor(g_conf->dwDrawCol);
    }

    return TRUE;
}

//! タブウィジェットセット

void CColorWin::_setTabWidget(int no,BOOL bLayout)
{
    //現在のものを取り外す

    if(m_pwinTab)
    {
        //取り外す前の処理
        m_pwinTab->beforeRemoveTab();

        m_pwinTab->removeFromLayout();
        delete m_pwinTab;

        m_pwinTab = NULL;
    }

    //作成

    switch(no)
    {
        case 0: m_pwinTab = new CColWin_bar(this); break;
        case 1: m_pwinTab = new CColWin_hsv(this); break;
        case 2: m_pwinTab = new CColWin_hls(this); break;
        case 3: m_pwinTab = new CColWin_grad(this); break;
        case 4: m_pwinTab = new CColWin_pal(this); break;
    }

    //セット

    if(m_pwinTab)
    {
        getLayout()->addItem(m_pwinTab);

        if(bLayout)
        {
            m_pwinTab->calcDefSize();
            layout();
        }
    }
}


//***********************************
// CColWin_tabbase : 基本クラス
//***********************************


CColWin_tabbase::CColWin_tabbase(AXWindow *pParent)
    : AXWindow(pParent, WS_BK_FACE, LF_EXPAND_WH)
{
}

void CColWin_tabbase::setColor(DWORD col)
{
}

void CColWin_tabbase::beforeRemoveTab()
{
}


//***********************************
// CColWin_bar : RGB/HSVバー
//***********************************


CColWin_bar::CColWin_bar(AXWindow *pParent)
    : CColWin_tabbase(pParent)
{
    AXLayout *plTop,*pl;
    int i;

    setLayout(plTop = new AXLayoutVert(LF_EXPAND_WH, 0));
    plTop->setSpacing(8);

    //バー＋エディット

    plTop->addItem(pl = new AXLayoutMatrix(3, LF_EXPAND_W));

    for(i = 0; i < 3; i++)
    {
        //ラベル

        m_plabel[i] = new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,5,7), "-");

        pl->addItem(m_plabel[i]);

        //バー

        m_pbar[i] = new CValBarSimple(this, 0, LF_EXPAND_W|LF_CENTER_Y,
                            ID_BAR1 + i, MAKE_DW4(0,0,6,7), 0, 255, 0);

        m_pbar[i]->setMinHeight(12);

        pl->addItem(m_pbar[i]);

        //エディット

        m_pedit[i] = new AXLineEdit(this, AXLineEdit::ES_SPIN, LF_CENTER_Y, ID_EDIT1 + i, MAKE_DW4(0,0,0,7));
        m_pedit[i]->setWidthFromLen(5);

        pl->addItem(m_pedit[i]);
    }

    //RGB/HSV

    plTop->addItem(pl = new AXLayoutHorz(0, 6));

    m_pckType[0] = new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0, ID_RGB, 0, "RGB", (g_conf->btColBarType == 0));
    m_pckType[1] = new AXCheckButton(this, AXCheckButton::CBS_RADIO, 0, ID_HSV, 0, "HSV", (g_conf->btColBarType == 1));

    pl->addItem(m_pckType[0]);
    pl->addItem(m_pckType[1]);

    //

    _setColorType();
}

//! RGB/HSV 変更時

void CColWin_bar::_setColorType()
{
    char name[2][3] = {{'R','G','B'},{'H','S','V'}},str[2] = {0,0};
    int i,type,max;

    type = g_conf->btColBarType;

    for(i = 0; i < 3; i++)
    {
        //ラベル

        str[0] = name[type][i];

        m_plabel[i]->setText(str);

        //バー・エディット最大値

        max = (type == 1 && i == 0)? 359: 255;

        m_pbar[i]->setRange(0, max);
        m_pedit[i]->setValStatus(0, max);
    }

    //ラベルが変わったので再レイアウト

    layout();

    //値セット

    _setColorVal(g_conf->dwDrawCol);
}

//! 色の値セット

void CColWin_bar::_setColorVal(DWORD col)
{
    int i,val[3];

    //--------- 値取得

    if(g_conf->btColBarType == 0)
    {
        //RGB

        val[0] = _GETR(col);
        val[1] = _GETG(col);
        val[2] = _GETB(col);
    }
    else
    {
        //HSV

        double hsv[3];

        AXRGBtoHSV(col, hsv);

        val[0] = (int)(hsv[0] * 360 + 0.5);
        val[1] = (int)(hsv[1] * 255 + 0.5);
        val[2] = (int)(hsv[2] * 255 + 0.5);
    }

    //----- バー、エディットにセット

    for(i = 0; i < 3; i++)
    {
        m_pbar[i]->setPos(val[i]);
        m_pedit[i]->setVal(val[i]);
    }
}

//! バー・エディットから値が変更された時

void CColWin_bar::_changeVal()
{
    int i,val[3];

    //バーから値取得

    for(i = 0; i < 3; i++)
        val[i] = m_pbar[i]->getPos();

    //HSV -> RGB変換

    if(g_conf->btColBarType == 1)
        AXHSVtoRGB(val[0], val[1] / 255.0, val[2] / 255.0, val);

    //

    getNotify()->onNotify(this, 0, _RGB(val[0], val[1], val[2]));
}

//!< 色変更時

void CColWin_bar::setColor(DWORD col)
{
    _setColorVal(col);
}

//!< 通知

BOOL CColWin_bar::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    if(id >= ID_BAR1 && id < ID_BAR1 + 3)
    {
        //バー

        m_pedit[id - ID_BAR1]->setVal(m_pbar[id - ID_BAR1]->getPos());

        _changeVal();
    }
    else if(id >= ID_EDIT1 && id < ID_EDIT1 + 3)
    {
        //エディット

        if(uNotify == AXLineEdit::EN_CHANGE)
        {
            m_pbar[id - ID_EDIT1]->setPos(m_pedit[id - ID_EDIT1]->getVal());

            _changeVal();
        }
    }
    else if(id == ID_RGB || id == ID_HSV)
    {
        //RGB/HSV

        g_conf->btColBarType = id - ID_RGB;

        _setColorType();
    }

    return TRUE;
}


//***********************************
// CColWin_hsv : HSVピッカー
//***********************************


CColWin_hsv::CColWin_hsv(AXWindow *pParent)
    : CColWin_tabbase(pParent)
{
    AXLayout *plTop,*pl;
    int i;
    char name[3] = { 'H', 'S', 'V' },str[2] = {0,0};
    short max[3] = {359,255,255};

    setLayout(plTop = new AXLayoutHorz(LF_EXPAND_W, 15));
    plTop->setSpacing(8);

    //HSVバー

    plTop->addItem(pl = new AXLayoutMatrix(2, LF_EXPAND_W));

    for(i = 0; i < 3; i++)
    {
        //HSVラベル

        str[0] = name[i];

        pl->addItem(new AXLabel(this, 0, LF_CENTER_Y, MAKE_DW4(0,0,5,7), str));

        //バー

        m_pbar[i] = new CValBar(this, 0, LF_EXPAND_W|LF_CENTER_Y, ID_BAR1 + i, MAKE_DW4(0,0,0,7),
                                0, 0, max[i], 0);

        pl->addItem(m_pbar[i]);
    }

    //HSVピッカー

    plTop->addItem(m_pPicker = new AXHSVPicker(this, 0, 0, 90));

    //セット

    setColor(g_conf->dwDrawCol);
}

//! バーの値セット

void CColWin_hsv::_setColorBar(DWORD col)
{
    double hsv[3];

    AXRGBtoHSV(col, hsv);

    m_pbar[0]->setPos((int)(hsv[0] * 360 + 0.5));
    m_pbar[1]->setPos((int)(hsv[1] * 255 + 0.5));
    m_pbar[2]->setPos((int)(hsv[2] * 255 + 0.5));
}

//!< 色変更時

void CColWin_hsv::setColor(DWORD col)
{
    _setColorBar(col);

    m_pPicker->setColor(col);
}

//!< 通知

BOOL CColWin_hsv::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();
    DWORD col;

    if(pwin == m_pPicker)
    {
        //ピッカー

        col = m_pPicker->getColor();

        _setColorBar(col);

        getNotify()->onNotify(this, 0, col);
    }
    else if(id >= ID_BAR1 && id < ID_BAR1 + 3)
    {
        //バー

        double s,v;

        s = m_pbar[1]->getPos() / 255.0;
        v = m_pbar[2]->getPos() / 255.0;

        col = AXHSVtoRGB(m_pbar[0]->getPos(), s, v);

        m_pPicker->setColor(col);

        getNotify()->onNotify(this, 0, col);
    }

    return TRUE;
}


//***********************************
// CColWin_hls : HLSパレット
//***********************************


CColWin_hls::CColWin_hls(AXWindow *pParent)
    : CColWin_tabbase(pParent)
{
    AXLayout *pl;

    setLayout(pl = new AXLayoutHorz(0, 8));
    pl->setSpacing(8);

    pl->addItem(m_pPal = new CColSub_hlspal(this));
    pl->addItem(m_pbtt = new AXArrowButton(this, AXArrowButton::ARBTS_DOWN, 0));

    m_pbtt->setMinSize(11, 11);
}

//!< 通知

BOOL CColWin_hls::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pPal)
        //パレットから色の変更
        getNotify()->onNotify(this, 0, lParam);
    else if(pwin == m_pbtt)
        //メニューボタン
        _runMenu();

    return TRUE;
}

//!< メニュー実行

void CColWin_hls::_runMenu()
{
    AXMenu *pmenu;
    int i;
    const char *name[3] = {"H:SL", "L:SH", "S:LH"};
    AXString str;
    AXRectSize rcs;

    pmenu = new AXMenu;

    for(i = 0; i < 3; i++)
    {
        str = name[i];
        pmenu->addCheck(i, str, (i == g_conf->nColHLSVal[0]));
    }

    m_pbtt->getWindowRect(&rcs);

    i = pmenu->popup(NULL, rcs.x, rcs.y + rcs.h, 0);

    delete pmenu;

    //

    if(i != -1)
    {
        g_conf->nColHLSVal[0] = i;

        m_pPal->drawAll();
    }
}


//***********************************
// CColWin_grad : グラデーションパレット
//***********************************


CColWin_grad::CColWin_grad(AXWindow *pParent)
    : CColWin_tabbase(pParent)
{
    AXLayout *pl;
    int i;

    setLayout(pl = new AXLayoutVert(LF_EXPAND_W, 8));
    pl->setSpacing(8);

    for(i = 0; i < CConfig::GRADPAL_NUM; i++)
    {
        m_pGrad[i] = new CColSub_grad(this, ID_GRAD1 + i,
                            g_conf->dwGradPalCol[i * 2], g_conf->dwGradPalCol[i * 2 + 1]);

        pl->addItem(m_pGrad[i]);
    }
}

//!< 通知

BOOL CColWin_grad::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();

    if(id >= ID_GRAD1 && id < ID_GRAD1 + CConfig::GRADPAL_NUM)
    {
        id -= ID_GRAD1;

        switch(uNotify)
        {
            //左の色が変った
            case CColSub_grad::NOTIFY_CHANGE_LEFT:
                g_conf->dwGradPalCol[id * 2] = m_pGrad[id]->getColLeft();
                break;
            //右の色が変わった
            case CColSub_grad::NOTIFY_CHANGE_RIGHT:
                g_conf->dwGradPalCol[id * 2 + 1] = m_pGrad[id]->getColRight();
                break;
            //色
            case CColSub_grad::NOTIFY_COLOR:
                getNotify()->onNotify(this, 0, lParam);
                break;
        }
    }

    return TRUE;
}


//***********************************
// CColWin_pal : パレット
//***********************************


CColWin_pal::CColWin_pal(AXWindow *pParent)
    : CColWin_tabbase(pParent)
{
    AXLayout *plTop,*pl;

    setLayout(plTop = new AXLayoutHorz(LF_EXPAND_W, 0));
    plTop->setSpacing(7);

    m_pScr = new AXScrollBar(this, AXScrollBar::SBS_VERT, LF_EXPAND_H);

    plTop->addItem(m_pPal = new CColSub_pal(this, m_pScr));
    plTop->addItem(m_pScr);

    plTop->addItem(pl = new AXLayoutVert(LF_EXPAND_H, 0));

    pl->addItem(m_pbtt = new AXArrowButton(this, AXArrowButton::ARBTS_DOWN, LF_EXPAND_Y, 0, MAKE_DW4(6,0,0,6)));
    pl->addItem(m_pSel = new CColSub_paltab(this));

    //スクロール初期位置セット

    m_pScr->setStatus(0, 5000, 1);
    m_pScr->setPos(COLORWIN->getPalScrPos(g_conf->btColPalSel));
}

//! タブから取り除く前の処理

void CColWin_pal::beforeRemoveTab()
{
    //スクロール位置保存
    COLORWIN->setPalScrPos(g_conf->btColPalSel, m_pScr->getPos());
}

//!< 通知

BOOL CColWin_pal::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pPal)
        //パレットから色変更
        getNotify()->onNotify(this, 0, lParam);
    else if(pwin == m_pScr)
    {
        //スクロール

        if(uNotify == AXScrollBar::SBN_SCROLL)
            m_pPal->draw();
    }
    else if(pwin == m_pSel)
    {
        //タブ選択変更
        //(スクロール位置は各タブごとに保存)

        COLORWIN->setPalScrPos(g_conf->btColPalSel, m_pScr->getPos());

        g_conf->btColPalSel = (BYTE)lParam;

        m_pPal->changeAll();
        m_pScr->setPos(COLORWIN->getPalScrPos(g_conf->btColPalSel));
    }
    else if(pwin == m_pbtt)
        //メニュー
        _runMenu();

    return TRUE;
}

//!< メニュー実行

void CColWin_pal::_runMenu()
{
    AXMenu *pmenu;
    WORD dat[] = {
        strid::COLWIN_PALMENU_RESIZE, 0xffff, strid::COLWIN_PALMENU_LOAD, strid::COLWIN_PALMENU_SAVE,
        0xffff, strid::COLWIN_PALMENU_FROMIMG, strid::COLWIN_PALMENU_CLEAR, 0xffff,
        strid::COLWIN_PALMENU_DISPLAYSIZE, strid::COLWIN_PALMENU_HELP
    };
    int id;
    AXRectSize rcs;

    //

    pmenu = new AXMenu;

    _trgroup(strid::GROUP_COLORWIN);
    pmenu->addTrArray(dat, sizeof(dat)/sizeof(WORD));

    m_pbtt->getWindowRect(&rcs);
    id = pmenu->popup(NULL, rcs.x + rcs.w - 1, rcs.y + rcs.h, AXMenu::POPF_RIGHT);

    delete pmenu;

    //

    if(id == -1) return;

    switch(id)
    {
        //個数変更
        case strid::COLWIN_PALMENU_RESIZE:
            _cmd_resize();
            break;
        //読み込み
        case strid::COLWIN_PALMENU_LOAD:
            _cmd_load();
            break;
        //保存
        case strid::COLWIN_PALMENU_SAVE:
            _cmd_save();
            break;
        //画像からパレット作成
        case strid::COLWIN_PALMENU_FROMIMG:
            _cmd_createFromImg();
            break;
        //クリア
        case strid::COLWIN_PALMENU_CLEAR:
            if(AXMessageBox::message(m_pTopLevel, NULL,
                _string(strid::GROUP_COLORWIN, strid::COLWIN_MES_PAL_CLEAR),
                AXMessageBox::YESNO,
                AXMessageBox::YES) == AXMessageBox::YES)
            {
                PALETTE->m_pal[g_conf->btColPalSel].clearWhite();
                m_pPal->draw();
            }
            break;
        //表示幅の設定
        case strid::COLWIN_PALMENU_DISPLAYSIZE:
            _cmd_displaySize();
            break;
        //操作ヘルプ
        case strid::COLWIN_PALMENU_HELP:
            AXMessageBox::message(m_pTopLevel, NULL,
                _string(strid::GROUP_COLORWIN, strid::COLWIN_MES_PAL_HELP),
                AXMessageBox::OK, AXMessageBox::OK);
            break;
    }
}

//!< 個数変更

void CColWin_pal::_cmd_resize()
{
    AXString str,strMes;

    _trgroup(strid::GROUP_COLORWIN);

    strMes = _str(strid::COLWIN_DLG_PALRESIZE_MESSAGE);
    strMes += " [1-";
    strMes += (int)CPalette::MAXCNT;
    strMes += ']';

    str.setInt(PALETTE->m_pal[g_conf->btColPalSel].getCnt());

    if(AXStrDialog::getString(m_pTopLevel, _str(strid::COLWIN_DLG_PALRESIZE_TITLE), strMes, &str))
    {
        PALETTE->m_pal[g_conf->btColPalSel].resize(str.toInt());

        m_pPal->changeAll();
    }
}

//! 読み込み

void CColWin_pal::_cmd_load()
{
    AXString str,filter;

    filter = "Palette File (APL/ACO)\t*.apl;*.aco\tAzPainter palette (*.apl)\t*.apl\tPhotoShop palette (*.aco)\t*.aco";

    if(AXFileDialog::openFile(m_pTopLevel, filter, 0, g_conf->strOpenDir, 0, &str))
    {
        if(!PALETTE->m_pal[g_conf->btColPalSel].loadFile(str))
            ((AXTopWindow *)m_pTopLevel)->errMes(strid::GROUP_MESSAGE, strid::MES_FAILED_LOAD);
        else
            m_pPal->changeAll();

        PALETTE->updateON();
    }
}

//! 保存

void CColWin_pal::_cmd_save()
{
    AXString str,filter;
    int type;
    BOOL ret;

    filter = "AzPainter palette (*.apl)\t*.apl\tPhotoShop palette (*.aco)\t*.aco";

    if(AXFileDialog::saveFile(m_pTopLevel, filter, 0, g_conf->strOpenDir, 0, &str, &type))
    {
        if(type == 0)
        {
            str.path_setExt("apl");
            ret = PALETTE->m_pal[g_conf->btColPalSel].saveAPL(str);
        }
        else
        {
            str.path_setExt("aco");
            ret = PALETTE->m_pal[g_conf->btColPalSel].saveACO(str);
        }

        if(!ret)
            ((AXTopWindow *)m_pTopLevel)->errMes(strid::GROUP_MESSAGE, strid::MES_FAILED);
    }
}

//! 画像からパレット作成

void CColWin_pal::_cmd_createFromImg()
{
    AXString str,filter;

    CAPP->getFileFilter_normalimg(&filter);

    if(AXFileDialog::openFile(m_pTopLevel, filter, 0, g_conf->strOpenDir, 0, &str))
    {
        if(!PALETTE->m_pal[g_conf->btColPalSel].createFromImg(str))
            ((AXTopWindow *)m_pTopLevel)->errMes(strid::GROUP_MESSAGE, strid::MES_FAILED);
        else
            m_pPal->changeAll();
    }
}

//! 表示幅の設定

void CColWin_pal::_cmd_displaySize()
{
    AXString str;
    int w,h;
    LPUSTR pu;

    str.setInt(g_conf->wPalOneSize >> 8);
    str += ',';
    str += (int)(g_conf->wPalOneSize & 255);

    if(AXStrDialog::getString(m_pTopLevel,
            _string(strid::GROUP_COLORWIN, strid::COLWIN_DLG_DISPSIZE_TITLE),
            _string(strid::GROUP_COLORWIN, strid::COLWIN_DLG_DISPSIZE_MESSAGE),
            &str))
    {
        w = str.toInt();
        if(w < 5) w = 5; else if(w > 50) w = 50;

        for(pu = str; *pu && *pu >= '0' && *pu <= '9'; pu++);
        for(; *pu && (*pu < '0' || *pu > '9'); pu++);

        h = AXUStrToInt(pu);
        if(h < 5) h = 5; else if(h > 50) h = 50;

        g_conf->wPalOneSize = (w << 8) | h;

        m_pPal->changeOneSize();
    }
}
