/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CCanvasWin, CCanvasWinArea - キャンバスのウィンドウ
*/


#include "CCanvasWin.h"

#include "CConfig.h"
#include "CDevList.h"
#include "CKeyDat.h"
#include "CMainWin.h"
#include "CApp.h"
#include "CXImage.h"
#include "cursor.h"

#include "drawdat.h"
#include "drawOpDef.h"
#include "draw_main.h"
#include "draw_calc.h"
#include "draw_update.h"
#include "draw_opmain.h"
#include "draw_opsub.h"

#include "global.h"

#include "AXScrollBar.h"
#include "AXCursor.h"
#include "AXKey.h"



//********************************
// CCanvasWin
//********************************


CCanvasWin *CCanvasWin::m_pSelf = NULL;


CCanvasWin::CCanvasWin(AXWindow *pParent)
    : AXScrollView(pParent, SVS_HORZVERT | SVS_SUNKEN, LF_EXPAND_WH)
{
    m_pSelf = this;

    m_uFlags |= FLAG_TAKE_FOCUS;    //キー入力フォーカスを受け取る

    m_uLastDownKey = 0;

    m_pScrArea = new CCanvasWinArea(this);
}

//! スペースキーが押されているか

BOOL CCanvasWin::isDownKeySpace()
{
    return (m_uLastDownKey == KEY_SPACE || m_uLastDownKey == KEY_NUM_SPACE);
}

//! スクロール範囲セット

void CCanvasWin::setScroll()
{
    AXSize size;

    draw::getCanvasScrollMax(&size);

    //スクロールバー中央位置

    m_nScrCtX = (size.w - g_draw->view.szCanvas.w) >> 1;
    m_nScrCtY = (size.h - g_draw->view.szCanvas.h) >> 1;

    //セット

    m_pScrH->setStatus(0, size.w, g_draw->view.szCanvas.w);
    m_pScrH->setPos(m_nScrCtX);

    m_pScrV->setStatus(0, size.h, g_draw->view.szCanvas.h);
    m_pScrV->setPos(m_nScrCtY);
}

//! スクロール位置セット

void CCanvasWin::setScrollPos()
{
    m_pScrH->setPos(g_draw->view.ptScr.x + m_nScrCtX);
    m_pScrV->setPos(g_draw->view.ptScr.y + m_nScrCtY);
}

//---------------------

//! すべてのキーを受け付ける

BOOL CCanvasWin::isAcceptKey(UINT keytype)
{
    return TRUE;
}

//! キー押し

BOOL CCanvasWin::onKeyDown(AXHD_KEY *phd)
{
    UINT key;

    key = phd->keysym;

    //最後に押されたキー保存（装飾キーは除く）
    /* グラブ中も判定 */

    if(key != KEY_CTRL_L && key != KEY_CTRL_R &&
       key != KEY_SHIFT_L && key != KEY_CTRL_R &&
       key != KEY_ALT_L && key != KEY_ALT_R)
        m_uLastDownKey = key;

    //

    if(!g_draw->isNowOpNone())
    {
        //操作中時、各ツールに対応した処理
        //（キャンセル処理などの場合、グラブ解放）

        if(draw::onKey_inOp(key))
            CAPP->ungrabBoth();
    }
    else
    {
        //ショートカットキー

        int cmd = KEYDAT->getCmd(key);

        if(cmd != -1)
            MAINWIN->sendCommand(cmd, 0, 0);
    }

    return TRUE;
}

//! キー離し

BOOL CCanvasWin::onKeyUp(AXHD_KEY *phd)
{
    if(phd->keysym == m_uLastDownKey)
        m_uLastDownKey = 0;

    return TRUE;
}

//! フォーカス消去

BOOL CCanvasWin::onFocusOut(int detail)
{
    m_uLastDownKey = 0;

    return TRUE;
}

//! コマンド

BOOL CCanvasWin::onCommand(UINT uID,ULONG lParam,int from)
{
    //ダイアログ開始
    /* ボタン押し時にダイアログを表示するものは、ここで実行させる。
       グラブ解放やXI2のデータを解放させてから行わせるため。 */

    switch(uID)
    {
        case CMDID_TEXTDLG:
            draw::op_text_down();
            break;
        case CMDID_SCALEROTDLG:
            draw::draw_boxedit_scaleRot();
            break;
    }

    return TRUE;
}

//! 通知

BOOL CCanvasWin::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    //スクロール

    if(pwin == m_pScrH && uNotify == AXScrollBar::SBN_SCROLL)
        g_draw->view.ptScr.x = m_pScrH->getPos() - m_nScrCtX;
    else if(pwin == m_pScrV && uNotify == AXScrollBar::SBN_SCROLL)
        g_draw->view.ptScr.y = m_pScrV->getPos() - m_nScrCtY;

    draw::updateCanvas(TRUE);

    return TRUE;
}


//*****************************************
// CCanvasWinArea
//*****************************************


CCanvasWinArea *CCanvasWinArea::m_pSelf = NULL;


CCanvasWinArea::CCanvasWinArea(AXWindow *pParent)
    : AXScrollArea(pParent, 0)
{
    m_pSelf = this;

    //ホイール動作を onButtonDown/Up で処理する
    m_uFlags |= FLAG_WHEELEVENT_NORMAL;

    m_rcsTimerUpdate.x = -1;

    //カーソル

    m_pcurTool = new AXCursor;
    m_pcurDrag = new AXCursor;
    m_pcurWait = new AXCursor;

    setCursorTool();

    cursor::create(m_pcurWait, cursor::WAIT);
}

CCanvasWinArea::~CCanvasWinArea()
{
    delete m_pcurTool;
    delete m_pcurDrag;
    delete m_pcurWait;
}

//! スクロールを表示するか

BOOL CCanvasWinArea::isShowScroll(int size,BOOL bHorz)
{
    return g_conf->isScrollBar();
}


//===========================
//カーソル
//===========================


//! キャンバスに現在のツールのカーソルセット

void CCanvasWinArea::setCursorTool()
{
    int no = cursor::DRAW;

    switch(g_draw->tool.toolno)
    {
        case draw::TOOL_BRUSH:
        case draw::TOOL_DOTPEN:
            if(g_conf->memDrawCursor.isExist())
                no = -1;
            break;
        case draw::TOOL_SEL:
            no = cursor::SEL;
            break;
        case draw::TOOL_BOXEDIT:
            no = cursor::BOXEDIT;
            break;
        case draw::TOOL_TEXT:
            no = cursor::TEXT;
            break;
        case draw::TOOL_MOVE:
            no = cursor::MOVE;
            break;
        case draw::TOOL_CANVROTATE:
            no = cursor::ROTATE;
            break;
        case draw::TOOL_CANVMOVE:
            no = cursor::HAND;
            break;
        case draw::TOOL_SCALE:
            no = cursor::LOUPE;
            break;
        case draw::TOOL_SPOIT:
            no = cursor::SPOIT;
            break;
    }

    //セット

    unsetCursor();

    if(no == -1)
        m_pcurTool->create(g_conf->memDrawCursor);
    else
        cursor::create(m_pcurTool, no);

    setCursor(m_pcurTool);
}

//! ドラッグ中のカーソルセット

void CCanvasWinArea::setCursorDrag(int type)
{
    cursor::create(m_pcurDrag, type);

    setCursor(m_pcurDrag);
}

//! カーソルを砂時計に

void CCanvasWinArea::setCursorWait()
{
    setCursor(m_pcurWait);
    axapp->flush();
}

//! カーソルをツールのものに戻す（ドラッグ中・砂時計共通）

void CCanvasWinArea::restoreCursorTool()
{
    setCursor(m_pcurTool);

    m_pcurDrag->free();
}


//===========================
//タイマー
//===========================


//! タイマー更新クリア（残っていた場合は処理する）

void CCanvasWinArea::clearTimerUpdate(UINT uTimerID)
{
    if(isTimerExist(uTimerID))
        onTimer(uTimerID, NULL);
}

//! UPDATECANVAS タイマークリア
/*!
    タイマーでの更新中は低画質のため、最後に高品質で更新
*/

void CCanvasWinArea::clearTimer_updateCanvas()
{
    delTimer(TIMERID_UPDATECANVAS);
    draw::updateCanvas();
}

//! UPDATERECT タイマークリア

void CCanvasWinArea::clearTimer_updateRect()
{
    delTimer(TIMERID_UPDATERECT);
    m_rcsTimerUpdate.x = -1;
}

//-----------

void CCanvasWinArea::setTimer_updateCanvas()
{
    addTimer(TIMERID_UPDATECANVAS, 5);
}

void CCanvasWinArea::setTimer_updateRect(const AXRectSize &rcs,int time)
{
    draw::unionRectSize(&m_rcsTimerUpdate, rcs);

    if(!AXWindow::isTimerExist(TIMERID_UPDATERECT))
        addTimer(TIMERID_UPDATERECT, time);
}

void CCanvasWinArea::setTimer_updateMove()
{
    addTimer(TIMERID_UPDATE_MOVE, 5);
}

void CCanvasWinArea::setTimer_update()
{
    addTimer(TIMERID_UPDATE, 5);
}


//===========================
//ハンドラ
//===========================


//! サイズ変更時

BOOL CCanvasWinArea::onSize()
{
    //メインウィンドウが初期化状態の時は更新なし

    draw::changeCanvasWinSize(MAINWIN->isShowMain());

    return TRUE;
}

//! 描画

BOOL CCanvasWinArea::onPaint(AXHD_PAINT *phd)
{
    g_draw->pimgCanvas->put(m_id, phd->x, phd->y, phd->x, phd->y, phd->w, phd->h);

    return TRUE;
}

//! タイマー

BOOL CCanvasWinArea::onTimer(UINT uTimerID,ULONG lParam)
{
    switch(uTimerID)
    {
        //キャンバス更新（低品質）
        case TIMERID_UPDATECANVAS:
            draw::updateCanvas(TRUE, FALSE);
            break;
        //イメージ・キャンバス更新（範囲）
        case TIMERID_UPDATERECT:
            draw::updateRect(m_rcsTimerUpdate);
            axapp->update();

            m_rcsTimerUpdate.x = -1;
            break;
        //イメージ/選択範囲移動時の更新
        case TIMERID_UPDATE_MOVE:
            if(g_draw->work.nNowOpNo == draw::OPNO_SELPOSMOVE)
                draw::updateCanvas(g_draw->work.rcfDrawUp);
            else
                draw::updateRect(g_draw->work.rcfDrawUp);

            g_draw->work.rcfDrawUp.clear();
            break;
        //更新
        case TIMERID_UPDATE:
            axapp->update();
            break;
    }

    delTimer(uTimerID);

    return TRUE;
}

//! ボタン押し

BOOL CCanvasWinArea::onButtonDown(AXHD_MOUSE *phd)
{
    ::DRAWPOINT pt;
    UINT btt;

    pt.x = phd->x;
    pt.y = phd->y;
    pt.press = 1;

    btt = DEVLIST->getButtonAction(2, phd->button,
                        phd->state & STATE_CTRL, phd->state & STATE_SHIFT, phd->state & STATE_ALT);

    if(draw::onDown(pt, btt))
        grabPointer();

    return TRUE;
}

//! ボタン離し

BOOL CCanvasWinArea::onButtonUp(AXHD_MOUSE *phd)
{
    ::DRAWPOINT pt;

    pt.x = phd->x;
    pt.y = phd->y;
    pt.press = 0;

    if(draw::onUp(pt, phd->button))
        ungrabPointer();

    return TRUE;
}

//! マウス移動

BOOL CCanvasWinArea::onMouseMove(AXHD_MOUSE *phd)
{
    ::DRAWPOINT pt;

    pt.x = phd->x;
    pt.y = phd->y;
    pt.press = 1;

    draw::onMove(pt, phd->state & STATE_CTRL, phd->state & STATE_SHIFT);

    return TRUE;
}

//! ダブルクリック

BOOL CCanvasWinArea::onDblClk(AXHD_MOUSE *phd)
{
    if(phd->button == BUTTON_LEFT)
    {
        if(draw::onLDblClk())
        {
            ungrabPointer();
            return TRUE;
        }
    }

    return FALSE;
}

//! ダイアログ中の操作

BOOL CCanvasWinArea::onMouseInSkip(AXHD_MOUSE *phd)
{
    //描画位置変更

    if(phd->type == EVENT_BUTTONDOWN && phd->button == BUTTON_LEFT)
        draw::onLDownInDlg(phd->x, phd->y);

    return TRUE;
}
