/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CBrushWin_value - ブラシウィンドウの設定項目部分
*/


#include "CBrushWin_value.h"

#include "CBrushList.h"
#include "CBrushItem.h"
#include "CConfig.h"

#include "CValBar.h"
#include "CSelImgBar.h"
#include "CPressCurve.h"

#include "AXLayout.h"
#include "AXLabel.h"
#include "AXLineEdit.h"
#include "AXButton.h"
#include "AXCheckButton.h"
#include "AXArrowButton.h"
#include "AXComboBox.h"
#include "AXScrollBar.h"
#include "AXExpander.h"
#include "AXMenu.h"
#include "AXApp.h"
#include "AXUtilStr.h"

#include "global.h"
#include "strid.h"

#include "CBrushWin_value_sub.h"



//************************************
// CBrushWin_value
//************************************



CBrushWin_value::CBrushWin_value(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_EXPAND_WH)
{
    AXLayout *pl;

    setLayout(pl = new AXLayoutHorz);

    pl->addItem(m_pArea = new CBrushValueArea(this));
    pl->addItem(m_pScrV = new AXScrollBar(this, AXScrollBar::SBS_VERT, LF_EXPAND_H));

    m_pArea->setScroll(m_pScrV);

    m_pBase = m_pArea->getBaseWidget();
}

//! 選択アイテムの値セット

void CBrushWin_value::setValue()
{
    m_pBase->setValue(BRUSHLIST->getEditItem());
}

//! 半径の値をセット

void CBrushWin_value::setValRadius()
{
    m_pBase->setValRadius();
}

//! 通知

BOOL CBrushWin_value::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pScrV)
        m_pBase->move(0, -m_pScrV->getPos());

    return TRUE;
}



//*******************************************************
// CBrushValueArea - CBrushValueBase を表示部分だけ見せる
//*******************************************************



CBrushValueArea::CBrushValueArea(AXWindow *pParent)
    : AXWindow(pParent, WS_BK_FACE, LF_EXPAND_WH)
{
    m_pBase = new CBrushValueBase(this);
}

//! スクロール情報セット

void CBrushValueArea::setScrollInfo()
{
    m_pScr->setStatus(0, m_pBase->getHeight(), m_nH);

    m_pBase->move(0, -m_pScr->getPos());
}

//! サイズ変更時

BOOL CBrushValueArea::onSize()
{
    m_pBase->resize(m_nW, m_pBase->getDefH());

    setScrollInfo();

    return TRUE;
}

//! 通知 (CBrushValueBase の再レイアウト時)

BOOL CBrushValueArea::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    setScrollInfo();

    return TRUE;
}


//************************************
// CBrushValueBase - 項目の全体
//************************************


CBrushValueBase::CBrushValueBase(AXWindow *pParent)
    : AXWindow(pParent, WS_BK_FACE, 0)
{
    AXLayout *plTop,*pl;
    AXWindow *pwin;

    _trgroup(strid::GROUP_BRUSHITEM);

    setLayout(plTop = new AXLayoutVert);
    plTop->setSpacing(4);

    //名前

    plTop->addItem(m_pwidName = new AXLabel(this, AXLabel::LS_BORDER, LF_EXPAND_W, MAKE_DW4(0,0,0,4), (LPCSTR)NULL));

    //手動保存/保存ボタン

    plTop->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 5));

    pl->addItem(m_pckManualSave = new AXCheckButton(this, 0, LF_EXPAND_X|LF_CENTER_Y, WID_CK_MANUALSAVE, 0, STRID_MANUALSAVE, FALSE));
    pl->addItem(pwin = new AXButton(this, AXButton::BS_REAL_WH, LF_EXPAND_H, WID_BT_SAVE, 0, STRID_SAVE));

    pwin->setMinSize(50, 16);

    //各項目

    _createMain();
    _createMinInterval();
    _createRandom();
    _createWater();
    _createBrushStyle();
    _createPress();
    _createEtc();

    //

    calcDefSize();
}

//! 半径の値をセット

void CBrushValueBase::setValRadius()
{
    CBrushItem *p = BRUSHLIST->getEditItem();

    m_pbarRadius->setRange(CBrushItem::RADIUS_MIN, p->wRadiusCtlMax);
    m_pbarRadius->setPos(p->wRadius);
}

//! ブラシアイテムの値セット

void CBrushValueBase::setValue(CBrushItem *p)
{
    BOOL isExistSel = (BRUSHLIST->getSelItem() != NULL);

    //名前

    if(isExistSel)
        m_pwidName->setText(p->strName);
    else
        m_pwidName->setText((LPCUSTR)NULL);

    //手動保存

    m_pckManualSave->enable(isExistSel);
    enableWidget(WID_BT_SAVE, (isExistSel && p->isManualSave()));

    m_pckManualSave->setCheck(p->isManualSave());

    //半径
    m_pbarRadius->setRange(CBrushItem::RADIUS_MIN, p->wRadiusCtlMax);
    m_pbarRadius->setPos(p->wRadius);

    //濃度
    m_pbarOpacity->setPos(p->btOpacity);

    //塗り
    m_pcbPix->setCurSel(p->btPixType);

    //補正
    m_pcbHoseiType->setCurSel(p->btHoseiType);
    m_pcbHoseiStr->setCurSel(p->btHoseiStr);

    //最小
    m_pbarMinSize->setPos(p->wMinSize);
    m_pbarMinOpa->setPos(p->wMinOpacity);

    //間隔
    m_pbarInterval->setPos(p->wInterval);

    //ランダム
    m_pbarRandSize->setPos(p->wRan_sizeMin);
    m_pbarRandPos->setPos(p->wRan_posLen);

    //水彩

    m_pckWaterOn->setCheck(p->isWaterOn());
    m_pbarWater[0]->setPos(p->wWater[0]);
    m_pbarWater[1]->setPos(p->wWater[1]);
    m_pbarWater[2]->setPos(p->wWater[2]);

    //ブラシ形状

    m_pselBrushImg->setName(p->strBrushImg);
    m_pbarBrushHard->setPos(p->btHardness);
    m_pbarBrushRough->setPos(p->wRoughness);
    m_pcbBrushRot->setCurSel(p->isBrushRotDir()? 1: 0);
    m_pbarBrushRotAngle->setPos(p->wRotAngle);
    m_pbarBrushRotRand->setPos(p->wRotRandom);

    //筆圧

    m_pPressCurve[0]->setVal(p->wPressSize);
    m_pPressCurve[1]->setVal(p->wPressOpacity);
    m_peditPress[0]->setVal(p->wPressSize);
    m_peditPress[1]->setVal(p->wPressOpacity);

    //その他

    m_pselTexture->setName(p->strTexImg);
    m_pckEtc[0]->setCheck(p->isAntiAlias());
    m_pckEtc[1]->setCheck(p->isCurve());

    //-------- 有効/無効

    _enableFromPixType(p->btPixType);
    _enableFromBrushImg(p->strBrushImg.isEmpty());
}

//! 塗りタイプによるウィジェットの有効/無効

void CBrushValueBase::_enableFromPixType(int type)
{
    int f;

    //ぼかし時は、[濃度最小][間隔][筆圧濃度]を無効

    f = (type != CBrushItem::PIXTYPE_BLUR);

    m_pbarMinOpa->enable(f);
    m_pbarInterval->enable(f);
    m_pPressCurve[1]->enable(f);
    m_peditPress[1]->enable(f);
}

//! ブラシ画像によるウィジェットの有効/無効

void CBrushValueBase::_enableFromBrushImg(BOOL bNormal)
{
    //硬さ

    m_pbarBrushHard->enable(bNormal);
}

//! 通知

BOOL CBrushValueBase::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    UINT id = pwin->getItemID();
    int n;

    //水彩プリセット

    if(id >= WID_BT_WATER_PRESET && id < WID_BT_WATER_PRESET + CConfig::WATERPRESET_NUM)
    {
        n = g_conf->dwWaterPreset[id - WID_BT_WATER_PRESET];

        m_pbarWater[0]->setPos(n & 1023);
        m_pbarWater[1]->setPos((n >> 10) & 1023);
        m_pbarWater[2]->setPos((n >> 20) & 1023);

        BRUSHLIST->changeValWater1(n & 1023);
        BRUSHLIST->changeValWater2((n >> 10) & 1023);
        BRUSHLIST->changeValWater3((n >> 20) & 1023);

        return TRUE;
    }

    //

    switch(id)
    {
        //手動保存・チェック
        case WID_CK_MANUALSAVE:
            BRUSHLIST->changeValManualSave();

            enableWidget(WID_BT_SAVE, (BRUSHLIST->getEditItem())->isManualSave());
            break;
        //手動保存
        case WID_BT_SAVE:
            BRUSHLIST->manualSave();
            break;
        //半径
        case WID_BAR_RADIUS:
            BRUSHLIST->changeValRadius(m_pbarRadius->getPos());
            break;
        //サイズメニュー
        case WID_BT_RADIUS_MENU:
            _runSizeMenu();
            break;
        //濃度
        case WID_BAR_OPACITY:
            BRUSHLIST->changeValOpacity(m_pbarOpacity->getPos());
            break;
        //塗り
        case WID_CB_PIXTYPE:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
            {
                n = m_pcbPix->getCurSel();

                BRUSHLIST->changeValPixType(n);

                _enableFromPixType(n);
            }
            break;
        //補正タイプ
        case WID_CB_HOSEITYPE:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
                BRUSHLIST->changeValHoseiType(m_pcbHoseiType->getCurSel());
            break;
        //補正強さ
        case WID_CB_HOSEISTR:
            if(uNotify == AXComboBox::CBN_SELCHANGE)
                BRUSHLIST->changeValHoseiStr(m_pcbHoseiStr->getCurSel());
            break;
        //サイズ最小
        case WID_BAR_MINSIZE:
            BRUSHLIST->changeValMinSize(m_pbarMinSize->getPos());
            break;
        //濃度最小
        case WID_BAR_MINOPA:
            BRUSHLIST->changeValMinOpacity(m_pbarMinOpa->getPos());
            break;
        //間隔
        case WID_BAR_INTERVAL:
            BRUSHLIST->changeValInterval(m_pbarInterval->getPos());
            break;
        //ランダムサイズ
        case WID_BAR_RANDSIZE:
            BRUSHLIST->changeValRandSize(m_pbarRandSize->getPos());
            break;
        //ランダム位置
        case WID_BAR_RANDPOS:
            BRUSHLIST->changeValRandPos(m_pbarRandPos->getPos());
            break;
        //水彩ON
        case WID_CK_WATERON:
            BRUSHLIST->changeValWaterOn();
            break;
        //水彩1
        case WID_BAR_WATER1:
            BRUSHLIST->changeValWater1(m_pbarWater[0]->getPos());
            break;
        //水彩2
        case WID_BAR_WATER2:
            BRUSHLIST->changeValWater2(m_pbarWater[1]->getPos());
            break;
        //水彩3
        case WID_BAR_WATER3:
            BRUSHLIST->changeValWater3(m_pbarWater[2]->getPos());
            break;
        //水彩プリセットメニュー
        case WID_BT_WATERMENU:
            _runWaterMenu();
            break;
        //ブラシ画像
        case WID_SEL_BRUSHIMG:
            n = BRUSHLIST->changeValBrushImg(m_pselBrushImg->getName());

            _enableFromBrushImg(n);
            break;
        //硬さ
        case WID_BAR_BRUSHHARD:
            BRUSHLIST->changeValBrushHardness(m_pbarBrushHard->getPos());
            break;
        //荒さ
        case WID_BAR_BRUSHROUGH:
            BRUSHLIST->changeValBrushRoughness(m_pbarBrushRough->getPos());
            break;
        //画像回転タイプ
        case WID_CB_BRUSHROT:
            BRUSHLIST->changeValBrushRotType(m_pcbBrushRot->getCurSel());
            break;
        //画像回転角度
        case WID_BAR_BRUSHROTANGLE:
            BRUSHLIST->changeValBrushRotAngle(m_pbarBrushRotAngle->getPos());
            break;
        //画像回転ランダム
        case WID_BAR_BRUSHROTRAND:
            BRUSHLIST->changeValBrushRotRandom(m_pbarBrushRotRand->getPos());
            break;
        //筆圧・サイズ
        case WID_PRESS_SIZE:
            BRUSHLIST->changeValPressSize(m_pPressCurve[0]->getVal());
            m_peditPress[0]->setVal(m_pPressCurve[0]->getVal());
            break;
        //筆圧・サイズ(エディット)
        case WID_EDIT_PRESSSIZE:
            if(uNotify == AXLineEdit::EN_CHANGE)
            {
                BRUSHLIST->changeValPressSize(m_peditPress[0]->getVal());
                m_pPressCurve[0]->setVal(m_peditPress[0]->getVal());
            }
            break;
        //筆圧・濃度
        case WID_PRESS_OPACITY:
            BRUSHLIST->changeValPressOpacity(m_pPressCurve[1]->getVal());
            m_peditPress[1]->setVal(m_pPressCurve[1]->getVal());
            break;
        //筆圧・濃度(エディット)
        case WID_EDIT_PRESSOPACITY:
            if(uNotify == AXLineEdit::EN_CHANGE)
            {
                BRUSHLIST->changeValPressOpacity(m_peditPress[1]->getVal());
                m_pPressCurve[1]->setVal(m_peditPress[1]->getVal());
            }
            break;
        //テクスチャ
        case WID_SEL_TEXTUREIMG:
            BRUSHLIST->changeValTexture(m_pselTexture->getName());
            break;
        //アンチエイリアス
        case WID_CK_ANTIALIAS:
            BRUSHLIST->changeValAntiAlias();
            break;
        //曲線
        case WID_CK_CURVE:
            BRUSHLIST->changeValCurve();
            break;

        //AXExpander の開閉による再レイアウト
        case WID_EXP_MAIN:
        case WID_EXP_MININTER:
        case WID_EXP_RANDOM:
        case WID_EXP_WATER:
        case WID_EXP_BRUSHSTYLE:
        case WID_EXP_PRESS:
        case WID_EXP_ETC:
            g_conf->uBrushValExpand ^= 1 << (id - WID_EXP_MAIN);

            calcDefSize();
            resize(m_nW, m_nDefH);

            m_pParent->onNotify(this, 0, 0);
            break;
    }

    return TRUE;
}


//=====================================
// メニュー
//=====================================


//! ブラシサイズメニュー（最大値指定）表示

void CBrushValueBase::_runSizeMenu()
{
    AXMenu *pm;
    int ret,i,max,now;
    AXRectSize rcs;
    WORD size[8] = {100, 500, 1000, 2000, 3000, 4000, 5000, 6000};
    char m[32];
    AXString str;

    //------ メニュー

    pm = new AXMenu;

    for(i = 0; i < 8; i++)
    {
        AXIntToFloatStr(m, size[i], 1);

        str = "max ";
        str += m;

        pm->add(i, str);
    }

    getWidget(WID_BT_RADIUS_MENU)->getWindowRect(&rcs);

    ret = pm->popup(NULL, rcs.x, rcs.y + rcs.h, 0);

    delete pm;

    //------ 処理

    if(ret == -1) return;

    max = size[ret];
    now = m_pbarRadius->getPos();

    if(now > max) now = max;

    m_pbarRadius->setRange(CBrushItem::RADIUS_MIN, max);
    m_pbarRadius->setPos(now);

    BRUSHLIST->changeValRadiusCtlMax(now, max);
}

//! 水彩プリセットメニュー

void CBrushValueBase::_runWaterMenu()
{
    AXMenu *pm;
    int id,i;
    AXString str;
    LPCUSTR puReg;
    AXRectSize rcs;

    _trgroup(strid::GROUP_BRUSHITEM);

    pm = new AXMenu;

    puReg = _str(STRID_WATERMENU_REGIST);

    for(i = 0; i < CConfig::WATERPRESET_NUM; i++)
    {
        str = puReg;
        str += ' ';
        str += (char)('A' + i);

        pm->add(i, str);
    }

    pm->addSep();
    pm->add(100, _str(STRID_WATERMENU_RESET));

    //

    getWidget(WID_BT_WATERMENU)->getWindowRect(&rcs);

    id = pm->popup(NULL, rcs.x, rcs.y + rcs.h, 0);

    delete pm;

    //-----------------

    if(id == -1) return;

    if(id == 100)
    {
        //初期値に戻す

        g_conf->resetWaterPreset();
    }
    else
    {
        //登録

        g_conf->dwWaterPreset[id] = m_pbarWater[0]->getPos() |
                (m_pbarWater[1]->getPos() << 10) | (m_pbarWater[2]->getPos() << 20);
    }
}


//=====================================
// 各項目作成
//=====================================


//! メイン項目作成

void CBrushValueBase::_createMain()
{
    AXLayout *plm,*pl;
    AXExpander *pex;
    AXWindow *pwin;
    int i;
    AXString str;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_MAIN, MAKE_DW4(0,5,0,0), STRID_EXPANDER_MAIN));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(plm = new AXLayoutMatrix(2));

    //半径

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,3), STRID_RADIUS));

    plm->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 4));
    pl->setPaddingBottom(3);

    pl->addItem(m_pbarRadius = new CValBar(pex, 0, LF_EXPAND_W|LF_CENTER_Y, WID_BAR_RADIUS, 0,
                    1, CBrushItem::RADIUS_MIN, CBrushItem::RADIUS_MAX, CBrushItem::RADIUS_MIN));

    pl->addItem(pwin = new AXArrowButton(pex, AXArrowButton::ARBTS_DOWN, LF_CENTER_Y, WID_BT_RADIUS_MENU, 0));
    pwin->setMinSize(-15, -15);

    //濃度

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,3), STRID_OPACITY));
    plm->addItem(m_pbarOpacity = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_OPACITY, MAKE_DW4(0,0,0,3), 0, 1, 100, 100));

    //塗りタイプ

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,3), STRID_PIXTYPE));
    plm->addItem(m_pcbPix = new AXComboBox(pex, 0, LF_EXPAND_W, WID_CB_PIXTYPE, MAKE_DW4(0,0,0,3)));

    _trgroup(strid::GROUP_PIXTYPE);

    m_pcbPix->addItemMulTr(0, CBrushItem::PIXTYPE_NUM);

    _trgroup(strid::GROUP_BRUSHITEM);

    //補正

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), STRID_HOSEI));
    plm->addItem(pl = new AXLayoutHorz(LF_EXPAND_W, 4));

    pl->addItem(m_pcbHoseiType = new AXComboBox(pex, 0, 0, WID_CB_HOSEITYPE, 0));
    pl->addItem(m_pcbHoseiStr = new AXComboBox(pex, 0, LF_EXPAND_W, WID_CB_HOSEISTR, 0));

    m_pcbHoseiType->addItemMulTr(STRID_HOSEITYPE_TOP, 4);
    m_pcbHoseiType->setAutoWidth();

    for(i = 0; i <= CBrushItem::HOSEISTR_MAX; i++)
    {
        str.setInt(i + 1);
        m_pcbHoseiStr->addItem(str);
    }

    //

    pex->expand(g_conf->uBrushValExpand & 1);
}

//! 最小・間隔項目作成

void CBrushValueBase::_createMinInterval()
{
    AXLayout *pl;
    AXExpander *pex;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_MININTER, MAKE_DW4(0,5,0,0), STRID_EXPANDER_MININTER));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(pl = new AXLayoutVert(0, 3));

    //サイズ最小

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_MINSIZE));
    pl->addItem(m_pbarMinSize = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_MINSIZE, 0, 1, 0, 1000, 0));

    //濃度最小

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_MINOPACITY));
    pl->addItem(m_pbarMinOpa = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_MINOPA, 0, 1, 0, 1000, 0));

    //間隔

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_INTERVAL));
    pl->addItem(m_pbarInterval = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_INTERVAL, 0, 2, 5, 1000, 5));

    //

    pex->expand(g_conf->uBrushValExpand & 2);
}

//! ランダム項目作成

void CBrushValueBase::_createRandom()
{
    AXLayout *pl;
    AXExpander *pex;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_RANDOM, MAKE_DW4(0,5,0,0), STRID_EXPANDER_RANDOM));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(pl = new AXLayoutVert(0, 3));

    //ランダムサイズ

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_RANDSIZE));
    pl->addItem(m_pbarRandSize = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_RANDSIZE, 0, 1, 0, 1000, 0));

    //ランダム位置

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_RANDPOS));
    pl->addItem(m_pbarRandPos = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_RANDPOS, 0, 2, 0, 5000, 0));

    //

    pex->expand(g_conf->uBrushValExpand & 4);
}

//! 水彩項目作成

void CBrushValueBase::_createWater()
{
    AXLayout *pl,*plh;
    AXExpander *pex;
    AXWindow *pwin;
    int i;
    char m[2] = {0,0};

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_WATER, MAKE_DW4(0,5,0,0), STRID_EXPANDER_WATER));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(pl = new AXLayoutVert(0, 3));

    //ON

    pl->addItem(m_pckWaterOn = new AXCheckButton(pex, 0, 0, WID_CK_WATERON, 0, STRID_WATERON, FALSE));

    //プリセットボタン

    pl->addItem(plh = new AXLayoutHorz);

    plh->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), STRID_WATERPRESET));

    for(i = 0; i < 5; i++)
    {
        m[0] = 'A' + i;
        plh->addItem(pwin = new AXButton(pex, AXButton::BS_REAL_WH, 0, WID_BT_WATER_PRESET + i, 0, m));

        pwin->setMinSize(19, 19);
    }

    plh->addItem(new AXArrowButton(pex, AXArrowButton::ARBTS_DOWN, LF_CENTER_Y, WID_BT_WATERMENU, MAKE_DW4(5,0,0,0)));

    //バー

    for(i = 0; i < 3; i++)
    {
        pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_WATER1 + i));
        pl->addItem(m_pbarWater[i] = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_WATER1 + i, 0, 1, 0, 1000, 0));
    }

    //

    pex->expand(g_conf->uBrushValExpand & 8);
}

//! ブラシ形状項目作成

void CBrushValueBase::_createBrushStyle()
{
    AXLayout *plv,*plm;
    AXExpander *pex;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_BRUSHSTYLE, MAKE_DW4(0,5,0,0), STRID_EXPANDER_BRUSHSTYLE));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(plv = new AXLayoutVert(0, 3));

    //画像選択

    plv->addItem(m_pselBrushImg = new CSelImgBar(pex, 0, LF_EXPAND_W, WID_SEL_BRUSHIMG, 0, CSelImgBar::TYPE_BRUSH_BRUSH));

    //-------- 硬さ・荒さ・回転

    plv->addItem(plm = new AXLayoutMatrix(2, LF_EXPAND_W));

    //硬さ

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,3), STRID_BRUSHHARD));
    plm->addItem(m_pbarBrushHard = new CValBar(pex, 0, LF_CENTER_Y|LF_EXPAND_W, WID_BAR_BRUSHHARD, MAKE_DW4(0,0,0,3), 0, 0, 100, 0));

    //荒さ

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,3), STRID_BRUSHROUGH));
    plm->addItem(m_pbarBrushRough = new CValBar(pex, 0, LF_CENTER_Y|LF_EXPAND_W, WID_BAR_BRUSHROUGH, MAKE_DW4(0,0,0,3), 0, 0, 100, 0));

    //回転

    plm->addItem(new AXLabel(pex, 0, LF_CENTER_Y, MAKE_DW4(0,0,4,0), STRID_BRUSHROT));
    plm->addItem(m_pcbBrushRot = new AXComboBox(pex, 0, LF_CENTER_Y|LF_EXPAND_W, WID_CB_BRUSHROT, 0));

    m_pcbBrushRot->addItemMulTr(STRID_BRUSHROT_TYPE_TOP, 2);

    //--------

    //回転角度

    plv->addItem(new AXLabel(pex, 0, 0, 0, STRID_BRUSHROTANGLE));
    plv->addItem(m_pbarBrushRotAngle = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_BRUSHROTANGLE, 0, 0, 0, 359, 0));

    //回転ランダム幅

    plv->addItem(new AXLabel(pex, 0, 0, 0, STRID_BRUSHROTRAND));
    plv->addItem(m_pbarBrushRotRand = new CValBar(pex, 0, LF_EXPAND_W, WID_BAR_BRUSHROTRAND, 0, 0, 0, 180, 0));

    //

    pex->expand(g_conf->uBrushValExpand & 16);
}

//! 筆圧項目作成

void CBrushValueBase::_createPress()
{
    AXLayout *plh,*pl;
    AXExpander *pex;
    int i;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_PRESS, MAKE_DW4(0,5,0,0), STRID_EXPANDER_PRESS));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(plh = new AXLayoutHorz(0, 15));

    //サイズ/濃度

    for(i = 0; i < 2; i++)
    {
        plh->addItem(pl = new AXLayoutVert(0, 3));

        pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_PRESSSIZE + i));
        pl->addItem(m_pPressCurve[i] = new CPressCurve(pex, WID_PRESS_SIZE + i, 60));
        pl->addItem(m_peditPress[i] = new AXLineEdit(pex, AXLineEdit::ES_SPIN, 0, WID_EDIT_PRESSSIZE + i, 0));

        m_peditPress[i]->setWidthFromLen(5);
        m_peditPress[i]->setValStatus(CBrushItem::PRESS_MIN, CBrushItem::PRESS_MAX, 2);
    }

    //

    pex->expand(g_conf->uBrushValExpand & 32);
}

//! その他項目作成

void CBrushValueBase::_createEtc()
{
    AXLayout *pl;
    AXExpander *pex;
    int i;

    m_pLayout->addItem(pex = new AXExpander(this, AXExpander::EXPS_HEADER_DARK, LF_EXPAND_W, WID_EXP_ETC, MAKE_DW4(0,5,0,0), STRID_EXPANDER_ETC));

    pex->setInsidePadding(0,5,0,2);
    pex->setLayout(pl = new AXLayoutVert(0, 3));

    //テクスチャ

    pl->addItem(new AXLabel(pex, 0, 0, 0, STRID_TEXTURE));
    pl->addItem(m_pselTexture = new CSelImgBar(pex, 0, LF_EXPAND_W, WID_SEL_TEXTUREIMG, 0, CSelImgBar::TYPE_BRUSH_TEXTURE));

    //チェック

    for(i = 0; i < 2; i++)
        pl->addItem(m_pckEtc[i] = new AXCheckButton(pex, 0, 0, WID_CK_ANTIALIAS + i, 0, STRID_ANTIALIAS + i, FALSE));

    //

    pex->expand(g_conf->uBrushValExpand & 64);
}
