/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CBrushWin_brushList - ブラシウィンドウのブラシリスト部分
*/


#include <stdlib.h>

#include "CBrushWin_brushList.h"

#include "CBrushList.h"
#include "CBrushItem.h"
#include "CXImage.h"
#include "cursor.h"

#include "AXLayout.h"
#include "AXScrollBar.h"
#include "AXMenu.h"
#include "AXMessageBox.h"
#include "AXStrDialog.h"
#include "AXApp.h"
#include "AXClipboard.h"

#include "strid.h"


//----------------------------

class CBrushListArea:public AXWindow
{
protected:
    enum
    {
        ONE_W   = 41,
        ONE_H   = 35,
        GROUP_W = 15,

        COL_BKGND       = 0xbbbbbb,
        COL_ITEMEX      = 0xdddddd,
        COL_ITEM_NORMAL = 0xffffff,
        COL_BK_SEL      = _RGB(255,220,220),

        TIMERID_RESIZE = 0
    };

    enum
    {
        AREA_NONE,
        AREA_GROUPEXPAND,
        AREA_GROUPMENU,
        AREA_ITEM,
        AREA_ITEMEX
    };

    enum
    {
        DOWNF_NONE,
        DOWNF_DNDFIRST,
        DOWNF_DND
    };

    typedef struct
    {
        CBrushGroupItem *pGroup;
        CBrushItem      *pItem;
        int     x,y;
    }PTINFO;

protected:
    AXScrollBar *m_pScr;

    CXImage     m_img;

    int     m_nXCnt,
            m_fDown;
    AXPoint m_ptDown;
    PTINFO  m_infoMoveSrc,
            m_infoMoveDst;

protected:
    void _notify();
    int _getPtInfo(int x,int y,PTINFO *pinfo);
    void _drawSel(int x,int y,BOOL bErase);
    void _runGroupMenu(int x,int y,CBrushGroupItem *pGroup);
    void _runBrushExMenu(int x,int y,const PTINFO &info);
    void _runBrushMenu(int x,int y,const PTINFO &info);
    void _cmd_rename(CBrushItem *pItem);

public:
    CBrushListArea(AXWindow *pParent);

    int getXCnt() { return m_nXCnt; }
    void setScroll(AXScrollBar *p) { m_pScr = p; }
    void setScrollInfo();
    void updateAll();
    void draw();

    virtual BOOL onPaint(AXHD_PAINT *phd);
    virtual BOOL onSize();
    virtual BOOL onButtonDown(AXHD_MOUSE *phd);
    virtual BOOL onMouseMove(AXHD_MOUSE *phd);
    virtual BOOL onButtonUp(AXHD_MOUSE *phd);
    virtual BOOL onDblClk(AXHD_MOUSE *phd);
    virtual BOOL onMouseWheel(AXHD_MOUSE *phd,BOOL bUp);
    virtual BOOL onTimer(UINT uTimerID,ULONG lParam);
};

//----------------------------


//************************************
// CBrushWin_brushList
//************************************



CBrushWin_brushList::CBrushWin_brushList(AXWindow *pParent,int h,AXFont *pFont)
    : AXWindow(pParent, 0, LF_EXPAND_W | LF_FIX_H)
{
    AXLayout *pl;

    m_nH = h;

    //

    setLayout(pl = new AXLayoutHorz);

    pl->addItem(m_pArea = new CBrushListArea(this));
    pl->addItem(m_pScrV = new AXScrollBar(this, AXScrollBar::SBS_VERT, LF_EXPAND_H));

    m_pArea->setFont(pFont);
    m_pArea->setScroll(m_pScrV);
    m_pArea->setNotify(pParent);
}

//! 通知

BOOL CBrushWin_brushList::onNotify(AXWindow *pwin,UINT uNotify,ULONG lParam)
{
    if(pwin == m_pScrV)
        m_pArea->draw();

    return TRUE;
}

//! リスト部分描画

void CBrushWin_brushList::drawArea()
{
    m_pArea->draw();
}

//! ブラシ選択を左右に移動

BOOL CBrushWin_brushList::moveSelBrush(int dir)
{
    CBrushItem *p,*p2;
    int cnt;

    p = BRUSHLIST->getSelItem();

    if(dir == 0)
        p = p->prev();
    else
    {
        for(p2 = p, cnt = 0; p2; p2 = p2->prev(), cnt++);

        p = p->next();

        if(p && !(p->pGroup->m_bExpand) && cnt >= m_pArea->getXCnt())
            p = NULL;
    }

    if(p)
    {
        if(BRUSHLIST->changeSelItem(p))
        {
            m_pArea->draw();
            return TRUE;
        }
    }

    return FALSE;
}



//************************************
// CBrushListArea
//************************************



CBrushListArea::CBrushListArea(AXWindow *pParent)
    : AXWindow(pParent, 0, LF_EXPAND_WH)
{
    m_fDown = DOWNF_NONE;
    m_nXCnt = 1;
}

//! スクロール情報セット

void CBrushListArea::setScrollInfo()
{
    CBrushGroupItem *pgroup;
    int cnt = 0,page;

    //表示縦数

    for(pgroup = BRUSHLIST->getTopItem(); pgroup; pgroup = pgroup->next())
        cnt += pgroup->getViewAreaHCnt(m_nXCnt);

    //

    page = (m_nH - 1) / ONE_H;
    if(page <= 0) page = 1;

    m_pScr->setStatus(0, cnt, page);
}

//! すべて更新

void CBrushListArea::updateAll()
{
    setScrollInfo();
    draw();
}

//! 通知

void CBrushListArea::_notify()
{
    getNotify()->onNotify(m_pParent, 0, 0);
}


//========================
// ハンドラ
//========================


//! 描画

BOOL CBrushListArea::onPaint(AXHD_PAINT *phd)
{
    int w,h;

    w = phd->w, h = phd->h;

    if(w > m_img.getWidth()) w = m_img.getWidth();
    if(h > m_img.getHeight()) h = m_img.getHeight();

    m_img.put(m_id, phd->x, phd->y, phd->x, phd->y, w, h);

    return TRUE;
}

//! サイズ変更時

BOOL CBrushListArea::onSize()
{
    //横の表示数

    m_nXCnt = (m_nW - GROUP_W - 1) / ONE_W;
    if(m_nXCnt < 1) m_nXCnt = 1;

    addTimer(TIMERID_RESIZE, 5);

    return TRUE;
}

//! タイマー

BOOL CBrushListArea::onTimer(UINT uTimerID,ULONG lParam)
{
    m_img.recreate(m_nW, m_nH, 32, 32);
    updateAll();

    delTimer(uTimerID);

    return TRUE;
}

//! マウスホイール

BOOL CBrushListArea::onMouseWheel(AXHD_MOUSE *phd,BOOL bUp)
{
    if(m_pScr->movePos((bUp)? -1: 1))
        draw();

    return TRUE;
}

//! ダブルクリック時

BOOL CBrushListArea::onDblClk(AXHD_MOUSE *phd)
{
    PTINFO info;
    int area;

    //アイテム、名前変更

    if(!m_fDown && phd->button == BUTTON_LEFT)
    {
        area = _getPtInfo(phd->x, phd->y, &info);

        if(area == AREA_ITEM)
        {
            _cmd_rename(info.pItem);
            return TRUE;
        }
    }

    return FALSE;
}

//! ボタン押し時

BOOL CBrushListArea::onButtonDown(AXHD_MOUSE *phd)
{
    int area;
    PTINFO info;

    if(m_fDown) return TRUE;

    area = _getPtInfo(phd->x, phd->y, &info);

    switch(area)
    {
        //グループ開閉
        case AREA_GROUPEXPAND:
            if(phd->button == BUTTON_LEFT)
            {
                info.pGroup->m_bExpand = !info.pGroup->m_bExpand;
                updateAll();
            }
            break;
        //グループメニュー
        case AREA_GROUPMENU:
            if(phd->button == BUTTON_LEFT)
                _runGroupMenu(phd->rootx, phd->rooty, info.pGroup);
            break;
        //ブラシ空欄部分
        case AREA_ITEMEX:
            if(phd->button == BUTTON_RIGHT)
                _runBrushExMenu(phd->rootx, phd->rooty, info);
            break;
        //ブラシ
        case AREA_ITEM:
            if(phd->button == BUTTON_RIGHT)
                _runBrushMenu(phd->rootx, phd->rooty, info);
            else if(phd->button == BUTTON_LEFT)
            {
                //選択変更

                if(BRUSHLIST->changeSelItem(info.pItem))
                {
                    draw();
                    _notify();
                }

                //D&D開始

                m_fDown    = DOWNF_DNDFIRST;
                m_ptDown.x = phd->x;
                m_ptDown.y = phd->y;

                m_infoMoveSrc = m_infoMoveDst = info;

                grabPointer();
            }
            break;
    }

    return TRUE;
}

//! ボタン離し時

BOOL CBrushListArea::onButtonUp(AXHD_MOUSE *phd)
{
    if(m_fDown && phd->button == BUTTON_LEFT)
    {
        //DND移動

        if(m_fDown == DOWNF_DND)
        {
            BRUSHLIST->moveBrush(m_infoMoveSrc.pGroup, m_infoMoveSrc.pItem,
                    m_infoMoveDst.pGroup, m_infoMoveDst.pItem);

            updateAll();

            unsetCursor();
        }

        //

        m_fDown = DOWNF_NONE;
        ungrabPointer();
    }

    return TRUE;
}

//! マウス移動

BOOL CBrushListArea::onMouseMove(AXHD_MOUSE *phd)
{
    int area;
    PTINFO info;

    if(m_fDown == DOWNF_NONE) return TRUE;

    //D&D開始判定

    if(m_fDown == DOWNF_DNDFIRST)
    {
        if(::abs(phd->x - m_ptDown.x) > 10 || ::abs(phd->y - m_ptDown.y) > 10)
        {
            m_fDown = DOWNF_DND;

            setCursor(cursor::getImgDat(cursor::ITEMMOVE));
        }
    }

    //D&D中

    if(m_fDown == DOWNF_DND)
    {
        area = _getPtInfo(phd->x, phd->y, &info);

        if((area == AREA_ITEM || area == AREA_ITEMEX) &&
            (m_infoMoveDst.x != info.x || m_infoMoveDst.y != info.y))
        {
            //挿入線

            m_img.lineV(m_infoMoveDst.x - 1, m_infoMoveDst.y - 1, ONE_H + 2, COL_BKGND);
            m_img.lineV(info.x - 1, info.y - 1, ONE_H + 2, 0xff0000);
            redraw();

            //

            m_infoMoveDst = info;
        }
    }

    return TRUE;
}


//========================
// コマンド
//========================


//! 名前の変更

void CBrushListArea::_cmd_rename(CBrushItem *pItem)
{
    AXString str;

    str = pItem->strName;

    if(AXStrDialog::getString(m_pTopLevel,
        _string(strid::GROUP_BRUSHLIST, strid::BRUSHLIST_RENAME_TITLE),
        _string(strid::GROUP_WORD, strid::WORD_NAME), &str))
    {
        pItem->strName = str;
        draw();

        //選択アイテムなら、編集用データも変更＆名前の表示を更新

        if(BRUSHLIST->getSelItem() == pItem)
        {
            (BRUSHLIST->getEditItem())->strName = str;
            _notify();
        }
    }
}


//========================
// メニュー
//========================


//! グループメニュー実行

void CBrushListArea::_runGroupMenu(int x,int y,CBrushGroupItem *pGroup)
{
    AXMenu *pm;
    int id;

    _trgroup(strid::GROUP_BRUSHLIST);

    //

    pm = new AXMenu;

    pm->addTrMul(strid::BRUSHLIST_MENU_GROUP, 4);

    if(BRUSHLIST->getGroupCnt() == 1) pm->enable(strid::BRUSHLIST_MENU_GROUP + 1, FALSE);
    if(!pGroup->prev()) pm->enable(strid::BRUSHLIST_MENU_GROUP + 2, FALSE);
    if(!pGroup->next()) pm->enable(strid::BRUSHLIST_MENU_GROUP + 3, FALSE);

    id = pm->popup(NULL, x, y, 0);

    delete pm;

    //-----------

    if(id == -1) return;

    id -= strid::BRUSHLIST_MENU_GROUP;

    switch(id)
    {
        //新規グループ
        case 0:
            BRUSHLIST->addNewGroup();
            break;
        //削除
        case 1:
            if(AXMessageBox::message(m_pTopLevel, NULL, _str(strid::BRUSHLIST_MES_GROUPDEL),
                    AXMessageBox::YESNO, AXMessageBox::NO) == AXMessageBox::YES)
            {
                if(BRUSHLIST->deleteGroup(pGroup))
                    _notify();
            }
            break;
        //上へ/下へ
        case 2:
        case 3:
            BRUSHLIST->moveGroup(pGroup, (id == 2));
            break;
    }

    updateAll();
}

//! ブラシ空欄部分のメニュー実行

void CBrushListArea::_runBrushExMenu(int x,int y,const PTINFO &info)
{
    AXMenu *pm;
    int id;

    _trgroup(strid::GROUP_BRUSHLIST);

    pm = new AXMenu;

    pm->addTrMul(strid::BRUSHLIST_MENU_BRUSHEX, 2);

    _drawSel(info.x, info.y, FALSE);

    id = pm->popup(NULL, x, y, 0);

    _drawSel(info.x, info.y, TRUE);

    delete pm;

    //-----------

    if(id == -1) return;

    id -= strid::BRUSHLIST_MENU_BRUSHEX;

    if(id == 0)
        //新規追加
        BRUSHLIST->addNewBrush(info.pGroup);
    else
    {
        //貼り付け

        AXString str;

        if(axclipb->getText(m_id, &str))
            BRUSHLIST->pasteBrush(info.pGroup, NULL, str);
    }

    updateAll();
    _notify();
}

//! ブラシのメニュー実行

void CBrushListArea::_runBrushMenu(int x,int y,const PTINFO &info)
{
    AXMenu *pm;
    int id,bRegBrush;
    WORD wID[] = {
        1200, 0xffff, 1201, 0xffff, 1202,1203, 0xffff, 1204,1205
    };
    AXString str;
    CBrushItem *pItem;

    //

    _trgroup(strid::GROUP_BRUSHLIST);

    pm = new AXMenu;

    pm->addTrArray(wID, sizeof(wID)/sizeof(WORD));

    bRegBrush = BRUSHLIST->isItemRegBrush(info.pItem);

    pm->enable(1204, !bRegBrush);
    pm->enable(1205, bRegBrush);

    //

    _drawSel(info.x, info.y, FALSE);

    id = pm->popup(NULL, x, y, 0);

    _drawSel(info.x, info.y, TRUE);

    delete pm;

    //-----------

    if(id == -1) return;

    id -= strid::BRUSHLIST_MENU_BRUSH;

    switch(id)
    {
        //名前の変更
        case 0:
            _cmd_rename(info.pItem);
            break;
        //削除
        case 1:
            if(BRUSHLIST->deleteBrush(info.pGroup, info.pItem))
                _notify();

            updateAll();
            break;
        //コピー
        case 2:
            /* 選択アイテム時は、編集用データをコピーする */

            pItem = BRUSHLIST->getCopyBrush(info.pItem);

            if(pItem->getTextFormat(&str))
                axclipb->setText(m_id, str);
            break;
        //上書き貼り付け
        case 3:
            if(axclipb->getText(m_id, &str))
            {
                if(BRUSHLIST->pasteBrush(info.pGroup, info.pItem, str))
                {
                    draw();
                    _notify();
                }
            }
            break;
        //登録ブラシ指定
        case 4:
            BRUSHLIST->changeRegItem(info.pItem);
            draw();
            break;
        //登録ブラシ解除
        case 5:
            BRUSHLIST->changeRegItem(NULL);
            draw();
            break;
    }
}


//========================
// サブ
//========================


//! マウス位置からアイテム・情報取得

int CBrushListArea::_getPtInfo(int x,int y,PTINFO *pinfo)
{
    CBrushGroupItem *pgroup;
    int hcnt,ytop,nx,ny,no;

    pinfo->pGroup = NULL;
    pinfo->pItem  = NULL;

    ytop = 1 - m_pScr->getPos() * ONE_H;

    for(pgroup = BRUSHLIST->getTopItem(); pgroup && ytop < m_nH; pgroup = pgroup->next())
    {
        hcnt = pgroup->getViewAreaHCnt(m_nXCnt);

        //グループ範囲内

        if(ytop <= y && y < ytop + hcnt * ONE_H)
        {
            pinfo->pGroup = pgroup;

            if(x < GROUP_W)
            {
                //グループ操作部分

                if(y < ytop + 12)
                    return AREA_GROUPEXPAND;
                else if(x >= 2 && x < 2 + 11 && y >= ytop + 16 && y < ytop + 16 + 11)
                    return AREA_GROUPMENU;
                else
                    return AREA_NONE;
            }
            else
            {
                //アイテム範囲

                nx = (x - GROUP_W) / ONE_W;
                ny = (y - ytop) / ONE_H;

                if(nx >= m_nXCnt) return AREA_NONE;

                no = nx + ny * m_nXCnt;

                pinfo->x = GROUP_W + nx * ONE_W;
                pinfo->y = ytop + ny * ONE_H;

                if(no > pgroup->getItemCnt())
                    return AREA_NONE;
                else if(no == pgroup->getItemCnt())
                    return AREA_ITEMEX;
                else
                {
                    pinfo->pItem = pgroup->getNoItem(no);
                    return AREA_ITEM;
                }
            }
        }

        //

        ytop += hcnt * ONE_H;
    }

    return AREA_NONE;
}

//! 選択枠描画

void CBrushListArea::_drawSel(int x,int y,BOOL bErase)
{
    if(bErase)
        m_img.box(x - 1, y - 1, ONE_W + 1, ONE_H + 1, COL_BKGND);
    else
        m_img.drawDashBox(x - 1, y - 1, ONE_W + 1, ONE_H + 1);

    redraw(x - 1, y - 1, ONE_W + 1, ONE_H + 1);
}

//! 全体描画

void CBrushListArea::draw()
{
    CBrushGroupItem *pgroup;
    CBrushItem *p;
    int ytop,hcnt,no,x,y,bSel;
    BYTE pat[7] = { 0xfe,0x82,0xfe,0x00,0x7c,0x38,0x10 };

    m_img.clear(COL_BKGND);

    //

    pgroup = BRUSHLIST->getTopItem();
    ytop   = 1 - m_pScr->getPos() * ONE_H;

    for(; pgroup && ytop < m_nH; pgroup = pgroup->next(), ytop += hcnt * ONE_H)
    {
        hcnt = pgroup->getViewAreaHCnt(m_nXCnt);

        //表示範囲外

        if(ytop + hcnt * ONE_H < 1) continue;

        //--------- グループ左

        m_img.fillBox(1, ytop, GROUP_W - 2, hcnt * ONE_H - 1, 0xffffff);

        //展開矢印

        if(pgroup->m_bExpand)
        {
            //下矢印
            for(x = 0; x < 4; x++)
                m_img.lineH(GROUP_W / 2 - x, ytop + 3 + 5 - x, (x << 1) + 1, 0);
        }
        else
        {
            //右矢印
            for(x = 0; x < 4; x++)
                m_img.lineV(GROUP_W / 2 + 1 - x, ytop + 3 + 4 - x, (x << 1) + 1, 0);
        }

        //メニューボタン

        m_img.lineH(2, ytop + 16, 10, _RGB(187,187,187));
        m_img.lineV(2, ytop + 17, 9, _RGB(187,187,187));

        m_img.lineH(2, ytop + 26, 11, _RGB(68,68,68));
        m_img.lineV(12, ytop + 16, 11, _RGB(68,68,68));

        m_img.drawPattern(4, ytop + 18, 0, pat, 7, 7);

        //--------- アイテム

        x  = GROUP_W;
        y  = ytop;
        no = 0;

        for(p = pgroup->getTopItem(); 1; p = p->next())
        {
            //新規用の拡張分

            if(!p)
            {
                m_img.fillBox(x, y, ONE_W - 1, ONE_H - 1, COL_ITEMEX);
                break;
            }

            //範囲内だけ描画

            if(y >= 0 && y < m_nH)
            {
                bSel = (p == BRUSHLIST->getSelItem());

                //背景

                m_img.fillBox(x, y, ONE_W - 1, ONE_H - 1, (bSel)? COL_BK_SEL: COL_ITEM_NORMAL);

                //選択枠

                if(bSel)
                    m_img.box(x, y, ONE_W - 1, ONE_H - 1, 0xff0000);

                //登録ブラシ枠

                if(BRUSHLIST->isItemRegBrush(p))
                    m_img.box(x + 1, y + 1, ONE_W - 3, ONE_H - 3, 0x0000ff);

                //名前

                m_img.drawTextWrap(*m_pFont, x + 2, y + 2, ONE_W - 5, ONE_H - 5,
                    p->strName, p->strName.getLen(), (bSel)? _RGB(200,0,0): 0);
            }

            //次へ

            if(no == m_nXCnt - 1)
            {
                if(!pgroup->m_bExpand) break;

                x = GROUP_W;
                y += ONE_H;
                no = 0;
            }
            else
            {
                x += ONE_W;
                no++;
            }
        }
    }

    redraw();
}
