/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    AzPainter - main
*/


#include <unistd.h>

#include "CApp.h"
#include "CDevList.h"
#include "CConfig.h"
#include "CResource.h"
#include "CPalette.h"
#include "CGradList.h"
#include "CBrushSizeList.h"
#include "CBrushList.h"
#include "CKeyDat.h"
#include "CFilterConfigList.h"

#define _DEF_NOEXTERN
#include "global.h"

#include "drawdat.h"
#include "draw_main.h"
#include "draw_update.h"

#include "CMainWin.h"
#include "CCanvasWin.h"
#include "CToolWin.h"
#include "CBrushWin.h"
#include "CLayerWin.h"
#include "COptionWin.h"
#include "CColorWin.h"
#include "CPrevWin.h"
#include "CImgViewWin.h"
#include "CFilterListWin.h"

#include "AXByteString.h"
#include "AXConfRead.h"
#include "AXConfWrite.h"
#include "AXFreeType.h"
#include "AXFile.h"
#include "AXUtilFile.h"

#include "deflang.h"

//----------------------------

#ifndef _AZPT_RESDIR
#define _AZPT_RESDIR "/usr/share/azpainter"
#endif

//----------------------------

CDevList          g_devlist;
CPaletteList      g_pallist;
CGradList         g_gradlist;
CBrushSizeList    g_brushsizelist;
CBrushList        g_brushlist;
CKeyDat           g_keydat;
CFilterConfigList g_filterconf;
AXFreeType        g_freetype;

//----------------------------

void finish();
void init();
void initCreateWindow();
void initShowWindow();
void copyDefConfig(LPCSTR szFileName);
void createConfig();
void loadConfig();
void saveConfig();

//----------------------------


//! メイン

int main(int argc,char **argv)
{
    CApp app;

    //App 初期化
    
    app.setConfigDirHome(".azpainter");
    app.setResourceDir(_AZPT_RESDIR);

    if(!app.init(argc, argv)) return 1;

    app.loadTranslation(g_deflangdat);
    app.init2();

    //初期化

    init();

    app.initXI(CANVASAREA);

    //コマンドラインファイル

    if(optind < argc)
    {
        AXString str;

        str.setLocal(argv[optind]);
        MAINWIN->loadImage(str);
    }

    //実行

    app.run();

    //--------- 終了

    finish();

    app.end2();     //作業用ディレクトリ削除
    app.end();

    return 0;
}

//! 終了処理

void finish()
{
    //設定保存

    saveConfig();

    //削除

    delete g_res;
    delete g_draw;
    delete g_conf;
}

//! 初期化

void init()
{
    createConfig();

    //作成

    g_conf = new CConfig;
    g_draw = new draw::MAINDAT;

    //XInput デバイスリストセット

    CAPP->setDevice();

    //設定読み込み

    loadConfig();

    //リソース作成（※設定読み込み後）

    g_res = new CResource;

    //デフォルトフォントセット
    /* フォントは、ウィンドウ作成時にデフォルトでルートのフォントが継承されるので、
       ここでセットしておけばすべてこのフォントになる。 */

    if(g_res->pfontMain)
        (CAPP->getRootWindow())->setFont(g_res->pfontMain);

    //初期化

    draw::initAfterConfig();

	//ウィンドウ作成
	
	initCreateWindow();

    //新規イメージ

    draw::newImage(g_conf->nInitImgW, g_conf->nInitImgH, 300, TRUE);

    draw::updateImage();

    LAYERWIN->updateInfoParam();

    //ウィンドウ表示

    initShowWindow();
}

//! ウィンドウ作成

void initCreateWindow()
{
    //各ウィンドウ作成
    
    UINT addstyle =
		(g_conf->btToolboxType == CConfig::TOOLBOXTYPE_NORMAL)?
			AXWindow::WS_MENUBTT: AXWindow::WS_TRANSIENT_FOR; 

    new CMainWin;
    new CToolWin(MAINWIN, addstyle);
    new COptionWin(MAINWIN, addstyle);
    new CLayerWin(MAINWIN, addstyle);
    new CBrushWin(MAINWIN, addstyle);
    new CColorWin(MAINWIN, addstyle);
    new CPrevWin(MAINWIN, addstyle);
    new CImgViewWin(MAINWIN, addstyle);
    new CFilterListWin(MAINWIN, addstyle);
    
    //ウィンドウタイプセット
    
    long atom = axapp->getAtom(
		(g_conf->btToolboxType == CConfig::TOOLBOXTYPE_NORMAL)?
			"_NET_WM_WINDOW_TYPE_NORMAL": "_NET_WM_WINDOW_TYPE_UTILITY");
	
	TOOLWIN->setWindowType(atom);
	OPTIONWIN->setWindowType(atom);
	LAYERWIN->setWindowType(atom);
	BRUSHWIN->setWindowType(atom);
	COLORWIN->setWindowType(atom);
	PREVWIN->setWindowType(atom);
	IMGVIEWWIN->setWindowType(atom);
	FILTERLISTWIN->setWindowType(atom);
}

//! ウィンドウ初期表示

void initShowWindow()
{
    AXRectSize rcs;

    //メインウィンドウ

    MAINWIN->showMainWin();

    //ツール

    TOOLWIN->showInit(g_conf->ptToolWin, -1, -1, CConfig::WININITVAL, g_conf->isToolWin());

    //オプション

    rcs.set(-1, -1, 220, 150);
    OPTIONWIN->showInit(g_conf->rcsOptWin, rcs, CConfig::WININITVAL, g_conf->isOptionWin());

    //ブラシ

    rcs.set(-1, -1, 220, 380);
    BRUSHWIN->showInit(g_conf->rcsBrushWin, rcs, CConfig::WININITVAL, g_conf->isBrushWin());

    //レイヤ

    rcs.set(-1, -1, 200, 300);
    LAYERWIN->showInit(g_conf->rcsLayerWin, rcs, CConfig::WININITVAL, g_conf->isLayerWin());

    //カラー

    rcs.set(-1, -1, 220, 192);
    COLORWIN->showInit(g_conf->rcsColWin, rcs, CConfig::WININITVAL, g_conf->isColorWin());

    //プレビュー

    rcs.set(-1, -1, 220, 220);
    PREVWIN->showInit(g_conf->rcsPrevWin, rcs, CConfig::WININITVAL, g_conf->isPrevWin());
    PREVWIN->draw();

    //イメージビュー

    rcs.set(-1, -1, 200, 200);
    IMGVIEWWIN->showInit(g_conf->rcsImgViewWin, rcs, CConfig::WININITVAL, g_conf->isImgViewWin());

    //フィルタ一覧

    rcs.set(-1, -1, 200, 200);
    FILTERLISTWIN->showInit(g_conf->rcsFilterListWin, rcs, CConfig::WININITVAL, g_conf->isFilterWin());
}

//! デフォルト設定ファイルをコピー

void copyDefConfig(LPCSTR szFileName)
{
    AXString strSrc,strDst;
    AXByteString strb;

    strb = "def_";
    strb += szFileName;

    axapp->getResourcePath(&strSrc, strb);
    axapp->getConfigPath(&strDst, szFileName);

    AXFile::copyFile(strSrc, strDst);
}

//! 設定ファイルディレクトリなど作成

void createConfig()
{
    AXString str;

    //ディレクトリが存在するか

    axapp->getConfigPath(&str, "");

    if(AXIsExistFile(str, TRUE)) return;

    //----------

    //ディレクトリ作成

    axapp->createConfigDir();

    //デフォルト設定ファイルをコピー

    copyDefConfig("brush.dat");
    copyDefConfig("grad.dat");
    copyDefConfig("palette.dat");

    //brush ディレクトリ作成

    axapp->getConfigPath(&str, "brush");
    AXCreateDir(str);

    //texture ディレクトリ作成

    axapp->getConfigPath(&str, "texture");
    AXCreateDir(str);
}


//============================
// 設定ファイル
//============================


//! 設定ファイル読み込み

void loadConfig()
{
    AXConfRead cf;
    AXString str;

    //-------- main.conf

    axapp->getConfigPath(&str, "main.conf");

    cf.loadFile(str);

    //バージョン

    cf.setGroup("main");

    if(cf.getInt("ver", 0) != 1)
        cf.free();

    //CConfig

    g_conf->load(&cf);

    //draw

    g_draw->loadConfig(&cf);

    //デバイス設定

    DEVLIST->loadConfig(&cf);

    //フィルタ設定

    FILTERCONF->loadConfig(&cf);

    cf.free();

    //---------- ブラシ

    axapp->getConfigPath(&str, "brush.dat");

    BRUSHLIST->loadFile(str);
    BRUSHLIST->setDefault();

    //--------- パレット

    axapp->getConfigPath(&str, "palette.dat");

    PALETTE->loadFile(str);
    PALETTE->setDefault();

    //---------- グラデーション

    axapp->getConfigPath(&str, "grad.dat");

    GRADLIST->loadFile(str);

    //--------- キーデータ

    axapp->getConfigPath(&str, "key.dat");

    g_keydat.loadFile(str);
}

//! 設定ファイル書き込み

void saveConfig()
{
    AXConfWrite cf;
    AXString str;

    //-------------- ウィンドウ位置・サイズ

    MAINWIN->getTopWindowRect(&g_conf->rcsMainWin);
    g_conf->bMaximized = MAINWIN->isMaximized();

    TOOLWIN->getTopWindowPos(&g_conf->ptToolWin);
    OPTIONWIN->getTopWindowRect(&g_conf->rcsOptWin);
    LAYERWIN->getTopWindowRect(&g_conf->rcsLayerWin);
    BRUSHWIN->getTopWindowRect(&g_conf->rcsBrushWin);
    COLORWIN->getTopWindowRect(&g_conf->rcsColWin);
    PREVWIN->getTopWindowRect(&g_conf->rcsPrevWin);
    IMGVIEWWIN->getTopWindowRect(&g_conf->rcsImgViewWin);
    FILTERLISTWIN->getTopWindowRect(&g_conf->rcsFilterListWin);

    BRUSHWIN->getWinH();

    //-------------- main.conf

    axapp->getConfigPath(&str, "main.conf");

    if(!cf.open(str)) return;

    //バージョン

    cf.putGroup("main");
    cf.putInt("ver", 1);

    //CConfig

    g_conf->save(&cf);

    //draw

    g_draw->saveConfig(&cf);

    //デバイス設定

    DEVLIST->saveConfig(&cf);

    //フィルタ設定

    FILTERCONF->saveConfig(&cf);

    cf.close();

    //-------------- ブラシ

    axapp->getConfigPath(&str, "brush.dat");

    BRUSHLIST->saveFile(str);

    //-------------- パレット

    axapp->getConfigPath(&str, "palette.dat");

    PALETTE->saveFile(str);

    //------------- グラデーション

    axapp->getConfigPath(&str, "grad.dat");

    GRADLIST->saveFile(str);

    //-------------- キーデータ

    axapp->getConfigPath(&str, "key.dat");

    g_keydat.saveFile(str);
}
