/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    色の合成関数（背景とレイヤの合成時）

    src : 合成する色
    dst : 合成先の色

    # http://www.pegtop.net/delphi/articles/blendmodes/
    # http://dunnbypaul.net/blends/
    # http://web.archive.org/web/20110830103044/http://www.bea.hi-ho.ne.jp/gaku-iwa/color/conjn.html
*/

#include "struct.h"


namespace blendcol
{

//! 通常

void normal(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{

}

//! 乗算

void mul(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    psrc->r = psrc->r * dst.r >> 15;
    psrc->g = psrc->g * dst.g >> 15;
    psrc->b = psrc->b * dst.b >> 15;
}

//! 加算

void add(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = psrc->c[i] + dst.c[i];
        if(n > 0x8000) n = 0x8000;

        psrc->c[i] = n;
    }
}

//! 減算

void sub(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = dst.c[i] - psrc->c[i];
        if(n < 0) n = 0;

        psrc->c[i] = n;
    }
}

//! スクリーン

void screen(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i;

    for(i = 0; i < 3; i++)
        psrc->c[i] = psrc->c[i] + dst.c[i] - (psrc->c[i] * dst.c[i] >> 15);
}

//! オーバーレイ

void overlay(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i;

    for(i = 0; i < 3; i++)
    {
        if(dst.c[i] < 0x4000)
            psrc->c[i] = psrc->c[i] * dst.c[i] >> 14;
        else
            psrc->c[i] = 0x8000 - ((0x8000 - dst.c[i]) * (0x8000 - psrc->c[i]) >> 14);
    }
}

//! ハードライト

void hardlight(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i;

    for(i = 0; i < 3; i++)
    {
        if(psrc->c[i] < 0x4000)
            psrc->c[i] = psrc->c[i] * dst.c[i] >> 14;
        else
            psrc->c[i] = 0x8000 - ((0x8000 - dst.c[i]) * (0x8000 - psrc->c[i]) >> 14);
    }
}

//! ソフトライト

void softlight(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = psrc->c[i] * dst.c[i] >> 15;

        psrc->c[i] = n + (dst.c[i] * (0x8000 - n - ((0x8000 - psrc->c[i]) * (0x8000 - dst.c[i]) >> 15)) >> 15);
    }
}

//! 覆い焼き

void dodge(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        if(psrc->c[i] == 0x8000)
            n = 0x8000;
        else
        {
            n = (dst.c[i] << 15) / (0x8000 - psrc->c[i]);
            if(n > 0x8000) n = 0x8000;
        }

        psrc->c[i] = n;
    }
}

//! 焼き込み

void burn(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        if(psrc->c[i] == 0)
            n = 0;
        else
        {
            n = 0x8000 - ((0x8000 - dst.c[i]) << 15) / psrc->c[i];
            if(n < 0) n = 0;
        }

        psrc->c[i] = n;
    }
}

//! 焼き込みリニア

void linearburn(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = dst.c[i] + psrc->c[i] - 0x8000;
        if(n < 0) n = 0;

        psrc->c[i] = n;
    }
}

//! ビビットライト

void vividlight(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        if(psrc->c[i] < 0x4000)
        {
            n = 0x8000 - (psrc->c[i] << 1);

            if(dst.c[i] <= n || psrc->c[i] == 0)
                n = 0;
            else
                n = ((dst.c[i] - n) << 15) / (psrc->c[i] << 1);
        }
        else
        {
            n = 0x10000 - (psrc->c[i] << 1);

            if(dst.c[i] >= n || n == 0)
                n = 0x8000;
            else
                n = (dst.c[i] << 15) / n;
        }

        psrc->c[i] = n;
    }
}

//! リニアライト

void linearlight(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = (psrc->c[i] << 1) + dst.c[i] - 0x8000;

        if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

        psrc->c[i] = n;
    }
}

//! ピンライト

void pinlight(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        if(psrc->c[i] > 0x4000)
        {
            n = (psrc->c[i] << 1) - 0x8000;
            if(n < dst.c[i]) n = dst.c[i];
        }
        else
        {
            n = psrc->c[i] << 1;
            if(n > dst.c[i]) n = dst.c[i];
        }

        psrc->c[i] = n;
    }
}

//! 比較（暗）

void darken(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i;

    for(i = 0; i < 3; i++)
    {
        if(dst.c[i] < psrc->c[i])
            psrc->c[i] = dst.c[i];
    }
}

//! 比較（明）

void lighten(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i;

    for(i = 0; i < 3; i++)
    {
        if(dst.c[i] > psrc->c[i])
            psrc->c[i] = dst.c[i];
    }
}

//! 差の絶対値

void difference(RGBAFIX15 *psrc,const RGBFIX15 &dst)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = psrc->c[i] - dst.c[i];
        if(n < 0) n = -n;

        psrc->c[i] = n;
    }
}

};
