/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CXImage [AXImage の派生クラス。キャンバス用などに使う]
*/


#include <string.h>
#include <math.h>

#include "CXImage.h"

#include "struct.h"

#include "AXApp.h"

#include "alphabetimg.h"



//! チェック柄塗りつぶし（縦横に２個。色マスク用）

void CXImage::drawCheckTwo(int x,int y,int size)
{
    int ix,iy,fy,div;
    DWORD col[2];

    col[0] = axapp->rgbToPix(0xc0c0c0);
    col[1] = axapp->rgbToPix(0xf0f0f0);

    div = size >> 1;

    for(iy = 0; iy < size; iy++)
    {
        fy = iy / div;

        for(ix = 0; ix < size; ix++)
            setPixelPx(x + ix, y + iy, col[(ix / div) ^ fy]);
    }
}

//! 白黒破線の水平線描画（プレビューウィンドウ、ルーペ時）

void CXImage::drawDashLineH(int y,const AXRectSize &rcs)
{
    DWORD col[2];
    int x;

    col[0] = axapp->rgbToPix(0);
    col[1] = axapp->rgbToPix(0xffffff);

    if(rcs.y <= y && y < rcs.y + rcs.h)
    {
        for(x = rcs.x; x < rcs.x + rcs.w; x++)
            setPixelPx(x, y, col[(x >> 1) & 1]);
    }
}

//! 白黒破線の垂直線描画

void CXImage::drawDashLineV(int x,const AXRectSize &rcs)
{
    DWORD col[2];
    int y;

    col[0] = axapp->rgbToPix(0);
    col[1] = axapp->rgbToPix(0xffffff);

    if(rcs.x <= x && x < rcs.x + rcs.w)
    {
        for(y = rcs.y; y < rcs.y + rcs.h; y++)
            setPixelPx(x, y, col[(y >> 1) & 1]);
    }
}

//! 白黒破線の四角枠描画

void CXImage::drawDashBox(int x,int y,int w,int h)
{
    int i,n,cnt = 0;
    DWORD col[2];

    col[0] = axapp->rgbToPix(0);
    col[1] = axapp->rgbToPix(0xffffff);

    //上

    for(i = 0; i < w; i++, cnt++)
        setPixelPx(x + i, y, col[(cnt >> 1) & 1]);

    //右

    n = x + w - 1;

    for(i = 1; i < h; i++, cnt++)
        setPixelPx(n, y + i, col[(cnt >> 1) & 1]);

    //下

    n = y + h - 1;

    for(i = w - 2; i >= 0; i--, cnt++)
        setPixelPx(x + i, n, col[(cnt >> 1) & 1]);

    //左

    for(i = h - 2; i > 0; i--, cnt++)
        setPixelPx(x, y + i, col[(cnt >>1) & 1]);
}

//! 大文字アルファベット描画(6x7)

void CXImage::drawAlphabet(int x,int y,DWORD col,LPCSTR szText)
{
    int i,xx,yy,len,xpos;
    BYTE f,topf;
    const unsigned char *pSrc,*pSrcBk;

    col = axapp->rgbToPix(col);
    len = ::strlen(szText);

    for(i = 0, xpos = x; i < len; i++, xpos += 6)
    {
        xx = szText[i] - 'A';
        if(xx < 0 || xx >= 26) continue;

        xx *= 6;

        pSrc = g_img_alphabet + (xx >> 3);
        topf = 1 << (7 - (xx & 7));

        //

        for(yy = 0; yy < 7; yy++)
        {
            pSrcBk = pSrc;

            for(xx = 0, f = topf; xx < 6; xx++)
            {
                if(*pSrc & f) setPixelPx(xpos + xx, y + yy, col);

                f >>= 1;
                if(f == 0) { f = 0x80; pSrc++; }
            }

            pSrc = pSrcBk + ((26 * 6 + 7) / 8);
        }
    }
}

//! 選択範囲輪郭用に点描画

void CXImage::setPixelSelection(int x,int y,const AXRect &rcClip)
{
    BYTE pt[8] = { 0xf0,0x78,0x3c,0x1e,0x0f,0x87,0xc3,0xe1 };

    if(x >= rcClip.left && x <= rcClip.right && y >= rcClip.top && y <= rcClip.bottom)
    {
        if(pt[y & 7] & (1 << (x & 7)))
            setPixel(x, y, 0);
        else
            setPixel(x, y, 0xffffff);
    }
}

//! 選択範囲輪郭用に線描画

void CXImage::drawLineSelection(int x1,int y1,int x2,int y2,const AXRect &rcClip)
{
    int dx,dy,f,d,add = 1;

    if(x1 == x2 && y1 == y2)
    {
        setPixelSelection(x1, y1, rcClip);
        return;
    }

    dx = (x1 < x2)? x2 - x1: x1 - x2;
    dy = (y1 < y2)? y2 - y1: y1 - y2;

    if(dx > dy)
    {
        f = y1 << 16;
        d = ((y2 - y1) << 16) / (x2 - x1);
        if(x1 > x2) add = -1, d = -d;

        while(x1 != x2)
        {
            setPixelSelection(x1, (f + 0x8000) >> 16, rcClip);
            f  += d;
            x1 += add;
        }
    }
    else
    {
        f = x1 << 16;
        d = ((x2 - x1) << 16) / (y2 - y1);
        if(y1 > y2) add = -1, d = -d;

        while(y1 != y2)
        {
            setPixelSelection((f + 0x8000) >> 16, y1, rcClip);
            f  += d;
            y1 += add;
        }
    }
}

//! 線を引く（グリッド線用、指定色を合成する）

void CXImage::drawLineBlend(double x1,double y1,double x2,double y2,DWORD col)
{
    double dx,dy;
    int nx1,ny1,nx2,ny2,f,d,add = 1;
    void (AXImage::*func)(int,int,DWORD);

    if((col >> 24) == 255)
        func = &AXImage::setPixel;
    else
        func = &AXImage::blendPixel;

    //

    nx1 = (int)(x1 + 0.5), ny1 = (int)(y1 + 0.5);
    nx2 = (int)(x2 + 0.5), ny2 = (int)(y2 + 0.5);

    //

    if(nx1 == nx2 && ny1 == ny2)
    {
        (this->*func)(nx1, ny1, col);
        return;
    }

    dx = (x1 < x2)? x2 - x1: x1 - x2;
    dy = (y1 < y2)? y2 - y1: y1 - y2;

    if(dx > dy)
    {
        f = (int)(y1 * (1<<16));
        d = (int)((y2 - y1) / (x2 - x1) * (1<<16));
        if(nx1 > nx2) add = -1, d = -d;

        while(nx1 != nx2)
        {
            (this->*func)(nx1, (f + 0x8000) >> 16, col);
            f   += d;
            nx1 += add;
        }

        (this->*func)(nx1, (f + 0x8000) >> 16, col);
    }
    else
    {
        f = (int)(x1 * (1<<16));
        d = (int)((x2 - x1) / (y2 - y1) * (1<<16));
        if(ny1 > ny2) add = -1, d = -d;

        while(ny1 != ny2)
        {
            (this->*func)((f + 0x8000) >> 16, ny1, col);
            f   += d;
            ny1 += add;
        }

        (this->*func)((f + 0x8000) >> 16, ny1, col);
    }
}


//=============================
//グリッド描画
//=============================


//! グリッド線描画

void CXImage::drawGrid(const AXRectSize &rcsDst,const AXRectSize &rcsImg,
    int gridw,int gridh,DWORD colGrid,const DRAWCANVASINFO &info)
{
    int first,end,i;
    double dx1,dy1,dx2,dy2,add[4],addx,addy;
    AXRect rcClip;

    rcClip.set(rcsDst);

    info.getImgToCanvDiff(add);

    //--------- 縦線

    first = rcsImg.x / gridw * gridw;
    end   = (rcsImg.x + rcsImg.w - 1) / gridw * gridw;

    info.imgTocanv(&dx1, &dy1, first, rcsImg.y);
    info.imgTocanv(&dx2, &dy2, first, rcsImg.y + rcsImg.h);

    addx = add[0] * gridw;
    addy = add[1] * gridw;

    for(i = first; i <= end; i += gridw)
    {
        if(i)
            _drawGridLine(dx1, dy1, dx2, dy2, colGrid, rcClip);

        dx1 += addx, dy1 += addy;
        dx2 += addx, dy2 += addy;
    }

    //-------- 横線

    first = rcsImg.y / gridh * gridh;
    end   = (rcsImg.y + rcsImg.h - 1) / gridh * gridh;

    info.imgTocanv(&dx1, &dy1, rcsImg.x, first);
    info.imgTocanv(&dx2, &dy2, rcsImg.x + rcsImg.w, first);

    addx = add[2] * gridh;
    addy = add[3] * gridh;

    for(i = first; i <= end; i += gridh)
    {
        if(i)
            _drawGridLine(dx1, dy1, dx2, dy2, colGrid, rcClip);

        dx1 += addx, dy1 += addy;
        dx2 += addx, dy2 += addy;
    }
}

//! グリッド線の描画（クリッピング付き）

void CXImage::_drawGridLine(double x1,double y1,double x2,double y2,DWORD col,const AXRect &rcClip)
{
    int flag1,flag2,flag;
    double x,y;

    flag1 = _calcClipCode(x1, y1, rcClip);
    flag2 = _calcClipCode(x2, y2, rcClip);

    while(1)
    {
        //クリッピング必要なし

        if(!(flag1 | flag2)) break;

        //範囲外

        if(flag1 & flag2) return;

        //クリッピング

        flag = (flag1)? flag1: flag2;

        if(flag & 8)
        {
            x = x1 + (x2 - x1) * (rcClip.bottom - y1) / (y2 - y1);
            y = rcClip.bottom;
        }
        else if(flag & 4)
        {
            x = x1 + (x2 - x1) * (rcClip.top - y1) / (y2 - y1);
            y = rcClip.top;
        }
        else if(flag & 2)
        {
            y = y1 + (y2 - y1) * (rcClip.right - x1) / (x2 - x1);
            x = rcClip.right;
        }
        else
        {
            y = y1 + (y2 - y1) * (rcClip.left - x1) / (x2 - x1);
            x = rcClip.left;
        }

        if(flag == flag1)
        {
            x1 = x, y1 = y;
            flag1 = _calcClipCode(x, y, rcClip);
        }
        else
        {
            x2 = x, y2 = y;
            flag2 = _calcClipCode(x, y, rcClip);
        }
    }

    drawLineBlend(x1, y1, x2, y2, col);
}

//! クリッピングコード取得

int CXImage::_calcClipCode(double x,double y,const AXRect &rcClip)
{
    int xx,yy,flag = 0;

    xx = floor(x);
    yy = floor(y);

    if(xx < rcClip.left) flag |= 1; else if(xx > rcClip.right) flag |= 2;
    if(yy < rcClip.top) flag |= 4; else if(yy > rcClip.bottom) flag |= 8;

    return flag;
}
