/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImage [file] - 画像ファイル関連
*/

#include "CTileImage.h"

#include "CProgressDlg.h"

#include "AXBMPLoad.h"
#include "AXPNGLoad.h"
#include "AXGIFLoad.h"
#include "AXJPEG.h"
#include "AXPNGSave.h"
#include "AXFileWriteBuf.h"
#include "AXZlib.h"
#include "AXMem.h"
#include "AXUtil.h"
#include "AXUtilImg.h"



//! ヘッダを判定して画像ファイル読み込み

BOOL CTileImage::loadImage(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg)
{
    int type,ret;

    type = AXGetImageFileType(filename);

    if(type == 'B')
        ret = loadBMP(filename, pInfo, pdlg);
    else if(type == 'P')
        ret = loadPNG(filename, pInfo, pdlg);
    else if(type == 'J')
        ret = loadJPEG(filename, pInfo, pdlg);
    else if(type == 'G')
        ret = loadGIF(filename, pInfo, pdlg);
    else
        ret = FALSE;

    return ret;
}

//! 32bitバッファからY1列セット

void CTileImage::_loadimg_line(int y,int w,LPDWORD pSrc)
{
    int x;
    RGBAFIX15 col;

    for(x = 0; x < w; x++, pSrc++)
    {
        col.set(*pSrc);
        setPixel_create2(x, y, col);
    }
}


//============================


//! BMP読み込み

BOOL CTileImage::loadBMP(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg)
{
    AXBMPLoad bmp;
    AXMem memLine;
    int y,i;

    if(!bmp.openFile(filename)) return FALSE;

    if(!create(bmp.m_nWidth, bmp.m_nHeight)) return FALSE;

    //Y1列32bitバッファ

    if(!memLine.alloc(bmp.m_nWidth * sizeof(DWORD)))
        return FALSE;

    //読み込み＆変換

    pdlg->beginProgSub(30, bmp.m_nHeight, TRUE);

    for(i = bmp.m_nHeight; i > 0; i--)
    {
        y = bmp.readLine();
        if(y < 0) return FALSE;

        bmp.lineTo32bit(memLine);

        _loadimg_line(y, bmp.m_nWidth, memLine);

        pdlg->incProgSub();
    }

    //

    if(pInfo)
    {
        pInfo->nWidth  = bmp.m_nWidth;
        pInfo->nHeight = bmp.m_nHeight;
        pInfo->nTPCol  = -1;
        pInfo->nDPI    = AXDPMtoDPI(bmp.m_nResoH);
    }

    bmp.close();

    return TRUE;
}

//! PNG読み込み

BOOL CTileImage::loadPNG(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg)
{
    AXPNGLoad png;
    AXMem memLine;
    int y;

    if(!png.loadFile(filename)) return FALSE;

    //作成

    if(!create(png.m_nWidth, png.m_nHeight)) return FALSE;

    //Y1列バッファ

    if(!memLine.alloc(png.m_nWidth * sizeof(DWORD))) return FALSE;

    //変換

    pdlg->beginProgSub(30, png.m_nHeight, TRUE);

    for(y = 0; y < png.m_nHeight; y++)
    {
        png.lineTo32bit(memLine, y, FALSE);

        _loadimg_line(y, png.m_nWidth, memLine);

        pdlg->incProgSub();
    }

    //

    if(pInfo)
    {
        pInfo->nWidth  = png.m_nWidth;
        pInfo->nHeight = png.m_nHeight;
        pInfo->nTPCol  = png.m_nTPCol;

        if(png.m_nResoH == -1)
            pInfo->nDPI = 96;
        else
            pInfo->nDPI = AXDPMtoDPI(png.m_nResoH);
    }

    return TRUE;
}

//! JPEG読み込み

BOOL CTileImage::loadJPEG(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg)
{
    AXJPEG jpg;
    AXMem memLine;
    int i,y;

    if(!jpg.openFileLoad(filename)) return FALSE;

    if(!create(jpg.m_nWidth, jpg.m_nHeight)) return FALSE;

    //Y1列32bitバッファ

    if(!memLine.alloc(jpg.m_nWidth * sizeof(DWORD)))
        return FALSE;

    //変換

    pdlg->beginProgSub(30, jpg.m_nHeight, TRUE);

    for(i = jpg.m_nHeight; i > 0; i--)
    {
        y = jpg.readLine();
        if(y == -1) break;

        jpg.lineTo32bit(memLine);

        _loadimg_line(y, jpg.m_nWidth, memLine);

        pdlg->incProgSub();
    }

    //

    if(pInfo)
    {
        pInfo->nWidth  = jpg.m_nWidth;
        pInfo->nHeight = jpg.m_nHeight;
        pInfo->nTPCol  = -1;
        pInfo->nDPI    = (jpg.m_nDPI_H == -1)? 96: jpg.m_nDPI_H;
    }

    jpg.close();

    return TRUE;
}

//! GIF読み込み

BOOL CTileImage::loadGIF(const AXString &filename,IMGLOADINFO *pInfo,CProgressDlg *pdlg)
{
    AXGIFLoad gif;
    AXMem memLine;
    int y;

    if(!gif.loadFile(filename)) return FALSE;

    if(!gif.getNextImage()) return FALSE;

    //作成

    if(!create(gif.m_nWidth, gif.m_nHeight)) return FALSE;

    //Y1列バッファ

    if(!memLine.alloc(gif.m_nWidth * sizeof(DWORD))) return FALSE;

    //変換

    pdlg->beginProgSub(30, gif.m_nHeight, TRUE);

    for(y = 0; y < gif.m_nHeight; y++)
    {
        gif.lineTo32bit(memLine, y, FALSE);

        _loadimg_line(y, gif.m_nWidth, memLine);

        pdlg->incProgSub();
    }

    //

    if(pInfo)
    {
        pInfo->nWidth  = gif.m_nWidth;
        pInfo->nHeight = gif.m_nHeight;
        pInfo->nDPI    = 96;
        pInfo->nTPCol  = (gif.m_nTPIndex == -1)? -1: gif.getPalCol(gif.m_nTPIndex);
    }

    return TRUE;
}


//===================================
// ファイルに出力
//===================================


//! PNG(32bit)に出力

BOOL CTileImage::savePNG32bit(const AXString &filename,int nDPI,CProgressDlg *pdlg)
{
    AXPNGSave png;
    AXPNGSave::INFO info;
    AXMem mem;
    LPBYTE pdst;
    RGBAFIX15 col;
    int x,y;

    //

    info.nWidth  = m_pinfo->nImgW;
    info.nHeight = m_pinfo->nImgH;
    info.nBits   = 32;
    info.nPalCnt = 0;

    if(!png.openFile(filename, &info)) return FALSE;

    //バッファ

    if(!mem.alloc(png.getPitch())) return FALSE;

    //DPI

    x = AXDPItoDPM(nDPI);
    png.put_pHYs(x, x);

    //イメージ

    if(!png.startImg()) return FALSE;

    pdlg->beginProgSub(30, info.nHeight, TRUE);

    for(y = 0; y < info.nHeight; y++)
    {
        pdst = mem;

        *(pdst++) = 0;  //フィルタタイプ

        for(x = 0; x < info.nWidth; x++, pdst += 4)
        {
            getPixel(&col, x, y);

            pdst[0] = (col.r * 255 + 0x4000) >> 15;
            pdst[1] = (col.g * 255 + 0x4000) >> 15;
            pdst[2] = (col.b * 255 + 0x4000) >> 15;
            pdst[3] = (col.a * 255 + 0x4000) >> 15;
        }

        if(!png.putImg(mem)) return FALSE;

        pdlg->incProgSub();
    }

    png.endImg();
    png.close();

    return TRUE;
}


//===================================
// タイル単位読み書き
//===================================


//! 選択範囲コピーのファイル出力

BOOL CTileImage::saveCopyFile(const AXString &filename,DWORD col)
{
    AXFileWriteBuf file;
    AXZlib zlib;
    AXMem memBuf;

    if(!zlib.allocBuf(m_nTileSize)) return FALSE;
    if(!zlib.initEncBuf(6)) return FALSE;

    if(!memBuf.alloc(m_nTileSize)) return FALSE;

    //---------

    if(!file.open(filename, 8192)) return FALSE;

    file.putBYTE(m_nColType);
    file.putDWORDBE(col);

    saveTiles(&file, &zlib, memBuf, NULL, 0);

    file.close();

    return TRUE;
}

//! 選択範囲コピーファイルから読み込み

BOOL CTileImage::loadCopyFile(AXFile *pfile)
{
    AXZlib zlib;
    AXMem memBuf;

    if(!zlib.allocBuf(m_nTileSize)) return FALSE;
    if(!zlib.initDecBuf()) return FALSE;

    if(!memBuf.alloc(m_nTileSize)) return FALSE;

    return loadTiles(pfile, &zlib, memBuf, NULL, 0);
}

//! APD/APLD 用、タイルデータを出力

void CTileImage::saveTiles(AXFileWriteBuf *pfile,AXZlib *pzlib,LPBYTE pWorkBuf,
        CProgressDlg *pdlg,int step)
{
    FLAGRECT rcf;
    void **pp;
    int ix,iy,xpos,ypos,tilew,tcnt;
    DWORD encsize;

    //-------- 情報

    //イメージのある範囲

    getExistImgRectPx(&rcf, &tcnt);

    tilew = (rcf.x2 - rcf.x1 + 1) >> 6;

    //左上px
    pfile->putDWORDBE(rcf.x1);
    pfile->putDWORDBE(rcf.y1);

    //タイル幅・高さ
    pfile->putWORDBE(tilew);
    pfile->putWORDBE((rcf.y2 - rcf.y1 + 1) >> 6);

    //タイル数
    pfile->putDWORDBE(tcnt);

    //空の状態なら終了

    if(tcnt == 0)
    {
        if(pdlg) pdlg->addProgPos(step);
        return;
    }

    //-------- タイル

    if(pdlg) pdlg->beginProgSub(step, tcnt);

    pp   = m_ppTile;
    ypos = (m_nOffY - rcf.y1) / 64;

    for(iy = 0; iy < m_nArrayH; iy++, ypos++)
    {
        xpos = (m_nOffX - rcf.x1) / 64;

        for(ix = 0; ix < m_nArrayW; ix++, xpos++, pp++)
        {
            if(!(*pp)) continue;

            //保存用のタイルデータ取得

            getSaveTileBuf(pWorkBuf, *pp);

            //圧縮 (0で非圧縮)

            pzlib->reset();

            if(pzlib->putEncBuf(pzlib->getBuf(), m_nTileSize, pWorkBuf, m_nTileSize))
                encsize = pzlib->getEncSize();
            else
                encsize = 0;

            //位置

            pfile->putDWORDBE(ypos * tilew + xpos);

            //圧縮サイズ・データ

            if(encsize == 0)
            {
                pfile->putDWORDBE(m_nTileSize);
                pfile->put(pWorkBuf, m_nTileSize);
            }
            else
            {
                pfile->putDWORDBE(encsize);
                pfile->put(pzlib->getBuf(), encsize);
            }

            //

            if(pdlg) pdlg->incProgSub();
        }
    }
}

//! APD/APLD タイル読み込み

BOOL CTileImage::loadTiles(AXFile *pfile,AXZlib *pzlib,LPBYTE pWorkBuf,
    CProgressDlg *pdlg,int step)
{
    int i,tcnt;
    DWORD pos,encsize,maxpos;
    WORD tilew,tileh;
    void **pp;

    //左上px

    pfile->readDWORD(&m_nOffX);
    pfile->readDWORD(&m_nOffY);

    //タイル幅・高さ

    pfile->readWORD(&tilew);
    pfile->readWORD(&tileh);

    if(tilew == 0 || tileh == 0)
        tilew = tileh = 1;

    m_nArrayW = tilew;
    m_nArrayH = tileh;

    //タイル数

    pfile->readDWORD(&tcnt);

    //配列確保

    if(!_allocTileArray()) return FALSE;

    //空なら終了

    if(tcnt == 0)
    {
        if(pdlg) pdlg->addProgPos(step);
        return TRUE;
    }

    //-------- タイル

    maxpos = m_nArrayW * m_nArrayH;

    if(pdlg) pdlg->beginProgSub(step, tcnt);

    for(i = tcnt; i > 0; i--)
    {
        //位置

        if(!pfile->readDWORD(&pos)) return FALSE;

        //圧縮サイズ

        pfile->readDWORD(&encsize);

        if(pos >= maxpos || encsize > (DWORD)m_nTileSize) return FALSE;

        //タイル確保

        pp = getTileBuf() + pos;

        if(!(*pp))
        {
            *pp = _allocTile();
            if(!(*pp)) return FALSE;
        }

        //圧縮展開 -> pWorkBuf

        if(encsize == (DWORD)m_nTileSize)
            pfile->read(pWorkBuf, m_nTileSize);
        else
        {
            pfile->read(pzlib->getBuf(), encsize);

            if(!pzlib->reset()) return FALSE;

            if(!pzlib->getDecBuf(pWorkBuf, m_nTileSize, pzlib->getBuf(), encsize))
                return FALSE;
        }

        //タイルデータ変換

        setSaveTileBuf(*pp, pWorkBuf);

        //

        if(pdlg) pdlg->incProgSub();
    }

    return TRUE;
}
