/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CTileImage[edit] 編集関連
*/

#include <stdlib.h>
#include <math.h>

#include "CTileImageA1.h"

#include "CProgressDlg.h"


//! カラータイプ変更による変換

BOOL CTileImage::createConvert(const CTileImage &src,BOOL bLum,CProgressDlg *pdlg)
{
    void **ppDst;
    const void **ppSrc;
    RGBAFIX15 *pBuf;
    int i;

    free();

    if(!createSame(src)) return FALSE;

    //バッファ

    pBuf = (RGBAFIX15 *)::malloc(sizeof(RGBAFIX15) * 64 * 64);
    if(!pBuf) return FALSE;

    //タイル単位

    ppDst = m_ppTile;
    ppSrc = (const void **)src.getTileBuf();

    if(pdlg) pdlg->beginProgSub(30, m_nArrayW * m_nArrayH, TRUE);

    for(i = m_nArrayW * m_nArrayH; i; i--, ppDst++, ppSrc++)
    {
        if(*ppSrc)
        {
            src._getBufRGBA_tile(pBuf, *ppSrc);

            *ppDst = _allocTile();
            if(!*ppDst)
            {
                ::free(pBuf);
                return FALSE;
            }

            _setFromBufRGBA_tile(*ppDst, pBuf, bLum);

            //輝度->アルファ値の場合、すべて透明ならタイルなし

            if(bLum && _isTransparentTile(*ppDst))
                _freeOneTile(ppDst);
        }

        if(pdlg) pdlg->incProgSub();
    }

    ::free(pBuf);

    return TRUE;
}

//! イメージ結合

BOOL CTileImage::combine(const CTileImage &src,const AXRect &rc,int srcopa,int dstopa,
    void (*funcCol)(RGBAFIX15 *,const RGBFIX15 &),CProgressDlg *pdlg,int step)
{
    int x,y;
    RGBAFIX15 col;

    //配列リサイズ

    if(!resizeTileArray_double(src)) return FALSE;

    //

    pdlg->beginProgSub(step, rc.height(), FALSE);

    for(y = rc.top; y <= rc.bottom; y++)
    {
        for(x = rc.left; x <= rc.right; x++)
        {
            src.getPixel(&col, x, y);

            col.a = col.a * srcopa >> 7;

            setPixel_combine(x, y, col, dstopa, funcCol);
        }

        pdlg->incProgSub();
    }

    return TRUE;
}

//! 透明部分の色を置き換え

void CTileImage::replaceColor_tp(const RGBAFIX15 &colDst)
{
    AXRect rc;
    int x,y;

    getEnableDrawRectPixel(&rc);

    for(y = rc.top; y <= rc.bottom; y++)
    {
        for(x = rc.left; x <= rc.right; x++)
        {
            if(isPixelTransparent(x, y))
                setPixelDraw(x, y, colDst);
        }
    }
}

//! 色の置き換え (RGBのみ。アルファ値はそのまま)

void CTileImage::replaceColor(const RGBAFIX15 &colSrc,const RGBAFIX15 &colDst)
{
    void **pp = m_ppTile;
    int tx,ty,ix,iy,x,y;
    RGBAFIX15 col;

    for(ty = 0, y = m_nOffY; ty < m_nArrayH; ty++, y += 64)
    {
        for(tx = 0, x = m_nOffX; tx < m_nArrayW; tx++, pp++, x += 64)
        {
            if(!(*pp)) continue;

            for(iy = 0; iy < 64; iy++)
            {
                for(ix = 0; ix < 64; ix++)
                {
                    getPixel(&col, x + ix, y + iy);

                    if(col.a && col.compareRGB(colSrc))
                    {
                        col.r = colDst.r;
                        col.g = colDst.g;
                        col.b = colDst.b;

                        setPixelDraw(x + ix, y + iy, col);
                    }
                }
            }
        }
    }
}


//==========================


//! 全体を左右反転

void CTileImage::reverseHorz()
{
    int cnt,x,y;
    void **pp,**pp2,*pt;

    //オフセット

    m_nOffX = (m_pinfo->nImgW - 1) - m_nOffX - (m_nArrayW * 64 - 1);

    //各タイルのイメージを左右反転

    pp = m_ppTile;

    for(x = m_nArrayW * m_nArrayH; x; x--, pp++)
    {
        if(*pp) _reverseHorz_tile(*pp);
    }

    //配列のタイルを左右反転

    cnt	= m_nArrayW >> 1;

    for(y = 0; y < m_nArrayH; y++)
    {
        pp  = m_ppTile + y * m_nArrayW;
        pp2 = pp + m_nArrayW - 1;

        for(x = cnt; x; x--, pp++, pp2--)
        {
            pt   = *pp;
            *pp  = *pp2;
            *pp2 = pt;
        }
    }
}

//! 全体を上下反転

void CTileImage::reverseVert()
{
    void **pp,**pp2,*pt;
    int x,y;

    //オフセット

    m_nOffY = (m_pinfo->nImgH - 1) - m_nOffY - (m_nArrayH * 64 - 1);

    //各タイルのイメージを上下反転

    pp = m_ppTile;

    for(x = m_nArrayW * m_nArrayH; x; x--, pp++)
    {
        if(*pp) _reverseVert_tile(*pp);
    }

    //配列のタイルを上下反転

    pp  = m_ppTile;
    pp2 = m_ppTile + (m_nArrayH - 1) * m_nArrayW;

    for(y = m_nArrayH >> 1; y; y--)
    {
        for(x = m_nArrayW; x; x--, pp++, pp2++)
        {
            pt   = *pp;
            *pp  = *pp2;
            *pp2 = pt;
        }

        pp2 -= m_nArrayW << 1;
    }
}

//! 左に90度回転

void CTileImage::rotateLeft()
{
    int ix,iy,n;
    void **pp,**ppNew,**pp2;

    //新規タイル配列

    ppNew = _allocTileArrayNew(m_nArrayH, m_nArrayW, FALSE);
    if(!ppNew) return;

    //オフセット

    ix = m_pinfo->nImgW >> 1, iy = m_pinfo->nImgH >> 1;
    n = m_nOffX;

    m_nOffX = m_nOffY - iy + ix;
    m_nOffY = iy - (n + m_nArrayW * 64 - 1 - ix);

    //各タイルのイメージを回転

    pp = m_ppTile;

    for(ix = m_nArrayW * m_nArrayH; ix; ix--, pp++)
    {
        if(*pp) _rotateLeft_tile(*pp);
    }

    //タイル配列を回転

    pp  = ppNew;
    pp2 = m_ppTile + m_nArrayW - 1;
    n   = m_nArrayW * m_nArrayH + 1;

    for(iy = 0; iy < m_nArrayW; iy++, pp2 -= n)
        for(ix = 0; ix < m_nArrayH; ix++, pp2 += m_nArrayW)
            *(pp++) = *pp2;

    //

    ::free(m_ppTile);

    ix = m_nArrayW;

    m_ppTile  = ppNew;
    m_nArrayW = m_nArrayH;
    m_nArrayH = ix;
}

//! 右に90度回転

void CTileImage::rotateRight()
{
    int ix,iy,n;
    void **pp,**ppNew,**pp2;

    //新規タイル配列

    ppNew = _allocTileArrayNew(m_nArrayH, m_nArrayW, FALSE);
    if(!ppNew) return;

    //オフセット

    ix = m_pinfo->nImgW >> 1, iy = m_pinfo->nImgH >> 1;
    n = m_nOffX;

    m_nOffX = ix - (m_nOffY + m_nArrayH * 64 - 1 - iy);
    m_nOffY = n - ix + iy;

    //各タイルのイメージを回転

    pp = m_ppTile;

    for(ix = m_nArrayW * m_nArrayH; ix; ix--, pp++)
    {
        if(*pp) _rotateRight_tile(*pp);
    }

    //タイル配列を回転

    pp  = ppNew;
    pp2 = m_ppTile + (m_nArrayH - 1) * m_nArrayW;
    n   = m_nArrayW * m_nArrayH + 1;

    for(iy = 0; iy < m_nArrayW; iy++, pp2 += n)
        for(ix = 0; ix < m_nArrayH; ix++, pp2 -= m_nArrayW)
            *(pp++) = *pp2;

    //

    ::free(m_ppTile);

    ix = m_nArrayW;

    m_ppTile  = ppNew;
    m_nArrayW = m_nArrayH;
    m_nArrayH = ix;
}


//===========================
// 指定範囲の編集
//===========================


//! 矩形範囲内のイメージをコピー
/*
    矩形編集[90度回転][拡大縮小回転]やフィルタで使用する。
*/

void CTileImage::copyRect(CTileImage *pimgSrc,const AXRect &rc)
{
    int x,y;
    RGBAFIX15 col;

    for(y = rc.top; y <= rc.bottom; y++)
    {
        for(x = rc.left; x <= rc.right; x++)
        {
            pimgSrc->getPixel(&col, x, y);

            if(col.a) setPixel_create(x, y, col);
        }
    }
}

//! 範囲内のイメージを透明にする (setPixelDraw で 0 をセット)

void CTileImage::clearRect(const AXRect &rc)
{
    int x,y;
    RGBAFIX15 col;

    col.zero();

    for(y = rc.top; y <= rc.bottom; y++)
        for(x = rc.left; x <= rc.right; x++)
            setPixelDraw(x, y, col);
}

//! 範囲内のイメージを pimgCopy にクリアし、this の範囲をクリア

void CTileImage::copyClearRect(CTileImage *pimgCopy,const AXRect &rc)
{
    int x,y;
    RGBAFIX15 colSrc,colZero;

    colZero.zero();

    for(y = rc.top; y <= rc.bottom; y++)
    {
        for(x = rc.left; x <= rc.right; x++)
        {
            getPixel(&colSrc, x, y);

            if(colSrc.a)
            {
                pimgCopy->setPixel_create(x, y, colSrc);

                setPixel_create(x, y, colZero);
            }
        }
    }
}

//! 範囲左右反転
/*!
    @param rcs イメージ範囲
*/

void CTileImage::reverseHorzRect(const AXRectSize &rcs)
{
    int cnt,ix,iy,x1,x2,y;
    RGBAFIX15 col1,col2;

    x1  = rcs.x;
    x2  = rcs.x + rcs.w - 1;
    cnt = rcs.w >> 1;

    for(iy = rcs.h, y = rcs.y; iy > 0; iy--, y++)
    {
        for(ix = 0; ix < cnt; ix++)
        {
            getPixel(&col1, x1 + ix, y);
            getPixel(&col2, x2 - ix, y);

            setPixelDraw(x1 + ix, y, col2);
            setPixelDraw(x2 - ix, y, col1);
        }
    }
}

//! 範囲上下反転

void CTileImage::reverseVertRect(const AXRectSize &rcs)
{
    int ix,iy,y1,y2,x;
    RGBAFIX15 col1,col2;

    y1 = rcs.y;
    y2 = rcs.y + rcs.h - 1;

    for(iy = rcs.h >> 1; iy > 0; iy--, y1++, y2--)
    {
        for(ix = rcs.w, x = rcs.x; ix > 0; ix--, x++)
        {
            getPixel(&col1, x, y1);
            getPixel(&col2, x, y2);

            setPixelDraw(x, y1, col2);
            setPixelDraw(x, y2, col1);
        }
    }
}

//! 範囲を左に90度回転

void CTileImage::rotateLeftRect(CTileImage *pimgSrc,const AXRectSize &rcs)
{
    int ix,iy,w,h,sx,sy,dx,dy;
    RGBAFIX15 col;

    w = rcs.h, h = rcs.w;

    sx = rcs.x + rcs.w - 1;
    dx = rcs.x + (rcs.w >> 1) - (rcs.h >> 1);
    dy = rcs.y + (rcs.h >> 1) - (rcs.w >> 1);

    for(iy = 0; iy < h; iy++, sx--)
    {
        for(ix = 0, sy = rcs.y; ix < w; ix++, sy++)
        {
            pimgSrc->getPixel(&col, sx, sy);

            setPixelDraw(dx + ix, dy + iy, col);
        }
    }
}

//! 範囲を右に90度回転

void CTileImage::rotateRightRect(CTileImage *pimgSrc,const AXRectSize &rcs)
{
    int ix,iy,w,h,sx,sy,dx,dy;
    RGBAFIX15 col;

    w = rcs.h, h = rcs.w;

    sx = rcs.x;
    dx = rcs.x + (rcs.w >> 1) - (rcs.h >> 1);
    dy = rcs.y + (rcs.h >> 1) - (rcs.w >> 1);

    for(iy = 0; iy < h; iy++, sx++)
    {
        sy = rcs.y + rcs.h - 1;

        for(ix = 0; ix < w; ix++, sy--)
        {
            pimgSrc->getPixel(&col, sx, sy);

            setPixelDraw(dx + ix, dy + iy, col);
        }
    }
}


//=============================
// 拡大縮小＆回転
//=============================


//! 拡大縮小＆回転

void CTileImage::scaleAndRotate(CTileImage *pimgSrc,const AXRect &rc,double dScale,double dAngle,BOOL bHiQuality)
{
    AXRect rcDraw;
    int cx,cy,x,y,ix,iy,w,h,isx,isy,xx,yy,type;
    double dcos,dsin,dcos2,dsin2,wx[4],wy[4],wf,dcol[4];
    double nxxx,nyyy,nxx,nyy,nxxs,nyys,nxxs2,nyys2;
    RGBAFIX15 col,col2;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15 &) = m_pinfo->funcDrawPixel;

    //描画範囲

    rcDraw = rc;
    if(!getScaleRotRect(&rcDraw, dScale, dAngle)) return;

    //

    if(bHiQuality)
        type = (dScale <= 0.9)? 1: 2;
    else
        type = 0;

    //

    w = rc.width(), h = rc.height();

    cx = w >> 1, cy = h >> 1;

    dScale = 1.0 / dScale;
    dcos   = ::cos(-dAngle);
    dsin   = ::sin(-dAngle);

    x    = rcDraw.left - (rc.left + cx);
    y    = rcDraw.top  - (rc.top + cy);
    nxxx = (x * dcos - y * dsin) * dScale + cx;
    nyyy = (x * dsin + y * dcos) * dScale + cy;

    dcos *= dScale;
    dsin *= dScale;

    //----------------

    for(y = rcDraw.top; y <= rcDraw.bottom; y++)
    {
        nxx = nxxx;
        nyy = nyyy;

        for(x = rcDraw.left; x <= rcDraw.right; x++, nxx += dcos, nyy += dsin)
        {
            //範囲外

            ix = (int)nxx;
            iy = (int)nyy;

            if(ix < 0 || iy < 0 || ix >= w || iy >= h) continue;

            //色

            switch(type)
            {
                //低品質
                case 0:
                    pimgSrc->getPixel(&col, ix + rc.left, iy + rc.top);
                    break;

                //高品質 縮小用: 5x5
                case 1:
                    dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

                    dcos2 = dcos / 5;
                    dsin2 = dsin / 5;

                    nxxs2 = nxx, nyys2 = nyy;

                    for(isy = 0; isy < 5; isy++)
                    {
                        nxxs = nxxs2, nyys = nyys2;

                        for(isx = 0; isx < 5; isx++)
                        {
                            ix = (int)nxxs;
                            iy = (int)nyys;

                            if(ix < 0) ix = 0; else if(ix >= w) ix = w - 1;
                            if(iy < 0) iy = 0; else if(iy >= h) iy = h - 1;

                            pimgSrc->getPixel(&col2, ix + rc.left, iy + rc.top);

                            if(col2.a)
                            {
                                wf = (double)col2.a / 0x8000;

                                dcol[0] += col2.r * wf;
                                dcol[1] += col2.g * wf;
                                dcol[2] += col2.b * wf;
                                dcol[3] += wf;
                            }

                            nxxs += dcos2;
                            nyys += dsin2;
                        }

                        nxxs2 -= dsin2;
                        nyys2 += dcos2;
                    }

                    //

                    col.a = (WORD)(dcol[3] / 25.0 * 0x8000 + 0.5);

                    if(col.a)
                    {
                        wf = 1.0 / dcol[3];

                        col.r = (WORD)(dcol[0] * wf + 0.5);
                        col.g = (WORD)(dcol[1] * wf + 0.5);
                        col.b = (WORD)(dcol[2] * wf + 0.5);
                    }
                    break;

                //高品質 拡大用: Lagrange
                default:
                    //重み

                    for(isx = 0; isx < 4; isx++)
                    {
                        //X

                        wf = ::fabs(nxx - (ix - 1 + isx));

                        if(wf < 1) wx[isx] = 0.5 * (wf - 2.0) * (wf + 1.0) * (wf - 1.0);
                        else wx[isx] = -(wf - 3.0) * (wf - 2.0) * (wf - 1.0) / 6.0;

                        //Y

                        wf = ::fabs(nyy - (iy - 1 + isx));

                        if(wf < 1) wy[isx] = 0.5 * (wf - 2.0) * (wf + 1.0) * (wf - 1.0);
                        else wy[isx] = -(wf - 3.0) * (wf - 2.0) * (wf - 1.0) / 6.0;
                    }

                    //4x4 近傍

                    dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

                    for(isy = 0, yy = iy - 1; isy < 4; isy++, yy++)
                    {
                        for(isx = 0, xx = ix - 1; isx < 4; isx++, xx++)
                        {
                            if(xx >= 0 && xx < w && yy >= 0 && yy < h)
                            {
                                pimgSrc->getPixel(&col2, xx + rc.left, yy + rc.top);

                                if(col2.a)
                                {
                                    wf = ((double)col2.a / 0x8000) * wx[isx] * wy[isy];

                                    dcol[0] += col2.r * wf;
                                    dcol[1] += col2.g * wf;
                                    dcol[2] += col2.b * wf;
                                    dcol[3] += wf;
                                }
                            }
                        }
                    }

                    //

                    xx = (int)(dcol[3] * 0x8000 + 0.5);
                    if(xx< 0) xx = 0; else if(xx > 0x8000) xx = 0x8000;

                    col.a = xx;

                    if(xx)
                    {
                        wf = 1.0 / dcol[3];

                        for(xx = 0; xx < 3; xx++)
                        {
                            yy = (int)(dcol[xx] * wf + 0.5);
                            if(yy < 0) yy = 0; else if(yy > 0x8000) yy = 0x8000;

                            col.c[xx] = yy;
                        }
                    }
                    break;
            }

            //セット

            if(col.a) (this->*funcPix)(x, y, col);
        }

        nxxx -= dsin;
        nyyy += dcos;
    }
}

//! 拡大・回転後の範囲取得
/*!
    @param prc 元矩形範囲を入れておく。結果が入る。
    @return FALSE で描画範囲外
*/

BOOL CTileImage::getScaleRotRect(AXRect *prc,double dScale,double dAngle)
{
    AXPoint pt[4],ptMin,ptMax;
    int i,cx,cy,ix,iy;
    double xx,yy,dcos,dsin;

    cx = (prc->right - prc->left + 1) / 2 + prc->left;
    cy = (prc->bottom - prc->top + 1) / 2 + prc->top;

    dcos = ::cos(dAngle);
    dsin = ::sin(dAngle);

    //四隅の点を拡大・回転

    pt[0].x = pt[3].x = prc->left;
    pt[0].y = pt[1].y = prc->top;
    pt[1].x = pt[2].x = prc->right + 1;
    pt[2].y = pt[3].y = prc->bottom + 1;

    for(i = 0; i < 4; i++)
    {
        xx = (pt[i].x - cx) * dScale;
        yy = (pt[i].y - cy) * dScale;

        ix = (int)(xx * dcos - yy * dsin) + cx;
        iy = (int)(xx * dsin + yy * dcos) + cy;

        if(i == 0)
        {
            ptMin.x = ptMax.x = ix;
            ptMin.y = ptMax.y = iy;
        }
        else
        {
            if(ix < ptMin.x) ptMin.x = ix;
            if(iy < ptMin.y) ptMin.y = iy;
            if(ix > ptMax.x) ptMax.x = ix;
            if(iy > ptMax.y) ptMax.y = iy;
        }
    }

    //セット＆調整

    prc->set(ptMin.x - 1, ptMin.y - 1, ptMax.x + 1, ptMax.y + 1);

    return clipRectInEnableDraw(prc);
}


//===========================
// 選択範囲用
//===========================


//! 選択範囲のイメージコピー
/*
    pimgSel で点がある範囲を pimgSrc から this へコピー。
    setPixel_create() で点をセット。
*/

void CTileImage::copySelectImage(CTileImage *pimgSrc,const CTileImageA1 *pimgSel)
{
    void **ppSel;
    int ix,iy,jx,jy,aw,ah,offx,x,y,xx,yy;
    RGBAFIX15 col;

    ppSel = pimgSel->getTileBuf();
    aw    = pimgSel->getArrayW();
    ah    = pimgSel->getArrayH();
    offx  = pimgSel->getOffX();

    //

    y = pimgSel->getOffY();

    for(iy = ah; iy > 0; iy--, y += 64)
    {
        x = offx;

        for(ix = aw; ix > 0; ix--, x += 64, ppSel++)
        {
            if(!(*ppSel)) continue;

            for(jy = 0, yy = y; jy < 64; jy++, yy++)
            {
                for(jx = 0, xx = x; jx < 64; jx++, xx++)
                {
                    if(pimgSel->isPixelOn(xx, yy))
                    {
                        pimgSrc->getPixel(&col, xx, yy);

                        if(col.a)
                            setPixel_create(xx, yy, col);
                    }
                }
            }
        }
    }
}

//! 選択範囲のイメージコピー&切り取り
/*
    pimgSel で点がある範囲を pimgSrc から this へコピーした後、
    pimgSrc の選択範囲内を透明に。
    setPixel_create2() で点をコピーし、setPixelDraw() で透明で上書き。
*/

void CTileImage::copyCutSelectImage(CTileImage *pimgSrc,const CTileImageA1 *pimgSel)
{
    void **ppSel;
    int ix,iy,jx,jy,aw,ah,offx,x,y,xx,yy;
    RGBAFIX15 col,colZero;

    ppSel = pimgSel->getTileBuf();
    aw    = pimgSel->getArrayW();
    ah    = pimgSel->getArrayH();
    offx  = pimgSel->getOffX();

    colZero.zero();

    //

    y = pimgSel->getOffY();

    for(iy = ah; iy > 0; iy--, y += 64)
    {
        x = offx;

        for(ix = aw; ix > 0; ix--, x += 64, ppSel++)
        {
            if(!(*ppSel)) continue;

            for(jy = 0, yy = y; jy < 64; jy++, yy++)
            {
                for(jx = 0, xx = x; jx < 64; jx++, xx++)
                {
                    if(pimgSel->isPixelOn(xx, yy))
                    {
                        pimgSrc->getPixel(&col, xx, yy);

                        if(col.a)
                        {
                            //this にコピー
                            setPixel_create2(xx, yy, col);

                            //透明に
                            pimgSrc->setPixelDraw(xx, yy, colZero);
                        }
                    }
                }
            }
        }
    }
}

//! 選択範囲イメージ移動後の合成

void CTileImage::blendSelectImage(const CTileImage *pimgSrc,const CTileImageA1 *pimgSel,BOOL bOverwrite)
{
    void **ppSrc;
    int ix,iy,jx,jy,aw,ah,offx,x,y,xx,yy;
    RGBAFIX15 col;

    ppSrc = pimgSrc->getTileBuf();
    aw    = pimgSrc->getArrayW();
    ah    = pimgSrc->getArrayH();
    offx  = pimgSrc->getOffX();

    //

    y = pimgSrc->getOffY();

    for(iy = ah; iy > 0; iy--, y += 64)
    {
        x = offx;

        for(ix = aw; ix > 0; ix--, x += 64, ppSrc++)
        {
            if(!(*ppSrc)) continue;

            for(jy = 0, yy = y; jy < 64; jy++, yy++)
            {
                for(jx = 0, xx = x; jx < 64; jx++, xx++)
                {
                    pimgSrc->getPixel(&col, xx, yy);

                    if(bOverwrite)
                    {
                        if(pimgSel->isPixelOn(xx, yy))
                            setPixelDraw(xx, yy, col);
                    }
                    else
                    {
                        if(col.a)
                            setPixelDraw(xx, yy, col);
                    }
                }
            }
        }
    }
}
