/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    CImage32 - 8bitRGBA カラーイメージ
*/


#include "CImage32.h"

#include "AXBMPLoad.h"
#include "AXPNGLoad.h"
#include "AXGIFLoad.h"
#include "AXJPEG.h"
#include "AXRect.h"
#include "AXMem.h"
#include "AXImage.h"
#include "AXUtil.h"
#include "AXUtilImg.h"
#include "AXApp.h"



//================================
// 読み込み
//================================


//! ヘッダで判別して読み込み

BOOL CImage32::loadImage(const AXString &filename,LOADINFO *pInfo)
{
    int type,ret = FALSE;

    type = AXGetImageFileType(filename);

    if(type == 'B')
        ret = loadBMP(filename, pInfo);
    else if(type == 'P')
        ret = loadPNG(filename, pInfo);
    else if(type == 'J')
        ret = loadJPEG(filename, pInfo);
    else if(type == 'G')
        ret = loadGIF(filename, pInfo);

    return ret;
}

//! BMP読み込み

BOOL CImage32::loadBMP(const AXString &filename,LOADINFO *pInfo)
{
    AXBMPLoad bmp;
    int y,i;

    if(!bmp.openFile(filename)) return FALSE;

    if(!create(bmp.m_nWidth, bmp.m_nHeight)) return FALSE;

    for(i = m_nHeight; i > 0; i--)
    {
        y = bmp.readLine();
        if(y < 0) return FALSE;

        bmp.lineTo32bit(getBufPt(0, y));
    }

    bmp.close();

    //

    if(pInfo)
    {
        pInfo->nDPI   = AXDPMtoDPI(bmp.m_nResoH);
        pInfo->bAlpha = FALSE;
    }

    return TRUE;
}

//! PNG読み込み

BOOL CImage32::loadPNG(const AXString &filename,LOADINFO *pInfo)
{
    AXPNGLoad png;

    if(!png.loadFile(filename)) return FALSE;

    if(!png.toImage32(this, FALSE)) return FALSE;

    //

    if(pInfo)
    {
        if(png.m_nResoH == -1)
            pInfo->nDPI = 96;
        else
            pInfo->nDPI = AXDPMtoDPI(png.m_nResoH);

        pInfo->bAlpha = (png.m_nBits == 32);
    }

    return TRUE;
}

//! GIF読み込み

BOOL CImage32::loadGIF(const AXString &filename,LOADINFO *pInfo)
{
    AXGIFLoad gif;

    if(!gif.loadFile(filename)) return FALSE;

    if(!gif.getNextImage()) return FALSE;

    if(!create(gif.m_nWidth, gif.m_nHeight)) return FALSE;

    gif.to32bit(m_pBuf, FALSE);

    //

    if(pInfo)
    {
        pInfo->nDPI   = 96;
        pInfo->bAlpha = FALSE;
    }

    return TRUE;
}

//! JPEG読み込み

BOOL CImage32::loadJPEG(const AXString &filename,LOADINFO *pInfo)
{
    AXJPEG jpg;
    int i,y;

    if(!jpg.openFileLoad(filename)) return FALSE;

    if(!create(jpg.m_nWidth, jpg.m_nHeight)) return FALSE;

    for(i = m_nHeight; i > 0; i--)
    {
        y = jpg.readLine();
        if(y == -1) break;

        jpg.lineTo32bit(getBufPt(0, y));
    }

    jpg.close();

    //

    if(pInfo)
    {
        pInfo->nDPI   = (jpg.m_nDPI_H == -1)? 96: jpg.m_nDPI_H;
        pInfo->bAlpha = FALSE;
    }

    return TRUE;
}


//================================
//
//================================


//! 画像からカーソルイメージ作成

BOOL CImage32::createCursor(const AXString &filename,AXMem *pmem)
{
    AXImage32::PIXEL *pSrc;
    LPBYTE pDst,pDstCol,pDstMask;
    int pitchDst,ix,iy,f,xpos;
    DWORD col;

    //読み込み

    if(!loadImage(filename, NULL)) return FALSE;

    //サイズ

    if(m_nWidth > 32 || m_nHeight > 32) return FALSE;

    //

    pitchDst = (m_nWidth + 7) >> 3;

    //確保

    if(!pmem->allocClear(4 + pitchDst * m_nHeight * 2))
        return FALSE;

    pDst = *pmem;

    pDst[0] = m_nWidth;
    pDst[1] = m_nHeight;

    //----- 変換

    pSrc = (AXImage32::PIXEL *)m_pBuf;

    pDstCol  = pDst + 4;
    pDstMask = pDst + 4 + pitchDst * m_nHeight;

    for(iy = 0; iy < m_nHeight; iy++)
    {
        for(ix = 0, f = 1, xpos = 0; ix < m_nWidth; ix++, pSrc++)
        {
            col = (pSrc->c & 0xffffff);

            if(col == 0 || col == 0xffffff)
            {
                if(col == 0) pDstCol[xpos] |= f;
                pDstMask[xpos] |= f;
            }

            f <<= 1;
            if(f == 256) f = 1, xpos++;
        }

        pDstCol  += pitchDst;
        pDstMask += pitchDst;
    }

    return TRUE;
}


//================================
// キャンバスへの描画
//================================

#define FIXF_BIT    14
#define FIXF_VAL    (1 << FIXF_BIT)


//! キャンバスへの描画（ニアレストネイバー）

void CImage32::drawCanvasNormal(AXImage *pimgDst,const AXRectSize &rcsDst,
    const CImage32::DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,n;
    long add,stx,fx,fy;
    LPBYTE pDst;
    LPDWORD pSrcY;
    DWORD pixEx;

    w = rcsDst.w;
    h = rcsDst.h;

    pDst   = pimgDst->getBufPt(rcsDst.x, rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;
    pixEx  = axapp->rgbToPix(info.dwExCol);

    add = (long)(info.dScaleDiv * FIXF_VAL);

    stx = (long)(((rcsDst.x - info.nScrollX) * info.dScaleDiv + info.nBaseX) * FIXF_VAL);
    fy  = (long)(((rcsDst.y - info.nScrollY) * info.dScaleDiv + info.nBaseY) * FIXF_VAL);

    //

    for(iy = h; iy > 0; iy--, fy += add)
    {
        n = fy >> FIXF_BIT;

        //Yが範囲外

        if(fy < 0 || n >= m_nHeight)
        {
            pimgDst->lineHBuf(pDst, w, info.dwExCol);
            pDst += pimgDst->getPitch();
            continue;
        }

        //X

        pSrcY = m_pBuf + n * m_nWidth;

        for(ix = w, fx = stx; ix > 0; ix--, fx += add, pDst += bpp)
        {
            n = fx >> FIXF_BIT;

            if(fx >= 0 && n < m_nWidth)
                pimgDst->setPixelBuf(pDst, *(pSrcY + n));
            else
                pimgDst->setPixelBufPx(pDst, pixEx);
        }

        pDst += pitchd;
    }
}

//! キャンバスへの描画（縮小[4x4オーバーサンプリング]）

void CImage32::drawCanvasScaleDown(AXImage *pimgDst,const AXRectSize &rcsDst,
    const CImage32::DRAWCANVASINFO &info)
{
    int w,h,bpp,pitchd,ix,iy,i,j,n,r,g,b,tblX[4];
    long add,add2,stx,fx,fy,f;
    LPBYTE pDst;
    AXImage32::PIXEL *pSrcY[4],*pSrc;
    DWORD pixEx;

    w = rcsDst.w;
    h = rcsDst.h;

    pixEx = axapp->rgbToPix(info.dwExCol);

    pDst   = pimgDst->getBufPt(rcsDst.x, rcsDst.y);
    bpp    = pimgDst->getBytes();
    pitchd = pimgDst->getPitch() - w * bpp;

    add  = (long)(info.dScaleDiv * FIXF_VAL);
    add2 = add >> 2;

    stx = (long)(((rcsDst.x - info.nScrollX) * info.dScaleDiv + info.nBaseX) * FIXF_VAL);
    fy  = (long)(((rcsDst.y - info.nScrollY) * info.dScaleDiv + info.nBaseY) * FIXF_VAL);

    //

    for(iy = h; iy > 0; iy--, fy += add)
    {
        n = fy >> FIXF_BIT;

        //Yが範囲外

        if(n < 0 || n >= m_nHeight)
        {
            pimgDst->lineHBuf(pDst, w, info.dwExCol);
            pDst += pimgDst->getPitch();
            continue;
        }

        //Y位置テーブル

        for(i = 0, f = fy; i < 4; i++, f += add2)
        {
            n = f >> FIXF_BIT;
            if(n < 0) n = 0; else if(n >= m_nHeight) n = m_nHeight - 1;

            pSrcY[i] = (AXImage32::PIXEL *)(m_pBuf + n * m_nWidth);
        }

        //-------- X

        for(ix = w, fx = stx; ix > 0; ix--, fx += add, pDst += bpp)
        {
            n = fx >> FIXF_BIT;

            if(n < 0 || n >= m_nWidth)
                //Xが範囲外
                pimgDst->setPixelBufPx(pDst, pixEx);
            else
            {
                //X位置テーブル

                for(i = 0, f = fx; i < 4; i++, f += add2)
                {
                    n = f >> FIXF_BIT;
                    if(n < 0) n = 0; else if(n >= m_nWidth) n = m_nWidth - 1;

                    tblX[i] = n;
                }

                //4x4 平均値

                r = g = b = 0;

                for(i = 0; i < 4; i++)
                {
                    for(j = 0; j < 4; j++)
                    {
                        pSrc = pSrcY[i] + tblX[j];

                        r += pSrc->r;
                        g += pSrc->g;
                        b += pSrc->b;
                    }
                }

                pimgDst->setPixelBuf(pDst, r >> 4, g >> 4, b >> 4);
            }
        }

        pDst += pitchd;
    }
}
