/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - 変形
*/

#include <math.h>

#include "CTileImage.h"

#include "AXMem.h"
#include "AXRand.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"


namespace filterdraw
{

//! 波形

BOOL trans_wave(FILTERDRAW &info)
{
    int ix,iy,type;
    double dx,dy,factor,freq;
    RGBAFIX15 col;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    factor = info.valbar[0];
    freq   = info.valbar[1] * 0.1 * M_PI / 180;
    type   = info.valcombo[0] + 1;

    //

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dx = ix, dy = iy;

            if(type & 1) dx += factor * ::sin(freq * iy);
            if(type & 2) dy += factor * ::sin(freq * ix);

            getLinerCol(&col, pimgSrc, dx, dy, info.valcombo[1]);

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 波紋

BOOL trans_ripple(FILTERDRAW &info)
{
    int ix,iy,mode;
    double dx,dy,factor,freq,r,a;
    RGBAFIX15 col;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    freq   = info.valbar[0] * 0.01;
    factor = info.valbar[1] * 0.1;
    mode   = (info.bClipping)? 1: 0;

    //

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dx = ix - info.tmpx;
            dy = iy - info.tmpy;

            r = ::sqrt(dx * dx + dy * dy);
            a = ::atan2(dy, dx);

            r += ::sin(r * freq) * factor;

            dx = r * ::cos(a) + info.tmpx;
            dy = r * ::sin(a) + info.tmpy;

            getLinerCol(&col, pimgSrc, dx, dy, mode);

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 極座標

BOOL trans_polar(FILTERDRAW &info)
{
    int ix,iy,right,bottom,bkgnd;
    double dx,dy,xx,yy,mx,my;
    RGBAFIX15 col;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    right  = CTileImage::m_pinfo->nImgW - 1; if(right == 0) right = 1;
    bottom = CTileImage::m_pinfo->nImgH - 1;

    mx = CTileImage::m_pinfo->nImgW * 0.5;
    my = CTileImage::m_pinfo->nImgH * 0.5;

    bkgnd = info.valcombo[1];

    //

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            if(info.valcombo[0])
            {
                //極座標 -> 直交座標

                xx = (double)(right - ix) / right * M_PI * 2 - M_PI / 2;
                yy = iy * 0.5;

                dx = yy * ::cos(xx) + mx;
                dy = yy * ::sin(xx) + my;
            }
            else
            {
                //直交座標 -> 極座標

                xx = ix - mx;
                yy = my - (bottom - iy);

                dx = (::atan2(xx, yy) + M_PI) / (M_PI * 2) * right;
                dy = ::sqrt(xx * xx + yy * yy) * 2;
            }

            getLinerColBkgnd(&col, pimgSrc, dx, dy, ix, iy, bkgnd);

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 放射状ずらし

BOOL trans_radialshift(FILTERDRAW &info)
{
    int ix,iy,n;
    RGBAFIX15 col;
    double dx,dy,r,a;
    AXMem mem;
    short *pbuf;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    //角度ごとのずらす量

    if(!mem.alloc(sizeof(short) * 512)) return FALSE;

    pbuf = (short *)mem.getBuf();
    iy = info.valbar[0] + 1;

    for(ix = 0; ix < 512; ix++)
        pbuf[ix] = CTileImage::m_prand->getDWORD() % iy;

    //

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dx = ix - info.tmpx;
            dy = iy - info.tmpy;

            r = ::sqrt(dx * dx + dy * dy);
            a = ::atan2(dy, dx);

            n = (int)(a * 256 / M_PI) & 511;

            r -= pbuf[n];

            if(r <= 0)
                pimgSrc->getPixel(&col, info.tmpx, info.tmpy);
            else
            {
                dx = r * ::cos(a) + info.tmpx;
                dy = r * ::sin(a) + info.tmpy;

                getLinerCol(&col, pimgSrc, dx, dy, 0);
            }

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 渦巻き

BOOL trans_swirl(FILTERDRAW &info)
{
    int ix,iy,bkgnd;
    RGBAFIX15 col;
    double dx,dy,zoom,factor,r,a;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    factor = info.valbar[0] * 0.001;
    zoom   = 1.0 / (info.valbar[1] * 0.01);
    bkgnd  = info.valcombo[0];

    //

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            dx = ix - info.tmpx;
            dy = iy - info.tmpy;

            r = ::sqrt(dx * dx + dy * dy) * zoom;
            a = ::atan2(dy, dx) + factor * r;

            dx = r * ::cos(a) + info.tmpx;
            dy = r * ::sin(a) + info.tmpy;

            if(bkgnd == 3)
                //クリッピングなし
                getLinerCol(&col, pimgSrc, dx, dy, 0);
            else
                getLinerColBkgnd(&col, pimgSrc, dx, dy, ix, iy, bkgnd);

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

};
