/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - サブ2
*/

#include "filterdraw.h"
#include "filterdrawfunc.h"

#include "CTileImage.h"

#include "AXRand.h"

#include "drawdat.h"
#include "global.h"


namespace filterdraw
{

//! 共通処理1

BOOL common1(FILTERDRAW &info,COMMON1FUNC func)
{
    int ix,iy;
    RGBAFIX15 col;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimgSrc->getPixel(&col, ix, iy);

            if((*func)(info, &col, ix, iy))
                (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 3x3フィルタ

BOOL filter3x3(FILTERDRAW &info,const FILTER3X3DAT &dat,BOOL bGrayscale)
{
    int ix,iy,i,j,no;
    double dcol[3],d;
    RGBAFIX15 col,col2;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimgSrc->getPixel(&col, ix, iy);
            if(!col.a) continue;

            //3x3

            dcol[0] = dcol[1] = dcol[2] = 0;

            for(i = -1, no = 0; i <= 1; i++)
            {
                for(j = -1; j <= 1; j++, no++)
                {
                    pimgSrc->getPixel(&col2, ix + j, iy + i, info.bClipping);

                    d = dat.mul[no];

                    dcol[0] += col2.r * d;
                    dcol[1] += col2.g * d;
                    dcol[2] += col2.b * d;
                }
            }

            //RGB

            d = dat.divmul;

            for(i = 0; i < 3; i++)
            {
                j = (int)(dcol[i] * d + dat.add);
                if(j < 0) j = 0; else if(j > 0x8000) j = 0x8000;

                col.c[i] = j;
            }

            //グレイスケール

            if(bGrayscale)
                col.r = col.g = col.b = _GETV(col.r, col.g, col.b);

            //

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}


//====================================
// 点を打つ
//====================================


//! 描画色を取得

void drawPoint_getColor(RGBAFIX15 *pcol,const DRAWPOINTDAT &dat)
{
    switch(dat.coltype)
    {
        //描画色
        case 0:
            *pcol = g_draw->colDraw;
            break;
        //ランダム(グレイスケール)
        case 1:
            pcol->r = pcol->g = pcol->b = CTileImage::m_prand->getRange(0, 0x8000);
            break;
        //ランダム(色相)
        case 2:
            HSVtoRGB_hue(pcol, CTileImage::m_prand->getRange(0, 359));
            break;
        //ランダム(RGB)
        default:
            pcol->r = CTileImage::m_prand->getRange(0, 0x8000);
            pcol->g = CTileImage::m_prand->getRange(0, 0x8000);
            pcol->b = CTileImage::m_prand->getRange(0, 0x8000);
            break;
    }

    pcol->a = dat.opacity;
}

//! 点を打つ
/*
    masktype :
        [0] 透明部分に描画するので、比較上書き
        [1] 不透明部分に描画するので、通常合成
*/

void drawPoint_setPixel(int x,int y,const RGBAFIX15 &col,const DRAWPOINTDAT &dat)
{
    //マスク

    if(dat.masktype)
    {
        switch(dat.masktype)
        {
            //pimgSrc の透明部分のみ
            case 1:
                if(!dat.pimgSrc->isPixelTransparent(x, y)) return;
                break;
            //pimgSrc の不透明部分のみ
            case 2:
                if(dat.pimgSrc->isPixelTransparent(x, y)) return;
                break;
        }
    }

    //

    (dat.pimgDst->*(CTileImage::m_pinfo->funcDrawPixel))(x, y, col);
}

//! 点を打つ (ドット円)

void drawPoint_dot(int x,int y,const DRAWPOINTDAT &dat)
{
    int ix,iy,yy,rr;
    RGBAFIX15 col;

    drawPoint_getColor(&col, dat);

    rr = dat.radius * dat.radius;

    for(iy = -dat.radius; iy <= dat.radius; iy++)
    {
        yy = iy * iy;

        for(ix = -dat.radius; ix <= dat.radius; ix++)
        {
            if(ix * ix + yy <= rr)
                drawPoint_setPixel(x + ix, y + iy, col, dat);
        }
    }
}

//! 点を打つ (アンチエイリアス円)

void drawPoint_aa(int x,int y,const DRAWPOINTDAT &dat)
{
    int ix,iy,i,j,xTbl[4],yTbl[4],rr,cnt,srca;
    RGBAFIX15 col;

    drawPoint_getColor(&col, dat);
    srca = col.a;

    rr = dat.radius << 2;
    rr *= rr;

    for(iy = -dat.radius; iy <= dat.radius; iy++)
    {
        //y テーブル (-2 は中央位置をずらす分)

        for(i = 0, j = (iy << 2) - 2; i < 4; i++, j++)
            yTbl[i] = j * j;

        //

        for(ix = -dat.radius; ix <= dat.radius; ix++)
        {
            //x テーブル

            for(i = 0, j = (ix << 2) - 2; i < 4; i++, j++)
                xTbl[i] = j * j;

            //4x4

            cnt = 0;

            for(i = 0; i < 4; i++)
            {
                for(j = 0; j < 4; j++)
                {
                    if(xTbl[j] + yTbl[i] < rr) cnt++;
                }
            }

            if(cnt)
            {
                col.a = srca * cnt >> 4;
                drawPoint_setPixel(x + ix, y + iy, col, dat);
            }
        }
    }
}

//! 点を打つ (柔らかい円)

void drawPoint_soft(int x,int y,const DRAWPOINTDAT &dat)
{
    int ix,iy,i,j,xTbl[4],yTbl[4],srca;
    double da,dr,rr;
    RGBAFIX15 col;

    drawPoint_getColor(&col, dat);
    srca = col.a;

    rr = 1.0 / ((dat.radius << 2) * (dat.radius << 2));

    for(iy = -dat.radius; iy <= dat.radius; iy++)
    {
        //y テーブル (-2 は中央位置をずらす分)

        for(i = 0, j = (iy << 2) - 2; i < 4; i++, j++)
            yTbl[i] = j * j;

        //

        for(ix = -dat.radius; ix <= dat.radius; ix++)
        {
            //x テーブル

            for(i = 0, j = (ix << 2) - 2; i < 4; i++, j++)
                xTbl[i] = j * j;

            //4x4

            da = 0;

            for(i = 0; i < 4; i++)
            {
                for(j = 0; j < 4; j++)
                {
                    dr = (xTbl[j] + yTbl[i]) * rr;

                    if(dr < 1.0) da += 1 - dr;
                }
            }

            if(da != 0)
            {
                col.a = (WORD)(srca * (da * 0.0625) + 0.5);
                drawPoint_setPixel(x + ix, y + iy, col, dat);
            }
        }
    }
}

//! 点を打つ (カレントブラシ)

void drawPoint_brush(int x,int y,const DRAWPOINTDAT &dat)
{
    drawPoint_getColor(&CTileImage::m_pinfo->colDraw, dat);

    dat.pimgDst->drawBrush_point(x, y, dat.radius, dat.opacity);
}

//! 点を打つ関数を指定タイプから取得

void getDrawPointFunc(DRAWPOINTFUNC *pfunc,int type)
{
    DRAWPOINTFUNC func[4] = {
        drawPoint_dot, drawPoint_aa, drawPoint_soft, drawPoint_brush
    };

    *pfunc = func[type];
}

};
