/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - ほか
*/

#include <math.h>

#include "CTileImage.h"

#include "AXMem.h"
#include "AXRand.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"



namespace filterdraw
{

//! モザイク

BOOL mozaic(FILTERDRAW &info)
{
    int ix,iy,size,xx,yy;
    double dcol[4],weight;
    RGBAFIX15 col;
    CTileImage *pimg;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    size   = info.valbar[0];
    weight = 1.0 / (size * size);
    pimg   = info.pimgDst;

    //

    info.progBeginOneStep(50, (info.rs.h + size - 1) / size);

    for(iy = info.rs.y1; iy <= info.rs.y2; iy += size)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix += size)
        {
            //平均

            dcol[0] = dcol[1] = dcol[2] = dcol[3] = 0;

            for(yy = 0; yy < size; yy++)
            {
                for(xx = 0; xx < size; xx++)
                    addAdvColor(dcol, info.pimgSrc, ix + xx, iy + yy, info.bClipping);
            }

            getAdvColor(&col, dcol, weight);

            //セット

            for(yy = 0; yy < size; yy++)
            {
                if(iy + yy > info.rs.y2) break;

                for(xx = 0; xx < size; xx++)
                {
                    if(ix + xx > info.rs.x2) break;

                    (pimg->*funcPix)(ix + xx, iy + yy, col);
                }
            }
        }

        info.progIncSub();
    }

    return TRUE;
}

//-------------------

typedef struct
{
    int x,y,cnt;
    double dcol[4];
    RGBAFIX15 col;
}FILTERDAT_CRYSTAL;

//-------------------

//! 水晶

BOOL crystal(FILTERDRAW &info)
{
    int ix,iy,i,j,xx,yy,size,xnum,ynum,minx,miny,mind,d;
    AXMem memBuf;
    FILTERDAT_CRYSTAL *pbuf,*p;
    CTileImage *pimg;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    size = info.valbar[0];

    xnum = (info.rs.w + size - 1) / size;
    ynum = (info.rs.h + size - 1) / size;

    //メモリ確保

    if(!memBuf.allocClear(xnum * ynum * sizeof(FILTERDAT_CRYSTAL)))
        return FALSE;

    pbuf = (FILTERDAT_CRYSTAL *)memBuf.getBuf();

    //各ブロックの中央位置をランダムで決める

    p = pbuf;

    for(iy = 0; iy < ynum; iy++)
    {
        for(ix = 0; ix < xnum; ix++, p++)
        {
            xx = info.rs.x1 + ix * size + CTileImage::m_prand->getRange(0, size - 1);
            yy = info.rs.y1 + iy * size + CTileImage::m_prand->getRange(0, size - 1);

            if(xx > info.rs.x2) xx = info.rs.x2;
            if(yy > info.rs.y2) yy = info.rs.y2;

            p->x = xx;
            p->y = yy;
        }
    }

    //平均色の場合：ブロック内の色を加算

    pimg = info.pimgSrc;

    if(info.valcheck[0])
    {
        for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
        {
            yy = (iy - info.rs.y1) / size;

            for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
            {
                xx = (ix - info.rs.x1) / size;

                //中央位置からの距離が一番短いブロック

                mind = 0x7fffffff;
                minx = miny = 0;

                for(i = yy - 1; i < yy + 2; i++)
                {
                    if(i < 0 || i >= ynum) continue;

                    for(j = xx - 1; j < xx + 2; j++)
                    {
                        if(j < 0 || j >= xnum) continue;

                        p = pbuf + i * xnum + j;

                        d = (p->x - ix) * (p->x - ix) + (p->y - iy) * (p->y - iy);

                        if(d < mind)
                        {
                            mind = d;
                            minx = j;
                            miny = i;
                        }
                    }
                }

                //

                p = pbuf + miny * xnum + minx;

                addAdvColor(p->dcol, pimg, ix, iy, info.bClipping);
                p->cnt++;
            }
        }
    }

    //ブロック色を計算

    p = pbuf;

    for(i = 0; i < ynum; i++)
    {
        for(j = 0; j < xnum; j++, p++)
        {
            if(info.valcheck[0])
                //平均色
                getAdvColor(&p->col, p->dcol, 1.0 / p->cnt);
            else
                //中央色
                pimg->getPixel(&p->col, p->x, p->y);
        }
    }

    //色セット

    pimg = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        yy = (iy - info.rs.y1) / size;

        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            xx = (ix - info.rs.x1) / size;

            mind = 0x7fffffff;
            minx = miny = 0;

            for(i = yy - 1; i < yy + 2; i++)
            {
                if(i < 0 || i >= ynum) continue;

                for(j = xx - 1; j < xx + 2; j++)
                {
                    if(j < 0 || j >= xnum) continue;

                    p = pbuf + i * xnum + j;

                    d = (p->x - ix) * (p->x - ix) + (p->y - iy) * (p->y - iy);

                    if(d < mind)
                    {
                        mind = d;
                        minx = j;
                        miny = i;
                    }
                }
            }

            p = pbuf + miny * xnum + minx;

            (pimg->*funcPix)(ix, iy, p->col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! ハーフトーン

BOOL halftone(FILTERDRAW &info)
{
    int x,y,ix,iy,i,c;
    double dsin[3],dcos[3],xx[3],yy[3],len,len_div,len_half,rr,dx,dy;
    double rrtmp,dx2,dy2,dyy;
    RGBAFIX15 col;
    CTileImage *pimg;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    len = info.valbar[0] * 0.1;

    len_div  = 1.0 / len;
    len_half = len * 0.5;
    rrtmp    = len * len / M_PI / 0x8000;

    for(i = 0; i < 3; i++)
    {
        c = info.valbar[1 + i];
        if(i && info.valcheck[0]) c = info.valbar[1];

        rr = -c * M_PI / 180.0;
        dcos[i] = ::cos(rr);
        dsin[i] = ::sin(rr);
    }

    //----------------

    pimg = info.pimgDst;

    for(y = info.rs.y1; y <= info.rs.y2; y++)
    {
        //最初の位置

        dx = info.rs.x1;
        dy = y;

        for(i = 0; i < 3; i++)
        {
            xx[i] = dx * dcos[i] - dy * dsin[i];
            yy[i] = dx * dsin[i] + dy * dcos[i];
        }

        //

        for(x = info.rs.x1; x <= info.rs.x2; x++)
        {
            info.pimgSrc->getPixel(&col, x, y);

            //透明部分は処理なし

            if(col.a == 0)
            {
                for(i = 0; i < 3; i++)
                {
                    xx[i] += dcos[i];
                    yy[i] += dsin[i];
                }
                continue;
            }

            //グレイスケール

            if(info.valcheck[2])
                col.r = col.g = col.b = _GETV(col.r, col.g, col.b);

            //RGB

            for(i = 0; i < 3; i++)
            {
                //枠の中央位置からの距離

                dx = xx[i] - (::floor(xx[i] * len_div) * len + len_half);
                dy = yy[i] - (::floor(yy[i] * len_div) * len + len_half);

                //RGB値から半径取得

                rr = ::sqrt(col.c[i] * rrtmp);
                rr *= rr;

                //色取得

                if(col.c[i] == 0x8000)
                    c = 0x8000;
                else if(info.valcheck[1])
                {
                    //アンチエイリアス(サブピクセルごとに計算しないとはみ出している部分が問題になる)

                    c = 0;

                    for(iy = 0, dy2 = dy; iy < 5; iy++, dy2 += 0.2)
                    {
                        if(dy2 >= len_half) dy2 -= len;
                        dyy = dy2 * dy2;

                        for(ix = 0, dx2 = dx; ix < 5; ix++, dx2 += 0.2)
                        {
                            if(dx2 >= len_half) dx2 -= len;

                            if(dx2 * dx2 + dyy < rr) c++;
                        }
                    }

                    c = (c << 15) / 25;
                }
                else
                {
                    //非アンチエイリアス

                    if((int)(dx * dx + dy * dy + 0.5) < (int)(rr + 0.5))
                        c = 0x8000;
                    else
                        c = 0;
                }

                col.c[i] = c;

                //

                xx[i] += dcos[i];
                yy[i] += dsin[i];
            }

            //描画

            (pimg->*funcPix)(x, y, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! シャープ

BOOL sharp(FILTERDRAW &info)
{
    FILTER3X3DAT dat;
    int i;

    for(i = 0; i < 9; i++)
        dat.mul[i] = info.valbar[0] * -0.01;

    dat.mul[4] = info.valbar[0] * 0.08 + 1;
    dat.divmul = 1;
    dat.add    = 0;

    return filter3x3(info, dat);
}

//! アンシャープマスク

BOOL unsharpmask(FILTERDRAW &info)
{
    CTileImage *pimgTmp,*pimg;
    int ix,iy,i,c,amount;
    RGBAFIX15 cols,colt;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    pimgTmp = allocTmpImage(info, info.rs);
    if(!pimgTmp) return FALSE;

    //ガウスぼかし (src -> tmp)

    if(!tmpGaussBlur(info, info.pimgSrc, pimgTmp, info.valbar[0], &CTileImage::setPixel_create2))
        return FALSE;

    //適用

    amount = (info.valbar[1] << 7) / 100;
    pimg   = info.pimgDst;

    info.progBeginOneStep(50, info.rs.h);

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            info.pimgSrc->getPixel(&cols, ix, iy);

            if(cols.a == 0) continue;

            pimgTmp->getPixel(&colt, ix, iy);

            //RGB

            for(i = 0; i < 3; i++)
            {
                c = cols.c[i] + ((cols.c[i] - colt.c[i]) * amount >> 7);
                if(c < 0) c = 0; else if(c > 0x8000) c = 0x8000;

                cols.c[i] = c;
            }

            (pimg->*funcPix)(ix, iy, cols);
        }

        info.progIncSub();
    }

    delete pimgTmp;

    return TRUE;
}

//! ハイパス

BOOL highpass(FILTERDRAW &info)
{
    CTileImage *pimgTmp,*pimg;
    int ix,iy,i,c;
    RGBAFIX15 cols,colt;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    pimgTmp = allocTmpImage(info, info.rs);
    if(!pimgTmp) return FALSE;

    //ガウスぼかし (src -> tmp)

    if(!tmpGaussBlur(info, info.pimgSrc, pimgTmp, info.valbar[0], &CTileImage::setPixel_create2))
        return FALSE;

    //適用

    pimg = info.pimgDst;

    info.progBeginOneStep(50, info.rs.h);

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            info.pimgSrc->getPixel(&cols, ix, iy);

            if(cols.a == 0) continue;

            pimgTmp->getPixel(&colt, ix, iy);

            //RGB

            for(i = 0; i < 3; i++)
            {
                c = cols.c[i] - colt.c[i] + 0x4000;
                if(c < 0) c = 0; else if(c > 0x8000) c = 0x8000;

                cols.c[i] = c;
            }

            (pimg->*funcPix)(ix, iy, cols);
        }

        info.progIncSub();
    }

    delete pimgTmp;

    return TRUE;
}

//! 輪郭抽出:sobel

BOOL edge_sobel(FILTERDRAW &info)
{
    CTileImage *pimg,*pimgSrc;
    int ix,iy,jx,jy,i,no;
    RGBAFIX15 cols,col;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;
    double ch[3],cv[3];
    int hmask[9] = { -1,-2,-1, 0,0,0, 1,2,1 },
        vmask[9] = { -1,0,1, -2,0,2, -1,0,1 };

    pimgSrc = info.pimgSrc;
    pimg    = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimgSrc->getPixel(&cols, ix, iy);
            if(!cols.a) continue;

            //

            ch[0] = ch[1] = ch[2] = 0;
            cv[0] = cv[1] = cv[2] = 0;

            for(jy = -1, no = 0; jy <= 1; jy++)
            {
                for(jx = -1; jx <= 1; jx++, no++)
                {
                    pimgSrc->getPixel(&col, ix + jx, iy + jy, info.bClipping);

                    for(i = 0; i < 3; i++)
                    {
                        ch[i] += (double)(col.c[i] * hmask[no]) / 0x8000;
                        cv[i] += (double)(col.c[i] * vmask[no]) / 0x8000;
                    }
                }
            }

            //

            for(i = 0; i < 3; i++)
            {
                no = (int)(::sqrt(ch[i] * ch[i] + cv[i] * cv[i]) * 0x8000 + 0.5);
                if(no > 0x8000) no = 0x8000;

                cols.c[i] = no;
            }

            (pimg->*funcPix)(ix, iy, cols);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 輪郭抽出:Laplacian

BOOL edge_laplacian(FILTERDRAW &info)
{
    FILTER3X3DAT dat;
    int i;

    for(i = 0; i < 9; i++)
        dat.mul[i] = -1;

    dat.mul[4] = 8;
    dat.divmul = 1;
    dat.add    = 0x8000;

    return filter3x3(info, dat);
}

//! 輝度から線画抽出

BOOL lumtoalpha_func(FILTERDRAW &info,RGBAFIX15 *p,int x,int y)
{
    if(!p->a)
        return FALSE;
    else
    {
        p->a = 0x8000 - _GETV(p->r, p->g, p->b);
        p->r = p->g = p->b = 0;

        return TRUE;
    }
}

BOOL lumtoalpha(FILTERDRAW &info)
{
    return common1(info, lumtoalpha_func);
}

//! 立体枠

BOOL button3D(FILTERDRAW &info)
{
    int i,j,cnt,v,vsrc,bSmooth,w,h,bw;
    CTileImage *pimg = info.pimgDst;

    w = CTileImage::m_pinfo->nImgW;
    h = CTileImage::m_pinfo->nImgH;

    bw = info.valbar[0];
    bSmooth = info.valcheck[0];

    //横線

    vsrc = (bSmooth)? 17348: 10666;
    if(info.valcheck[1]) vsrc = -vsrc;

    for(i = 0, cnt = w; i < bw; i++, cnt -= 2)
    {
        if((i << 1) >= h)  break;

        v = (bSmooth)? vsrc - vsrc * i / bw: vsrc;

        for(j = 0; j < cnt; j++)
            pimg->setPixelDraw_addsub(i + j, i, v);

        for(j = 0; j < cnt - 2; j++)
            pimg->setPixelDraw_addsub(i + 1 + j, h - 1 - i, -v);
    }

    //縦線

    vsrc = (bSmooth)? 13493: 8224;
    if(info.valcheck[1]) vsrc = -vsrc;

    for(i = 0, cnt = h - 1; i < bw; i++, cnt -= 2)
    {
        if((i << 1) >= w) break;

        v = (bSmooth)? vsrc - vsrc * i / bw: vsrc;

        for(j = 0; j < cnt; j++)
            pimg->setPixelDraw_addsub(i, i + 1 + j, v);

        for(j = 0; j < cnt; j++)
            pimg->setPixelDraw_addsub(w - 1 - i, i + 1 + j, -v);
    }

    return TRUE;
}

//! シフト

BOOL shift(FILTERDRAW &info)
{
    int ix,iy,w,h,x,y;
    RGBAFIX15 col;
    CTileImage *pimgSrc,*pimgDst;
    void (CTileImage::*funcPix)(int,int,const RGBAFIX15&) = CTileImage::m_pinfo->funcDrawPixel;

    w = CTileImage::m_pinfo->nImgW;
    h = CTileImage::m_pinfo->nImgH;

    pimgSrc = info.pimgSrc;
    pimgDst = info.pimgDst;

    info.progBeginOneStep(50, h);

    for(iy = 0; iy < h; iy++)
    {
        y = iy - info.valbar[1];

        while(y < 0) y += h;
        while(y >= h) y -= h;

        for(ix = 0; ix < w; ix++)
        {
            x = ix - info.valbar[0];

            while(x < 0) x += w;
            while(x >= w) x -= w;

            pimgSrc->getPixel(&col, x, y);

            (pimgDst->*funcPix)(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

};
