/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - 漫画用
*/

#include <math.h>

#include "CTileImage.h"

#include "AXRand.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"

#include "drawdat.h"
#include "global.h"


namespace filterdraw
{

//! 集中線/フラッシュ描画
/*
    type : [0]集中線/フラッシュ (円の外周から中心へ向けて)
           [1]吹き出しフラッシュ (円の外周から外へ向けて&中を塗りつぶし)
           [2]ウニフラッシュ
    valbar : [0]半径(px)
             [1]縦横比(-1.000〜+1.000)
             [2]密度最小(角度の分割数/2)
             [3]間隔ランダム幅(0-99, 角度ステップに対する%)
             [4]線の太さ(px)
             [5]線の太さランダム(px, 0.0-)
             [6]線の長さ(0.1-100.0, 半径に対する%)
             [7]線の長さランダム幅(0.0-100.0, 半径に対する%)
    描画タイプ : [0]ブラシ・アンチエイリアスあり
               [1]ブラシ・アンチエイリアスなし
               [2]1pxドット線
*/

BOOL _comic_flash_common(FILTERDRAW &info,int type)
{
    double angle,angleStep,angleRnd,r,len,lenRnd,xscale,yscale,size,sizeRnd,sizeRes;
    double xpos,ypos,x1,x2,y1,y2,sind,cosd,tmp,line_r1,xx,yy;
    WORD wHeadTail;
    int drawtype;
    BOOL bSample;
    RGBAFIX15 col;
    BRUSHDRAWPARAM *pbrush = CTileImage::m_pbrush;
    CTileImage *pimg = info.pimgDst;

    //簡易プレビュー

    bSample = (info.isPrev() && info.valcheck[0]);

    //色

    if(info.isPrev() && info.valcheck[1])
        col.r = 0x8000, col.g = col.b = 0;
    else
        col = g_draw->colDraw;

    col.a = 0x8000;

    CTileImage::m_pinfo->colDraw = col;

    //

    drawtype  = info.valcombo[0];
    xpos      = info.tmpx;
    ypos      = info.tmpy;
    r         = info.valbar[0];
    size      = info.valbar[4] * 0.1;
    sizeRnd   = info.valbar[5] * 0.1;
    len       = r * (info.valbar[6] * 0.001);
    lenRnd    = r * (info.valbar[7] * 0.001);
    angleStep = M_PI * 2 / (info.valbar[2] * 2);
    angleRnd  = angleStep * (info.valbar[3] * 0.01);

    if(info.valbar[1] < 0)
        xscale = 1, yscale = 1.0 + info.valbar[1] * 0.001;
    else
        xscale = 1.0 - info.valbar[1] * 0.001, yscale = 1;

    wHeadTail = (type == 2)? (50 << 8)|50: 100;

    //描画準備

    pbrush->dMinOpacity = (drawtype == 0)? 0: 1;
    pbrush->dwFlags     = (drawtype == 1)? BRUSHDF_NONEAA: 0;

    g_draw->drawinfo.funcColor = (drawtype == 0 && !bSample)? &CTileImage::colfunc_normal: &CTileImage::colfunc_overwrite;


    //-------- 描画

    info.progBeginOneStep(50, info.valbar[2] * 2);

    for(angle = 0; angle < M_PI * 2; angle += angleStep)
    {
        //角度

        tmp = angle;
        if(info.valbar[3]) tmp += CTileImage::m_prand->getDouble() * angleRnd;

        cosd = ::cos(tmp);
        sind = ::sin(tmp);

        //位置

        switch(type)
        {
            //集中線・フラッシュ
            case 0:
                line_r1 = r;

                tmp = r - len;
                if(info.valbar[7])
                    tmp -= CTileImage::m_prand->getDouble() * lenRnd;
                break;
            //吹き出しフラッシュ
            case 1:
                line_r1 = r;

                tmp = r + len;
                if(info.valbar[7])
                    tmp += CTileImage::m_prand->getDouble() * lenRnd;
                break;
            //ウニフラッシュ
            case 2:
                tmp = len;
                if(info.valbar[7])
                    tmp += CTileImage::m_prand->getDouble() * lenRnd;

                tmp *= 0.5;

                line_r1 = r + tmp;
                tmp     = r - tmp;
                break;
        }

        xx = cosd * xscale;
        yy = sind * yscale;

        x1 = line_r1 * xx + xpos;
        y1 = line_r1 * yy + ypos;
        x2 = tmp * xx + xpos;
        y2 = tmp * yy + ypos;

        //線の半径

        sizeRes = size;

        if(info.valbar[5])
            sizeRes += CTileImage::m_prand->getDouble() * sizeRnd;

        pbrush->dRadius = sizeRes;

        //描画

        if(drawtype == 2)
            pimg->drawLineB(x1, y1, x2, y2, col, FALSE);
        else if(!bSample)
            pimg->drawBrush_lineHeadTail(x1, y1, x2, y2, wHeadTail);
        else
        {
            //簡易プレビュー

            if(sizeRes <= 0.5)
                pimg->drawLineB(x1, y1, x2, y2, col, FALSE);
            else if(type == 2)
            {
                //ウニフラッシュ

                xx = x1 + (x2 - x1) * 0.5;
                yy = y1 + (y2 - y1) * 0.5;

                sind = sizeRes * -sind * xscale;
                cosd = sizeRes *  cosd * yscale;

                pimg->drawLineB(x1, y1, xx + sind, yy + cosd, col, FALSE);
                pimg->drawLineB(xx + sind, yy + cosd, x2, y2, col, FALSE);
                pimg->drawLineB(x2, y2, xx - sind, yy - cosd, col, FALSE);
                pimg->drawLineB(xx - sind, yy - cosd, x1, y1, col, FALSE);
            }
            else
            {
                //フラッシュ

                sind = sizeRes * -sind * xscale;
                cosd = sizeRes *  cosd * yscale;

                pimg->drawLineB(x2, y2, x1 + sind, y1 + cosd, col, FALSE);
                pimg->drawLineB(x1 + sind, y1 + cosd, x1 - sind, y1 - cosd, col, FALSE);
                pimg->drawLineB(x1 - sind, y1 - cosd, x2, y2, col, FALSE);
            }
        }

        //

        info.progIncSub();
    }

    //吹き出しフラッシュ、内側塗りつぶし

    if(type == 1)
    {
        CANVASVIEWPARAM param;

        param.dCos = 1, param.dSin = 0;
        param.dCosRev = 1, param.dSinRev = 0;

        if(bSample)
            pimg->drawCircle(xpos, ypos, r * xscale, r * yscale, param, FALSE, FALSE);
        else
            pimg->drawFillCircle(xpos, ypos, r * xscale, r * yscale, param, FALSE, 100, FALSE);
    }

    return TRUE;
}

//! 集中線/フラッシュ

BOOL comic_concline_flash(FILTERDRAW &info)
{
    return _comic_flash_common(info, 0);
}

//! 吹き出しフラッシュ

BOOL comic_popupflash(FILTERDRAW &info)
{
    return _comic_flash_common(info, 1);
}

//! ウニフラッシュ

BOOL comic_uniflash(FILTERDRAW &info)
{
    return _comic_flash_common(info, 2);
}

//! ウニフラッシュ(波)
/*
    valbar : [0]半径(px)
             [1]縦横比(-1.000〜+1.000)
             [2]波の密度(角度の分割数)
             [3]波の長さ(半径に対する%)
             [4]線の太さ(px)
             [5]線の長さ(0.1-100.0, 半径に対する%)
             [6]抜きの最小(%)
*/

BOOL comic_uniflash_wave(FILTERDRAW &info)
{
    double angle,angleStep,angleStep2,r,len,xscale,yscale;
    double xpos,ypos,x1,x2,y1,y2,sind,cosd,tmp,tblR[6];
    int i,bSample,drawtype;
    RGBAFIX15 col;
    BRUSHDRAWPARAM *pbrush = CTileImage::m_pbrush;
    CTileImage *pimg = info.pimgDst;

    //簡易プレビュー

    bSample = (info.isPrev() && info.valcheck[0]);

    //色

    if(info.isPrev() && info.valcheck[1])
        col.r = 0x8000, col.g = col.b = 0;
    else
        col = g_draw->colDraw;

    col.a = 0x8000;

    CTileImage::m_pinfo->colDraw = col;

    //

    drawtype   = info.valcombo[0];
    xpos       = info.tmpx;
    ypos       = info.tmpy;
    r          = info.valbar[0];
    len        = r * (info.valbar[5] * 0.001);
    angleStep  = M_PI * 2 / info.valbar[2];
    angleStep2 = angleStep / 6;

    if(info.valbar[1] < 0)
        xscale = 1, yscale = 1.0 + info.valbar[1] * 0.001;
    else
        xscale = 1.0 - info.valbar[1] * 0.001, yscale = 1;

    //

    tmp = r * (info.valbar[3] * 0.001);

    tblR[0] = 0;
    tblR[1] = tmp / 3;
    tblR[2] = tmp * 2 / 3;
    tblR[3] = tmp;
    tblR[4] = tblR[2];
    tblR[5] = tblR[1];

    //描画準備

    g_draw->drawinfo.funcColor = (drawtype == 0 && !bSample)? &CTileImage::colfunc_normal: &CTileImage::colfunc_overwrite;

    pbrush->dRadius     = info.valbar[4] * 0.1;
    pbrush->dMinSize    = info.valbar[6] * 0.01;
    pbrush->dMinOpacity = 1;
    pbrush->dwFlags     = (drawtype == 1)? BRUSHDF_NONEAA: 0;


    //-------- 描画

    info.progBeginOneStep(50, info.valbar[2]);

    for(angle = 0; angle < M_PI * 2; angle += angleStep)
    {
        //1つの波に6個描画

        for(i = 0; i < 6; i++)
        {
            tmp = angle + i * angleStep2;

            cosd = ::cos(tmp) * xscale;
            sind = ::sin(tmp) * yscale;

            //位置

            tmp = r + tblR[i];

            x1 = tmp * cosd + xpos;
            y1 = tmp * sind + ypos;

            tmp -= len;

            x2 = tmp * cosd + xpos;
            y2 = tmp * sind + ypos;

            //描画

            if(drawtype == 2)
                pimg->drawLineB(x1, y1, x2, y2, col, FALSE);
            else if(!bSample)
                pimg->drawBrush_lineHeadTail(x1, y1, x2, y2, 100);
            else
            {
                //簡易プレビュー

                sind = pbrush->dRadius * -sind * xscale;
                cosd = pbrush->dRadius *  cosd * yscale;

                pimg->drawLineB(x2, y2, x1 + sind, y1 + cosd, col, FALSE);
                pimg->drawLineB(x1 + sind, y1 + cosd, x1 - sind, y1 - cosd, col, FALSE);
                pimg->drawLineB(x1 - sind, y1 - cosd, x2, y2, col, FALSE);
            }
        }

        info.progIncSub();
    }

    return TRUE;
}

};
