/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    フィルタ描画関数 - カラー/アルファ操作
*/

#include <math.h>

#include "CTileImage.h"
#include "CTileImageA1.h"
#include "CImage8.h"
#include "CGradList.h"

#include "CLayerList.h"
#include "CLayerItem.h"

#include "filterdraw.h"
#include "filterdrawfunc.h"

#include "drawdat.h"
#include "global.h"

#include "AXRand.h"
#include "AXUtilColor.h"


typedef void (*COMMONCOLFUNC)(FILTERDRAW &,int *);


namespace filterdraw
{

//==============================
// 共通処理
//==============================


//! 色操作の共通処理（キャンバスプレビュー用）
/*
    レイヤのカラータイプは RGBA か グレイスケール。アルファ値のみの場合は除く。
*/

BOOL commonColor_prev(FILTERDRAW &info,COMMONCOLFUNC funcCol)
{
    CTileImage::TILERECTINFO tinfo;
    AXRectSize rcs;
    void **ppSrc,**ppDst;
    int tx,ty,px,py,ix,iy,bRGBA,col[3];
    LPWORD pwSrc,pwDst;

    //処理する範囲の情報

    rcs.set(info.rs.x1, info.rs.y1, info.rs.w, info.rs.h);

    ppSrc = info.pimgSrc->getTileRect_inImageInfo(&tinfo, rcs);
    if(!ppSrc) return TRUE;

    //---------

    ppDst = info.pimgDst->getTileBufPt(tinfo.rcTile.left, tinfo.rcTile.top);
    bRGBA = (info.pimgSrc->getColType() == CTileImage::COLTYPE_RGBA);

    py = tinfo.ptTopPx.y;

    for(ty = tinfo.rcTile.bottom - tinfo.rcTile.top + 1; ty > 0; ty--, py += 64)
    {
        px = tinfo.ptTopPx.x;

        for(tx = tinfo.rcTile.right - tinfo.rcTile.left + 1; tx > 0; tx--, px += 64, ppSrc++, ppDst++)
        {
            if(!(*ppSrc)) continue;

            //タイルで処理

            pwSrc = (LPWORD)(*ppSrc);
            pwDst = (LPWORD)(*ppDst);

            if(bRGBA)
            {
                //RGBA

                for(iy = 0; iy < 64; iy++)
                {
                    info.tmpy = py + iy;

                    for(ix = 0; ix < 64; ix++, pwSrc += 4, pwDst += 4)
                    {
                        if(info.pimgSel && !info.pimgSel->isPixelOn(px + ix, py + iy))
                            continue;

                        if(pwSrc[3])
                        {
                            col[0] = pwSrc[0];
                            col[1] = pwSrc[1];
                            col[2] = pwSrc[2];

                            info.tmpx = px + ix;

                            (*funcCol)(info, col);

                            pwDst[0] = col[0];
                            pwDst[1] = col[1];
                            pwDst[2] = col[2];
                        }
                    }
                }
            }
            else
            {
                //グレイスケール

                for(iy = 0; iy < 64; iy++)
                {
                    info.tmpy = py + iy;

                    for(ix = 0; ix < 64; ix++, pwSrc += 2, pwDst += 2)
                    {
                        if(info.pimgSel && !info.pimgSel->isPixelOn(px + ix, py + iy))
                            continue;

                        if(pwSrc[1])
                        {
                            col[0] = col[1] = col[2] = *pwSrc;

                            info.tmpx = px + ix;

                            (*funcCol)(info, col);

                            *pwDst = col[0];
                        }
                    }
                }
            }
        }

        ppSrc += tinfo.pitch;
        ppDst += tinfo.pitch;
    }

    return TRUE;
}

//! 色操作の共通処理（実際の描画用）

BOOL commonColor_real(FILTERDRAW &info,COMMONCOLFUNC funcCol)
{
    int ix,iy,c[3];
    RGBAFIX15 col;
    CTileImage *pimg = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        info.tmpy = iy;

        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimg->getPixel(&col, ix, iy);

            if(col.a)
            {
                c[0] = col.r;
                c[1] = col.g;
                c[2] = col.b;

                info.tmpx = ix;

                (*funcCol)(info, c);

                col.r = c[0];
                col.g = c[1];
                col.b = c[2];

                pimg->setPixelDraw(ix, iy, col);
            }
        }

        info.progIncSub();
    }

    return TRUE;
}

//! 色操作の共通処理

BOOL commonColor(FILTERDRAW &info,COMMONCOLFUNC funcCol)
{
    if(info.bPreview)
        return commonColor_prev(info, funcCol);
    else
        return commonColor_real(info, funcCol);
}


//==============================


//! 明度・コントラスト調整

void colfunc_brightcont(FILTERDRAW &info,int *pcol)
{
    int i,n;

    //コントラスト

    for(i = 0; i < 3; i++)
    {
        n = pcol[i];
        n += (n - 0x4000) * info.ntmp[1] >> 8;

        if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

        pcol[i] = n;
    }

    //明度

    RGBtoYCrCb(pcol);
    pcol[0] += info.ntmp[0];
    YCrCbtoRGB(pcol);
}

BOOL color_brightcont(FILTERDRAW &info)
{
    info.ntmp[0] = ((info.valbar[0] << 15) + 50) / 100;
    info.ntmp[1] = ((info.valbar[1] << 8) + 50) / 100;

    return commonColor(info, colfunc_brightcont);
}

//! ガンマ補正

void colfunc_gamma(FILTERDRAW &info,int *pcol)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = ::pow((double)pcol[i] / 0x8000, info.dtmp[0]) * 0x8000;

        if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

        pcol[i] = n;
    }
}

BOOL color_gamma(FILTERDRAW &info)
{
    info.dtmp[0] = 1.0 / (info.valbar[0] * 0.01);

    return commonColor(info, colfunc_gamma);
}

//! レベル補正
/*
    [0]入力最小 [1]入力中間 [2]入力最大 [3]出力最小 [4]出力最大
*/

void colfunc_level(FILTERDRAW &info,int *pcol)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        //入力値制限

        n = pcol[i];

        if(n < info.valbar[0]) n = info.valbar[0];
        else if(n > info.valbar[2]) n = info.valbar[2];

        //補正

        if(n < info.valbar[1])
            n = (n - info.valbar[0]) * info.ntmp[2] / info.ntmp[0];
        else
            n = (n - info.valbar[1]) * info.ntmp[3] / info.ntmp[4] + info.ntmp[2];

        pcol[i] = n + info.valbar[3];
    }
}

BOOL color_level(FILTERDRAW &info)
{
    info.ntmp[0] = info.valbar[1] - info.valbar[0];
    info.ntmp[1] = info.valbar[4] - info.valbar[3];
    info.ntmp[2] = info.ntmp[1] >> 1;
    info.ntmp[3] = info.ntmp[1] - info.ntmp[2];
    info.ntmp[4] = info.valbar[2] - info.valbar[1];

    return commonColor(info, colfunc_level);
}

//! RGB調整

void colfunc_rgb(FILTERDRAW &info,int *pcol)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        n = pcol[i] + info.ntmp[i];
        if(n < 0) n = 0; else if(n > 0x8000) n = 0x8000;

        pcol[i] = n;
    }
}

BOOL color_rgb(FILTERDRAW &info)
{
    int i;

    for(i = 0; i < 3; i++)
        info.ntmp[i] = ((info.valbar[i] << 15) + 500) / 1000;

    return commonColor(info, colfunc_rgb);
}

//! HSV調整

void colfunc_hsv(FILTERDRAW &info,int *pcol)
{
    RGBtoHSV(pcol);

    pcol[0] += info.valbar[0];
    pcol[1] += info.ntmp[0];
    pcol[2] += info.ntmp[1];

    if(pcol[0] < 0) pcol[0] += 360; else if(pcol[0] >= 360) pcol[0] -= 360;
    if(pcol[1] < 0) pcol[1] = 0; else if(pcol[1] > 0x8000) pcol[1] = 0x8000;
    if(pcol[2] < 0) pcol[2] = 0; else if(pcol[2] > 0x8000) pcol[2] = 0x8000;

    HSVtoRGB(pcol);
}

BOOL color_hsv(FILTERDRAW &info)
{
    int i;

    for(i = 0; i < 2; i++)
        info.ntmp[i] = ((info.valbar[i + 1] << 15) + 50) / 100;

    return commonColor(info, colfunc_hsv);
}

//! ネガポジ反転

void colfunc_nega(FILTERDRAW &info,int *pcol)
{
    pcol[0] = 0x8000 - pcol[0];
    pcol[1] = 0x8000 - pcol[1];
    pcol[2] = 0x8000 - pcol[2];
}

BOOL color_nega(FILTERDRAW &info)
{
    return commonColor_real(info, colfunc_nega);
}

//! グレイスケール

void colfunc_grayscale(FILTERDRAW &info,int *pcol)
{
    int n;

    n = _GETV(pcol[0], pcol[1], pcol[2]);

    pcol[0] = pcol[1] = pcol[2] = n;
}

BOOL color_grayscale(FILTERDRAW &info)
{
    return commonColor_real(info, colfunc_grayscale);
}

//! セピアカラー

void colfunc_sepia(FILTERDRAW &info,int *pcol)
{
    int v;

    v = _GETV(pcol[0], pcol[1], pcol[2]);

    pcol[0] = v + 1928; if(pcol[0] > 0x8000) pcol[0] = 0x8000;
    pcol[1] = v;
    pcol[2] = v - 964; if(pcol[2] < 0) pcol[2] = 0;
}

BOOL color_sepia(FILTERDRAW &info)
{
    return commonColor_real(info, colfunc_sepia);
}

//! 2値化

void colfunc_threshold(FILTERDRAW &info,int *pcol)
{
    int v = _GETV(pcol[0], pcol[1], pcol[2]);

    if(v < info.ntmp[0]) v = 0;
    else v = 0x8000;

    pcol[0] = pcol[1] = pcol[2] = v;
}

BOOL color_threshold(FILTERDRAW &info)
{
    info.ntmp[0] = ((info.valbar[0] << 15) + 500) / 1000;

    return commonColor(info, colfunc_threshold);
}

//! 2値化(ディザ)
/*
    [0]bayer2x2 [1]bayer4x4 [2]渦巻き4x4 [3]網点4x4 [4]ランダム
*/

void colfunc_threshold_dither(FILTERDRAW &info,int *pcol)
{
    BYTE val_bayer2x2[4] = {0,2,3,1};
    BYTE val_4x4[3][16] = {
        {0,8,2,10, 12,4,14,6, 3,11,1,9, 15,7,13,5},
        {13,7,6,12, 8,1,0,5, 9,2,3,4, 14,10,11,15},
        {10,4,6,8, 12,0,2,14, 7,9,11,5, 3,15,13,1}
    };
    int v,cmp;

    switch(info.valcombo[0])
    {
        case 0:
            cmp = val_bayer2x2[((info.tmpy & 1) << 1) + (info.tmpx & 1)] << 13;
            break;
        case 1:
        case 2:
        case 3:
            cmp = val_4x4[info.valcombo[0] - 1][((info.tmpy & 3) << 2) + (info.tmpx & 3)] << 11;
            break;
        case 4:
            cmp = CTileImage::m_prand->getDWORD() & 0x7fff;
            break;
    }

    v = _GETV(pcol[0], pcol[1], pcol[2]);
    v = (v <= cmp)? 0: 0x8000;

    pcol[0] = pcol[1] = pcol[2] = v;
}

BOOL color_threshold_dither(FILTERDRAW &info)
{
    return commonColor(info, colfunc_threshold_dither);
}

//! ポスタリゼーション

void colfunc_posterize(FILTERDRAW &info,int *pcol)
{
    int i,n;

    for(i = 0; i < 3; i++)
    {
        if(pcol[i] == 0x8000)
            n = info.ntmp[0];
        else
            n = pcol[i] * info.valbar[0] >> 15;

        pcol[i] = (n << 15) / info.ntmp[0];
    }
}

BOOL color_posterize(FILTERDRAW &info)
{
    info.ntmp[0] = info.valbar[0] - 1;

    return commonColor(info, colfunc_posterize);
}

//! グラデーションマップ

BOOL color_gradmap(FILTERDRAW &info)
{
    int ix,iy,v;
    RGBAFIX15 col;
    CTileImage *pimg;
    AXMem mem;
    LPBYTE pDat;

    //選択されているカスタムグラデーションのデータ

    if(!GRADLIST->setDrawDat(&mem, (g_draw->tool.optGrad >> 16) & 255,
            g_draw->colDraw, g_draw->colBack, g_draw->tool.optGrad & (1<<12), FALSE))
        return FALSE;

    pDat = mem;

    //

    pimg = info.pimgDst;

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimg->getPixel(&col, ix, iy);
            if(!col.a) continue;

            v = _GETV(col.r, col.g, col.b);

            pimg->drawGradient_pix(ix, iy, (double)v / 0x8000, 256, pDat);
        }

        info.progIncSub();
    }

    return TRUE;
}


//===================================
// 色操作
//===================================


//! 描画色の置換
/*
    描画色の判定は RGB8bit に変換して行う。
    16bit で直接判定すると、微妙に色が違う部分が置換できない。
*/

void colopfunc_drawcol_rep(FILTERDRAW &info,int *pcol)
{
    int i;
    DWORD c[3];

    for(i = 0; i < 3; i++)
        c[i] = (pcol[i] * 255 + 0x4000) >> 15;

    if(info.dwDrawCol == _RGB(c[0], c[1], c[2]))
    {
        pcol[0] = info.ntmp[0];
        pcol[1] = info.ntmp[1];
        pcol[2] = info.ntmp[2];
    }
}

BOOL colorop_drawcol_rep(FILTERDRAW &info)
{
    int i;

    AXHSVtoRGB(info.valbar[0], info.valbar[1] / 255.0, info.valbar[2] / 255.0, info.ntmp);

    for(i = 0; i < 3; i++)
        info.ntmp[i] = ((info.ntmp[i] << 15) + 127) / 255;

    return commonColor(info, colopfunc_drawcol_rep);
}

//! 色置換・共通
/*
    ntmp[0] : タイプ
*/

BOOL coloropfunc_common(FILTERDRAW &info,RGBAFIX15 *p,int x,int y)
{
    BOOL ret = FALSE,bDrawCol = FALSE;
    DWORD col;

    //描画色判定

    if(p->a && info.ntmp[0] < 3)
    {
        col = p->toDWORD() & 0xffffff;
        bDrawCol = (col == info.dwDrawCol);
    }

    //

    switch(info.ntmp[0])
    {
        //描画色 -> 透明
        case 0:
            if(bDrawCol)
            {
                p->zero();
                ret = TRUE;
            }
            break;
        //描画色以外を透明
        case 1:
            if(p->a && !bDrawCol)
            {
                p->zero();
                ret = TRUE;
            }
            break;
        //描画色を背景色に
        case 2:
            if(bDrawCol)
            {
                p->r = g_draw->colBack.r;
                p->g = g_draw->colBack.g;
                p->b = g_draw->colBack.b;
                ret = TRUE;
            }
            break;
        //透明部分を描画色に
        case 3:
            if(!p->a)
            {
                *p = g_draw->colDraw;
                p->a = 0x8000;
                ret = TRUE;
            }
            break;
    }

    return ret;
}

BOOL colorop_common(FILTERDRAW &info)
{
    return common1(info, coloropfunc_common);
}

//! アルファ操作1・共通

BOOL alphaop1_common(FILTERDRAW &info)
{
    int ix,iy,type,f,flag;
    RGBAFIX15 col,col2;
    CTileImage *pimg = info.pimgDst;
    CLayerItem *ptop,*p;

    //チェックレイヤのリンクセット

    ptop = g_draw->player->setLinkCheck();
    if(!ptop) return TRUE;

    //

    type = info.ntmp[0];

    for(iy = info.rs.y1; iy <= info.rs.y2; iy++)
    {
        for(ix = info.rs.x1; ix <= info.rs.x2; ix++)
        {
            pimg->getPixel(&col, ix, iy);

            flag = FALSE;

            switch(type)
            {
                //透明な部分を透明に,不透明な部分を透明に
                case 0:
                case 1:
                    for(p = ptop, f = 0; p; p = p->m_pLink)
                    {
                        p->m_pimg->getPixel(&col2, ix, iy);

                        if(col2.a) { f = 1; break; }
                    }

                    if((type == 0 && !f) || (type == 1 && f))
                    {
                        col.zero();
                        flag = TRUE;
                    }
                    break;
                //値をコピー
                case 2:
                    if(col.a)
                    {
                        flag = TRUE;
                        col.a = 0;

                        for(p = ptop; p; p = p->m_pLink)
                        {
                            p->m_pimg->getPixel(&col2, ix, iy);

                            col.a = col.a + col2.a - (col.a * col2.a >> 15);
                        }
                    }
                    break;
                //値を足す
                case 3:
                    if(col.a < 0x8000)
                    {
                        flag = TRUE;

                        for(p = ptop; p; p = p->m_pLink)
                        {
                            p->m_pimg->getPixel(&col2, ix, iy);

                            if(col.a + col2.a < 0x8000)
                                col.a += col2.a;
                            else
                            {
                                col.a = 0x8000;
                                break;
                            }
                        }
                    }
                    break;
                //値を引く
                case 4:
                    if(col.a)
                    {
                        flag = TRUE;

                        for(p = ptop; p; p = p->m_pLink)
                        {
                            p->m_pimg->getPixel(&col2, ix, iy);

                            if(col.a - col2.a > 0)
                                col.a -= col2.a;
                            else
                            {
                                col.a = 0;
                                break;
                            }
                        }
                    }
                    break;
                //値を乗算
                case 5:
                    if(col.a)
                    {
                        flag = TRUE;

                        for(p = ptop; p; p = p->m_pLink)
                        {
                            p->m_pimg->getPixel(&col2, ix, iy);

                            col.a = (col.a * col2.a + 0x4000) >> 15;
                        }
                    }
                    break;
                //輝度からセット
                case 6:
                    ptop->m_pimg->getPixel(&col2, ix, iy);

                    if(col2.a)
                    {
                        col.a = _GETV(col2.r, col2.g, col2.b);
                        flag = TRUE;
                    }
                    break;
                //輝度を反転してセット
                case 7:
                    ptop->m_pimg->getPixel(&col2, ix, iy);

                    if(col2.a)
                    {
                        col.a = 0x8000 - _GETV(col2.r, col2.g, col2.b);
                        flag = TRUE;
                    }
                    break;
            }

            if(flag) pimg->setPixelDraw(ix, iy, col);
        }

        info.progIncSub();
    }

    return TRUE;
}

//! アルファ操作2・共通

BOOL alphaop2func_common(FILTERDRAW &info,RGBAFIX15 *p,int x,int y)
{
    BOOL ret = FALSE;
    int n;

    switch(info.ntmp[0])
    {
        //明るい色ほど透明に
        case 0:
            if(p->a)
            {
                p->a = 0x8000 - _GETV(p->r, p->g, p->b);
                ret = TRUE;
            }
            break;
        //暗い色ほど透明に
        case 1:
            if(p->a)
            {
                p->a = _GETV(p->r, p->g, p->b);
                ret = TRUE;
            }
            break;
        //すべて不透明に (透明は白)
        case 2:
            if(p->a != 0x8000)
            {
                if(p->a == 0)
                    p->r = p->g = p->b = 0x8000;

                p->a = 0x8000;
                ret = TRUE;
            }
            break;
        //テクスチャ適用
        case 3:
            if(p->a)
            {
                n = g_draw->pimg8OptTex->getPixelTexture(x, y);

                p->a = (p->a * n + 127) / 255;

                ret = TRUE;
            }
            break;
        //アルファ値からグレイスケール作成
        case 4:
            p->r = p->g = p->b = p->a;
            p->a = 0x8000;
            ret = TRUE;
            break;
    }

    return ret;
}

BOOL alphaop2_common(FILTERDRAW &info)
{
    return common1(info, alphaop2func_common);
}

};
