/*$
Copyright (C) 2013-2016 Azel.

This file is part of AzPainter.

AzPainter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

AzPainter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
$*/
/*
    アンドゥ用圧縮関数

    - データは 2Byte 単位。
*/

#include <string.h>

#include "AXDef.h"


//------------------------

class CBitsBuf
{
protected:
    DWORD   m_val;
    int     m_bits,
            m_bufsize,
            m_nowpos;

    LPBYTE  m_pbuf;

public:
    int getSize() { return m_nowpos; }

    void init(LPBYTE pbuf,int bufsize);
    void put(int val,int bits);
    void putSerialWord(int val,int len);
    BOOL putend();

    int get(int bits);
};

//------------------------


//! 圧縮後のサイズ取得

int getCompressUndoSize(const void *pSrc,int size)
{
    LPWORD p,pbk,pend;
    WORD val;
    DWORD bits = 0;
    int dsize;

    p    = (LPWORD)pSrc;
    pend = (LPWORD)((LPBYTE)pSrc + size);

    while(p < pend)
    {
        if(p == pend - 1 || *p != p[1])
        {
            //非連続

            pbk = p;

            for(; p - pbk < 4096; p++)
            {
                if(p == pend - 1)
                {
                    p++;
                    break;
                }
                else if(*p == p[1])
                    break;
            }

            bits += 1 + 12 + ((p - pbk) << 4);
        }
        else
        {
            //連続

            val = *p;
            pbk = p;

            for(p += 2; p < pend && p - pbk < 4096 && *p == val; p++);

            bits += 1 + 12 + 16;
        }
    }

    //

    dsize = (bits + 7) >> 3;

    if(dsize >= size)
        return size;
    else
        return dsize;
}

//! 圧縮
/*
    圧縮サイズが size 以上の場合はソースをコピーして無圧縮。

    [1bit]0:非連続,1:連続 , [12bit]長さ , 非連続時:WORDx長さ分。連続時:WORD
*/

int compressUndo(unsigned char *pDst,const unsigned char *pSrc,int size)
{
    CBitsBuf buf;
    LPWORD p,pbk,pend;
    WORD val;

    buf.init(pDst, size);

    p    = (LPWORD)pSrc;
    pend = (LPWORD)(pSrc + size);

    while(p < pend)
    {
        if(p == pend - 1 || *p != p[1])
        {
            //非連続

            pbk = p;

            for(; p - pbk < 4096; p++)
            {
                if(p == pend - 1)
                {
                    p++;
                    break;
                }
                else if(*p == p[1])
                    break;
            }

            buf.put(p - pbk - 1, 13);

            for(; pbk < p; pbk++)
                buf.put(*pbk, 16);
        }
        else
        {
            //連続

            val = *p;
            pbk = p;

            for(p += 2; p < pend && p - pbk < 4096 && *p == val; p++);

            buf.putSerialWord(val, p - pbk);
        }
    }

    //

    if(buf.putend())
        return buf.getSize();
    else
    {
        //無圧縮

        ::memcpy(pDst, pSrc, size);
        return size;
    }
}

//! 展開

void uncompUndo(unsigned char *pDst,unsigned char *pSrc,int dstsize,int srcsize)
{
    CBitsBuf buf;
    LPWORD pd;
    int type,len,val,pos = 0;

    buf.init(pSrc, srcsize);

    pd = (LPWORD)pDst;
    dstsize >>= 1;

    while(1)
    {
        type = buf.get(1);
        if(type == -1) break;

        //長さ

        len = buf.get(12);
        if(len == -1) break;

        len++;

        //

        if(pos + len > dstsize) break;

        if(type == 1)
        {
            //連続

            val = buf.get(16);
            if(val == -1) break;

            for(; len; len--)
                pd[pos++] = (WORD)val;
        }
        else
        {
            //非連続

            for(; len; len--)
            {
                val = buf.get(16);
                if(val == -1) return;

                pd[pos++] = (WORD)val;
            }
        }
    }
}



//**************************************
// CBitsBuf
//**************************************


//! 初期化

void CBitsBuf::init(LPBYTE pbuf,int bufsize)
{
    m_val     = 0;
    m_bits    = 0;
    m_bufsize = bufsize;
    m_nowpos  = 0;
    m_pbuf    = pbuf;
}

//! ビット出力

void CBitsBuf::put(int val,int bits)
{
    if(m_nowpos == m_bufsize) return;

    m_val = (m_val << bits) | val;
    m_bits += bits;

    while(m_bits >= 8)
    {
        if(m_nowpos == m_bufsize) return;

        m_pbuf[m_nowpos++] = (BYTE)(m_val >> (m_bits - 8));

        m_bits -= 8;
    }
}

//! WORD 連続値出力

void CBitsBuf::putSerialWord(int val,int len)
{
    put(1, 1);
    put(len - 1, 12);
    put(val, 16);
}

//! 出力終了

BOOL CBitsBuf::putend()
{
    //残り

    if(m_bits && m_nowpos < m_bufsize)
        m_pbuf[m_nowpos++] = (BYTE)(m_val << (8 - m_bits));

    return (m_nowpos < m_bufsize);
}

//! ビット値取得

int CBitsBuf::get(int bits)
{
    int ret;

    while(m_bits < bits)
    {
        if(m_nowpos == m_bufsize) return -1;

        m_val = (m_val << 8) | m_pbuf[m_nowpos++];
        m_bits += 8;
    }

    ret = (m_val >> (m_bits - bits)) & ((1 << bits) - 1);

    m_bits -= bits;

    return ret;
}
